/********************************************************************************
*                                                                               *
*                  S t r i n g    T o k e n i z e r     C l a s s               *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXString.h>
#include <fox/FXStream.h>
using namespace FX;
#include "FXStringTokenizer.h"
using namespace FXEX;
namespace FXEX {

// ctor
FXStringTokenizer::FXStringTokenizer(const FXString& s,const FXString& seps,FXuint opts) {
  str=s;
  seperators=seps;
  index=-1;
  options=opts;
  }

// ctor
FXStringTokenizer::FXStringTokenizer(const FXString& s,const FXchar* seps,FXuint opts) {
  str=s;
  seperators=seps;
  index=-1;
  options=opts;
  }

// ctor
FXStringTokenizer::FXStringTokenizer(const FXString& s,const FXchar sep,FXuint opts) {
  str=s;
  seperators=FXString(sep,1);
  index=-1;
  options=opts;
  }

// dtor
FXStringTokenizer::~FXStringTokenizer() {}

// get the next token from the string
FXString FXStringTokenizer::next() {

  // Make sure index isn't beyond the end of the string
  // and that we actually have tokens to return
  // by searching for the start of the next token.
  // If we dont find anything, we return a null string.
  while (++index<str.length() && isSep(str[index])) {}
  if (index==str.length()) return FXString::null;
  FXString s(str[index],1);
  
  // We have found something that is not a seperator, so start building
  // a list until we find another seperator, or the end of the string.
  // Note: we skip over seperators that are enclosed in quotes
  FXuint quoteMode=0;
  register FXchar c;
  while(++index<str.length()){
    c=str[index];
    // handle quoting options
    if (c == '"'){
      if (quoteMode==0 && options&TOKENIZER_QUOTE_DOUBLE) quoteMode=TOKENIZER_QUOTE_DOUBLE;
      else if (quoteMode==TOKENIZER_QUOTE_DOUBLE) quoteMode=0;
      }
    else if (c == '\'') {
      if (quoteMode==0 && options&TOKENIZER_QUOTE_SINGLE) quoteMode=TOKENIZER_QUOTE_SINGLE;
      else if (quoteMode==TOKENIZER_QUOTE_SINGLE) quoteMode=0;
      }
    // if not in quoting mode...
    if (!quoteMode && isSep(c)) break;
    s+=c;
    }
  return s;
  }

// get the previous token from the string
FXString FXStringTokenizer::prev(){

  // Make sure index isn't beyond the start of the string
  // and that we actually have tokens to return
  // by searching for the start of the next token.
  // If we dont find anything, we return a null string.
  while (--index>-1 && isSep(str[index])) {}
  if (index==-1) return FXString::null;
  FXString s(str[index],1);
  
  // We have found something that is not a seperator, so start building
  // a list until we find another seperator, or the end of the string.
  // Note: we skip over seperators that are enclosed in quotes
  FXuint quoteMode=0;
  register FXchar c;
  while(--index>-1){
    c=str[index];
    // handle quoting options
    if (c == '"'){
      if (quoteMode==0 && options&TOKENIZER_QUOTE_DOUBLE) quoteMode=TOKENIZER_QUOTE_DOUBLE;
      else if (quoteMode==TOKENIZER_QUOTE_DOUBLE) quoteMode=0;
      }
    else if (c == '\'') {
      if (quoteMode==0 && options&TOKENIZER_QUOTE_SINGLE) quoteMode=TOKENIZER_QUOTE_SINGLE;
      else if (quoteMode==TOKENIZER_QUOTE_SINGLE) quoteMode=0;
      }
    // if not in quoting mode...
    if (!quoteMode && isSep(c)) break;
    s+=c;
    }
  return s;
  }

// indicates whether there are more tokens to be returned
FXbool FXStringTokenizer::hasNext(){
  FXint next=index;
  while (++next<str.length() && isSep(str[next])) {}
  if (next==str.length()) return FALSE;
  return TRUE;
  }

// indicates whether there are more tokens to be returned
FXbool FXStringTokenizer::hasPrev(){
  FXint prev=index;
  while (--prev>-1 && isSep(str[prev])) {}
  if (prev==-1) return FALSE;
  return TRUE;
  }

// indicates if the character is a seperator
FXbool FXStringTokenizer::isSep(const FXchar c){
  for (register FXint sep=0; sep<seperators.length(); sep++) {
    if (c == seperators[sep]) return TRUE;
    }
  return FALSE;
  }

// set to a new string
void FXStringTokenizer::setText(const FXString& s){
  if (str!=s){
    reset();
    str=s;
    }
  }

// if someone assigns a string tokenizer to another string tokenizer
FXStringTokenizer& FXStringTokenizer::operator=(const FXStringTokenizer& s){
  if (str!=s.str){
    reset();
    str=s.str;
    }
  return *this;
  }

// just in case someone assigns a string to the string tokenizer
FXStringTokenizer& FXStringTokenizer::operator=(const FXString& s){
  if (str!=s){
    reset();
    str=s;
    }
  return *this;
  }

// comparison operators ==
FXbool operator==(const FXStringTokenizer &s1,const FXStringTokenizer &s2){
  return s1.str==s2.str;
  }
FXbool operator==(const FXString &s1,const FXStringTokenizer &s2){
  return s1==s2.str;
  }
FXbool operator==(const FXStringTokenizer &s1,const FXString &s2){
  return s1.str==s2;
  }

// comparison operators !=
FXbool operator!=(const FXStringTokenizer &s1,const FXStringTokenizer &s2){
  return s1.str!=s2.str;
  }
FXbool operator!=(const FXString &s1,const FXStringTokenizer &s2){
  return s1!=s2.str;
  }
FXbool operator!=(const FXStringTokenizer &s1,const FXString &s2){
  return s1.str!=s2;
  }

// save to stream
FXStream& operator<<(FXStream& store,const FXStringTokenizer& s){
  store << s.str;
  store << s.seperators;
  store << s.index;
  store << s.options;
  return store;
  }

// load from stream
FXStream& operator>>(FXStream& store,FXStringTokenizer& s){
  store >> s.str;
  store >> s.seperators;
  store >> s.index;
  store >> s.options;
  return store;
  }

}

