/********************************************************************************
*                                                                               *
*                          T r e e L i s t   O b j e c t                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997,2002 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: FXCheckTreeList.cpp,v 1.78 2001/11/22 00:02:06 jeroen Exp $              *
********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include "fox/fxkeys.h"
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
#include <fox/FXFont.h>
#include <fox/FXIcon.h>
#include <fox/FXDC.h>
#include <fox/FXScrollBar.h>
#include <fox/FXDCWindow.h>
using namespace FX;
#include "FXCheckTreeList.h"
using namespace FXEX;
namespace FXEX {

/*
  Notes:
  - Tooltip should pop up exactly on top of current item.
  - Clicking on + does not make it current.
  - Need translate right-clicks into message with item figured out...
  - In autoselect mode, all items are expanded.
  - Sortfunc's will be hard to serialize.
  - Perhaps simplify things by having fixed root item embedded inside,
    which is not visible (i.e. no icon or text); just an idea at this stage.
  - It may be convenient to have ways to move items around.
  - Need insertSorted() API to add item in the right place based on current
    sort function.
*/


#define ICON_SPACING        4         // Spacing between parent and child in x direction
#define TEXT_SPACING        4         // Spacing between icon and text
#define SIDE_SPACING        4         // Spacing between side and item

#define DEFAULT_INDENT      8         // Indent between parent and child


#define SELECT_MASK     (CHECKTREELIST_SINGLESELECT|CHECKTREELIST_BROWSESELECT)
#define TREELIST_MASK   (SELECT_MASK|CHECKTREELIST_AUTOSELECT|CHECKTREELIST_SHOWS_LINES|CHECKTREELIST_SHOWS_BOXES|CHECKTREELIST_ROOT_BOXES)

/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXCheckTreeItem,FXObject,NULL,0)



// Draw item
void FXCheckTreeItem::draw(const FXCheckTreeList* list,FXDC& dc,FXint x,FXint y,FXint,FXint h) const {
  register FXFont *font=list->getFont();
  register FXIcon *icon=(state&OPENED)?openIcon:closedIcon;
  register FXint th=0,ih=0,tw,len;
  if(icon) ih=icon->getHeight();
  if(!label.empty()) th=4+font->getFontHeight();
  x+=SIDE_SPACING/2;
  if(icon){
    dc.drawIcon(icon,x,y+(h-ih)/2);
    x+=ICON_SPACING+icon->getWidth();
    }

  ////////////////////////////
  FXint ix,iy;
  FXbool check=(state&CHECKED)!=0;
  ix=x; iy=y+(h-13)/2;

  dc.setForeground(list->getApp()->getShadowColor());
  dc.fillRectangle(ix,iy,12,1);
  dc.fillRectangle(ix,iy,1,12);

  dc.setForeground(list->getApp()->getBorderColor());
  dc.fillRectangle(ix+1,iy+1,10,1);
  dc.fillRectangle(ix+1,iy+1,1,10);

  dc.setForeground(list->getApp()->getHiliteColor());
  dc.fillRectangle(ix,iy+12,13,1);
  dc.fillRectangle(ix+12,iy,1,13);

  dc.setForeground(list->getApp()->getBaseColor());
  dc.fillRectangle(ix+1,iy+11,11,1);
  dc.fillRectangle(ix+11,iy+1,1,11);

  if(!isEnabled())
    dc.setForeground(list->getApp()->getBaseColor());
  else
    dc.setForeground(list->getBackColor());
  dc.fillRectangle(ix+2,iy+2,9,9);

  if(check!=FALSE){
    FXSegment seg[6];
#ifndef WIN32
    seg[0].x1=3+ix; seg[0].y1=5+iy; seg[0].x2=5+ix; seg[0].y2=7+iy;
    seg[1].x1=3+ix; seg[1].y1=6+iy; seg[1].x2=5+ix; seg[1].y2=8+iy;
    seg[2].x1=3+ix; seg[2].y1=7+iy; seg[2].x2=5+ix; seg[2].y2=9+iy;
    seg[3].x1=5+ix; seg[3].y1=7+iy; seg[3].x2=9+ix; seg[3].y2=3+iy;
    seg[4].x1=5+ix; seg[4].y1=8+iy; seg[4].x2=9+ix; seg[4].y2=4+iy;
    seg[5].x1=5+ix; seg[5].y1=9+iy; seg[5].x2=9+ix; seg[5].y2=5+iy;
#else
    seg[0].x1=3+ix; seg[0].y1=5+iy; seg[0].x2=5+ix; seg[0].y2=7+iy;
    seg[1].x1=3+ix; seg[1].y1=6+iy; seg[1].x2=5+ix; seg[1].y2=8+iy;
    seg[2].x1=3+ix; seg[2].y1=7+iy; seg[2].x2=5+ix; seg[2].y2=9+iy;
    seg[3].x1=5+ix; seg[3].y1=7+iy; seg[3].x2=10+ix; seg[3].y2=2+iy;
    seg[4].x1=5+ix; seg[4].y1=8+iy; seg[4].x2=10+ix; seg[4].y2=3+iy;
    seg[5].x1=5+ix; seg[5].y1=9+iy; seg[5].x2=10+ix; seg[5].y2=4+iy;
#endif
    if(isEnabled()){
      if(check==MAYBE)
        dc.setForeground(list->getApp()->getShadowColor());
      else
        dc.setForeground(list->getTextColor());
      }
    else{
      dc.setForeground(list->getApp()->getShadowColor());
      }
    dc.drawLineSegments(seg,6);
    }

  x+=13+SIDE_SPACING/2;
  ///////////////////////

  if(!label.empty()){
    len=label.length();
    tw=4+font->getTextWidth(label.text(),len);
    y+=(h-th)/2;
    if(isSelected()){
      dc.setForeground(list->getSelBackColor());
      dc.fillRectangle(x,y,tw,th);
      if(!isEnabled())
        dc.setForeground(makeShadowColor(list->getBackColor()));
      else
        dc.setForeground(list->getSelTextColor());
      }
    else{
      if(!isEnabled())
        dc.setForeground(makeShadowColor(list->getBackColor()));
      else
        dc.setForeground(list->getTextColor());
      }
    dc.drawText(x+2,y+font->getFontAscent()+2,label.text(),len);
    if(hasFocus()){
      drawFocus(list,dc,x,y,tw,th);
      }
    }
  }


// Draw dotted rectangle for focus
void FXCheckTreeItem::drawFocus(const FXCheckTreeList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setFillStyle(FILL_OPAQUESTIPPLED);
  dc.setStipple(STIPPLE_GRAY,x,y);
  dc.setForeground(list->getTextColor());
  dc.setBackground(list->getBackColor());
  dc.drawRectangle(x+1,y+1,w-3,h-3);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }



// See if item got hit, and where:- 1 is icon, 2 is text
// 4 is check
FXint FXCheckTreeItem::hitItem(const FXCheckTreeList* list,FXint x,FXint y) const {
  register FXint oiw=0,ciw=0,oih=0,cih=0,tw=0,th=0,iw,ih,ix,iy,tx,ty,h;
  register FXFont *font=list->getFont();
  if(openIcon){
    oiw=openIcon->getWidth();
    oih=openIcon->getHeight();
    }
  if(closedIcon){
    ciw=closedIcon->getWidth();
    cih=closedIcon->getHeight();
    }
  if(!label.empty()){
    tw=4+font->getTextWidth(label.text(),label.length());
    th=4+font->getFontHeight();
    }
  iw=FXMAX(oiw,ciw);
  ih=FXMAX(oih,cih);
  h=FXMAX(th,ih);
  ix=SIDE_SPACING/2;
  tx=SIDE_SPACING/2;
  if(iw) tx+=iw+ICON_SPACING;
  iy=(h-ih)/2;
  ty=(h-th)/2;

  // In icon?
  if(ix<=x && iy<=y && x<ix+iw && y<iy+ih) return 1;

  // In check?
  if(tx<=x && ty<=y && x<tx+13 && y<ty+th) return 4;

  // In text?
  if(tx<=x && ty<=y && x<tx+tw && y<ty+th) return 2;

  // Outside
  return 0;
  }


// Set or kill focus
void FXCheckTreeItem::setFocus(FXbool focus){
  if(focus) state|=FOCUS; else state&=~FOCUS;
  }

// Select or deselect item
void FXCheckTreeItem::setSelected(FXbool selected){
  if(selected) state|=SELECTED; else state&=~SELECTED;
  }

// Check or uncheck item
void FXCheckTreeItem::setChecked(FXbool checked){
  if(checked) state|=CHECKED; else state&=~CHECKED;
  }

// Set item opened
void FXCheckTreeItem::setOpened(FXbool opened){
  if(opened) state|=OPENED; else state&=~OPENED;
  }

// Set item expanded
void FXCheckTreeItem::setExpanded(FXbool expanded){
  if(expanded) state|=EXPANDED; else state&=~EXPANDED;
  }

// Enable or disable the item
void FXCheckTreeItem::setEnabled(FXbool enabled){
  if(enabled) state&=~DISABLED; else state|=DISABLED;
  }

// Icon is draggable
void FXCheckTreeItem::setDraggable(FXbool draggable){
  if(draggable) state|=DRAGGABLE; else state&=~DRAGGABLE;
  }

// Icons owner by item
void FXCheckTreeItem::setIconOwned(FXuint owned){
  state=(state&~(OPENICONOWNED|CLOSEDICONOWNED))|(owned&(OPENICONOWNED|CLOSEDICONOWNED));
  }


// Create icon
void FXCheckTreeItem::create(){
  if(openIcon) openIcon->create();
  if(closedIcon) closedIcon->create();
  }


// Destroy icon
void FXCheckTreeItem::destroy(){
  if((state&OPENICONOWNED) && openIcon) openIcon->destroy();
  if((state&CLOSEDICONOWNED) && closedIcon) closedIcon->destroy();
  }


// Detach from icon resource
void FXCheckTreeItem::detach(){
  if(openIcon) openIcon->detach();
  if(closedIcon) closedIcon->detach();
  }


// Get number of child items
FXint FXCheckTreeItem::getNumChildren() const {
  register FXCheckTreeItem *item=first;
  register FXint n=0;
  while(item){item=item->next;n++;}
  return n;
  }



// Get item (logically) below this one
FXCheckTreeItem* FXCheckTreeItem::getBelow() const {
  register FXCheckTreeItem* item=(FXCheckTreeItem*)this;
  if(first) return first;
  while(!item->next && item->parent) item=item->parent;
  return item->next;
  }


// Get item (logically) above this one
FXCheckTreeItem* FXCheckTreeItem::getAbove() const {
  register FXCheckTreeItem* item=prev;
  if(!item) return parent;
  while(item->last) item=item->last;
  return item;
  }


// Get item width
FXint FXCheckTreeItem::getWidth(const FXCheckTreeList* list) const {
  register FXint w=0,oiw=0,ciw=0;
  if(openIcon)   oiw=openIcon->getWidth();
  if(closedIcon) ciw=closedIcon->getWidth();
  w=FXMAX(oiw,ciw);
  if(!label.empty()){
    if(w) w+=ICON_SPACING;
    w+=4+list->getFont()->getTextWidth(label.text(),label.length());
    }
  return SIDE_SPACING+w;
  }


// Get item height
FXint FXCheckTreeItem::getHeight(const FXCheckTreeList* list) const {
  register FXint th=0,oih=0,cih=0;
  if(openIcon)   oih=openIcon->getHeight();
  if(closedIcon) cih=closedIcon->getHeight();
  if(!label.empty()) th=4+list->getFont()->getFontHeight();
  return FXMAX3(th,oih,cih);
  }


// Save data
void FXCheckTreeItem::save(FXStream& store) const {
  FXObject::save(store);
  store << prev;
  store << next;
  store << parent;
  store << first;
  store << last;
  store << label;
  store << openIcon;
  store << closedIcon;
  store << state;
  }


// Load data
void FXCheckTreeItem::load(FXStream& store){
  FXObject::load(store);
  store >> prev;
  store >> next;
  store >> parent;
  store >> first;
  store >> last;
  store >> label;
  store >> openIcon;
  store >> closedIcon;
  store >> state;
  }


// Delete icons if owned
FXCheckTreeItem::~FXCheckTreeItem(){
  if(state&OPENICONOWNED) delete openIcon;
  if(state&CLOSEDICONOWNED) delete closedIcon;
  }

/*******************************************************************************/

// Map
FXDEFMAP(FXCheckTreeList) FXCheckTreeListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXCheckTreeList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXCheckTreeList::onMotion),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXCheckTreeList::onAutoScroll),
  FXMAPFUNC(SEL_TIMEOUT,FXCheckTreeList::ID_TIPTIMER,FXCheckTreeList::onTipTimer),
  FXMAPFUNC(SEL_TIMEOUT,FXCheckTreeList::ID_LOOKUPTIMER,FXCheckTreeList::onLookupTimer),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXCheckTreeList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXCheckTreeList::onLeftBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXCheckTreeList::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXCheckTreeList::onRightBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXCheckTreeList::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXCheckTreeList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXCheckTreeList::onKeyRelease),
  FXMAPFUNC(SEL_ENTER,0,FXCheckTreeList::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXCheckTreeList::onLeave),
  FXMAPFUNC(SEL_FOCUSIN,0,FXCheckTreeList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXCheckTreeList::onFocusOut),
  FXMAPFUNC(SEL_SELECTED,0,FXCheckTreeList::onSelected),
  FXMAPFUNC(SEL_DESELECTED,0,FXCheckTreeList::onDeselected),
  FXMAPFUNC(SEL_OPENED,0,FXCheckTreeList::onOpened),
  FXMAPFUNC(SEL_CLOSED,0,FXCheckTreeList::onClosed),
  FXMAPFUNC(SEL_EXPANDED,0,FXCheckTreeList::onExpanded),
  FXMAPFUNC(SEL_COLLAPSED,0,FXCheckTreeList::onCollapsed),
  FXMAPFUNC(SEL_CLICKED,0,FXCheckTreeList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXCheckTreeList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXCheckTreeList::onTripleClicked),
  FXMAPFUNC(SEL_COMMAND,0,FXCheckTreeList::onCommand),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXCheckTreeList::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXCheckTreeList::onQueryHelp),
  };


// Object implementation
FXIMPLEMENT(FXCheckTreeList,FXScrollArea,FXCheckTreeListMap,ARRAYNUMBER(FXCheckTreeListMap))


/*******************************************************************************/


// Tree List
FXCheckTreeList::FXCheckTreeList(){
  flags|=FLAG_ENABLED;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  extentitem=NULL;
  cursoritem=NULL;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  lineColor=0;
  treeWidth=0;
  treeHeight=0;
  visible=0;
  indent=DEFAULT_INDENT;
  grabx=0;
  graby=0;
  state=FALSE;
  }


// Tree List
FXCheckTreeList::FXCheckTreeList(FXComposite *p,FXint nvis,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  extentitem=NULL;
  cursoritem=NULL;
  font=getApp()->getNormalFont();
  sortfunc=NULL;
  textColor=getApp()->getForeColor();
  selbackColor=getApp()->getSelbackColor();
  seltextColor=getApp()->getSelforeColor();
  lineColor=getApp()->getShadowColor();
  treeWidth=0;
  treeHeight=0;
  visible=FXMAX(nvis,0);
  indent=DEFAULT_INDENT;
  grabx=0;
  graby=0;
  state=FALSE;
  }


// Create window
void FXCheckTreeList::create(){
  register FXCheckTreeItem *item=firstitem;
  FXScrollArea::create();
  while(item){
    item->create();
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  font->create();
  }


// Detach window
void FXCheckTreeList::detach(){
  register FXCheckTreeItem *item=firstitem;
  FXScrollArea::detach();
  while(item){
    item->detach();
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  font->detach();
  }


// Can have focus
FXbool FXCheckTreeList::canFocus() const { return TRUE; }


// Get default width
FXint FXCheckTreeList::getDefaultWidth(){
  return FXScrollArea::getDefaultWidth();
  }


// Get default height
FXint FXCheckTreeList::getDefaultHeight(){
  if(visible) return visible*(4+font->getFontHeight());
  return FXScrollArea::getDefaultHeight();
  }


// Propagate size change
void FXCheckTreeList::recalc(){
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  cursoritem=NULL;
  }


// List is multiple of nitems
void FXCheckTreeList::setNumVisible(FXint nvis){
  if(nvis<0) nvis=0;
  if(visible!=nvis){
    visible=nvis;
    recalc();
    }
  }


// Get number of toplevel items
FXint FXCheckTreeList::getNumItems() const {
  register FXCheckTreeItem *item=firstitem;
  register FXint n=0;
  while(item){
    item=item->next;
    n++;
    }
  return n;
  }


// Recompute interior
void FXCheckTreeList::recompute(){
  register FXCheckTreeItem* item;
  register FXint x,y,w,h;
  x=y=0;
  treeWidth=0;
  treeHeight=0;
  item=firstitem;
  if(options&CHECKTREELIST_ROOT_BOXES) x+=(4+indent);
  while(item){
    item->x=x;
    item->y=y;
    w=item->getWidth(this);
    h=item->getHeight(this);
    if(x+w>treeWidth) treeWidth=x+w;
    y+=h;
    if(item->first && ((options&CHECKTREELIST_AUTOSELECT) || item->isExpanded())){
      x+=(indent+h/2);
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      item=item->parent;
      x-=(indent+item->getHeight(this)/2);
      }
    item=item->next;
    }
  treeHeight=y;
  flags&=~FLAG_RECALC;
  }


// Determine content width of tree list
FXint FXCheckTreeList::getContentWidth(){
  if(flags&FLAG_RECALC) recompute();
  return treeWidth;
  }


// Determine content height of tree list
FXint FXCheckTreeList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return treeHeight;
  }


// Recalculate layout
void FXCheckTreeList::layout(){

  // Repaint when content size changed
  //if(flags&FLAG_RECALC) update();

  // Calculate contents
  FXScrollArea::layout();

  // Set line size based on item size
  if(firstitem){
    vertical->setLine(firstitem->getHeight(this));
    horizontal->setLine(firstitem->getWidth(this)/10);
    }

  // Force repaint
  update();

  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Set item text
void FXCheckTreeList::setItemText(FXCheckTreeItem* item,const FXString& text){
  if(item==NULL){ fxerror("%s::setItemText: item is NULL.\n",getClassName()); }
  item->setText(text);
  recalc();
  }


// Get item text
FXString FXCheckTreeList::getItemText(const FXCheckTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemText: item is NULL.\n",getClassName()); }
  return item->getText();
  }


// Set item open icon
void FXCheckTreeList::setItemOpenIcon(FXCheckTreeItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemOpenIcon: item is NULL.\n",getClassName()); }
  item->setOpenIcon(icon);
  recalc();
  }


// Get item open icon
FXIcon* FXCheckTreeList::getItemOpenIcon(const FXCheckTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemOpenIcon: item is NULL.\n",getClassName()); }
  return item->getOpenIcon();
  }


// Set item closed icon
void FXCheckTreeList::setItemClosedIcon(FXCheckTreeItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemClosedIcon: item is NULL.\n",getClassName()); }
  item->setClosedIcon(icon);
  recalc();
  }


// Get item closed icon
FXIcon* FXCheckTreeList::getItemClosedIcon(const FXCheckTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemClosedIcon: item is NULL.\n",getClassName()); }
  return item->getClosedIcon();
  }


// Set item data
void FXCheckTreeList::setItemData(FXCheckTreeItem* item,void* ptr) const {
  if(item==NULL){ fxerror("%s::setItemData: item is NULL.\n",getClassName()); }
  item->setData(ptr);
  }


// Get item data
void* FXCheckTreeList::getItemData(const FXCheckTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemData: item is NULL.\n",getClassName()); }
  return item->getData();
  }


// True if item is selected
FXbool FXCheckTreeList::isItemSelected(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemSelected: item is NULL.\n",getClassName()); }
  return item->isSelected();
  }


// True if item is checked
FXbool FXCheckTreeList::isItemChecked(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemChecked: item is NULL.\n",getClassName()); }
  return item->isChecked();
  }


// True if item is current
FXbool FXCheckTreeList::isItemCurrent(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemCurrent: item is NULL.\n",getClassName()); }
  return currentitem==item;
  }


// Check if item is expanded
FXbool FXCheckTreeList::isItemExpanded(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemExpanded: item is NULL.\n",getClassName()); }
  return (options&CHECKTREELIST_AUTOSELECT) || item->isExpanded();
  }


// Is item a leaf item
FXbool FXCheckTreeList::isItemLeaf(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemLeaf: item is NULL.\n",getClassName()); }
  return item->first==NULL;
  }


// Check if item is enabled
FXbool FXCheckTreeList::isItemEnabled(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemEnabled: item is NULL.\n",getClassName()); }
  return item->isEnabled();
  }


// Check item is open
FXbool FXCheckTreeList::isItemOpened(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemOpen: item is NULL.\n",getClassName()); }
  return item->isOpened();
  }


// True if item (partially) visible
FXbool FXCheckTreeList::isItemVisible(const FXCheckTreeItem* item) const {
  if(!item){ fxerror("%s::isItemVisible: item is NULL.\n",getClassName()); }
  return 0<(pos_y+item->y+item->getHeight(this)) && (pos_y+item->y)<viewport_h;
  }


// Make item fully visible
void FXCheckTreeList::makeItemVisible(FXCheckTreeItem* item){
  FXint x,y,w,h;
  if(item){

    // Expand parents of this node
    if(!(options&CHECKTREELIST_AUTOSELECT)){
      FXCheckTreeItem *par=item->parent;
      FXbool expanded=FALSE;
      while(par){
        if(!par->isExpanded()){
          par->setExpanded(TRUE);
          expanded=TRUE;
          }
        par=par->parent;
        }

      // If any nodes have expanded that weren't previously, recompute list size
      if(expanded){
        recalc();
        if(xid) layout();
        }
      }

    // Now we adjust the scrolled position to fit everything
    if(xid){
      x=pos_x;
      y=pos_y;

      w=item->getWidth(this);
      h=item->getHeight(this);

      if(viewport_w<=x+item->x+w) x=viewport_w-item->x-w;
      if(x+item->x<=0) x=-item->x;

      if(viewport_h<=y+item->y+h) y=viewport_h-item->y-h;
      if(y+item->y<=0) y=-item->y;

      setPosition(x,y);
      }
    }
  }


// Get item at position x,y
FXCheckTreeItem* FXCheckTreeList::getItemAt(FXint,FXint y) const {
  register FXCheckTreeItem* item=firstitem;
  register FXint ix,iy,iw,ih;
  ix=pos_x;
  iy=pos_y;
  if(options&CHECKTREELIST_ROOT_BOXES) ix+=(4+indent);
  while(item && iy<=y){
    iw=item->getWidth(this);
    ih=item->getHeight(this);
    if(y<iy+ih) return item;
    iy+=ih;
    if(item->first && ((options&CHECKTREELIST_AUTOSELECT) || item->isExpanded())){
      ix+=(indent+ih/2);
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      item=item->parent;
      ix-=(indent+item->getHeight(this)/2);
      }
    item=item->next;
    }
  return NULL;
  }


// Did we hit the item, and which part of it did we hit (0=outside, 1=icon, 2=text, 3=box)
// 4 hit the check
FXint FXCheckTreeList::hitItem(const FXCheckTreeItem* item,FXint x,FXint y) const {
  FXint ix,iy,iw,ih,xh,yh,hit=0;
  if(item){
    x-=pos_x;
    y-=pos_y;
    ix=item->x;
    iy=item->y;
    iw=item->getWidth(this);
    ih=item->getHeight(this);
    if(iy<=y && y<iy+ih){
      if((options&CHECKTREELIST_SHOWS_BOXES) && ((item->state&FXCheckTreeItem::HASITEMS) || item->first)){
        xh=ix-indent+(SIDE_SPACING/2);
        yh=iy+ih/2;
        if(xh-4<=x && x<=xh+4 && yh-4<=y && y<=yh+4) return 3;
        }
      hit=item->hitItem(this,x-ix,y-iy);
      }
    }
  return hit;
  }


// Repaint
void FXCheckTreeList::updateItem(FXCheckTreeItem* item){
  if(xid && item){
    update(0,pos_y+item->y,content_w,item->getHeight(this));
    }
  }


// Enable one item
FXbool FXCheckTreeList::enableItem(FXCheckTreeItem* item){
  if(!item){ fxerror("%s::enableItem: item is NULL.\n",getClassName()); }
  if(!item->isEnabled()){
    item->setEnabled(TRUE);
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Disable one item
FXbool FXCheckTreeList::disableItem(FXCheckTreeItem* item){
  if(!item){ fxerror("%s::disableItem: item is NULL.\n",getClassName()); }
  if(item->isEnabled()){
    item->setEnabled(FALSE);
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Select one item
FXbool FXCheckTreeList::selectItem(FXCheckTreeItem* item,FXbool notify){
  if(!item){ fxerror("%s::selectItem: item is NULL.\n",getClassName()); }
  if(!item->isSelected()){
    switch(options&SELECT_MASK){
      case CHECKTREELIST_SINGLESELECT:
      case CHECKTREELIST_BROWSESELECT:
        killSelection(notify);
        item->setSelected(TRUE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
        break;
      case CHECKTREELIST_EXTENDEDSELECT:
      case CHECKTREELIST_MULTIPLESELECT:
        item->setSelected(TRUE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// Deselect one item
FXbool FXCheckTreeList::deselectItem(FXCheckTreeItem* item,FXbool notify){
  if(!item){ fxerror("%s::deselectItem: item is NULL.\n",getClassName()); }
  if(item->isSelected()){
    switch(options&SELECT_MASK){
      case CHECKTREELIST_EXTENDEDSELECT:
      case CHECKTREELIST_MULTIPLESELECT:
      case CHECKTREELIST_SINGLESELECT:
        item->setSelected(FALSE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)item);}
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// toggle one item
FXbool FXCheckTreeList::toggleItem(FXCheckTreeItem* item,FXbool notify){
  if(!item){ fxerror("%s::toggleItem: item is NULL.\n",getClassName()); }
  switch(options&SELECT_MASK){
    case CHECKTREELIST_BROWSESELECT:
      if(!item->isSelected()){
        killSelection(notify);
        item->setSelected(TRUE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
        }
      break;
    case CHECKTREELIST_SINGLESELECT:
      if(!item->isSelected()){
        killSelection(notify);
        item->setSelected(TRUE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
        }
      else{
        item->setSelected(FALSE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)item);}
        }
      break;
    case CHECKTREELIST_EXTENDEDSELECT:
    case CHECKTREELIST_MULTIPLESELECT:
      if(!item->isSelected()){
        item->setSelected(TRUE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
        }
      else{
        item->setSelected(FALSE);
        updateItem(item);
        if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)item);}
        }
      break;
    }
  return TRUE;
  }


// Check one item
FXbool FXCheckTreeList::checkItem(FXCheckTreeItem* item,FXbool notify){
  if(!item){ fxerror("%s::checkItem: item is NULL.\n",getClassName()); }
  if(!item->isChecked()){
    item->setChecked(TRUE);
    updateItem(item);
    if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
    return TRUE;
    }
  return FALSE;
  }


// uncheck one item
FXbool FXCheckTreeList::uncheckItem(FXCheckTreeItem* item,FXbool notify){
  if(!item){ fxerror("%s::uncheckItem: item is NULL.\n",getClassName()); }
  if(item->isChecked()){
    item->setChecked(FALSE);
    updateItem(item);
    if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)item);}
    return TRUE;
    }
  return FALSE;
  }


// Toggle check item
FXbool FXCheckTreeList::toggleCheckItem(FXCheckTreeItem* item,FXbool notify){
  if(!item){ fxerror("%s::toggleCheckItem: item is NULL.\n",getClassName()); }
  if(item->isChecked()){
    item->setChecked(FALSE);
    updateItem(item);
    if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)item);}
    return TRUE;
    }
  else{
    item->setChecked(TRUE);
    updateItem(item);
    if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
    return TRUE;
    }
  return FALSE;
  }


// Open item
FXbool FXCheckTreeList::openItem(FXCheckTreeItem* item,FXbool notify){
  if(item==NULL){ fxerror("%s::openItem: item is NULL.\n",getClassName()); }
  if(!item->isOpened()){
    item->setOpened(TRUE);
    updateItem(item);
    if(notify){handle(this,FXSEL(SEL_OPENED,0),(void*)item);}
    return TRUE;
    }
  return FALSE;
  }


// Close item
FXbool FXCheckTreeList::closeItem(FXCheckTreeItem* item,FXbool notify){
  if(item==NULL){ fxerror("%s::closeItem: item is NULL.\n",getClassName()); }
  if(item->isOpened()){
    item->setOpened(FALSE);
    updateItem(item);
    if(notify){handle(this,FXSEL(SEL_CLOSED,0),(void*)item);}
    return TRUE;
    }
  return FALSE;
  }


// Collapse all subtrees under item
FXbool FXCheckTreeList::collapseTree(FXCheckTreeItem* tree,FXbool notify){
  if(tree==NULL){ fxerror("%s::collapseTree: tree is NULL.\n",getClassName()); }
  if(tree->isExpanded()){
    tree->setExpanded(FALSE);
    if(!(options&CHECKTREELIST_AUTOSELECT)){     // In autoselect, already shown as expanded!
      if(tree->first){
        recalc();
        }
      else{
        updateItem(tree);
        }
      }
    if(notify){handle(this,FXSEL(SEL_COLLAPSED,0),(void*)tree);}
    return TRUE;
    }
  return FALSE;
  }


// Expand subtree under item
FXbool FXCheckTreeList::expandTree(FXCheckTreeItem* tree,FXbool notify){
  if(tree==NULL){ fxerror("%s::expandTree: tree is NULL.\n",getClassName()); }
  if(!tree->isExpanded()){
    tree->setExpanded(TRUE);
    if(!(options&CHECKTREELIST_AUTOSELECT)){     // In autoselect, already shown as expanded!
      if(tree->first){
        recalc();
        }
      else{
        updateItem(tree);
        }
      }
    if(notify){handle(this,FXSEL(SEL_EXPANDED,0),(void*)tree);}
    return TRUE;
    }
  return FALSE;
  }


// Reparent item under a new parent
void FXCheckTreeList::reparentItem(FXCheckTreeItem* item,FXCheckTreeItem* p){
  if(!item){ fxerror("%s::reparentItem: item is NULL.\n",getClassName()); }
  if(item->parent!=p){
    if(item->prev) item->prev->next=item->next; else if(item->parent) item->parent->first=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else if(item->parent) item->parent->last=item->prev; else lastitem=item->prev;
    if(p){
      item->prev=p->last;
      item->next=NULL;
      if(item->prev) item->prev->next=item; else p->first=item;
      p->last=item;
      }
    else{
      item->prev=lastitem;
      item->next=NULL;
      if(item->prev) item->prev->next=item; else firstitem=item;
      lastitem=item;
      }
    item->parent=p;
    recalc();
    }
  }


// Start motion timer while in this window
long FXCheckTreeList::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onEnter(sender,sel,ptr);
  getApp()->removeTimeout(this,ID_TIPTIMER);
  getApp()->addTimeout(this,ID_TIPTIMER,getApp()->getMenuPause());
  cursoritem=NULL;
  return 1;
  }


// Stop motion timer when leaving window
long FXCheckTreeList::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onLeave(sender,sel,ptr);
  getApp()->removeTimeout(this,ID_TIPTIMER);
  cursoritem=NULL;
  return 1;
  }


// Gained focus
long FXCheckTreeList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(currentitem){
    currentitem->setFocus(TRUE);
    updateItem(currentitem);
    }
  return 1;
  }


// Lost focus
long FXCheckTreeList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(currentitem){
    currentitem->setFocus(FALSE);
    updateItem(currentitem);
    }
  return 1;
  }


// Draw item list
long FXCheckTreeList::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXCheckTreeItem* item=firstitem;
  FXint yh,xh,x,y,w,h,xp,hh;
  FXDCWindow dc(this,event);
  dc.setFont(font);
  x=pos_x;
  y=pos_y;
  if(options&CHECKTREELIST_ROOT_BOXES) x+=(4+indent);
  while(item && y<event->rect.y+event->rect.h){
    w=item->getWidth(this);
    h=item->getHeight(this);
    if(event->rect.y<=y+h){

      // Draw item
      dc.setForeground(backColor);
      dc.fillRectangle(pos_x,y,content_w,h);
      item->draw(this,dc,x,y,w,h);

      // Show other paraphernalia such as dotted lines and expand-boxes
      if((options&(CHECKTREELIST_SHOWS_LINES|CHECKTREELIST_SHOWS_BOXES)) && (item->parent || (options&CHECKTREELIST_ROOT_BOXES))){
        hh=h/2;
        yh=y+hh;
        xh=x-indent+(SIDE_SPACING/2);
        dc.setForeground(lineColor);
        dc.setFillStyle(FILL_STIPPLED);
        dc.setStipple(STIPPLE_GRAY,pos_x&1,pos_y&1);
        if(options&CHECKTREELIST_SHOWS_LINES){                   // Connect items with lines
          FXCheckTreeItem* p=item->parent;
          xp=xh;
          while(p){
            xp-=(indent+p->getHeight(this)/2);
            if(p->next) dc.fillRectangle(xp,y,1,h);
            p=p->parent;
            }
          if((options&CHECKTREELIST_SHOWS_BOXES) && ((item->state&FXCheckTreeItem::HASITEMS) || item->first)){
            if(item->prev || item->parent) dc.fillRectangle(xh,y,1,yh-y-4);
            if(item->next) dc.fillRectangle(xh,yh+4,1,y+h-yh-4);
            }
          else{
            if(item->prev || item->parent) dc.fillRectangle(xh,y,1,hh);
            if(item->next) dc.fillRectangle(xh,yh,1,h);
            dc.fillRectangle(xh,yh,x+(SIDE_SPACING/2)-2-xh,1);
            }
          dc.setFillStyle(FILL_SOLID);
          }

        // Boxes before items for expand/collapse of item
        if((options&CHECKTREELIST_SHOWS_BOXES) && ((item->state&FXCheckTreeItem::HASITEMS) || item->first)){
          dc.setFillStyle(FILL_STIPPLED);
          dc.fillRectangle(xh+4,yh,(SIDE_SPACING/2)-2,1);
          dc.setFillStyle(FILL_SOLID);
          dc.drawRectangle(xh-4,yh-4,8,8);
          dc.setForeground(textColor);
          dc.fillRectangle(xh-2,yh,5,1);
          if(!(options&CHECKTREELIST_AUTOSELECT) && !item->isExpanded()){
            dc.fillRectangle(xh,yh-2,1,5);
            }
          }
        }
      }

    y+=h;

    // Move on to the next item
    if(item->first && ((options&CHECKTREELIST_AUTOSELECT) || item->isExpanded())){
      x+=(indent+h/2);
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      item=item->parent;
      x-=(indent+item->getHeight(this)/2);
      }
    item=item->next;
    }
  if(y<event->rect.y+event->rect.h){
    dc.setForeground(backColor);
    dc.fillRectangle(event->rect.x,y,event->rect.w,event->rect.y+event->rect.h-y);
    }
  return 1;
  }

// Zero out lookup string
long FXCheckTreeList::onLookupTimer(FXObject*,FXSelector,void*){
  lookup=FXString::null;
  return 1;
  }


// We were asked about tip text
long FXCheckTreeList::onQueryTip(FXObject* sender,FXSelector,void*){
  FXint x,y; FXuint state;
  if((flags&FLAG_TIP) && !(options&CHECKTREELIST_AUTOSELECT)){   // No tip when autoselect!
    getCursorPosition(x,y,state);
    FXTRACE((250,"%s::onQueryTip %p (%d,%d)\n",getClassName(),this,x,y));
    FXCheckTreeItem *item=getItemAt(x,y);
    if(item){
      FXString string=item->getText();
      sender->handle(this,FXSEL(SEL_COMMAND,ID_SETSTRINGVALUE),(void*)&string);
      return 1;
      }
    }
  return 0;
  }


// We were asked about status text
long FXCheckTreeList::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(!help.empty() && (flags&FLAG_HELP)){
    FXTRACE((200,"%s::onQueryHelp %p\n",getClassName(),this));
    sender->handle(this,FXSEL(SEL_COMMAND,ID_SETSTRINGVALUE),(void*)&help);
    return 1;
    }
  return 0;
  }


// Key Press
long FXCheckTreeList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXCheckTreeItem *item=currentitem;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,FXSEL(SEL_KEYPRESS,message),ptr)) return 1;
  if(item==NULL) item=firstitem;
  switch(event->code){
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(flags&FLAG_DODRAG){handle(this,FXSEL(SEL_DRAGGED,0),ptr);}
      return 1;
    case KEY_Page_Up:
    case KEY_KP_Page_Up:
      lookup=FXString::null;
      setPosition(pos_x,pos_y+verticalScrollBar()->getPage());
      return 1;
    case KEY_Page_Down:
    case KEY_KP_Page_Down:
      lookup=FXString::null;
      setPosition(pos_x,pos_y-verticalScrollBar()->getPage());
      return 1;
    case KEY_Up:                          // Move up
    case KEY_KP_Up:
      if(item){
        if(item->prev){
          item=item->prev;
          while(item->first && ((options&CHECKTREELIST_AUTOSELECT) || item->isExpanded())) item=item->last;
          }
        else if(item->parent){
          item=item->parent;
          }
        }
      goto hop;
    case KEY_Down:                        // Move down
    case KEY_KP_Down:
      if(item){
        if(item->first && ((options&CHECKTREELIST_AUTOSELECT) || item->isExpanded())){
          item=item->first;
          }
        else{
          while(!item->next && item->parent) item=item->parent;
          item=item->next;
          }
        }
      goto hop;
    case KEY_Right:                       // Move right/down and open subtree
    case KEY_KP_Right:
      if(item){
        if(!(options&CHECKTREELIST_AUTOSELECT) && !item->isExpanded() && ((item->state&FXCheckTreeItem::HASITEMS) || item->first)){
          expandTree(item,TRUE);
          }
        else if(item->first){
          item=item->first;
          }
        else{
          while(!item->next && item->parent) item=item->parent;
          item=item->next;
          }
        }
      goto hop;
    case KEY_Left:                        // Move left/up and close subtree
    case KEY_KP_Left:
      if(item){
        if(!(options&CHECKTREELIST_AUTOSELECT) && item->isExpanded() && ((item->state&FXCheckTreeItem::HASITEMS) || item->first)){
          collapseTree(item,TRUE);
          }
        else if(item->parent){
          item=item->parent;
          }
        else if(item->prev){
          item=item->prev;
          }
        }
      goto hop;
    case KEY_Home:                        // Move to first
    case KEY_KP_Home:
      item=firstitem;
      goto hop;
    case KEY_End:                         // Move to last
    case KEY_KP_End:
      item=lastitem;
      while(item){
        if(item->last && ((options&CHECKTREELIST_AUTOSELECT) || item->isExpanded())){
          item=item->last;
          }
        else if(item->next){
          item=item->next;
          }
        else{
          break;
          }
        }
hop:  lookup=FXString::null;
      if(item){
        setCurrentItem(item,TRUE);
        makeItemVisible(item);
        if((options&SELECT_MASK)==CHECKTREELIST_EXTENDEDSELECT){
          if(item->isEnabled()){
            if(event->state&SHIFTMASK){
              if(anchoritem){
                selectItem(anchoritem,TRUE);
                extendSelection(item,TRUE);
                }
              else{
                selectItem(item,TRUE);
                setAnchorItem(item);
                }
              }
            else if(!(event->state&CONTROLMASK)){
              killSelection(TRUE);
              selectItem(item,TRUE);
              setAnchorItem(item);
              }
            }
          }
        }
      handle(this,FXSEL(SEL_CLICKED,0),(void*)currentitem);
      if(currentitem && currentitem->isEnabled()){
        handle(this,FXSEL(SEL_COMMAND,0),(void*)currentitem);
        }
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      lookup=FXString::null;
      if(item && item->isEnabled()){
        switch(options&SELECT_MASK){
          case CHECKTREELIST_EXTENDEDSELECT:
            if(event->state&SHIFTMASK){
              if(anchoritem){
                selectItem(anchoritem,TRUE);
                extendSelection(item,TRUE);
                }
              else{
                selectItem(item,TRUE);
                }
              }
            else if(event->state&CONTROLMASK){
              toggleItem(item,TRUE);
              }
            else{
              killSelection(TRUE);
              selectItem(item,TRUE);
              toggleCheckItem(item,TRUE);
              }
            break;
          case CHECKTREELIST_MULTIPLESELECT:
          case CHECKTREELIST_SINGLESELECT:
            toggleItem(item,TRUE);
            toggleCheckItem(item,TRUE);
            break;
          }
        setAnchorItem(item);
        }
      handle(this,FXSEL(SEL_CLICKED,0),(void*)currentitem);
      if(currentitem && currentitem->isEnabled()){
        handle(this,FXSEL(SEL_COMMAND,0),(void*)currentitem);
        }
      return 1;
    case KEY_Return:
    case KEY_KP_Enter:
      lookup=FXString::null;
      handle(this,FXSEL(SEL_DOUBLECLICKED,0),(void*)currentitem);
      if(currentitem && currentitem->isEnabled()){
        handle(this,FXSEL(SEL_COMMAND,0),(void*)currentitem);
        }
      return 1;
    default:
      if((event->state&(CONTROLMASK|ALTMASK)) || ((FXuchar)event->text[0]<32)) return 0;
      lookup.append(event->text);
      getApp()->removeTimeout(this,ID_LOOKUPTIMER);
      getApp()->addTimeout(this,ID_LOOKUPTIMER,getApp()->getTypingSpeed());
      item=findItem(lookup,currentitem,SEARCH_FORWARD|SEARCH_WRAP|SEARCH_PREFIX);
      if(item){
	setCurrentItem(item,TRUE);
	makeItemVisible(item);
	if((options&SELECT_MASK)==CHECKTREELIST_EXTENDEDSELECT){
	  if(item->isEnabled()){
	    killSelection(TRUE);
	    selectItem(item,TRUE);
	    }
	  }
	setAnchorItem(item);
        }
      handle(this,FXSEL(SEL_CLICKED,0),(void*)currentitem);
      if(currentitem && currentitem->isEnabled()){
	handle(this,FXSEL(SEL_COMMAND,0),(void*)currentitem);
	}
      return 1;
    }
  return 0;
  }


// Key Release
long FXCheckTreeList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,FXSEL(SEL_KEYRELEASE,message),ptr)) return 1;
  switch(event->code){
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(flags&FLAG_DODRAG){handle(this,FXSEL(SEL_DRAGGED,0),ptr);}
      return 1;
    }
  return 0;
  }


// We timed out, i.e. the user didn't move for a while
long FXCheckTreeList::onTipTimer(FXObject*,FXSelector,void*){
  FXTRACE((200,"%s::onTipTimer %p\n",getClassName(),this));
  flags|=FLAG_TIP;
  return 1;
  }


// Scroll timer
long FXCheckTreeList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXCheckTreeItem *item;
  FXint xx,yy;

  // Scroll the content
  FXScrollArea::onAutoScroll(sender,sel,ptr);

  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    handle(this,FXSEL(SEL_DRAGGED,0),ptr);
    return 1;
    }

  // In autoselect mode, stop scrolling when mouse outside window
  if((flags&FLAG_PRESSED) || (options&CHECKTREELIST_AUTOSELECT)){

    // Validated position
    xx=event->win_x; if(xx<0) xx=0; else if(xx>=viewport_w) xx=viewport_w-1;
    yy=event->win_y; if(yy<0) yy=0; else if(yy>=viewport_h) yy=viewport_h-1;

    // Find item
    item=getItemAt(xx,yy);

    // Got item and different from last time
    if(item && item!=currentitem){

      // Make it the current item
      setCurrentItem(item,TRUE);

      // Extend the selection
      if((options&SELECT_MASK)==CHECKTREELIST_EXTENDEDSELECT){
        state=FALSE;
        extendSelection(item,TRUE);
        }
      }
    return 1;
    }
  return 0;
  }


// Mouse motion
long FXCheckTreeList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXCheckTreeItem *oldcursoritem=cursoritem;
  FXuint flg=flags;
  FXCheckTreeItem *item;

  // Kill the tip
  flags&=~FLAG_TIP;

  // Kill the tip timer
  getApp()->removeTimeout(this,ID_TIPTIMER);

  // Right mouse scrolling
  if(flags&FLAG_SCROLLING){
    setPosition(event->win_x-grabx,event->win_y-graby);
    return 1;
    }

  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    if(startAutoScroll(event,TRUE)) return 1;
    handle(this,FXSEL(SEL_DRAGGED,0),ptr);
    return 1;
    }

  // Tentative drag and drop
  if((flags&FLAG_TRYDRAG) && event->moved){
    flags&=~FLAG_TRYDRAG;
    if(handle(this,FXSEL(SEL_BEGINDRAG,0),ptr)){
      flags|=FLAG_DODRAG;
      }
    return 1;
    }

  // Normal operation
  if((flags&FLAG_PRESSED) || (options&CHECKTREELIST_AUTOSELECT)){

    // Start auto scrolling?
    if(startAutoScroll(event,FALSE)) return 1;

    // Find item
    item=getItemAt(event->win_x,event->win_y);

    // Got an item different from before
    if(item && item!=currentitem){

      // Make it the current item
      setCurrentItem(item,TRUE);

      // Extend the selection
      if((options&SELECT_MASK)==CHECKTREELIST_EXTENDEDSELECT){
        state=FALSE;
        extendSelection(item,TRUE);
        }
      }
    return 1;
    }

  // Reset tip timer if nothing's going on
  getApp()->removeTimeout(this,ID_TIPTIMER);
  getApp()->addTimeout(this,ID_TIPTIMER,getApp()->getMenuPause());

  // Get item we're over
  cursoritem=getItemAt(event->win_x,event->win_y);

  // Force GUI update only when needed
  return (cursoritem!=oldcursoritem)||(flg&FLAG_TIP);
  }


// Pressed a button
long FXCheckTreeList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXCheckTreeItem *item;
  FXint code;
  flags&=~FLAG_TIP;
  handle(this,FXSEL(SEL_FOCUS_SELF,0),ptr);
  if(isEnabled()){
    grab();
    flags&=~FLAG_UPDATE;

    // First change callback
    if(target && target->handle(this,FXSEL(SEL_LEFTBUTTONPRESS,message),ptr)) return 1;

    // Not autoselect mode
    if(options&CHECKTREELIST_AUTOSELECT) return 1;

    // Locate item
    item=getItemAt(event->win_x,event->win_y);

    // No item
    if(item==NULL) return 1;

    // Find out where hit
    code=hitItem(item,event->win_x,event->win_y);

    // Maybe clicked on box
    if(code==3){
      if(isItemExpanded(item))
        collapseTree(item,TRUE);
      else
        expandTree(item,TRUE);
      return 1;
      }

    // Change current item
    setCurrentItem(item,TRUE);

    if(code==4){
      toggleCheckItem(item,TRUE);
      }
    else{
    // Change item selection
    state=item->isSelected();
    switch(options&SELECT_MASK){
      case CHECKTREELIST_EXTENDEDSELECT:
        if(event->state&SHIFTMASK){
          if(anchoritem){
            if(anchoritem->isEnabled()) selectItem(anchoritem,TRUE);
            extendSelection(item,TRUE);
            }
          else{
            if(item->isEnabled()) selectItem(item,TRUE);
            setAnchorItem(item);
            }
          }
        else if(event->state&CONTROLMASK){
          if(item->isEnabled() && !state) selectItem(item,TRUE);
          setAnchorItem(item);
          }
        else{
          if(item->isEnabled() && !state){ killSelection(TRUE); selectItem(item,TRUE); }
          setAnchorItem(item);
          }
        break;
      case CHECKTREELIST_MULTIPLESELECT:
      case CHECKTREELIST_SINGLESELECT:
        if(item->isEnabled() && !state) selectItem(item,TRUE);
        break;
      }
    }
    
    // Start drag if actually pressed text or icon only
    if(code && item->isSelected() && item->isDraggable()){
      flags|=FLAG_TRYDRAG;
      }

    flags|=FLAG_PRESSED;
    return 1;
    }
  return 0;
  }


// Released button
long FXCheckTreeList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXuint flg=flags;
  if(isEnabled()){
    ungrab();
    stopAutoScroll();
    flags|=FLAG_UPDATE;
    flags&=~(FLAG_PRESSED|FLAG_TRYDRAG|FLAG_DODRAG);

    // First chance callback
    if(target && target->handle(this,FXSEL(SEL_LEFTBUTTONRELEASE,message),ptr)) return 1;

    // No activity
    if(!(flg&FLAG_PRESSED) && !(options&CHECKTREELIST_AUTOSELECT)) return 1;

    // Was dragging
    if(flg&FLAG_DODRAG){
      handle(this,FXSEL(SEL_ENDDRAG,0),ptr);
      return 1;
      }

    // Select only enabled item
    switch(options&SELECT_MASK){
      case CHECKTREELIST_EXTENDEDSELECT:
        if(currentitem && currentitem->isEnabled()){
          if(event->state&CONTROLMASK){
            if(state) deselectItem(currentitem,TRUE);
            }
          else if(!(event->state&SHIFTMASK)){
            if(state){ killSelection(TRUE); selectItem(currentitem,TRUE); }
            }
          }
        break;
      case CHECKTREELIST_MULTIPLESELECT:
      case CHECKTREELIST_SINGLESELECT:
        if(currentitem && currentitem->isEnabled()){
          if(state) deselectItem(currentitem,TRUE);
          }
        break;
      }

    // Scroll to make item visibke
    makeItemVisible(currentitem);

    // Update anchor
    setAnchorItem(currentitem);

    // Generate clicked callbacks
    if(event->click_count==1){
      handle(this,FXSEL(SEL_CLICKED,0),(void*)currentitem);
      }
    else if(event->click_count==2){
      handle(this,FXSEL(SEL_DOUBLECLICKED,0),(void*)currentitem);
      }
    else if(event->click_count==3){
      handle(this,FXSEL(SEL_TRIPLECLICKED,0),(void*)currentitem);
      }

    // Command callback only when clicked on item
    if(currentitem && currentitem->isEnabled()){
      handle(this,FXSEL(SEL_COMMAND,0),(void*)currentitem);
      }
    return 1;
    }
  return 0;
  }


// Pressed right button
long FXCheckTreeList::onRightBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  handle(this,FXSEL(SEL_FOCUS_SELF,0),ptr);
  if(isEnabled()){
    grab();
    flags&=~FLAG_UPDATE;
    if(target && target->handle(this,FXSEL(SEL_RIGHTBUTTONPRESS,message),ptr)) return 1;
    flags|=FLAG_SCROLLING;
    grabx=event->win_x-pos_x;
    graby=event->win_y-pos_y;
    return 1;
    }
  return 0;
  }


// Released right button
long FXCheckTreeList::onRightBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    flags&=~FLAG_SCROLLING;
    flags|=FLAG_UPDATE;
    if(target && target->handle(this,FXSEL(SEL_RIGHTBUTTONRELEASE,message),ptr)) return 1;
    return 1;
    }
  return 0;
  }



// The widget lost the grab for some reason
long FXCheckTreeList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~(FLAG_DODRAG|FLAG_TRYDRAG|FLAG_PRESSED|FLAG_CHANGED|FLAG_SCROLLING);
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }


// Command message
long FXCheckTreeList::onCommand(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_COMMAND,message),ptr);
  }


// Clicked in list
long FXCheckTreeList::onClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_CLICKED,message),ptr);
  }


// Double clicked in list; ptr may or may not point to an item
long FXCheckTreeList::onDoubleClicked(FXObject*,FXSelector,void* ptr){

  // Double click anywhere in the widget
  if(target && target->handle(this,FXSEL(SEL_DOUBLECLICKED,message),ptr)) return 1;

  // Double click on an item
  if(ptr){
    if(isItemExpanded((FXCheckTreeItem*)ptr))
      collapseTree((FXCheckTreeItem*)ptr,TRUE);
    else
      expandTree((FXCheckTreeItem*)ptr,TRUE);
    }
  return 0;
  }


// Triple clicked in list; ptr may or may not point to an item
long FXCheckTreeList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_TRIPLECLICKED,message),ptr);
  }


// Item opened
long FXCheckTreeList::onOpened(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_OPENED,message),ptr);
  }


// Item closed
long FXCheckTreeList::onClosed(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_CLOSED,message),ptr);
  }


// Item expanded
long FXCheckTreeList::onExpanded(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_EXPANDED,message),ptr);
  }


// Item collapsed
long FXCheckTreeList::onCollapsed(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_COLLAPSED,message),ptr);
  }


// Selected item
long FXCheckTreeList::onSelected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_SELECTED,message),ptr);
  }


// Deselected item
long FXCheckTreeList::onDeselected(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,FXSEL(SEL_DESELECTED,message),ptr);
  }



// Extend selection
FXbool FXCheckTreeList::extendSelection(FXCheckTreeItem* item,FXbool notify){
  register FXCheckTreeItem *it,*i1,*i2,*i3;
  register FXbool changes=FALSE;
  if(item && anchoritem && extentitem){
    it=firstitem;
    i1=i2=i3=NULL;
    FXTRACE((100,"extendSelection: anchor=%s extent=%s item=%s\n",anchoritem->label.text(),extentitem->label.text(),item->label.text()));

    // Find segments
    while(it){
      if(it==item){i1=i2;i2=i3;i3=it;}
      if(it==anchoritem){i1=i2;i2=i3;i3=it;}
      if(it==extentitem){i1=i2;i2=i3;i3=it;}
      it=it->getBelow();
      }

    FXASSERT(i1 && i2 && i3);

    // First segment
    it=i1;
    while(it!=i2){

      // item = extent - anchor
      // item = anchor - extent
      if(i1==item){
        if(!it->isSelected()){
          it->setSelected(TRUE);
          updateItem(it);
          changes=TRUE;
          if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)it);}
          }
        }

      // extent = anchor - item
      // extent = item   - anchor
      else if(i1==extentitem){
        if(it->isSelected()){
          it->setSelected(FALSE);
          updateItem(it);
          changes=TRUE;
          if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)it);}
          }
        }
      it=it->getBelow();
      }

    // Second segment
    it=i2;
    while(it!=i3){
      it=it->getBelow();

      // extent - anchor = item
      // anchor - extent = item
      if(i3==item){
        if(!it->isSelected()){
          it->setSelected(TRUE);
          updateItem(it);
          changes=TRUE;
          if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)it);}
          }
        }

      // item   - anchor = extent
      // anchor - item   = extent
      else if(i3==extentitem){
        if(it->isSelected()){
          it->setSelected(FALSE);
          updateItem(it);
          changes=TRUE;
          if(notify){handle(this,FXSEL(SEL_DESELECTED,0),(void*)it);}
          }
        }
      }
    extentitem=item;
    }
  return changes;
  }


// Kill selection
FXbool FXCheckTreeList::killSelection(FXbool notify){
  register FXCheckTreeItem *item=firstitem;
  register FXbool changes=FALSE;
  while(item){
    if(item->isSelected()){
      item->setSelected(FALSE);
      updateItem(item);
      changes=TRUE;
      if(notify){handle(this,FXSEL(SEL_SELECTED,0),(void*)item);}
      }
    item=item->getBelow();
    }
  return changes;
  }


// Sort items in ascending order
FXint FXCheckTreeList::ascending(const FXCheckTreeItem* a,const FXCheckTreeItem* b){
  return compare(a->label,b->label);
  }


// Sort items in descending order
FXint FXCheckTreeList::descending(const FXCheckTreeItem* a,const FXCheckTreeItem* b){
  return compare(b->label,a->label);
  }


// Sort items
void FXCheckTreeList::sort(FXCheckTreeItem*& f1,FXCheckTreeItem*& t1,FXCheckTreeItem*& f2,FXCheckTreeItem*& t2,int n){
  FXCheckTreeItem *ff1,*tt1,*ff2,*tt2,*q;
  FXint m;
  if(f2==NULL){
    f1=NULL;
    t1=NULL;
    return;
    }
  if(n>1){
    m=n/2;
    n=n-m;
    sort(ff1,tt1,f2,t2,n);  // 1 or more
    sort(ff2,tt2,f2,t2,m);  // 0 or more
    FXASSERT(ff1);
    if(ff2 && sortfunc(ff1,ff2)>0){
      f1=ff2;
      ff2->prev=NULL;
      ff2=ff2->next;
      }
    else{
      f1=ff1;
      ff1->prev=NULL;
      ff1=ff1->next;
      }
    t1=f1;
    t1->next=NULL;
    while(ff1 || ff2){
      if(ff1==NULL){ t1->next=ff2; ff2->prev=t1; t1=tt2; break; }
      if(ff2==NULL){ t1->next=ff1; ff1->prev=t1; t1=tt1; break; }
      if(sortfunc(ff1,ff2)>0){
        t1->next=ff2;
        ff2->prev=t1;
        t1=ff2;
        ff2=ff2->next;
        }
      else{
        t1->next=ff1;
        ff1->prev=t1;
        t1=ff1;
        ff1=ff1->next;
        }
      t1->next=NULL;
      }
    return;
    }
  FXASSERT(f2);
  f1=f2;
  t1=f2;
  f2=f2->next;
  while(f2){
    f2->prev=NULL;
    if(sortfunc(f2,t1)>0){
      t1->next=f2;
      f2->prev=t1;
      t1=f2;
      f2=f2->next;
      continue;
      }
    if(sortfunc(f1,f2)>0){
      q=f2;
      f2=f2->next;
      q->next=f1;
      f1->prev=q;
      f1=q;
      continue;
      }
    break;
    }
  FXASSERT(f1);
  FXASSERT(t1);
  f1->prev=NULL;
  t1->next=NULL;
  }


// Sort the items based on the sort function
void FXCheckTreeList::sortItems(){
  if(sortfunc){
    FXCheckTreeItem* f=firstitem;
    FXCheckTreeItem* l=lastitem;
    sort(firstitem,lastitem,f,l,getNumItems());
    recalc();
    }
  }


// Sort child items
void FXCheckTreeList::sortChildItems(FXCheckTreeItem* item){
  if(sortfunc){
    FXCheckTreeItem* f=item->first;
    FXCheckTreeItem* l=item->last;
    sort(item->first,item->last,f,l,item->getNumChildren());
    if(item->isExpanded()) recalc();     // No need to recalc if it ain't visible!
    }
  }


// Set current item
void FXCheckTreeList::setCurrentItem(FXCheckTreeItem* item,FXbool notify){
  if(item!=currentitem){

    // Deactivate old item
    if(currentitem){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        currentitem->setFocus(FALSE);
        updateItem(currentitem);
        }

      // Close old item
      closeItem(currentitem,notify);
      }

    currentitem=item;

    // Activate new item
    if(currentitem){

      // No visible change if it doen't have the focus
      if(hasFocus()){
        currentitem->setFocus(TRUE);
        updateItem(currentitem);
        }

      // Open new item
      openItem(currentitem,notify);
      }

    // Notify item change
    if(notify && target){target->handle(this,FXSEL(SEL_CHANGED,message),(void*)currentitem);}
    }

  // Select if browse mode
  if((options&SELECT_MASK)==CHECKTREELIST_BROWSESELECT && currentitem && currentitem->isEnabled()){
    selectItem(currentitem,notify);
    }
  }


// Set anchor item
void FXCheckTreeList::setAnchorItem(FXCheckTreeItem* item){
  anchoritem=item;
  extentitem=item;
  }



// Create item
FXCheckTreeItem* FXCheckTreeList::createItem(const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  return new FXCheckTreeItem(text,oi,ci,ptr);
  }


// Add item as first one under parent p
FXCheckTreeItem* FXCheckTreeList::addItemFirst(FXCheckTreeItem* p,FXCheckTreeItem* item,FXbool notify){
  register FXCheckTreeItem* olditem=currentitem;

  // Must have item
  if(!item){ fxerror("%s::addItemFirst: item is NULL.\n",getClassName()); }

  // Add item to list
  if(p){
    item->prev=NULL;
    item->next=p->first;
    if(item->next) item->next->prev=item; else p->last=item;
    p->first=item;
    }
  else{
    item->prev=NULL;
    item->next=firstitem;
    if(item->next) item->next->prev=item; else lastitem=item;
    firstitem=item;
    }
  item->parent=p;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;

  // Make current if just added
  if(!currentitem && item==lastitem) currentitem=item;

  // Notify item has been inserted
  if(notify && target){target->handle(this,FXSEL(SEL_INSERTED,message),(void*)item);}

  // Current item may have changed
  if(olditem!=currentitem){
    if(notify && target){target->handle(this,FXSEL(SEL_CHANGED,message),(void*)currentitem);}
    }

  // Was new item
  if(currentitem==item){
    if(hasFocus()){
      currentitem->setFocus(TRUE);
      }
    if((options&SELECT_MASK)==CHECKTREELIST_BROWSESELECT && currentitem->isEnabled()){
      selectItem(currentitem,notify);
      }
    }

  // Redo layout
  recalc();
  return item;
  }


// Add item as first one under parent p
FXCheckTreeItem* FXCheckTreeList::addItemFirst(FXCheckTreeItem* p,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr,FXbool notify){
  return addItemFirst(p,createItem(text,oi,ci,ptr),notify);
  }


// Add item as last one under parent p
FXCheckTreeItem* FXCheckTreeList::addItemLast(FXCheckTreeItem* p,FXCheckTreeItem* item,FXbool notify){
  register FXCheckTreeItem* olditem=currentitem;

  // Must have item
  if(!item){ fxerror("%s::addItemLast: item is NULL.\n",getClassName()); }

  // Add item to list
  if(p){
    item->prev=p->last;
    item->next=NULL;
    if(item->prev) item->prev->next=item; else p->first=item;
    p->last=item;
    }
  else{
    item->prev=lastitem;
    item->next=NULL;
    if(item->prev) item->prev->next=item; else firstitem=item;
    lastitem=item;
    }
  item->parent=p;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;

  // Make current if just added
  if(!currentitem && item==firstitem) currentitem=item;

  // Notify item has been inserted
  if(notify && target){target->handle(this,FXSEL(SEL_INSERTED,message),(void*)item);}

  // Current item may have changed
  if(olditem!=currentitem){
    if(notify && target){target->handle(this,FXSEL(SEL_CHANGED,message),(void*)currentitem);}
    }

  // Was new item
  if(currentitem==item){
    if(hasFocus()){
      currentitem->setFocus(TRUE);
      }
    if((options&SELECT_MASK)==CHECKTREELIST_BROWSESELECT && currentitem->isEnabled()){
      selectItem(currentitem,notify);
      }
    }

  // Redo layout
  recalc();
  return item;
  }


// Add item as last one under parent p
FXCheckTreeItem* FXCheckTreeList::addItemLast(FXCheckTreeItem* p,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr,FXbool notify){
  return addItemLast(p,createItem(text,oi,ci,ptr),notify);
  }


// Link item after other
FXCheckTreeItem* FXCheckTreeList::addItemAfter(FXCheckTreeItem* other,FXCheckTreeItem* item,FXbool notify){

  // Must have items
  if(!item){ fxerror("%s::addItemAfter: item is NULL.\n",getClassName()); }
  if(!other){ fxerror("%s::addItemAfter: other item is NULL.\n",getClassName()); }

  // Add item to list
  item->prev=other;
  item->next=other->next;
  other->next=item;
  if(item->next) item->next->prev=item; else if(other->parent) other->parent->last=item; else lastitem=item;
  item->parent=other->parent;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;

  // Notify item has been inserted
  if(notify && target){target->handle(this,FXSEL(SEL_INSERTED,message),(void*)item);}

  // Redo layout
  recalc();
  return item;
  }


// Link item after other
FXCheckTreeItem* FXCheckTreeList::addItemAfter(FXCheckTreeItem* other,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr,FXbool notify){
  return addItemAfter(other,createItem(text,oi,ci,ptr),notify);
  }


// Link item before other
FXCheckTreeItem* FXCheckTreeList::addItemBefore(FXCheckTreeItem* other,FXCheckTreeItem* item,FXbool notify){

  // Must have items
  if(!item){ fxerror("%s::addItemBefore: item is NULL.\n",getClassName()); }
  if(!other){ fxerror("%s::addItemBefore: other item is NULL.\n",getClassName()); }

  // Add item to list
  item->next=other;
  item->prev=other->prev;
  other->prev=item;
  if(item->prev) item->prev->next=item; else if(other->parent) other->parent->first=item; else firstitem=item;
  item->parent=other->parent;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;

  // Notify item has been inserted
  if(notify && target){target->handle(this,FXSEL(SEL_INSERTED,message),(void*)item);}

  // Redo layout
  recalc();
  return item;
  }


// Link item before other
FXCheckTreeItem* FXCheckTreeList::addItemBefore(FXCheckTreeItem* other,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr,FXbool notify){
  return addItemBefore(other,createItem(text,oi,ci,ptr),notify);
  }


// Remove node from list
void FXCheckTreeList::removeItem(FXCheckTreeItem* item,FXbool notify){
  register FXCheckTreeItem* olditem=currentitem;
  if(item){

    // First remove children
    removeItems(item->first,item->last,notify);

    // Notify item will be deleted
    if(notify && target){target->handle(this,FXSEL(SEL_DELETED,message),(void*)item);}

    // Adjust pointers
    if(anchoritem==item){ anchoritem=(anchoritem->next ? anchoritem->next : anchoritem->prev); }
    if(extentitem==item){ extentitem=(extentitem->next ? extentitem->next : extentitem->prev); }
    if(currentitem==item){ currentitem=(currentitem->next ? currentitem->next : currentitem->prev); }

    // Remove item from list
    if(item->prev) item->prev->next=item->next; else if(item->parent) item->parent->first=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else if(item->parent) item->parent->last=item->prev; else lastitem=item->prev;

    // Hasta la vista, baby!
    delete item;

    // Current item has changed
    if(olditem!=currentitem){
      if(notify && target){target->handle(this,FXSEL(SEL_CHANGED,message),(void*)currentitem);}
      }

    // Deleted current item
    if(currentitem && item==olditem){
      if(hasFocus()){
        currentitem->setFocus(TRUE);
        }
      if((options&SELECT_MASK)==CHECKTREELIST_BROWSESELECT && currentitem->isEnabled()){
        selectItem(currentitem,notify);
        }
      }

    // Redo layout
    recalc();
    }
  }


// Remove all siblings from [fm,to]
void FXCheckTreeList::removeItems(FXCheckTreeItem* fm,FXCheckTreeItem* to,FXbool notify){
  register FXCheckTreeItem *item;
  if(fm && to){
    do{
      item=fm;
      fm=fm->next;
      removeItem(item,notify);
      }
    while(item!=to);
    }
//   if(fm && to){
//     FXCheckTreeItem *item;
//     if(fm->prev) fm->prev->next=to->next; else if(fm->parent) fm->parent->first=to->next; else firstitem=to->next;
//     if(to->next) to->next->prev=fm->prev; else if(to->parent) to->parent->last=fm->prev; else lastitem=fm->prev;
//     do{
//       item=fm;
//       if(currentitem==item) currentitem=NULL;
//       if(anchoritem==item) anchoritem=NULL;
//       if(extentitem==item) extentitem=NULL;
//       removeItems(item->first,item->last,notify);
//       fm=fm->next;
//       delete item;
//       }
//     while(item!=to);
//     recalc();
//     }
  }


// Remove all items
void FXCheckTreeList::clearItems(FXbool notify){
  removeItems(firstitem,lastitem,notify);
  }


typedef FXint (*FXCompareFunc)(const FXString&,const FXString &,FXint);


// Get item by name
FXCheckTreeItem* FXCheckTreeList::findItem(const FXString& text,FXCheckTreeItem* start,FXuint flags) const {
  register FXCompareFunc comparefunc;
  register FXCheckTreeItem *item,*s,*f,*l;
  register FXint len;
  if(firstitem){
    comparefunc=(flags&SEARCH_IGNORECASE) ? (FXCompareFunc)comparecase : (FXCompareFunc)compare;
    len=(flags&SEARCH_PREFIX)?text.length():2147483647;
    if(!(flags&SEARCH_BACKWARD)){
      s=f=firstitem;
      if(start){s=start;if(s->parent){f=s->parent->first;}}
      item=s;
      while(item){
        if((*comparefunc)(item->label,text,len)==0) return item;
        item=item->next;
        }
      if(!(flags&SEARCH_WRAP)) return NULL;
      item=f;
      while(item && item!=s){
        if((*comparefunc)(item->label,text,len)==0) return item;
        item=item->next;
        }
      }
    else{
      s=l=lastitem;
      if(start){s=start;if(s->parent){l=s->parent->last;}}
      item=s;
      while(item){
        if((*comparefunc)(item->label,text,len)==0) return item;
        item=item->prev;
        }
      if(!(flags&SEARCH_WRAP)) return NULL;
      item=l;
      while(item && item!=s){
        if((*comparefunc)(item->label,text,len)==0) return item;
        item=item->prev;
        }
      }
    }
  return NULL;
  }


// Change the font
void FXCheckTreeList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Change help text
void FXCheckTreeList::setHelpText(const FXString& text){
  help=text;
  }


// Set text color
void FXCheckTreeList::setTextColor(FXColor clr){
  if(clr!=textColor){
    textColor=clr;
    update();
    }
  }


// Set select background color
void FXCheckTreeList::setSelBackColor(FXColor clr){
  if(clr!=selbackColor){
    selbackColor=clr;
    update();
    }
  }


// Set selected text color
void FXCheckTreeList::setSelTextColor(FXColor clr){
  if(clr!=seltextColor){
    seltextColor=clr;
    update();
    }
  }


// Set line color
void FXCheckTreeList::setLineColor(FXColor clr){
  if(clr!=lineColor){
    lineColor=clr;
    update();
    }
  }


// Set parent to child indent amount
void FXCheckTreeList::setIndent(FXint in){
  if(indent!=in){
    indent=in;
    recalc();
    }
  }


// Change list style
void FXCheckTreeList::setListStyle(FXuint style){
  FXuint opts=(options&~TREELIST_MASK) | (style&TREELIST_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Get list style
FXuint FXCheckTreeList::getListStyle() const {
  return (options&TREELIST_MASK);
  }


// Save data
void FXCheckTreeList::save(FXStream& store) const {
  FXScrollArea::save(store);
  store << firstitem;
  store << lastitem;
  store << anchoritem;
  store << currentitem;
  store << extentitem;
  store << font;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << lineColor;
  store << treeWidth;
  store << treeHeight;
  store << visible;
  store << indent;
  store << help;
  }


// Load data
void FXCheckTreeList::load(FXStream& store){
  FXScrollArea::load(store);
  store >> firstitem;
  store >> lastitem;
  store >> anchoritem;
  store >> currentitem;
  store >> extentitem;
  store >> font;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> lineColor;
  store >> treeWidth;
  store >> treeHeight;
  store >> visible;
  store >> indent;
  store >> help;
  }


// Cleanup
FXCheckTreeList::~FXCheckTreeList(){
  getApp()->removeTimeout(this,ID_TIPTIMER);
  getApp()->removeTimeout(this,ID_LOOKUPTIMER);
  clearItems(FALSE);
  firstitem=(FXCheckTreeItem*)-1;
  lastitem=(FXCheckTreeItem*)-1;
  anchoritem=(FXCheckTreeItem*)-1;
  currentitem=(FXCheckTreeItem*)-1;
  extentitem=(FXCheckTreeItem*)-1;
  font=(FXFont*)-1;
  }

}

