/********************************************************************************
*                                                                               *
*                   Thread object for worker functions                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXTHREADFUNCTION_H
#define FXTHREADFUNCTION_H

#ifndef FXRUNNABLE_H
#include "FXRunnable.h"
#endif
namespace FXEX {

/**
 * You can start a worker thread for running your task.  It runs with a lower priority
 * than your main thread.  It runs as a detached thread (ie you main thread cant/shouldn't
 * wait for it to finish).  example:
 * 
 * // declare file scope pointer to variable
 * FXThreadFunction *thread;
 * // declare some variables...
 * 
 * // do some work
 * void workerFunction(){
 *   while (...) {
 *     // use global variables, including some atomic's
 *     ...
 *     if (..) thread->signal();
 *     ...
 *     }
 *   }
 *
 * // FOX message map - handle a thread event
 * ...
 * FXMAPFUNC(SEL_CREATE,MyApp::ID_WORKER,MyApp:onWorker),
 * FXMAPFUNC(SEL_THREAD,MyApp::ID_WORKER,MyApp:onWorker),
 * FXMAPFUNC(SEL_DESTROY,MyApp::ID_WORKER,MyApp:onWorker),
 * ...
 *
 * // main window/thread
 * MyApp::MyApp(a):FXMainWindow(a){
 *   ...
 *   // setup variables
 *   ...
 *   // create the thread
 *   thread = new FXThreadFunction(workerFunction,this,ID_WORKER);
 *   ...
 *   }
 *
 * MyApp::create(){
 *   thread->start();
 *   }
 *
 * // handle thread event
 * long MyApp::onWorker(,,){
 *   // update something
 *   // reset variables
 *   return 1;
 *   }
 *
 *
 *
 * alternatively:
 *
 *
 *
 * FXAtomicInt atomicInt;
 *
 * MyApp::MyApp(a):FXMainWindow(a){
 *   ...
 *   atomicInt=0;
 *   atomicIntDataTarget.connect(atomicInt);
 *   thread = new FXThreadFunction(workerFunction,...);
 *   ...
 *   new FXTextField(this,10,atomicIntDataTarget,FXDataTarget::ID_VALUE,...);
 *   ...
 *   }
 *
 * void MyApp::workerFunction(){
 *   while (...) {
 *     // do some work
 *     ...
 *     atomicInt++;
 *     }
 *   }
 *
 */
class FXAPI FXThreadFunction : public FXRunnable {
  FXDECLARE(FXThreadFunction)

private:
  FXThreadFunction(const FXThreadFunction&);
  FXThreadFunction& operator=(const FXThreadFunction&);

private:
  FXThreadHandle thread;       // the thread

protected:
  FXThreadFunction() {}

public:
  /// Note: you dont normally need to call these, they are used internally but must be public

  /// reset the state of the thread object, so that it can be reused
  void reset();

  /// handle to thread function
  void (*threadFunction)();

public:
  /// Pass a reference to your static method to this ctor
  FXThreadFunction(void (*func)(),FXObject *tgt=NULL,FXSelector sel=0);

  /// set to new function
  void setFunc(void (*func)());

  /// start running the object
  void start();

  /// dtor
  virtual ~FXThreadFunction();
  };

} // namespace FXEX
#endif // FXTHREADFUNCTION_H
