/////////////////////////////////////////////////////////////////////////////
//
// DbControl
//
/////////////////////////////////////////////////////////////////////////////
//
// Hakki Dogusan
// dogusanh@tr.net
// http://www.dynaset.org/dogusanh
//
/////////////////////////////////////////////////////////////////////////////
#ifndef DBCONTROL_H
#define DBCONTROL_H

#ifndef FXBASEOBJECT_H
#include "FXBaseObject.h"
#endif
namespace FXEX {
class DbField;

/**
 * DbControl is an object which is a storage unit for some type of in-memory database (I think?)
 */
class FXAPI DbControl : public FXBaseObject {
  FXDECLARE(DbControl)
  typedef FXObjectListOf<DbField> DbFields;

  public:
    enum {
      ID_DB_FIRST=50000,
      ID_OPEN,
      ID_CLOSE,
      ID_FIELD,                       // field operation
      ID_FOCUSIN,
      ID_FOCUSOUT,
      ID_INSERT,                      // record operation
      ID_EDIT,
      ID_KILL,
      ID_POST,
      ID_CANCEL,
      ID_QUERY,                      // navigation
      ID_SCROLL,
      ID_FIRST,
      ID_PRIOR,
      ID_NEXT,
      ID_LAST,
      ID_PRINT,                      // misc operations
      ID_DB_LAST,
      };

  public:
    /// Current mode of operation
    enum DbMode {
      DB_INACTIVE,
      DB_BROWSE,
      DB_INSERT,
      DB_EDIT
      };

    // Current position of data pointer
    enum DbPosition {
      DB_BOF = -4,
      DB_EOF = -3,
      DB_FIRST = -2,
      DB_PRIOR = -1,
      DB_CURRENT = 0,
      DB_NEXT = 1,
      DB_LAST = 2,
      DB_FIND = 3
      };

  public:
    /// ctor
    DbControl(FXWindow* own,FXObject* tgt=NULL,FXSelector sel=0);

    /// dtor
    virtual ~DbControl();

  public:
    /// accessor methods
    FXbool open();
    FXbool close();
    FXbool insert();
    FXbool post();
    FXbool cancel();
    FXbool kill(); //Delete
    FXbool edit();
    FXbool query();
    FXbool first();
    FXbool prior();
    FXbool next();
    FXbool last();
    FXbool print();

    /// are we at the bottom of the dataset
    FXbool isBof() const;

    /// are we are the end of the file
    FXbool isEof() const;

  public:
    /// get the current mode of operation
    DbMode getMode() const { return mode; }

    /// get pointer to specific column (by column number), from current row
    DbField* getField(FXint i=-1) const; // i=-1 => current focused field

    /// get pointer to specific column (by column name), from current row
    DbField* getField(const FXString&) const; // NULL => current focused field

    /// return the query applied to the data
    FXString getQuery() const { return query_; }

    /// set the query applied to the data
    void setQuery(const FXString& q) { query_=q; }

  protected:
    /// add specified field to database
    void addField(DbField* field) { fields.append(field); }

    /// add field using field name/title for FXchar type
    void addField(const FXString& name,const FXString& title,FXchar& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXuchar type
    void addField(const FXString& name,const FXString& title,FXuchar& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXshort type
    void addField(const FXString& name,const FXString& title,FXshort& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXushort type
    void addField(const FXString& name,const FXString& title,FXushort& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXint type
    void addField(const FXString& name,const FXString& title,FXint& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXuint type
    void addField(const FXString& name,const FXString& title,FXuint& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXFloat type
    void addField(const FXString& name,const FXString& title,FXfloat& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXdouble type
    void addField(const FXString& name,const FXString& title,FXdouble& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for FXString type
    void addField(const FXString& name,const FXString& title,FXString& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for DateFiled type
    void addField(const FXString& name,const FXString& title,DateField& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

    /// add field using field name/title for CurrencyField type
    void addField(const FXString& name,const FXString& title,CurrencyField& value) { fields.append(new DbField(name,title,value,this,ID_FIELD)); }

  protected:

    /// return widget owner
    FXWindow* getOwner() const { return owner; }

    /// return handle to list of database fields
    DbFields& getFields() { return fields; }

  private:
    /// these virtual methods are meant to be handled by the actual database instance object
    virtual FXbool doOpen() { return FALSE; }
    virtual FXbool doClose() { return FALSE; }
    virtual FXbool doClear() { return FALSE; }
    virtual FXbool doInsert() { return FALSE; }
    virtual FXbool doPost() { return FALSE; }
    virtual FXbool doCancel() { return FALSE; }
    virtual FXbool doKill() { return FALSE; }
    virtual FXbool doEdit() { return FALSE; }
    virtual FXbool doQuery() { return FALSE; }
    virtual FXbool doFetch(DbPosition&) { return FALSE; }
    virtual FXbool doPrint() { return FALSE; }
    virtual FXbool doShowError(const FXString&) { return FALSE; }

  public:
    long onFieldChanged(FXObject*, FXSelector, void*);
    long onFieldUpdate(FXObject*, FXSelector, void*);
    long onFieldFocusIn(FXObject*, FXSelector, void*);
    long onFieldFocusOut(FXObject*, FXSelector, void*);
    long onCmdOpen(FXObject*, FXSelector, void*);
    long onUpdOpen(FXObject*, FXSelector, void*);
    long onCmdClose(FXObject*, FXSelector, void*);
    long onUpdClose(FXObject*, FXSelector, void*);
    long onCmdInsert(FXObject*, FXSelector, void*);
    long onUpdInsert(FXObject*, FXSelector, void*);
    long onCmdKill(FXObject*, FXSelector, void*);
    long onUpdKill(FXObject*, FXSelector, void*);
    long onCmdPost(FXObject*, FXSelector, void*);
    long onUpdPost(FXObject*, FXSelector, void*);
    long onCmdCancel(FXObject*, FXSelector, void*);
    long onUpdCancel(FXObject*, FXSelector, void*);
    long onCmdQuery(FXObject*, FXSelector, void*);
    long onUpdQuery(FXObject*, FXSelector, void*);
    long onCmdFirst(FXObject*, FXSelector, void*);
    long onUpdFirst(FXObject*, FXSelector, void*);
    long onCmdPrior(FXObject*, FXSelector, void*);
    long onUpdPrior(FXObject*, FXSelector, void*);
    long onCmdNext(FXObject*, FXSelector, void*);
    long onUpdNext(FXObject*, FXSelector, void*);
    long onCmdLast(FXObject*, FXSelector, void*);
    long onUpdLast(FXObject*, FXSelector, void*);
    long onCmdPrint(FXObject*, FXSelector, void*);
    long onUpdPrint(FXObject*, FXSelector, void*);

  private:
    FXWindow*   owner;
    DbMode      mode;
    DbFields    fields;
    DbField*    focusedField;
    FXString    query_;
    DbPosition  position;
    void setMode(DbMode m) { mode=m; }
    FXbool before(FXuint sel, void* ptr);
    FXbool after(FXuint sel, void* ptr);

  protected:
    /// serialisation
    DbControl(){}

  private:
    /// disable access to these operations
    DbControl(const DbControl&);
    DbControl& operator=(const DbControl&);
  };

} // namespace FXEX
#endif // DBCONTROL_H
