/* rbmake.c
 *
 * Use this "class" when you want to create a new .rb file.  The actual
 * creation of the individual pages (via RbPage objects) is handled in
 * rbpage.c and rbhtml.c.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <ctype.h>
#include <libxml/uri.h>
#include <rbmake/rbmake.h>
#include "rbmake.h"
#include "rbpage.h"
#include "rbfile.h"
#include "mbuf.h"

static char *TextConvOptNames[RB_TEXTCONV_OPT_CNT] = {
    "none", "preformatted", "simple-para",
};
static char *IncludeOptNames[RB_INCLUDE_OPT_CNT] = {
    "no", "match", "yes",
};

static char mmStartTags[] = "\
<HTML><HEAD>\n\
<TITLE>MenuMark</TITLE>\n\
</HEAD><BODY>\n\
<P>\n\
";

static int incPageNameHash(RbMake *me, const char *tn);

void
RbMake_init()
{
    RbHtml_init();
}

void
RbMake_cleanup()
{
    RbHtml_cleanup();
}

RbMake *
RbMake_new(RbMakeAllowUrlFunc allowFunc, RbMakeScheduleUrlFunc scheduleFunc,
	   RbMakeFinishInfoFunc finishInfoFunc)
{
    RbMake *me = Mem_calloc(1, sizeof *me);
    me->shouldAllowURL = allowFunc;
    me->scheduleURL = scheduleFunc;
    me->finishInfoPage = finishInfoFunc;
    me->tocNameHash = HashTable_new(501, false, true);
    me->tocUrlHash = HashTable_new(501, false, true);
    me->dropUrlHash = HashTable_new(501, false, false);
    me->infoHash = RbInfoHash_newFromMBuf(NULL);
    me->menuItems = MArray_new(8, 0);
    me->textConversionMode = RB_TEXTCONV_PRE;
    me->includeImages = me->includeAudio = RB_INCLUDE_NO;

    incPageNameHash(me, "cover.png"); /* Reserve this ToC name */
    
    return me;
}

void
RbMake_setGenerator(RbMake *me, const char *gen)
{
    me->generator = gen;
}

const char *
RbMake_getGenerator(RbMake *me)
{
    static char *genstr = NULL;
    if (!genstr) {
	MBuf *tmp = MBuf_new(128, 0);
	if (me->generator)
	    MBuf_vwrite(tmp, me->generator,-1, " using ",7, NULL);
	MBuf_puts(tmp, "the rbmake library v" RBMAKE_VERSION);
	genstr = MBuf_toBuffer(tmp, NULL);
    }
    return genstr;
}

char *
RbMake_mungeBookName(RbMake *me, const char *bn)
{
    const char *title;
    char *nn, *suf;

    if (strnEQ(bn, "_Put_Title_Here_", 16)
     && (title = RbInfoHash_fetch(me->infoHash, "TITLE")) != NULL) {
	const char *f;
	char *t;
	nn = Mem_alloc(strlen(title) + 3 + 1);
	for (t = nn, f = title; *f; f++) {
	    if (*f == '\'' || uc(f,0) == RB_CH_RSQUO)
		continue;
	    if (!ISALPHA(*f) && !ISDIGIT(*f) && *f != '-' && *f != '+') {
		if (t != nn && t[-1] != '_')
		    *t++ = '_';
	    }
	    else
		*t++ = *f;
	}
	if (t != nn && t[-1] == '_')
	    t--;
	*t = '\0';
    }
    else {
	nn = Mem_alloc(strlen(bn) + 3 + 1);
	strcpy(nn, bn);
    }

#ifdef DOS_FILENAMES
    for (suf = nn; *suf; suf++) {
	if (*suf == '\\')
	    *suf = '/';
    }
#endif
    suf = rbGetFileSuffix(nn);
    if (rbIsRbSuf(suf)||rbIsHtmlSuf(suf)||rbIsTextSuf(suf)||rbIsInfoSuf(suf))
	strcpy(suf, "rb");
    else
	strcat(nn, ".rb");

    return nn;
}

bool
RbMake_create(RbMake *me, const char *fn, int maxTocCnt)
{
    me->file = RbFile_create(fn, maxTocCnt > 0? maxTocCnt : RB_MAX_TOC_CNT);
    if (!me->file)
	return false;
    if (me->coverImage) {
	me->binaryCnt++;
	me->scheduleURL(me, NULL, me->coverImage, RB_PAGETYPE_COVER_IMAGE);
    }
    return true;
}

void
RbMake_finish(RbMake *me)
{
    if (!me->foundMenumark && me->menuItems->totalLen) {
	char *mmf = rbBuildURL("menumark.html", me->rootURL);
	RbPage *pg = RbPage_new(me, NULL, mmf, RB_PAGETYPE_HTML);
	MBuf *mb = MBuf_new(512, 0);
	char *item, *eq, *url;
	HashTable *ht = HashTable_new(11, false, false);

	RbPage_appendContent(pg, mmStartTags, STATICLEN(mmStartTags));
	MArray_setFetchPos(me->menuItems, 0);
	while ((item = MArray_fetchPtr(me->menuItems)) != NULL) {
	    if ((eq = strchr(item, '=')) == NULL)
		continue;
	    if (!HashTable_fetch(ht, eq)) {
		*eq = '\0';
		if (!HashTable_fetch(ht, item)) {
		    url = rbBuildURL(eq+1, pg->url);
		    MBuf_truncate(mb, 0);
		    MBuf_vwrite(mb, "<A HREF=\"",9, url,-1,
				"\" Rel=\"toc\">",12, item,-1, "</A><BR>\n",9,
				NULL);
		    rbFreeURL(url);
		    RbPage_appendContent(pg, MBuf_dataPtr(mb, NULL), mb->totalLen);
		    HashTable_store(ht, item, "");
		    *eq = '=';
		    HashTable_store(ht, eq, "");
		}
	    }
	}
	HashTable_delete(ht);
	MBuf_delete(mb);
	RbPage_finishContent(pg);
	RbPage_write(pg);
	rbFreeURL(mmf);
    }

    RbFile_close(me->file);
    HashTable_delete(me->tocNameHash);
    HashTable_delete(me->tocUrlHash);
    HashTable_delete(me->dropUrlHash);
    RbInfoHash_delete(me->infoHash);
}

NameWithType *
RbMake_addPageName(RbMake *me, const char *arg, int pt)
{
    int oldHtmlCnt = me->htmlCnt, oldBinaryCnt = me->binaryCnt;
    NameWithType *nwt;
    char *tn, *suf, *cp;
    int tnlen, tnum;
    char buf[32];
    xmlURIPtr uri;

    if (HashTable_fetch(me->dropUrlHash, arg))
	return NULL;
    uri = xmlParseURI(arg);
    if (!uri->scheme
     || (strNE(uri->scheme, "file") && strNE(uri->scheme, "http")
      && strNE(uri->scheme, "rb"))) {
	xmlFreeURI(uri);
	return NULL;
    }
    if (!uri->path || !(tn = strrchr(uri->path, '/')) || !*++tn) {
	tn = "index.html";
	suf = tn + 5;
	if (pt == RB_PAGETYPE_UNKNOWN)
	    pt = RB_PAGETYPE_MAYBE_HTML;
    }
    else if ((suf = strrchr(tn, '.')) == NULL) {
	if (pt == RB_PAGETYPE_UNKNOWN)
	    pt = RB_PAGETYPE_MAYBE_HTML;
	suf = tn + strlen(tn);
    }
    else if (pt == RB_PAGETYPE_UNKNOWN) {
	if ((pt = rbSuffixToPageType(suf+1)) == RB_PAGETYPE_UNKNOWN)
	    pt = RB_PAGETYPE_MAYBE_HTML;
    }
    tnlen = suf - tn;
    if (tnlen > 15)
	tnlen = 15;
    strncpy(buf, tn, tnlen);
    tn = buf;
    xmlFreeURI(uri);
    switch (pt) {
      case RB_PAGETYPE_HTML:
      case RB_PAGETYPE_TEXT:
      case RB_PAGETYPE_MAYBE_HTML:
      case RB_PAGETYPE_RAW_TEXT:
	if (!me->pageJoiningMode)
	    me->htmlCnt++;
	else
	    me->joinedHtmlCnt++;
	if (strnEQ(buf, "menumark", 8)) {
	    if (me->foundMenumark)
		return NULL;
	    me->foundMenumark = 1;
	    tn = "menumark.html";
	    goto hash_it;
	}
	suf = "html";
	break;
      case RB_PAGETYPE_IMAGE:
	suf = "png";
	me->binaryCnt++;
	break;
      case RB_PAGETYPE_AUDIO:
	suf = "wav";
	me->binaryCnt++;
	break;
      default:
	return NULL;
    }
    if (me->htmlCnt*2 + me->binaryCnt + 3 > (me->file? me->file->maxTocCnt : RB_MAX_TOC_CNT)) {
	HashTable_store(me->dropUrlHash, arg, "");
	me->htmlDropCnt += me->htmlCnt - oldHtmlCnt;
	me->binaryDropCnt += me->binaryCnt - oldBinaryCnt;
	me->htmlCnt = oldHtmlCnt;
	me->binaryCnt = oldBinaryCnt;
	return NULL;
    }
    sprintf(buf+tnlen, ".%s", suf);
    if (me->pageJoiningMode)
	while ((cp = strchr(buf, RB_JOIN_NAME_SEP)) != NULL) *cp = '_';
    for (cp = buf; *cp; cp++) {
	if (uc(cp,0) <= ' ' || strchr("\"';&^!`[]{}()*?<>\\|$#",*cp))
	    *cp = '_';
    }
    if ((tnum = incPageNameHash(me, buf)) > 1) {
	char numbuf[32], dupbuf[32];
	int alen;
	strcpy(dupbuf, buf);
	while (1) {
	    sprintf(numbuf, "%d", tnum);
	    alen = strlen(numbuf) + 1;
	    if (tnlen > 15 - alen)
		tnlen = 15 - alen;
	    sprintf(buf+tnlen, "_%d.%s", tnum, suf);
	    if (incPageNameHash(me, buf) == 1)
		break;
	    tnum = incPageNameHash(me, dupbuf);
	}
    }
    if (me->pageJoiningMode && strEQ(suf, "html")) {
	char dupbuf[32];
	int i;
	buf[strlen(buf) - 5] = '\0';
	strcpy(dupbuf, buf);
	if (me->pageJoiningMode == 1) {
	    i = 0;
	    sprintf(buf, "joined.html#%s", dupbuf);
	}
	else {
	    i = me->joinNamesCnt++ / me->pageJoiningMode;
	    sprintf(buf, "j%d.html#%s", i+1, dupbuf);
	}
	if (!me->joinGroupsTail || me->joinGroupsTail->ord < i) {
	    JoinGroup *jg = Mem_alloc(sizeof *jg);
	    jg->pageList = NULL;
	    jg->ord = i;
	    jg->todo = 1;
	    jg->pageCount = 0;
	    jg->next = NULL;
	    if (me->joinGroupsTail)
		me->joinGroupsTail->next = jg;
	    else
		me->joinGroupsHead = jg;
	    me->joinGroupsTail = jg;
	    me->htmlCnt++;
	    me->joinedHtmlCnt--;
	}
	else
	    me->joinGroupsTail->todo++;
    }
  hash_it:
    nwt = Mem_alloc(sizeof *nwt + strlen(tn));
    strcpy(nwt->name, tn);
    nwt->type = pt;
    HashTable_store(me->tocUrlHash, arg, nwt);
    return nwt;
}

NameWithType *
RbMake_findPageName(RbMake *me, const char *arg)
{
    return (NameWithType*)HashTable_fetch(me->tocUrlHash, arg);
}

static int
incPageNameHash(RbMake *me, const char *tn)
{
    int *obj;
    if (!(obj = HashTable_fetch(me->tocNameHash, tn))) {
	obj = Mem_alloc(sizeof (int));
	*obj = 0;
	HashTable_store(me->tocNameHash, tn, (void*)obj);
    }
    return ++*obj;
}

int
RbMake_getPageCount(RbMake *me)
{
    return me->htmlCnt + me->joinedHtmlCnt + me->binaryCnt;
}

int
RbMake_getBinaryCount(RbMake *me)
{
    return me->binaryCnt;
}

int
RbMake_getDoneCount(RbMake *me)
{
    return me->joinCnt + me->doneCnt;
}

int
RbMake_getHtmlDropCount(RbMake *me)
{
    return me->htmlDropCnt;
}

int
RbMake_getBinaryDropCount(RbMake *me)
{
    return me->binaryDropCnt;
}

void
RbMake_addMenuItem(RbMake *me, const char *text, const char *url)
{
    char *item;
    item = Mem_alloc(strlen(text) + 1 + strlen(url) + 1);
    sprintf(item, "%s=%s", text, url);
    MArray_appendPtr(me->menuItems, item);
}

char *
RbMake_getTextConvOptName(int opt)
{
    if (opt < 0 || opt >= RB_TEXTCONV_OPT_CNT)
	return NULL;
    return TextConvOptNames[opt];
}

int
RbMake_findTextConvOpt(const char *str, int len)
{
    int i;
    for (i = 0; i < RB_TEXTCONV_OPT_CNT; i++) {
	if (strncaseEQ(str, TextConvOptNames[i], len))
	    return i;
    }
    return -1;
}

char *
RbMake_getIncludeOptName(int opt)
{
    if (opt < 0 || opt >= RB_INCLUDE_OPT_CNT)
	return NULL;
    return IncludeOptNames[opt];
}

int
RbMake_findIncludeOpt(const char *str, int len)
{
    int i;
    for (i = 0; i < RB_INCLUDE_OPT_CNT; i++) {
	if (strncaseEQ(str, IncludeOptNames[i], len))
	    return i;
    }
    return -1;
}

void
RbMake_setAllowHRSize0PageBreaks(RbMake *me, bool trueOrFalse)
{
    me->allowHRSize0PageBreaks = trueOrFalse;
}

void
RbMake_setCoverImage(RbMake *me, const char *url)
{
    me->coverImage = url;
}

void
RbMake_setCreateHkeyFile(RbMake *me, bool trueOrFalse)
{
    me->createHkeyFile = trueOrFalse;
}

void
RbMake_setEnhancePunctuation(RbMake *me, int enhanceFlags)
{
    if (enhanceFlags == 1)
	enhanceFlags = RB_ENHANCE_ALL;
    me->enhancePunctuation = enhanceFlags;
}

void
RbMake_setFollowLinks(RbMake *me, int depth)
{
    me->followLinks = depth;
}

void
RbMake_setIncludeAudio(RbMake *me, int incl)
{
    me->includeAudio = incl;
}

void
RbMake_setIncludeImages(RbMake *me, int incl)
{
    me->includeImages = incl;
}

void
RbMake_setPageJoining(RbMake *me, int join)
{
    me->pageJoiningMode = join;
}

void
RbMake_setTextConversion(RbMake *me, int conv)
{
    me->textConversionMode = conv;
}

void
RbMake_setUseBookParagraphs(RbMake *me, bool trueOrFalse)
{
    me->bookParagraphDepth = trueOrFalse? 3 : 0;
}

void
RbMake_setBookParagraphDepth(RbMake *me, int depth)
{
    me->bookParagraphDepth = depth > 0? depth + 2 : 0;
}

void
RbMake_setVerboseOutput(RbMake *me, bool trueOrFalse)
{
    me->verboseOutput = trueOrFalse;
}

const char *
RbMake_addSubstRules(RbMake *me, MBuf *mb)
{
    char *txt = MBuf_dataPtrAt(mb, 0, NULL);
    if (!txt)
	return NULL;
    if (!me->substRules)
	me->substRules = MArray_new(32, 0);
    return Subst_parseRules(me->substRules, txt);
}

const char *
RbMake_getFileName(RbMake *me)
{
    return me->file? me->file->fileName : NULL;
}

const char *
RbMake_getNewFileName(RbMake *me)
{
    return me->file? me->file->newFileName : NULL;
}

RbInfoHash *
RbMake_getInfoHash(RbMake *me)
{
    return me->infoHash;
}

bool
RbMake_getAllowHRSize0PageBreaks(RbMake *me)
{
    return me->allowHRSize0PageBreaks;
}

const char *
RbMake_getCoverImage(RbMake *me)
{
    return me->coverImage;
}

bool
RbMake_getCreateHkeyFile(RbMake *me)
{
    return me->createHkeyFile;
}

int
RbMake_getEnhancePunctuation(RbMake *me)
{
    return me->enhancePunctuation;
}

int
RbMake_getFollowLinks(RbMake *me)
{
    return me->followLinks;
}

int
RbMake_getIncludeAudio(RbMake *me)
{
    return me->includeAudio;
}

int
RbMake_getIncludeImages(RbMake *me)
{
    return me->includeImages;
}

MArray *
RbMake_getMenuItems(RbMake *me)
{
    return me->menuItems;
}

int
RbMake_getPageJoining(RbMake *me)
{
    return me->pageJoiningMode;
}

int
RbMake_getTextConversion(RbMake *me)
{
    return me->textConversionMode;
}

bool
RbMake_getUseBookParagraphs(RbMake *me)
{
    return me->bookParagraphDepth? true : false;
}

int
RbMake_getBookParagraphDepth(RbMake *me)
{
    return me->bookParagraphDepth? me->bookParagraphDepth - 2 : 0;
}


bool
RbMake_getVerboseOutput(RbMake *me)
{
    return me->verboseOutput;
}
