/* rbfile.c
 *
 * These are the low-level routines for reading and writing a .rb file.
 * You can open an existing .rb file, get a list of the ToC (Table of
 * Contents) objects, and then read any item's data into an MBuf (or push
 * the data to a call-back function).  You can also create a new .rb file,
 * write all the page buffers (which creates the relevant ToC objects),
 * and when you close the RbFile object, the new .rb file will be
 * finalized.  If you want assistance creating the right combination of
 * associated pages, look at rbmake.c, rbpage.c, and rbhtml.c.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <time.h>
#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#endif
#include <sys/types.h>
#include <ctype.h>
#include <fcntl.h>
#include <zlib.h>
#include <sys/stat.h>
#include <rbmake/rbfile.h>
#include "rbfile.h"
#include "mbuf.h"

char rbID[] = { 0xb0, 0x0c, 0xb0, 0x0c };

static char *PageTypeName[RB_PAGETYPE_COUNT] = {
    "UNKNOWN", "html", "text", "RAW-TEXT", "image", "audio", "info",
    "hidx", "hkey", "rb", "COVER-IMAGE", "MAYBE-HTML", "IGNORE"
};

static void finishCreate(RbFile *me);
static char *getBaseName4MarkupFile(RbFile *me, const char *path);
static void addMarkup(RbFile *me, ToC *toc, RbMarkup *add);
static unsigned int hexValue(const char *s);
static void addToC(RbFile *me, ToC *toc);
static void unjoinPushFunc(void *userPtr, const char *bp, int len);

RbFile *
RbFile_open(const char *fn, int openFlags)
{
    const char *cp;
    struct stat sb;
    int32 i;
    ToC *toc;
    RbFile *rb = Mem_calloc(1, sizeof *rb);

    if ((rb->fp = fopen(fn, "rb")) == NULL) {
	Mem_free(rb);
	RbError_warn("Unable to open %s\n", fn);
	return NULL;
    }

    for (cp = rbID, i = sizeof rbID; i--; cp++) {
	if (RbFile_readByte(rb) != uc(cp,0))
	    break;
    }
    if (i >= 0 || RbFile_readInt16(rb) != RB_FILE_VERSION) {
	fclose(rb->fp);
	Mem_free(rb);
	RbError_warn("%s is not in the proper format\n", fn);
	return NULL;
    }

    RbFile_seek(rb, 24, 0);
    rb->tocOffset = RbFile_readInt32(rb);
    rb->fileSize = RbFile_readInt32(rb);
    fstat(fileno(rb->fp), &sb);
    if (sb.st_size != rb->fileSize) {
	fclose(rb->fp);
	Mem_free(rb);
	RbError_warn("%s's size does not match the header\n", fn);
	return NULL;
    }

    RbFile_seek(rb, rb->tocOffset, 0);
    for (i = RbFile_readInt32(rb); i--; ) {
	toc = RbFile_readToc(rb);
	if (toc->flags & RB_TOCFLAG_INFOPAGE)
	    addToC(rb, toc);
	else {
	    switch (toc->type) {
	      case RB_PAGETYPE_HIDX:
	      case RB_PAGETYPE_HKEY:
		if (!(openFlags & RB_OPENFLAG_INCLUDE_HIDDEN)) {
		    Mem_free(toc);
		    continue;
		}
		break;
	      case RB_PAGETYPE_IMAGE:
		 if (!(openFlags & RB_OPENFLAG_INCLUDE_IMAGES)) {
		     Mem_free(toc);
		     continue;
		 }
		 break;
	      case RB_PAGETYPE_AUDIO:
		 if (!(openFlags & RB_OPENFLAG_INCLUDE_AUDIO)) {
		     Mem_free(toc);
		     continue;
		 }
		 break;
	    }
	    addToC(rb, toc);
	}
    }
    if (!rb->tocHead) {
	fclose(rb->fp);
	Mem_free(rb);
	RbError_warn("%s has no contents", fn);
	return NULL;
    }
    if (!(rb->tocHead->flags & RB_TOCFLAG_INFOPAGE)) {
	addToC(rb, ToC_new("bogus.info", RB_PAGETYPE_INFO, 0, 0,
			   RB_TOCFLAG_INFOPAGE, 0));
    }
    rb->fileName = Mem_strdup(fn);

    if (openFlags & RB_OPENFLAG_UNJOIN) {
	ToC *tail, *prev = rb->tocHead, *unjoinPrev = NULL;
	while ((toc = prev->next) != NULL) {
	    char *tn = toc->name;
	    int ord = 1;
	    if ((strEQ(tn, "joined.html")
	      || sscanf(tn, "j%d.html", &ord) == 1)) {
		toc->ord = ord;
		prev->next = toc->next;
		toc->next = NULL;
		if (unjoinPrev)
		    unjoinPrev->next = toc;
		else
		    rb->tocUnjoin = toc;
		unjoinPrev = toc;
	    }
	    else
		prev = toc;
	}
	prev = rb->tocHead->next;
	rb->tocHead->next = NULL;
	tail = rb->tocTail;
	rb->tocTail = rb->tocHead;
	for (toc = rb->tocUnjoin; toc; toc = toc->next) {
	    rb->joinOrd = toc->ord;
	    RbFile_readPage(rb, toc, rb, unjoinPushFunc);
	    unjoinPushFunc(rb, NULL, 0);
	}
	if (prev) {
	    rb->tocTail->next = prev;
	    rb->tocTail = tail;
	}
    }

    return rb;
}

RbFile *
RbFile_create(const char *fn, int32 maxTocCnt)
{
    time_t now = time((time_t*)NULL);
    struct tm *tp;
    char *cp;
    int i;
    RbFile *rb = Mem_calloc(1, sizeof *rb);

#ifdef WIN32
    if (!getenv("TZ"))
	putenv("TZ=GMT0");
#endif
    tp = localtime(&now);
    rb->newFileName = Mem_alloc(strlen(fn)+5);
    sprintf(rb->newFileName, "%s.new", fn);
    if (!(rb->fp = fopen(rb->newFileName, "wb+"))) {
	Mem_free(rb->newFileName);
	Mem_free(rb);
	RbError_warn("Unable to write %s\n", rb->newFileName);
	return NULL;
    }
    rb->tocOffset = RB_TOC_POS;
    rb->fileName = Mem_strdup(fn);

    for (cp = rbID, i = sizeof rbID; i--; cp++)
	RbFile_writeByte(rb, *cp);
    RbFile_writeInt16(rb, RB_FILE_VERSION);
    RbFile_writeString(rb, "NUVO");
    RbFile_writePad(rb, 4, 0);

    /* Set creation date */
    RbFile_writeInt16(rb, tp->tm_year + 1900);
    RbFile_writeByte(rb, tp->tm_mon+1);
    RbFile_writeByte(rb, tp->tm_mday);

    RbFile_writePad(rb, 6, 0);
    RbFile_writeInt32(rb, RB_TOC_POS);
    RbFile_writeInt32(rb, 0);

    /* Pad header out to ToC addrss (0x128), and then pad out the
    ** entire ToC area so that we can start appending data. */
    if (maxTocCnt <= 0)
	maxTocCnt = RB_MAX_TOC_CNT;
    RbFile_writePad(rb, RB_TOC_POS-ftell(rb->fp)+4 + maxTocCnt*RB_TOC_SIZE, 0);
    rb->maxTocCnt = maxTocCnt;

    return rb;
}

void
RbFile_close(RbFile *me)
{
    if (me->fp) {
	ToC *toc, *tocNext;
	if (me->newFileName)
	    finishCreate(me);
	else
	    fclose(me->fp);
	for (toc = me->tocHead; toc; toc = tocNext) {
	    tocNext = toc->next;
	    Mem_free(toc);
	}
	Mem_free(me->fileName);
	Mem_free(me);
    }
}

static void
finishCreate(RbFile *me)
{
    ToC *toc;
    int32 posDiff = (me->maxTocCnt - me->tocCnt) * RB_TOC_SIZE;

    if (posDiff < 0 || !me->tocCnt) {
	fclose(me->fp);
	unlink(me->newFileName);
	me->newFileName = NULL;
	if (posDiff < 0)
	    RbError_exit("Overflowed maximum file count -- aborting.\n");
	else
	    RbError_exit("Book is empty -- aborting.\n");
    }

    /* Pad the EOF with 20 Ctrl-As */
    RbFile_writePad(me, 20, 1);

    /* If we padded the file's ToC, re-write the whole file to compact it. */
    if (posDiff > 0) {
	FILE *fp_in = fopen(me->newFileName, "rb");
	int len;

	if (fp_in) {
	    char buf[1024];
	    int bigbuflen = 32*1024;
	    char *bigbuf = Mem_alloc(bigbuflen);
	    if (!bigbuf) {
		bigbuf = buf;
		bigbuflen = sizeof buf;
	    }
	    fseek(fp_in, me->tocOffset + 4 + me->maxTocCnt * RB_TOC_SIZE, 0);
	    RbFile_seek(me, me->tocOffset + 4 + me->tocCnt * RB_TOC_SIZE, 0);
	    while ((len = fread(bigbuf, 1, bigbuflen, fp_in)) > 0)
		RbFile_writeBuf(me, bigbuf, len);
	    fclose(fp_in);
	    if (bigbuflen > sizeof buf)
		Mem_free(bigbuf);
	}
	else {
	    RbError_warn("Warning: unable to reopen %s to make it more compact\n",
			 me->newFileName);
	}
	me->fileSize = ftell(me->fp);
#ifdef HAVE_FTRUNCATE
	ftruncate(fileno(me->fp), me->fileSize);
#else
# ifdef HAVE_CHSIZE
	chsize(fileno(me->fp), me->fileSize);
# else
#  ifdef F_CHSIZE
	fcntl(fileno(me->fp), F_CHSIZE, me->fileSize);
#  else
#   error No known way to truncate a file!
#  endif
# endif
#endif
    }
    else
	me->fileSize = ftell(me->fp);

    /* Fill in the length of the file in the header */
    RbFile_seek(me, RB_FILELEN_POS, 0);
    RbFile_writeInt32(me, me->fileSize);

    /* Write out the Table of Contents */
    RbFile_seek(me, me->tocOffset, 0);
    RbFile_writeInt32(me, me->tocCnt);
    for (toc = me->tocHead; toc; toc = toc->next) {
	toc->pos -= posDiff;
	RbFile_writeToc(me, toc);
    }

    fclose(me->fp);

    unlink(me->fileName);
    rename(me->newFileName, me->fileName);
    Mem_free(me->newFileName);
    me->newFileName = NULL;
}

char *
RbFile_getLibDir(RbFile *me, const char *libDir, const char *rocketID)
{
    MBuf *tmp = MBuf_new(128, 0);
    struct stat sb;
    char *dir;

    if (!libDir && (libDir = getenv("RB_LIB_DIR")) == NULL) {
#ifdef DEFAULT_LIB_DIR
	libDir = DEFAULT_LIB_DIR;
#else
# ifdef DOS_FILENAMES
	libDir = "C:/Program Files/NuvoMedia/RocketLibrarian/Library/books";
# else
	libDir = "/mnt/DOS_hda1/Program Files/NuvoMedia/RocketLibrarian/Library/books";
# endif
#endif
    }
    if (!*libDir) {
	libDir = me->fileName;
	rocketID = NULL;
    }
    else if (!rocketID && (!(rocketID = getenv("RB_ID")))) {
	/*// try to find the first ID in the current "libDir"? */
    }
    else if (!*rocketID)
	rocketID = NULL;

    if (rocketID) {
	const char *cp = rocketID;
	while (ISALPHA(*cp)) cp++;
	if (!ISDIGIT(*cp))
	    cp = rocketID;
	else
	    while (ISDIGIT(*cp)) cp++;
	if (*cp) {
	    RbError_warn("`%s' does not look like a valid rocket-ID.\n",
			 rocketID);
	}
    }

    /* It's OK if rocketID is NULL -- it properly terminates this early: */
    MBuf_vwrite(tmp, libDir,-1, "/",1, rocketID,-1, "/",1, NULL);

    dir = MBuf_toBuffer(tmp, NULL);

    if (libDir != me->fileName) {
	sb.st_size = -1;
	if (stat(libDir, &sb) < 0 || !S_ISDIR(sb.st_mode)) {
	    if (sb.st_size <= 0 || !rbIsRbSuf(rbGetFileSuffix(libDir)))
		RbError_warn("Library-dir `%s' is not a directory.", libDir);
	}
	else if (rocketID) {
	    char *end = dir + strlen(dir) - 1;
	    *end = '\0';
	    if (stat(dir, &sb) < 0 || !S_ISDIR(sb.st_mode)) {
		RbError_warn("Rocket-ID `%s' does not have a directory in `%s'.\n",
			     rocketID, libDir);
	    }
	    *end = '/';
	}
    }

    return dir;
}

static int
readLine(FILE *fp, MBuf *mb, bool keepLineEndings)
{
    int ch;

    MBuf_truncate(mb, 0);
    while ((ch = getc(fp)) != EOF) {
	if (ch == '\r') {
	    if (!keepLineEndings)
		continue;
	}
	else if (ch == '\n') {
	    if (keepLineEndings)
		MBuf_putc(mb, ch);
	    return mb->totalLen;
	}
	MBuf_putc(mb, ch);
    }
    return mb->totalLen? mb->totalLen : EOF;
}

void
RbFile_readMarkup(RbFile *me, const char *path)
{
    char *fn = getBaseName4MarkupFile(me, path);
    char *cp, *end = fn + strlen(fn) - 1;
    char *tname;
    RbMarkup *mu;
    MBuf *buf = MBuf_new(256, 0);
    ToC *toc;
    FILE *fp;

    /* First: try to open the new, 1-file format */
    end[-1] = 'a'; *end = 'n';
    if ((fp = fopen(fn, "rb")) != NULL && readLine(fp, buf, false) > 0
     && strEQ(MBuf_dataPtrAt(buf, 0, NULL), "<?xml version=\"1.0\"?>")) {
	char type = '\0';
	/* While I could use libxml here, I am reluctant to do that
	 * because it would make programs that don't yet need to use it
	 * (like rbburst) dependent on it.  I may change my mind in the
	 * future, though... */
	while (readLine(fp, buf, false) > 0) {
	    char *bp = MBuf_dataPtrAt(buf, 0, NULL);
	    while (ISSPACE(*bp)) bp++;
	    if (strnEQ(bp, "<bookmark ", 10)) {
		bp += 10;
		type = RB_MARKUP_BOOKMARK;
	    }
	    else if (strnEQ(bp, "<note ", 6)) {
		bp += 6;
		type = RB_MARKUP_NOTE2;
	    }
	    else if (strnEQ(bp, "<underline ", 11)) {
		bp += 11;
		type = RB_MARKUP_UNDERLINE2;
	    }
	    else
		continue;
	    toc = NULL;
	    if (strnEQ(bp, "file=\"", 6)
	     && (cp = strchr(bp+=6, '"')) != NULL) {
		char *body = bp;
		*cp = '\0';
		bp = cp + 2;
		if (strnEQ(bp, "byte=\"", 6)) {
		    bp += 6;
		    mu = Mem_calloc(1, sizeof *mu);
		    mu->type = type;
		    mu->start = atoi(bp);
		    while (ISDIGIT(*bp)) bp++;
		    while (*bp == '"') bp++;
		    while (*bp == ' ') bp++;
		    if (strnEQ(bp, "endbyte=\"", 9))
			mu->end = atoi(bp+9);
		    if ((bp = strchr(bp, '>')) != NULL) {
			MBuf *tmp = MBuf_new(64, 0);
			bp++;
			while (1) {
			    char ch;
			    while ((ch = *bp++) != '\0') {
				if (ch == '<')
				    goto double_break;
				if (ch == '&') {
				    if (strnEQ(bp, "amp;", 4)) {
					bp += 4;
					MBuf_putc(tmp, '&');
				    }
				    else if (strnEQ(bp, "lt;", 3)) {
					bp += 3;
					MBuf_putc(tmp, '<');
				    }
				    else if (strnEQ(bp, "gt;", 3)) {
					bp += 3;
					MBuf_putc(tmp, '>');
				    }
				    else
					MBuf_putc(tmp, ch);
				}
				else
				    MBuf_putc(tmp, ch);
			    }
			    MBuf_putc(tmp, '\n');
			    if (readLine(fp, buf, false) < 0)
				break;
			    bp = MBuf_dataPtrAt(buf, 0, NULL);
			}
		      double_break:
			mu->text = MBuf_toBuffer(tmp, NULL);
			if (mu->end && !*mu->text) {
			    Mem_free(mu->text);
			    mu->text = NULL;
			}
			toc = RbFile_findWithPos(me, body, mu->start);
			if (toc) {
			    if (toc->flags & RB_TOCFLAG_UNJOINED_FRAGMENT) {
				mu->start -= toc->pos;
				if (mu->end)
				    mu->end -= toc->pos;
			    }
			    addMarkup(me, toc, mu);
			}
			else
			    Mem_free(mu);
		    }
		}
	    }
	    if (!toc) {
		RbError_warn("The annotation file `%s' is corrupt.\n", fn);
		break;
	    }
	}
	MBuf_delete(buf);
	return;
    }

    /* Second: try to open the old, 3-file format */
    end[-1] = 'r';
    tname = "bookmark";
    for (*end = RB_MARKUP_BOOKMARK; ; *end = RB_MARKUP_NOTE, tname = "note") {
	if ((fp = fopen(fn, "rb")) != NULL && readLine(fp, buf, true) > 0
	 && strEQ(MBuf_dataPtrAt(buf, 0, NULL), "REV=1\r\n")) {
	    while (readLine(fp, buf, true) > 0) {
		char *bp = MBuf_dataPtrAt(buf, 0, NULL);
		if (*bp == '\r')
		    continue;
		toc = NULL;
		if (strnEQ(bp, "Byte=", 5)) {
		    mu = Mem_calloc(1, sizeof *mu);
		    mu->type = *end;
		    mu->start = hexValue(bp+5) + (*end == RB_MARKUP_NOTE);
		    if (readLine(fp, buf, true) > 0
		     && strnEQ(MBuf_dataPtrAt(buf, 0, NULL), "Menu=", 5)) {
			MBuf *tmp = MBuf_new(64, 0);
			MBuf_puts(tmp, bp+5);
			while (1) {
			    int len = tmp->totalLen - 2;
			    if (len >= 0) {
				cp = MBuf_dataPtrAt(tmp, len, NULL);
				if (*cp == '\r') {
				    MBuf_truncate(tmp, len);
				    break;
				}
			    }
			    if (readLine(fp, buf, true) <= 0)
				break;
			    MBuf_puts(tmp, MBuf_dataPtrAt(buf, 0, NULL));
			}
			if (readLine(fp, buf, true) > 0
			 && strnEQ(bp = MBuf_dataPtrAt(buf,0,NULL),"Body=",5)) {
			    cp = bp + strlen(bp) - 2;
			    if (*cp == '\r')
				*cp = '\0';
			    toc = RbFile_findWithPos(me, bp+5, mu->start);
			}
			else
			    Mem_free(mu->text); //?? $$ !! Huh? XXX
			mu->text = MBuf_toBuffer(tmp, NULL);
		    }
		    if (toc) {
			if (toc->flags & RB_TOCFLAG_UNJOINED_FRAGMENT) {
			    mu->start -= toc->pos;
			    if (mu->end)
				mu->end -= toc->pos;
			}
			addMarkup(me, toc, mu);
		    }
		    else
			Mem_free(mu);
		}
		if (!toc) {
		    RbError_warn("The %s file `%s' is corrupt.\n", tname, fn);
		    break;
		}
	    }
	}
	if (*end == RB_MARKUP_NOTE)
	    break;
    }
    *end = RB_MARKUP_UNDERLINE;
    if ((fp = fopen(fn, "rb")) != NULL && readLine(fp, buf, true) > 0
     && strEQ(MBuf_dataPtrAt(buf, 0, NULL), "REV=2\r\n")) {
	while (readLine(fp, buf, true) > 0) {
	    char *bp = MBuf_dataPtrAt(buf, 0, NULL);
	    toc = NULL;
	    if (strnEQ(bp, "START=", 6)) {
		cp = bp + strlen(bp) - 2;
		if (*cp == '\r')
		    *cp = '\0';
		mu = Mem_calloc(1, sizeof *mu);
		mu->type = RB_MARKUP_UNDERLINE;
		mu->start = hexValue(bp+6);
		if ((cp = strchr(bp+6, ' ')) != NULL
		 && strnEQ(cp+1, "END=", 4)) {
		    mu->end = hexValue(cp+1+4);
		    if ((cp = strchr(cp+1+4, ' ')) != NULL
		     && strnEQ(cp+1, "BODY=", 5)) {
			toc = RbFile_findWithPos(me, cp+1+5, mu->start);
			if (toc->flags & RB_TOCFLAG_UNJOINED_FRAGMENT) {
			    mu->start -= toc->pos;
			    if (mu->end)
				mu->end -= toc->pos;
			}
		    }
		}
		if (toc)
		    addMarkup(me, toc, mu);
		else
		    Mem_free(mu);
	    }
	    if (!toc) {
		RbError_warn("The underline file `%s' is corrupt.\n", fn);
		break;
	    }
	}
    }
    MBuf_delete(buf);
}

const char *
RbFile_writeMarkup(RbFile *me, const char *path)
{
    char *fn = getBaseName4MarkupFile(me, path);
    char *cp, *end = fn + strlen(fn) - 1;
    RbMarkup *mu;
    ToC *toc;
    FILE *fp;
    int cnt;

    cnt = me->bookmarkCnt;
    cp = "Unable to write bookmark file";
    for (*end = RB_MARKUP_BOOKMARK; ; *end = RB_MARKUP_NOTE) {
	if (cnt) {
	    if (!(fp = fopen(fn, "wb")))
		return cp;
	    fprintf(fp, "REV=1\r\n");
	    for (toc = me->tocHead; toc; toc = toc->next) {
		for (mu = toc->markupHead; mu; mu = mu->next) {
		    if (mu->type != RB_MARKUP_BOOKMARK)
			continue;
		    fprintf(fp, "\r\nByte=%x\r\nMenu=%s\r\nBody=%s\r\n",
			    mu->start - (*end == RB_MARKUP_NOTE), mu->text,
			    toc->name);
		}
	    }
	}
	if (*end == RB_MARKUP_NOTE)
	    break;
	cnt = me->noteCnt;
	cp = "Unable to write note file";
    }
    *end = RB_MARKUP_UNDERLINE;
    if (me->underlineCnt) {
	if (!(fp = fopen(fn, "wb")))
	    return "Unable to write underline file";
	fprintf(fp, "REV=2\r\n");
	for (toc = me->tocHead; toc; toc = toc->next) {
	    for (mu = toc->markupHead; mu; mu = mu->next) {
		if (mu->type != RB_MARKUP_UNDERLINE)
		    continue;
		fprintf(fp, "START=%x END=%x BODY=%s\r\n",
			mu->start, mu->end, toc->name);
	    }
	}
    }

    return NULL;
}

/* We support having "path" be the actual name of the .rb file we want to
 * use as a base-name (which only works if the markup files are in the
 * same dir and using the same base-name). */
static char *
getBaseName4MarkupFile(RbFile *me, const char *path)
{
    MBuf *tmp = MBuf_new(128, 0);
    char *p, *cp;
    MBuf_puts(tmp, path);
    p = MBuf_dataPtr(tmp, NULL);
#ifdef DOS_FILENAMES
    for (cp = p; *cp; cp++) {
	if (*cp == '\\')
	    *cp = '/';
    }
    cp--;
#else
    cp = p + strlen(p) - 1;
#endif
    if (cp - p > 3 && cp[-3] == '.' && cp[-2] == 'r' && cp[-1] == 'b')
	*cp = '\0';
    else {
	RbInfoHash *ih = RbInfoHash_newFromRbFile(me);
	if ((cp = RbInfoHash_fetch(ih, "URL")) == NULL) {
	    if (!(cp = strrchr(me->fileName, '/')))
		cp = me->fileName;
	    else
		cp++;
	    MBuf_puts(tmp, cp);
	}
	else {
	    if (strnEQ(cp, "ebook:", 6))
		cp += 6;
	    MBuf_vwrite(tmp, cp,-1, ".rb",3, NULL);
	}
	RbInfoHash_delete(ih);
    }
    return MBuf_toBuffer(tmp, NULL);
}

static void
addMarkup(RbFile *me, ToC *toc, RbMarkup *add)
{
    RbMarkup *mu;
    if (!(mu = toc->markupHead) || add->start < mu->start) {
	add->next = toc->markupHead;
	toc->markupHead = add;
    }
    else {
	RbMarkup *prev;
	while (1) {
	    prev = mu;
	    if ((mu = mu->next) == NULL)
		break;
	    if (add->start > mu->start)
		continue;
	    switch (add->type) {
	      case RB_MARKUP_UNDERLINE:
	      case RB_MARKUP_UNDERLINE2:
		if (add->start == mu->start)
		    continue;
		break;
	    }
	    break;
	}
	add->next = mu;
	prev->next = add;
    }

    switch (add->type) {
      case RB_MARKUP_BOOKMARK:
	me->bookmarkCnt++;
	toc->bookmarkCnt++;
	break;
      case RB_MARKUP_UNDERLINE:
      case RB_MARKUP_UNDERLINE2:
	me->underlineCnt++;
	toc->underlineCnt++;
	break;
      case RB_MARKUP_NOTE:
      case RB_MARKUP_NOTE2:
	me->noteCnt++;
	toc->noteCnt++;
	break;
    }
}

static unsigned int
hexValue(const char *s)
{
    unsigned int val = 0;
    char ch;

    while (ISDIGIT(ch = *s) || (ISALPHA(ch) && (ch = (ch&0x1F)+9+'0') < 16+'0'))
	val = val * 16 + ch - '0', s++;

    return val;
}

ToC *
RbFile_find(RbFile *me, const char *tocName)
{
    ToC *toc;
    for (toc = me->tocHead; toc; toc = toc->next) {
	if (strEQ(toc->name, tocName))
	    return toc;
    }
    return NULL;
}

ToC *
RbFile_findWithPos(RbFile *me, const char *tocName, int pos)
{
    ToC *toc;
    for (toc = me->tocUnjoin; toc; toc = toc->next) {
	if (strEQ(toc->name, tocName)) {
	    int ord = toc->ord;
	    for (toc = me->tocHead; toc; toc = toc->next) {
		if (toc->ord == ord
		 && pos >= toc->pos && pos-toc->pos <= toc->length)
		    return toc;
	    }
	    break;
	}
    }
    return RbFile_find(me, tocName);
}

int
RbFile_readPage(RbFile *me, ToC *toc, void *userPtr,
		void (*pushFunc)(void *userPtr, const char *bp, int len))
{
    char mybuf[4096+128];
    int32 startAt, putLen;
    MBuf *mb = NULL;
    if (!pushFunc) {
	mb = userPtr;
	pushFunc = bufPushFunc;
    }
    if (toc->flags & RB_TOCFLAG_UNJOINED_FRAGMENT) {
	int ord = toc->ord;
	startAt = toc->pos;
	putLen = toc->length;
	for (toc = me->tocUnjoin; toc->ord != ord; toc = toc->next) {}
    }
    else
	startAt = putLen = 0;
    RbFile_seek(me, toc->pos, 0);
    if ((toc->flags & (RB_TOCFLAG_DEFLATED | RB_TOCFLAG_ENCRYPTED)) == RB_TOCFLAG_DEFLATED) {
	int32 i, len, seekBump, pieces = RbFile_readInt32(me);
	char compressedData[4096+128];
	int32 *plen = Mem_alloc(pieces * sizeof (int32));
	z_stream zs;

	len = RbFile_readInt32(me);
	if (!putLen)
	    putLen = len;
	if (mb)
	    MBuf_setUpcomingLength(mb, putLen);
	seekBump = 0;
	for (i = 0; i < pieces; ) {
	    if (startAt >= 4096) {
		startAt -= 4096;
		pieces--;
		seekBump += RbFile_readInt32(me);
	    }
	    else
		plen[i++] = RbFile_readInt32(me);
	}
	if (seekBump)
	    RbFile_seek(me, seekBump, 1);
	zs.zalloc = Z_NULL;
	zs.zfree = Z_NULL;
	zs.opaque = 0;
	zs.total_in = 0;
	for (i = 0; i < pieces; i++) {
	    RbFile_readBuf(me, compressedData, plen[i]);
	    zs.next_out = (Bytef*)mybuf;
	    zs.avail_out = sizeof mybuf;
	    zs.total_out = 0;
	    zs.next_in = (Bytef*)compressedData;
	    zs.avail_in = plen[i];
	    inflateInit2(&zs, 13);
	    inflate(&zs, Z_FINISH);
	    inflateEnd(&zs);
	    mybuf[zs.total_out] = '\0';
	    len = zs.total_out - startAt;
	    if ((putLen -= len) < 0)
		len += putLen;
	    pushFunc(userPtr, mybuf + startAt, len);
	    if (putLen <= 0)
		break;
	    startAt = 0;
	}
	Mem_free(plen);
    }
    else {
	int cnt;
	if (!putLen)
	    putLen = toc->length;
	else if (startAt)
	    RbFile_seek(me, startAt, 1);
	for ( ; putLen > 0; putLen -= cnt) {
	    cnt = putLen > sizeof mybuf? sizeof mybuf : putLen;
	    if ((cnt = RbFile_readBuf(me, mybuf, cnt)) <= 0)
		break;
	    pushFunc(userPtr, mybuf, cnt);
	}
    }
    return 0;
}

static void
addToC(RbFile *me, ToC *toc)
{
    if (toc->flags & RB_TOCFLAG_INFOPAGE) {
	toc->next = me->tocHead;
	if (!me->tocHead)
	    me->tocTail = toc;
	me->tocHead = toc;
    }
    else {
	if (me->tocTail)
	    me->tocTail->next = toc;
	else
	    me->tocHead = toc;
	me->tocTail = toc;
    }
    me->tocCnt++;
}

/* This also adds a ToC entry for the page to the list */
void
RbFile_writePage(RbFile *me, const char *tn, int type, int32 flags, MBuf *mb)
{
    int32 filePos;
    int totalSize = mb->totalLen;

    if (!me->newFileName) {
	RbError_exit("Unable to write %s:\n"
		     "file is was only opened for reading.\n", me->fileName);
    }

    filePos = RbFile_filePos(me);
    if (totalSize < 128)
	flags  &= ~RB_TOCFLAG_DEFLATED;

    if (flags & RB_TOCFLAG_DEFLATED) {
	int32 i, plenPos, todo, pieces = (totalSize + 4095) / 4096;
	int32 *plen = Mem_alloc(pieces * sizeof (int32));
	char compressedData[4096+128];
	z_stream zs;
	RbFile_writeInt32(me, pieces);
	RbFile_writeInt32(me, totalSize);
	plenPos = RbFile_filePos(me);
	RbFile_writePad(me, pieces * sizeof (int32), 0);
	zs.zalloc = Z_NULL;
	zs.zfree = Z_NULL;
	zs.opaque = 0;
	zs.total_in = 0;
	MBuf_setReadPos(mb, 0, 0);
	for (i = 0; i < pieces; i++) {
	    todo = totalSize < 4096? totalSize : 4096;
	    deflateInit2(&zs, Z_BEST_COMPRESSION, Z_DEFLATED, 13, 8,
			 Z_DEFAULT_STRATEGY);
	    zs.next_out = (Bytef*)compressedData;
	    zs.avail_out = sizeof compressedData;
	    zs.total_out = 0;
	    while (1) {
		int cnt = todo;
		char *bp;
		if (!(bp = MBuf_dataPtr(mb, &cnt))) {
		    todo = 0;
		    break;
		}
		totalSize -= cnt;
		zs.next_in = (Bytef*)bp;
		if (cnt == todo)
		    break;
		zs.avail_in = cnt;
		deflate(&zs, Z_NO_FLUSH);
		todo -= cnt;
	    }
	    zs.avail_in = todo;
	    deflate(&zs, Z_FINISH);
	    plen[i] = zs.total_out;
	    RbFile_writeBuf(me, compressedData, zs.total_out);
	    deflateEnd(&zs);
	}
	RbFile_seek(me, plenPos, 0);
	for (i = 0; i < pieces; i++)
	    RbFile_writeInt32(me, plen[i]);
	Mem_free(plen);
	RbFile_seek(me, 0, 2);
    }
    else {
	MBuf_setReadPos(mb, 0, 0);
	while (1) {
	    int cnt = mb->totalLen;
	    char *bp = MBuf_dataPtr(mb, &cnt);
	    if (bp == NULL)
		break;
	    RbFile_writeBuf(me, bp, cnt);
	}
    }

    totalSize = RbFile_filePos(me) - filePos;
    addToC(me, ToC_new(tn, type, totalSize, filePos, flags & 0xFF, 0));
}

int32
RbFile_uncompressedLength(RbFile *me, ToC *toc)
{
    if ((toc->flags & (RB_TOCFLAG_DEFLATED | RB_TOCFLAG_ENCRYPTED)) == RB_TOCFLAG_DEFLATED) {
	RbFile_seek(me, toc->pos + 4, 0);
	return RbFile_readInt32(me);
    }
    return toc->length;
}

char *
RbFile_getTypeName(int type)
{
    if (type < 0 || type > RB_PAGETYPE_IGNORE)
	return NULL;
    return PageTypeName[type];
}

void
bufPushFunc(void *userPtr, const char *bp, int len)
{
    MBuf_write((MBuf*)userPtr, bp, len);
}

static void
unjoinPushFunc(void *userPtr, const char *bp, int len)
{
    static char matchStr[] = "\n<HR SIZE=0><A NAME=\"";
    static char *matchPos = NULL;
    static ToC *prevToc = NULL;
    static int filePos = 0;
    static MBuf *tmp;
    RbFile *rb = userPtr;
    const char *mp;

    if (!bp) {
	if (prevToc) {
	    prevToc->length = filePos - prevToc->pos;
	    prevToc = NULL;
	}
	matchPos = NULL;
	filePos = 0;
	return;
    }
    if (!tmp)
	tmp = MBuf_new(64, 0);
    mp = bp;
    if (!prevToc) {
	while ((mp = strchr(bp, '<')) != NULL) {
	    if (strnEQ(++mp, "BODY>", 5)) {
		mp += 5;
		matchPos = matchStr + 12;
		break;
	    }
	    filePos += mp - bp;
	    bp = mp;
	}
    }
    while (1) {
	if (!matchPos) {
	    if ((mp = strchr(bp, '\n')) == NULL) {
		filePos += strlen(bp);
		return;
	    }
	    matchPos = matchStr;
	    MBuf_truncate(tmp, 0);
	}
	while (*mp == *matchPos && *matchPos) mp++, matchPos++;
	filePos += mp - bp;
	bp = mp;
	if (*matchPos == '\0') {
	    while (*mp && *mp != RB_JOIN_NAME_SEP && *mp != '"')
		MBuf_putc(tmp, *mp++);
	    if (*mp && *mp != RB_JOIN_NAME_SEP && tmp->totalLen < 16) {
		MBuf_puts(tmp, ".html");
		if (prevToc) {
		    prevToc->length = filePos - prevToc->pos
				    - STATICLEN(matchStr);
		}
		while (*mp++ != '\n') {}
		filePos += mp - bp;
		bp = mp;
		prevToc = ToC_new(MBuf_dataPtrAt(tmp,0,NULL), RB_PAGETYPE_HTML,
				  0, prevToc? filePos : 0,
				  RB_TOCFLAG_UNJOINED_FRAGMENT | RB_TOCFLAG_DEFLATED,
				  rb->joinOrd);
		addToC(rb, prevToc);
	    }
	}
	else if (!*mp)
	    return;
	matchPos = NULL;
    }
}

int
RbFile_readByte(RbFile *me)
{
    return fgetc(me->fp);
}

int
RbFile_readInt16(RbFile *me)
{
    int i = getc(me->fp);
    i += getc(me->fp) * 0x100;
    return i;
}

int32
RbFile_readInt32(RbFile *me)
{
    int32 i = getc(me->fp);
    i += getc(me->fp) << 8;
    i += getc(me->fp) << 16;
    i += getc(me->fp) << 24;
    return i;
}

int
RbFile_readBuf(RbFile *me, char *bp, int len)
{
    return fread(bp, 1, len, me->fp);
}

ToC *
RbFile_readToc(RbFile *me)
{
    char tn[RB_TOC_NAMELEN+1];
    int32 len, pos, flags;
    RbFile_readBuf(me, tn, RB_TOC_NAMELEN);
    len = RbFile_readInt32(me);
    pos = RbFile_readInt32(me);
    flags = RbFile_readInt32(me);
    return ToC_new(tn, RB_PAGETYPE_UNKNOWN, len, pos, flags, 0);
}

int
RbFile_seek(RbFile *me, int32 off, int from)
{
    return fseek(me->fp, off, from);
}

int32
RbFile_filePos(RbFile *me)
{
    return ftell(me->fp);
}

int
RbFile_writeByte(RbFile *me, char b)
{
    return fputc(b, me->fp);
}

int
RbFile_writePad(RbFile *me, int cnt, int pad)
{
    while (cnt-- > 0)
	fputc(pad, me->fp);
    return 0;
}

int
RbFile_writeInt16(RbFile *me, int i)
{
    fputc(i & 0xFF, me->fp);
    fputc((i >> 8) & 0xFF, me->fp);
    return 0;
}

int
RbFile_writeInt32(RbFile *me, int32 i)
{
    fputc(i & 0xFF, me->fp);
    fputc((i >> 8) & 0xFF, me->fp);
    fputc((i >> 16) & 0xFF, me->fp);
    fputc((i >> 24) & 0xFF, me->fp);
    return 0;
}

int
RbFile_writeString(RbFile *me, const char *str)
{
    return fputs(str, me->fp);
}

int
RbFile_writeBuf(RbFile *me, const char *bp, int32 len)
{
    return fwrite(bp, 1, len, me->fp);
}

int
RbFile_writeToc(RbFile *me, ToC *toc)
{
    RbFile_writeBuf(me, toc->name, RB_TOC_NAMELEN);
    RbFile_writeInt32(me, toc->length);
    RbFile_writeInt32(me, toc->pos);
    RbFile_writeInt32(me, toc->flags & 0xFF);
    return 0;
}

const char *
RbFile_getFileName(RbFile *me)
{
    return me->fileName;
}

const char *
RbFile_getNewFileName(RbFile *me)
{
    return me->newFileName;
}

ToC *
RbFile_getTocHead(RbFile *me)
{
    return me->tocHead;
}

int
RbFile_getTocCnt(RbFile *me)
{
    return me->tocCnt;
}

int
RbFile_getMaxTocCnt(RbFile *me)
{
    return me->maxTocCnt;
}

int
RbFile_getFileSize(RbFile *me)
{
    return me->fileSize;
}

unsigned int
RbFile_getUnderlineCnt(RbFile *me)
{
    return me->underlineCnt;
}

unsigned int
RbFile_getNoteCnt(RbFile *me)
{
    return me->noteCnt;
}

unsigned int
RbFile_getBookmarkCnt(RbFile *me)
{
    return me->bookmarkCnt;
}

ToC *
ToC_new(const char *tn, int type, int32 len, int32 pos, int32 flags, int ord)
{
    ToC *toc = Mem_calloc(1, sizeof (ToC));
    strcpy(toc->name, tn);
    toc->length = len;
    toc->pos = pos;
    toc->flags = flags;
    toc->ord = ord;
    if (type == RB_PAGETYPE_UNKNOWN)
	type = rbFileNameToPageType(tn);
    toc->type = type;
    return toc;
}

const char *
ToC_getName(ToC *me)
{
    return me->name;
}

int
ToC_getType(ToC *me)
{
    return me->type;
}

int32
ToC_getLength(ToC *me)
{
    return me->length;
}

int32
ToC_getPos(ToC *me)
{
    return me->pos;
}

int32
ToC_getFlags(ToC *me)
{
    return me->flags;
}

RbMarkup *
ToC_getMarkupHead(ToC *me)
{
    return me->markupHead;
}

int
ToC_getUnderlineCnt(ToC *me)
{
    return me->underlineCnt;
}

int
ToC_getNoteCnt(ToC *me)
{
    return me->noteCnt;
}

int
ToC_getBookmarkCnt(ToC *me)
{
    return me->bookmarkCnt;
}

ToC *
ToC_getNext(ToC *me)
{
    return me->next;
}

char
RbMarkup_getType(RbMarkup *me)
{
    return me->type;
}

unsigned int
RbMarkup_getStart(RbMarkup *me)
{
    return me->start;
}

unsigned int
RbMarkup_getEnd(RbMarkup *me)
{
    return me->end;
}

const char *
RbMarkup_getText(RbMarkup *me)
{
    return me->text;
}

RbMarkup *
RbMarkup_getNext(RbMarkup *me)
{
    return me->next;
}

void
RbMarkup_fixMarkups(ToC *toc, const char *pageBuf)
{
    const char *cp;
    RbMarkup *mu, *prev = NULL;

    for (mu = toc->markupHead; mu; prev = mu, mu = mu->next) {
	while (mu->type == RB_MARKUP_NOTE2) {
	    RbMarkup *next, *newMu = NULL;
	    cp = pageBuf + mu->start;
	    while (!ISSPACE(*cp) && *cp != '<') cp++, mu->start++;
	    while ((next = mu->next) != NULL && mu->start > next->start) {
		mu->next = next->next;
		next->next = mu;
		if (!newMu)
		    newMu = next;
		if (!prev)
		    toc->markupHead = next;
		else
		    prev->next = next;
		prev = next;
	    }
	    mu->type = RB_MARKUP_NOTE;
	    if (newMu)
		mu = newMu;
	}
	if (mu->type == RB_MARKUP_UNDERLINE2) {
	    cp = pageBuf + mu->end;
	    while (!ISSPACE(cp[1]) && cp[1] != '<') cp++, mu->end++;
	    mu->type = RB_MARKUP_UNDERLINE;
	}
    }
}
