/* hashtable.c
 *
 * A "class" that implements in-core hashing with a separate (null-
 * terminated) key string and a generic data ptr.  Options exist to have
 * the class automatically free the data objects when deleting the hash,
 * and to auto-strdup() the objects when storing them (which only works
 * if they are null-terminated strings, of course).
 */
/* This file is a heavily altered version of a set of hash routines by
 * Geoffrey Collyer.  See the end of this file for his copyright.  See the
 * LICENSE file for my copyright.
 */

#include <config.h>
#include <rbmake/rbfile.h>
#include "hashtable.h"

#define BADTBL(tbl)	((tbl) == NULL || (tbl)->ht_magic != HASHMAG)

#define HASHMAG  ((char)0257)

static HashEnt **hashfind(HashTable *me, const char *key, int keylen);
static unsigned hash(const char *key, int keylen);
static HashEnt *healloc(void);
static void hefree(HashEnt *hp);

/* grab this many hashents at a time (under 1024 for malloc overhead) */
#define HEBLOCKSIZE 1000

/* define the following if you actually want to free the hashents
 * You probably don't want to do this with the usual malloc...
 */
/* #define HASH_FREE_ENTRIES */

/* tunable parameters */
#define RETAIN 1000		/* retain & recycle this many HashEnts */

static HashEnt *hereuse = NULL;
static int reusables = 0;

HashTable *
HashTable_new(unsigned size, bool autoDupData, bool autoFreeData)
{
    HashTable *tbl;
    /* allocate HashTable and (HashEnt*) array together to reduce the
     * number of malloc calls. */
    struct alignalloc {
	HashTable ht;
	HashEnt *hepa[1];	/* longer than it looks */
    } *aap;

    if (size < 1)		/* size < 1 is nonsense */
	size = 1;
    aap = Mem_calloc(sizeof *aap+(size-1)*sizeof(HashEnt*),1);
    tbl = &aap->ht;
    tbl->ht_size = size;
    tbl->ht_count = 0;
    tbl->ht_magic = HASHMAG;
    tbl->ht_addr = aap->hepa;
    tbl->autoDupData = autoDupData;
    tbl->autoFreeData = autoFreeData;
    return tbl;
}

/* Free all the memory associated with this table, erase the pointers to
 * it, and invalidate it to prevent further use via any other pointers. */
void
HashTable_delete(HashTable *me)
{
    unsigned idx;
    HashEnt *hp, *next, **hepp;
    int tblsize;

    if (BADTBL(me))
	return;
    tblsize = me->ht_size;
    hepp = me->ht_addr;
    for (idx = 0; idx < tblsize; idx++) {
	for (hp = hepp[idx]; hp != NULL; hp = next) {
	    next = hp->he_next;
	    hp->he_next = NULL;
	    if (me->autoFreeData)
		Mem_free(hp->he_data);
	    hefree(hp);
	}
	hepp[idx] = NULL;
    }
    me->ht_magic = 0;			/* de-certify this table */
    me->ht_addr = NULL;
    Mem_free((char*)me);
}

void *
HashTable_store(HashTable *me, const char *key, void *data)
{
    HashEnt *hp, **nextp;
    int keylen = strlen(key);
    void *vp;

    nextp = hashfind(me, key, keylen);
    hp = *nextp;
    if (hp == NULL) {			/* absent; allocate an entry */
	hp = healloc();
	hp->he_next = NULL;
	hp->he_key = Mem_strdup(key);
	hp->he_keylen = keylen;
	*nextp = hp;			/* append to hash chain */
	me->ht_count++;
	vp = NULL;
    }
    else {
	vp = hp->he_data;
	if (me->autoFreeData)
	    Mem_free(vp);
    }
    /* supersede any old data for this key */
    hp->he_data = me->autoDupData? Mem_strdup(data) : data;
    return vp;
}

void *
HashTable_remove(HashTable *me, const char *key)
{
    HashEnt *hp, **nextp;
    int keylen = strlen(key);
    void *vp;

    nextp = hashfind(me, key, keylen);
    hp = *nextp;
    if (hp == NULL)			/* absent */
	return NULL;
    vp = hp->he_data;
    if (me->autoFreeData)
	Mem_free(vp);
    *nextp = hp->he_next;		/* skip this entry */
    hp->he_next = NULL;
    hp->he_data = NULL;
    hefree(hp);
    me->ht_count--;
    return vp;
}

void *				/* data corresponding to key */
HashTable_fetch(HashTable *me, const char *key)
{
    HashEnt *hp, **nextp;
    int keylen = strlen(key);

    nextp = hashfind(me, key, keylen);
    hp = *nextp;
    return hp? hp->he_data : NULL;
}

/* Visit each entry by calling nodefunc at each, with keylen, data,
 * and extra as arguments. */
void
HashTable_walk(HashTable *me, void *userPtr,
	  int (*nodefunc)(void *userPtr, const char *key, void *data))
{
    HashEnt *hp, *next, **hepp, **last_nextp;
    unsigned idx;
    int tblsize;

    if (BADTBL(me))
	return;
    hepp = me->ht_addr;
    tblsize = me->ht_size;
    for (idx = 0; idx < tblsize; idx++) {
	last_nextp = &hepp[idx];
	for (hp = *last_nextp; hp != NULL; hp = next) {
	    next = hp->he_next;
	    if ((*nodefunc)(userPtr, hp->he_key, hp->he_data) < 0) {
		*last_nextp = next;
		hp->he_next = NULL;
		if (me->autoFreeData)
		    Mem_free(hp->he_data);
		hefree(hp);
		me->ht_count--;
	    }
	    else
		last_nextp = &hp->he_next;
	}
    }
}

/* The key pointers are not freshley allocated, so (1) only the array
 * needs to be released, and (2) the values are only valid until you start
 * removing things. */
const char **
HashTable_keys(HashTable *me)
{
    unsigned idx;
    HashEnt *hp, **hepp;
    int tblsize;
    const char **keys, **cpp;

    if (BADTBL(me))
	return NULL;
    keys = cpp = Mem_alloc((me->ht_count + 1) * sizeof (char*));
    hepp = me->ht_addr;
    tblsize = me->ht_size;
    for (idx = 0; idx < tblsize; idx++) {
	for (hp = hepp[idx]; hp != NULL; hp = hp->he_next)
	    *cpp++ = hp->he_key;
    }
    *cpp = NULL;
    return keys;
}

unsigned
HashTable_itemCnt(HashTable *me)
{
    return me->ht_count;
}

/* The returned value is the address of the pointer that refers to the
 * found object.  Said pointer may be NULL if the object was not found;
 * if so, this pointer should be updated with the address of the object
 * to be inserted, if insertion is desired. */
static HashEnt **
hashfind(HashTable *me, const char *key, int keylen)
{
    HashEnt *hp;
    HashEnt *prevhp = NULL;
    HashEnt **hepp;
    unsigned size; 

    if (BADTBL(me))
	RbError_exit("Hash table is invalid.\n");
    size = me->ht_size;
    hepp = &me->ht_addr[hash(key,keylen) % size];
    for (hp = *hepp; hp != NULL; prevhp = hp, hp = hp->he_next) {
	if (hp->he_keylen == keylen && memcmp(key, hp->he_key, keylen) == 0)
	    break;
    }
    /* assert: *(returned value) == hp */
    return prevhp == NULL? hepp: &prevhp->he_next;
}

static unsigned				/* not yet taken modulus table size */
hash(const char *key, int keylen)
{
    unsigned hash = 0;

    while (keylen--)
	hash += *key++;
    return hash;
}

static HashEnt *
healloc()				/* allocate a hash entry */
{
    HashEnt *hp;

    if (hereuse == NULL) {
	int i;

	/* make a nice big block of hashents to play with */
	hp = Mem_alloc(HEBLOCKSIZE * sizeof (HashEnt));
	/* set up the pointers within the block */
	for (i = 0; i < HEBLOCKSIZE-1; i++)
	    (hp+i)->he_next = hp + i + 1;
	/* The last block is the end of the list */
	(hp+i)->he_next = NULL;
	hereuse = hp;		/* start of list is the first item */
	reusables += HEBLOCKSIZE;
    }

    /* pull the first reusable one off the pile */
    hp = hereuse;
    hereuse = hereuse->he_next;
    hp->he_next = NULL;			/* prevent accidents */
    reusables--;
    return hp;
}

static void				/* free a hash entry */
hefree(HashEnt *hp)
{
    Mem_free((char*)hp->he_key);
#ifdef HASH_FREE_ENTRIES
    if (reusables >= RETAIN)		/* compost heap is full? */
	Mem_free((char*)hp);		/* yup, just pitch this one */
    else {				/* no, just stash for reuse */
	++reusables;
	hp->he_next = hereuse;
	hereuse = hp;
    }
#else
    /* always add to list */
    ++reusables;
    hp->he_next = hereuse;
    hereuse = hp;
#endif
}

/*
 * Copyright (C) 1992 by Geoffrey Collyer
 * Copyright (C) 2000 by Wayne Davison
 * All rights reserved.
 * Written by Geoffrey Collyer and heavily modified by Wayne Davison.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company, the Regents of the University of California, or
 * the Free Software Foundation.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. The author is not responsible for the consequences of use of this
 *    software, no matter how awful, even if they arise from flaws in it.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Since few users ever read sources,
 *    credits must appear in the documentation.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.  Since few users
 *    ever read sources, credits must appear in the documentation.
 *
 * 4. This notice may not be removed or altered.
 */
