/* graburl.c
 *
 * A simple routine that grabs the indicated URL (either http: or file:)
 * and reads it into an MBuf.
 */
/* This software is copyrighted as detailed in the LICENSE file. */

#include <config.h>
#include <libxml/nanohttp.h>
#include <rbmake/rbmake.h>
#include "mbuf.h"

static char authHeaderPrefix[] = "Authorization: Basic ";

static GrabUrlAskForAuthInfoFunc askForAuthInfo;
static MArray *authUrlArray, *authRealmArray;
static MBuf *httpHeaders;

static char *combineUrlRealm(const char *url, const char *realm, int rlen);
static void appendAuthUrl(const char *url, const char *authHeader);
static void appendRealmInfo(const char *url, const char *realm,
			    const char *authHeader);

void
GrabUrl_init(GrabUrlAskForAuthInfoFunc authFunc)
{
    askForAuthInfo = authFunc;
    GrabUrl_setHttpHeader(""); /* Make sure that the mbuf is initialized. */
}

/* Try to guess the authorization based on this url's full path info.
 * If found, set the Authorization header in the HTTP headers and
 * return 1, else return 0. */
int
GrabUrl_guessAuthHeader(const char *url)
{
    const char *cp;
    char *urlPrefix;
    MBuf *mb;
    int j;

    if (!authUrlArray)
	return 0;

    mb = MBuf_new(32, 0);
    if ((cp = strrchr(url+7, '/')) != NULL)
	MBuf_write(mb, url, cp - url + 1);
    else
	MBuf_vwrite(mb, url,-1, "/",1, NULL);
    urlPrefix = MBuf_toBuffer(mb, NULL);
    for (j = MArray_itemCnt(authUrlArray) - 2; j >= 0; j -= 2) {
	if (strcaseEQ(MArray_fetchPtrAt(authUrlArray, j), urlPrefix)) {
	    GrabUrl_setHttpHeader(MArray_fetchPtrAt(authUrlArray, j+1));
	    break;
	}
    }
    Mem_free(urlPrefix);

    if (j < 0) {
	GrabUrl_setHttpHeader("Authorization:"); /* deletes header */
	return 0;
    }
    return 1;
}

/* If we get new Authorization info, set the header in the HTTP headers
 * and return a pointer to the new header's value.  Else, return NULL. */
const char *
GrabUrl_askForAuthInfo(const char *url, void *ctxt)
{
    const char *cp, *authHeader, *urlRealm = NULL;
    int rc, ret;

    if (!ctxt || ((rc = xmlNanoHTTPReturnCode(ctxt)) != 401 && rc != 407))
	return NULL;
    authHeader = xmlNanoHTTPAuthHeader(ctxt);
    while (*authHeader == ' ') authHeader++;
    if (strncaseEQ(authHeader, "basic ", 6)) {
	authHeader += 6;
	while (*authHeader == ' ') authHeader++;
	if (strncaseEQ(authHeader, "realm=\"", 7)
	 && (cp = strchr(authHeader += 7, '"')) != NULL) {
	    urlRealm = combineUrlRealm(url, authHeader, cp - authHeader);
	    if (authRealmArray) {
		int j;
		for (j = MArray_itemCnt(authRealmArray) - 2; j >= 0; j -= 2) {
		    if (strcaseEQ(urlRealm,MArray_fetchPtrAt(authRealmArray,j))) {
			cp = MArray_fetchPtrAt(authRealmArray, j+1);
			if (GrabUrl_setHttpHeader(cp) == 1) {
			    /* The auth header was changed. */
			    appendAuthUrl(url, cp);
			    Mem_free((void*)urlRealm);
			    return cp;
			}
			break;
		    }
		}
	    }
	}
    }

    if (!askForAuthInfo
     || !(cp = askForAuthInfo(url, urlRealm? strrchr(urlRealm,'/')+1 : "<Unknown>"))
     || GrabUrl_setHttpHeader(cp) != 1)
	cp = NULL;
    if (urlRealm)
	Mem_free((void*)urlRealm);
    return cp;
}

const char *
GrabUrl_setAuthInfo(const char *url, const char *realm, const char *b64str)
{
    int authHeaderLen = STATICLEN(authHeaderPrefix) + strlen(b64str) + 2;
    char *authHeader = Mem_alloc(authHeaderLen + 1);

    sprintf(authHeader, "%s%s\r\n", authHeaderPrefix, b64str);
    if (strcaseEQ(url, "proxy")) {
	char *proxyAuth = Mem_alloc(authHeaderLen + 6 + 1);
	sprintf(proxyAuth, "Proxy-%s", authHeader);
	GrabUrl_setHttpHeader(proxyAuth);
	Mem_free(proxyAuth);
    }
    else {
	appendAuthUrl(url, authHeader);

	if (realm && *realm)
	    appendRealmInfo(url, realm, authHeader);
    }

    return authHeader;
}

static char *
combineUrlRealm(const char *url, const char *realm, int rlen)
{
    const char *cp;
    MBuf *mb;

    if (strnNE(url, "http://", 7))
	return NULL;

    mb = MBuf_new(128, 0);
    if ((cp = strchr(url+7, '/')) == NULL)
	MBuf_vwrite(mb, url,-1, "/",1, realm,rlen, NULL);
    else
	MBuf_vwrite(mb, url,cp-url+1, realm,rlen, NULL);
    return MBuf_toBuffer(mb, NULL);
}

static void
appendAuthUrl(const char *url, const char *authHeader)
{
    const char *cp;
    MBuf *mb = MBuf_new(128, 0);

    if ((cp = strrchr(url+7, '/')) == NULL)
	MBuf_vwrite(mb, url,-1, "/",1, NULL);
    else
	MBuf_vwrite(mb, url,cp-url+1, NULL);

    if (!authUrlArray)
	authUrlArray = MArray_new(8, 0);
    MArray_appendPtr(authUrlArray, MBuf_toBuffer(mb, NULL));
    MArray_appendPtr(authUrlArray, authHeader);
}

static void
appendRealmInfo(const char *url, const char *realm, const char *authHeader)
{
    if (!authRealmArray)
	authRealmArray = MArray_new(8, 0);
    MArray_appendPtr(authRealmArray, combineUrlRealm(url, realm, -1));
    MArray_appendPtr(authRealmArray, authHeader);
}

/* Returns 1 if the header was updated, 0 if the header was already present
 * with the right value, and -1 if the header was not in the right format.
 * Note also that setting a header that has nothing after the colon deletes
 * that header (e.g. "Authorization:"). */
int
GrabUrl_setHttpHeader(const char *header)
{
    char *cp, buf[1024];
    int len, clen, hlen;

    if (!httpHeaders) {
	httpHeaders = MBuf_new(32, 0);
	MBuf_puts(httpHeaders, "User-Agent: rbmake/" RBMAKE_VERSION "\r\n");
    }

    if ((cp = strchr(header, ':')) == NULL)
	return -1; /* invalid header */
    clen = cp - header;

    hlen = strlen(header);
    while (hlen && (header[hlen-1] == '\r' || header[hlen-1] == '\n')) hlen--;

    MBuf_setReadPos(httpHeaders, 0, 0);
    while ((len = MBuf_gets(httpHeaders, buf, sizeof buf)) > 0) {
	if (len <= clen || strchr(buf, ':') - buf != clen
	 || strncaseNE(buf, header, clen))
	    continue;
	len--; /* Avoid counting the \n (note: gets turned \r\n into \n). */
	if (len == hlen && strnEQ(buf+clen, header+clen, hlen-clen))
	    return 0; /* Header was already set appropriately. */
	/* Get rid of the old version of this header, including \r\n. */
	MBuf_memcpy(httpHeaders, httpHeaders->readPos - len - 2,
		    httpHeaders->readPos,
		    httpHeaders->totalLen - httpHeaders->readPos);
	MBuf_truncate(httpHeaders, httpHeaders->totalLen - len - 2);
	break;
    }

    if (hlen <= clen+1)
	return len > 0; /* Non-zero if we deleted the header. */

    MBuf_vwrite(httpHeaders, header,hlen, "\r\n",2, NULL);
    return 1;
}

const char *
GrabUrl_getHttpHeaders(void)
{
    MBuf_setReadPos(httpHeaders, 0, 0);
    return MBuf_dataPtr(httpHeaders, NULL);
}

/* You can either pass in an existing MBuf object (in which case the
 * file's data will be appended to the buffer), or you can pass in a NULL
 * and let the buffer return a newly-created MBuf object. */
MBuf *
GrabUrl_read(const char *url, MBuf *mb)
{
    char buf[2048];
    int cnt;
    const char *fn;
    bool delMbOnError;

    if (!mb) {
	mb = MBuf_new(4096, 0);
	delMbOnError = true;
    }
    else
	delMbOnError = false;
    if ((fn = rbGetUrlFn(url)) != NULL || strEQ(url, "-")) {
	FILE *fp = fn? fopen(fn, "rb") : stdin;
	if (fn)
	    Mem_free((void*)fn);
	if (!fp) {
	    if (delMbOnError)
		MBuf_delete(mb);
	    return NULL;
	}
	while ((cnt = fread(buf, 1, sizeof buf, fp)) > 0)
	    MBuf_write(mb, buf, cnt);
	fclose(fp);
    }
    else {
	void *ctxt;

	GrabUrl_guessAuthHeader(url);

      try_http:
	ctxt = xmlNanoHTTPMethod(url, NULL, NULL, NULL,
				 GrabUrl_getHttpHeaders(), 0);
	if (!ctxt || xmlNanoHTTPReturnCode(ctxt) / 100 != 2) {
	    if (GrabUrl_askForAuthInfo(url, ctxt))
		goto try_http;
	    if (delMbOnError)
		MBuf_delete(mb);
	    return NULL;
	}
	while ((cnt = xmlNanoHTTPRead(ctxt, buf, sizeof buf)) > 0)
	    MBuf_write(mb, buf, cnt);
	xmlNanoHTTPClose(ctxt);
    }

    return mb;
}
