/*
 * Copyright (c) 2005 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RUMController.h"

@implementation RUMController

- ( id )init
{
    self = [ super init ];

    [ NSApp setDelegate: self ];
    statusItem = nil;
    animationTimer = nil;
    frameno = 1;
    
    return( self );
}

- ( void )awakeFromNib
{
    updateImageArray = [[ NSArray alloc ] initWithObjects:
			[ NSImage imageNamed: @"RUMUpdateFrame1" ],
			[ NSImage imageNamed: @"RUMUpdateFrame2" ],
			[ NSImage imageNamed: @"RUMUpdateFrame3" ], nil ];
    [ self activateStatusMenu ];
}

- ( void )activateStatusMenu
{
    NSStatusBar		*statusBar = [ NSStatusBar systemStatusBar ];
    
    statusItem = [ statusBar statusItemWithLength: NSSquareStatusItemLength ];
    [ statusItem retain ];
    
    statusMenu = [[ NSMenu alloc ] initWithTitle: @"" ];
    [ self addGeneralMenuItems ];
    
    [ statusItem setImage: [ NSImage imageNamed: @"NuclearBrain" ]];
    [ statusItem setHighlightMode: YES ];
    [ statusItem setMenu: statusMenu ];
    [ statusItem setToolTip:
	    NSLocalizedString( @"Radmind Update Monitor",
				@"Radmind Update Monitor" ) ];
    [ statusItem setTarget: self ];
}

- ( void )addGeneralMenuItems
{
    id			lastCheck;
    NSString		*lastCheckString;
    NSTimeInterval	secs;
    
    if ( statusMenu == nil ) {
	return;
    }
    
    lastCheck = [[ NSUserDefaults standardUserDefaults ]
		    objectForKey: @"RUMLastCheckDate" ];
    if ( lastCheck == nil || ![ lastCheck isKindOfClass: [ NSNumber class ]] ) {
	lastCheckString = @"never";
    } else {
	secs = [ lastCheck doubleValue ];
	lastCheckString = [[ NSDate dateWithTimeIntervalSince1970: secs ]
				descriptionWithCalendarFormat: @"%e %b %Y %I:%M %p"
				timeZone: nil locale: nil ];
	if ( lastCheckString == nil ) {
	    lastCheckString = @"failed to format date!";
	}
    }
    
    [ statusMenu addItemWithTitle:
	    [ NSString stringWithFormat: @"Last check: %@", lastCheckString ]
	    action: NULL keyEquivalent: @"" ];
    [ statusMenu addItem: [ NSMenuItem separatorItem ]];
    
    [ statusMenu addItemWithTitle: RUM_OPEN_ASSISTANT_STRING
		action: @selector( openRadmindAssistant: ) keyEquivalent: @"" ];
    [ statusMenu addItem: [ NSMenuItem separatorItem ]];
    [ statusMenu addItemWithTitle: RUM_QUIT_STRING
		action: @selector( quit: ) keyEquivalent: @"" ];
}

- ( void )saveCheckDate
{
    NSDate		*date = [ NSDate date ];
    NSNumber		*dateSeconds;
    
    dateSeconds = [ NSNumber numberWithDouble: [ date timeIntervalSince1970 ]];
    
    [[ NSUserDefaults standardUserDefaults ] setObject: dateSeconds
	    forKey: @"RUMLastCheckDate" ];
}

- ( void )addObservers
{
    [[ NSDistributedNotificationCenter defaultCenter ]
	    addObserver: self selector: @selector( radmindCheckingForUpdates: )
	    name: RUMUpdatesAvailableNotification object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ]
	    addObserver: self selector: @selector( radmindNoUpdatesAvailable: )
	    name: RUMUpdatesNoneNotification object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ]
	    addObserver: self selector: @selector( radmindUpdatesAvailable: )
	    name: RUMUpdatesAvailableNotification object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ]
	    addObserver: self selector: @selector( radmindUpdateCheckError: )
	    name: RUMUpdatesErrorNotification object: nil ];
}

- ( void )removeObservers
{
    [[ NSDistributedNotificationCenter defaultCenter ]
	    removeObserver: self
	    name: RUMUpdatesAvailableNotification object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ]
	    removeObserver: self
	    name: RUMUpdatesAvailableNotification object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ]
	    removeObserver: self
	    name: RUMUpdatesAvailableNotification object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ]
	    removeObserver: self
	    name: RUMUpdatesAvailableNotification object: nil ];
}

- ( void )applicationDidFinishLaunching: ( NSNotification * )notification
{
    [ self addObservers ];
}

- ( void )radmindCheckingForUpdates: ( id )sender
{
    NSArray	    *menuItems = [ statusMenu itemArray ];
    int		    i;
	    
    [ self saveCheckDate ];
    
    for ( i = ( [ menuItems count ] - 1 ); i >= 0; i-- ) {
	[ statusMenu removeItemAtIndex: i ];
    }

    [ statusMenu addItemWithTitle: RUM_UPDATE_CHECK_STRING
	    action: NULL keyEquivalent: @"" ];
    [ statusMenu addItem: [ NSMenuItem separatorItem ]];
    [ self addGeneralMenuItems ];
    
    if ( animationTimer ) {
	[ animationTimer invalidate ];
	animationTimer = nil;
    }
    animationTimer = [ NSTimer scheduledTimerWithTimeInterval: 0.5
			    target: self
			    selector: @selector( checkingAnimationUpdate: )
			    userInfo: nil
			    repeats: YES ];
}

- ( void )radmindUpdatesAvailable: ( id )sender
{
    NSArray	    *menuItems = [ statusMenu itemArray ];
    NSString	    *output;
    int		    i;
    
    [ self removeObservers ];
    [ self saveCheckDate ];
	    
    for ( i = ( [ menuItems count ] - 1 ); i >= 0; i-- ) {
	[ statusMenu removeItemAtIndex: i ];
    }
    
    if (( output = [[ sender userInfo ]
	    objectForKey: @"RUMServerOutput" ] ) == nil ) {
	output = RUM_UPDATES_AVAILABLE_STRING;
    } else {
	[ statusMenu addItemWithTitle: RUM_UPDATES_AVAILABLE_STRING
	    action: NULL keyEquivalent: @"" ];
    }
    
    [ statusMenu addItemWithTitle: output
	    action: NULL keyEquivalent: @"" ];
    [ statusMenu addItem: [ NSMenuItem separatorItem ]];
    [ self addGeneralMenuItems ];
    
    if ( animationTimer ) {
	[ animationTimer invalidate ];
	animationTimer = nil;
    }
    animationTimer = [ NSTimer scheduledTimerWithTimeInterval: 0.5
			    target: self
			    selector: @selector( updateBlink: )
			    userInfo: nil
			    repeats: YES ];
}

- ( void )radmindNoUpdatesAvailable: ( id )sender
{
    NSArray	    *menuItems = [ statusMenu itemArray ];
    NSString	    *output;
    int		    i;
    
    [ self saveCheckDate ];

    for ( i = ( [ menuItems count ] - 1 ); i >= 0; i-- ) {
	[ statusMenu removeItemAtIndex: i ];
    }
    
    if (( output = [[ sender userInfo ]
	    objectForKey: @"RUMServerOutput" ] ) == nil ) {
	output = RUM_NO_UPDATES_STRING;
    }
    
    [ statusMenu addItemWithTitle: output
	    action: NULL keyEquivalent: @"" ];
    [ statusMenu addItem: [ NSMenuItem separatorItem ]];
    [ self addGeneralMenuItems ];
    
    [ statusItem setImage: [ NSImage imageNamed: @"NuclearBrain" ]];
    
    if ( animationTimer ) {
	[ animationTimer invalidate ];
	animationTimer = nil;
    }
}

- ( void )radmindUpdateCheckError: ( id )sender
{
    NSImage	    *errorImage = [ NSImage imageNamed: @"RUMError" ];
    NSArray	    *menuItems = [ statusMenu itemArray ];
    NSString	    *output;
    int		    i;
    
    [ self saveCheckDate ];

    for ( i = ( [ menuItems count ] - 1 ); i >= 0; i-- ) {
	[ statusMenu removeItemAtIndex: i ];
    }
    
    if (( output = [[ sender userInfo ]
	    objectForKey: @"RUMServerOutput" ] ) == nil ) {
	output = RUM_ERROR_STRING;
    }
    
    [ statusMenu addItemWithTitle: output
	    action: NULL keyEquivalent: @"" ];
    [ statusMenu addItem: [ NSMenuItem separatorItem ]];
    [ self addGeneralMenuItems ];
    
    if ( animationTimer ) {
	[ animationTimer invalidate ];
	animationTimer = nil;
    }
    
    [ statusItem setImage: errorImage ];
}

- ( void )updateBlink: ( id )sender
{
    NSImage	    *animationImage = nil;
    
    switch ( frameno ) {
    case 1:
	animationImage = [ NSImage imageNamed: @"BrightNuclearBrain" ];
	break;
	
    case 2:
    default:
	animationImage = [ NSImage imageNamed: @"NuclearBrain" ];
	break;
    }
    
    if (( frameno++ ) > 2 ) {
	frameno = 1;
    }
    
    [ statusItem setImage: animationImage ];
}

- ( void )checkingAnimationUpdate: ( id )sender
{
    int		f = ( frameno - 1 );

    [ statusItem setImage: [ updateImageArray objectAtIndex: f ]];

    if (( frameno++ ) >= 3 ) {
	frameno = 1;
    }
}

- ( void )openRadmindAssistant: ( id )sender
{
    if ( animationTimer ) {
	[ animationTimer invalidate ];
	animationTimer = nil;
    }
    [ statusItem setImage: [ NSImage imageNamed: @"NuclearBrain" ]];
    frameno = 1;
    
    [[ NSWorkspace sharedWorkspace ] launchApplication: @"Radmind Assistant.app" ];
    
    [ self addObservers ];
}

- ( void )quit: ( id )sender
{
    [ NSApp terminate: self ];
}

- ( void )dealloc
{
    if ( statusItem ) {
	[ statusItem release ];
    }
    
    [ super dealloc ];
}

@end
