#include <Security/Authorization.h>
#include <Security/AuthorizationTags.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <dirent.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "selfrepair.h"

extern int		errno;

/* --repair setuid root--
 * This is moderately more secure way to use
 * Apple's AuthorizationExecuteWithPrivileges().
 * Apple's samples have the tool launching itself
 * with AEWP, but this opens a security hole:
 * there's a small window of opportunity in which
 * the tool can be swapped out for another,
 * and AEWP will execute anything as root
 * with impunity. This method calls chmod and chown with AEWP,
 * the assumption being that the user would have to
 * be root to swap them out in the first place.
 *
 * We reuse the AuthorizationRef we got from the GUI,
 * and pass the AuthorizationExternalForm received from
 * the GUI to ourselves after the self-repair.
 */
    int
selfrepair( int argc, char **argv, const char *rightname,
		AuthorizationRef authref,
		AuthorizationExternalForm extauth )
{
    OSStatus		err;
    int			status, ifd[ 2 ];
    pid_t		pid;
    struct statfs	stfs;
    FILE		*comm = NULL;
    const char		*chownargs[] = { "root:admin", argv[ 0 ], NULL };
    const char		*chmodargs[] = { "4555", argv[ 0 ], NULL };
    AuthorizationItem	right = { rightname, 0, NULL, 0 };
    AuthorizationRights	rights = { 1, &right };
    AuthorizationFlags	flags = kAuthorizationFlagDefaults |
				kAuthorizationFlagPreAuthorize |
				kAuthorizationFlagInteractionAllowed |
				kAuthorizationFlagExtendRights;

    /* make sure we're not running from a read-only filesystem */
    if ( statfs( argv[ 0 ], &stfs ) != 0 ) {
	fprintf( stderr, "statfs %s: %s\n", argv[ 0 ], strerror( errno ));
	return( 2 );
    }
    if ( stfs.f_flags & MNT_RDONLY ) {
	fprintf( stderr,
		"Cannot repair %s: read-only filesystem.\n", argv[ 0 ] );
	return( 2 );
    }

    /* this brings forward the auth dialog, if necessary */
    err = AuthorizationCopyRights( authref, &rights,
				kAuthorizationEmptyEnvironment,
				flags, NULL );

    if ( err != errAuthorizationSuccess ) {
	fprintf( stderr, "AuthorizationCopyRights failed with error %d.\n",
			( int )err );
	return( 2 );
    } 

    if ( access( "/usr/sbin/chown", X_OK | F_OK ) < 0 ) {
	fprintf( stderr, "Couldn't access /usr/sbin/chown for self-repair:"
			 " %s\n", strerror( errno ));
	return( errno );
    }

    err = AuthorizationExecuteWithPrivileges( authref,
		( char * const )"/usr/sbin/chown",
		kAuthorizationFlagDefaults,
		( char * const * )chownargs, &comm );

    fflush( comm );
    if ( comm != NULL ) {
	fclose( comm );
    }

    pid = wait( &status );

    if ( err ) {
	fprintf( stderr, "AuthorizationExecuteWithPrivileges failed to "
		"repair %s. Error %d.\n", argv[ 0 ], ( int )err );
	return( 2 );
    }

    if ( access( "/bin/chmod", X_OK | F_OK ) < 0 ) {
        fprintf( stderr, "Couldn't access chmod for self-repair:"     
                         " %s\n", strerror( errno ));
        return( errno );
    }

    err = AuthorizationExecuteWithPrivileges( authref,
                ( char * const )"/bin/chmod",
                kAuthorizationFlagDefaults,
		( char * const * )chmodargs, &comm );

    fflush( comm );
    if ( comm != NULL ) {
        fclose( comm );
    }

    pid = wait( &status );

    if ( err ) {
        fprintf( stderr, "AuthorizationExecuteWithPrivileges failed to "
                "repair %s. Error %d.\n", argv[ 0 ], ( int )err );
        return( 2 );
    }

    /* once we've repaired the tool, run it again */
    if ( pipe( ifd ) != 0 ) {
	fprintf( stderr, "pipe failed: %s\n", strerror( errno ));
	return( errno );
    }

    switch (( pid = fork())) {
    case 0:
	( void )close( ifd[ 1 ] );
	if ( dup2( ifd[ 0 ], 0 ) < 0 ) {
	    fprintf( stderr, "dup2 failed: %s\n", strerror( errno ));
	    _exit( errno );
	}

	execve( argv[ 0 ], argv, NULL );
	fprintf( stderr, "execve %s failed: %s\n", argv[ 0 ],
			strerror( errno ));
	_exit( errno );

    case -1:
	fprintf( stderr, "fork failed: %s\n", strerror( errno ));
	return( errno );

    default:
	break;
    }

    signal( SIGPIPE, SIG_IGN );

    ( void )close( ifd[ 0 ] );

    if ( write( ifd[ 1 ], &extauth, sizeof( extauth )) != sizeof( extauth )) {
	fprintf( stderr, "write failed: %s\n", strerror( errno ));
	( void )close( ifd[ 1 ] );
	return( errno );
    }

    pid = wait( &status );

    return( WEXITSTATUS( status ));
}
