/*
Copyright (c) 2002 Regents of The University of Michigan.
All Rights Reserved.

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appears in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of The University
of Michigan not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission. This software is supplied as is without expressed or
implied warranties of any kind.

Research Systems Unix Group
The University of Michigan
c/o Wesley Craig
4251 Plymouth Road B1F2, #2600
Ann Arbor, MI 48105-2785

http://rsug.itd.umich.edu/software/radmind
radmind@umich.edu
*/

#import "NSString(RXAdditions).h"
#include <errno.h>
#include <string.h>
#include "code.h"

extern int	errno;

@implementation NSString(RXAdditions)

- ( NSString * )stringFromOctal
{
    char	tmp[ 12 ] = { 0 };
    char	*p;
    mode_t      mode = strtol(( char * )[ self UTF8String ], NULL, 8 );
    
    strmode( mode, tmp );
    
    p = tmp;
    p++;
    
    return( [ NSString stringWithUTF8String: p ] );
}

- ( BOOL )containsString: ( NSString * )substring
{
    return( [ self containsString: substring useCaseInsensitiveComparison: NO ] );
}

- ( BOOL )containsString: ( NSString * )substring useCaseInsensitiveComparison: ( BOOL )ci
{
    unsigned		mask;
    
    if ( substring == nil ) return NO;
    
    if ( ci == YES ) mask = NSCaseInsensitiveSearch;
    else mask = NSLiteralSearch;
    
    if ( [ self rangeOfString: substring options: mask ].location != NSNotFound ) {
        return( YES );
    }
    return( NO );
}

+ ( NSString * )transcriptLineFromDictionary: ( NSDictionary * )dict
{
    NSMutableString     *line = [[ NSMutableString alloc ] init ];
    NSString            *transcriptLine = nil;
    char		ftype, pm, *path;
    
    ftype = [[ dict objectForKey: @"type" ]
                                characterAtIndex: 0 ];
              
    /* if this is an apply-able transcript line, print out the + and - signs */
    if (( pm = [[ dict objectForKey: @"pm" ] characterAtIndex: 0 ] ) != '0' ) {
        [ line appendFormat: @"%c ", pm ];
    }
    /* likewise, if line is transcript marker in an apply-able transcript... */
    if ( ftype == '0' ) {
        [ line release ];
        return( [ NSString stringWithFormat: @"%@\n", [ dict objectForKey: @"path" ]] );
    }

    path = ( char * )[[ dict objectForKey: @"path" ] UTF8String ];
    [ line appendFormat: @"%c %-37s\t", ftype, encode( path ) ];
    
    switch ( ftype ) {
    case 'h':
    case 'l':
        [ line appendFormat: @"%s",
                encode(( char * )[[ dict objectForKey: @"dest" ] UTF8String ] ) ];
        break;
        
    case 'd':
    case 'D':
    case 'p':
    case 's':
        [ line appendFormat: @"%.4s %5s %5s", 
                ( char * )[[ dict objectForKey: @"perm" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"owner" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"group" ] UTF8String ]];
        if ( [ dict objectForKey: @"finfo" ] != nil ) {
            [ line appendFormat: @" %s",
                ( char * )[[ dict objectForKey: @"finfo" ] UTF8String ]];
        }
        break;
        
    case 'a':
    case 'f':
        [ line appendFormat: @"%.4s %5s %5s %9s %7s %s",
                ( char * )[[ dict objectForKey: @"perm" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"owner" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"group" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"mtime" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"size" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"cksum" ] UTF8String ]];
        break;
        
    case 'b':
    case 'c':
        [ line appendFormat: @"%.4s %5s %5s %5s %5s",
                ( char * )[[ dict objectForKey: @"perm" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"owner" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"group" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"major" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"minor" ] UTF8String ]];
        break;
    }
    
    [ line appendFormat: @"\n" ];
    transcriptLine = [ NSString stringWithString: line ];
    [ line autorelease ];
    
    return( transcriptLine );
}

+ ( NSString * )formattedDateFromTimeInterval: ( NSString * )seconds
{
    const char		*secs = [ seconds UTF8String ];
    NSDate		*date = nil;
    NSString		*formattedDate = @"";
    unsigned long	s;
    
    if (( s = strtol( secs, NULL, 10 )) == 0 ) {
        if ( errno == EINVAL ) {
            formattedDate = [ NSString stringWithFormat: @"%@: %s", seconds,
                                                                    strerror( errno ) ];
        }
    } else {
        date = [ NSDate dateWithTimeIntervalSince1970: ( NSTimeInterval )s ];
        formattedDate = [ date descriptionWithCalendarFormat: @"%d %b %Y %H:%M"
                                        timeZone: nil locale: nil ];
    }
    
    return( formattedDate );
}

- ( NSString * )descriptiveSizeString
{
    const char			*s = [ self UTF8String ];
    off_t			size, m, n;
    float			fraction, mbytes;
    
    if (( size = strtoll( s, NULL, 10 )) == 0 ) {
        /* don't bother with an error, just return bytes */
        return( self );
    }
    
    /* if the size is less than 1K, return bytes */
    if (( m = ( size / 1024 )) == 0 ) {
        return( [ NSString stringWithFormat: @"%@ B  ", self ] );
    }
    
    /* if size is less than 1MB, return kbytes */
    if (( n = ( m / 1024 )) == 0 ) {
        /* round up, if necessary */
        if (( size % 1024 ) > 512 ) {
            m++;
        }
        return( [ NSString stringWithFormat: @"%lld KB  ", m ] );
    }
    
    /* otherwise, return in MB */
    m = ( m % 1024 );
    fraction = ( float )( m / 1024.0 );
    mbytes = ( float )n;
    mbytes += fraction;
    
    return( [ NSString stringWithFormat: @"%.2f MB  ", mbytes ] );
}

- ( NSString * )transcriptObjectTypeFromString
{
    NSString		*objectType = @"";
    
    if ( [ self length ] != 1 ) {
	return( @"" );
    }
    
    switch ( [ self characterAtIndex: 0 ] ) {
    case 'a':
	objectType = NSLocalizedString( @"applefile", @"applefile" );
	break;
	
    case 'b':
	objectType = NSLocalizedString( @"block special", @"block special" );
	break;
	
    case 'c':
	objectType = NSLocalizedString( @"character special", @"character special" );
	break;
	
    case 'D':
	objectType = NSLocalizedString( @"door", @"door" );
	break;
	
    case 'd':
	objectType = NSLocalizedString( @"directory", @"directory" );
	break;
	
    case 'f':
	objectType = NSLocalizedString( @"file", @"file" );
	break;
	
    case 'h':
	objectType = NSLocalizedString( @"hard link", @"hard link" );
	break;
	
    case 'l':
	objectType = NSLocalizedString( @"symbolic link", @"symbolic link" );
	break;
	
    case 'p':
	objectType = NSLocalizedString( @"named pipe", @"named pipe" );
	break;
	
    case 's':
	objectType = NSLocalizedString( @"socket", @"socket" );
	break;
	
    default:
	objectType = NSLocalizedString( @"unrecognized type", @"unrecognized type" );
	break;
    }
    
    return( objectType );
}

@end
