#import "RSMPreferences.h"
#import "RAServerAuth.h"
#import "RASTableView.h"
#import "NSArray(CreateArgv).h"
#import "NSPanel(Resizing).h"

#include <CoreFoundation/CoreFoundation.h>
#include <Security/Authorization.h>
#include <Security/AuthorizationDB.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>

#include "authtools.h"
#include "sessionrights.h"
#include "srvprocinfo.h"

#define RADMIND_RIGHT_START	"edu.umich.radmind.startserver"
#define RADMIND_RIGHT_STOP	"edu.umich.radmind.stopserver"
#define RADMIND_RIGHT_SET	"edu.umich.radmind.setright"
#define RADMIND_RIGHT_REMOVE	"edu.umich.radmind.removeright"
#define RADMIND_RIGHT_GENERIC	"edu.umich.radmind.generic"

#define RSMPrefServerIdentifier	    @"RSMPrefServer"
#define RSMPrefGeneralIdentifier    @"RSMPrefGeneral"
#define RSMPrefAdvancedIdentifier   @"RSMPrefAdvanced"

@implementation RSMPreferences

- ( id )init
{
    self = [ super init ];
    _rsmServerIsRunning = NO;
    _rsmServerPID = -1;
    _rsmServerStatusTimer = nil;
    _rsmPrefsSessionTmpDir = nil;
    _rsmPrefsAllowedUsers = nil;
    
    return( self );
}

- ( void )awakeFromNib
{
    [ self toolbarSetup ];

    [ self refreshServerStatusDisplay ];
    [ self reloadDefaults ];
    
    [ rsmPrefAllowedUsersTable setDataSource: self ];
    [ rsmPrefAllowedUsersTable setDelegate: self ];
    
    if ( [ rsmPrefWindow toolbar ] != nil &&
                [[ rsmPrefWindow toolbar ] respondsToSelector:
                @selector( setSelectedItemIdentifier: ) ] ) {
        [[ rsmPrefWindow toolbar ] setSelectedItemIdentifier:
                            RSMPrefServerIdentifier ];
    }
    
    _rsmServerStatusTimer = [ NSTimer scheduledTimerWithTimeInterval: 300
				target: self
				selector: @selector( refreshServerStatusDisplay )
				userInfo: nil repeats: YES ];
}

/* server status accessor methods */
- ( BOOL )serverIsRunning
{
    return( _rsmServerIsRunning );
}

- ( void )setServerIsRunning: ( BOOL )running
{
    _rsmServerIsRunning = running;
}

- ( pid_t )serverPID
{
    return( _rsmServerPID );
}

- ( void )setServerPID: ( pid_t )pid
{
    _rsmServerPID = pid;
}

- ( NSString * )sessionTemporaryDirectory
{
    return( _rsmPrefsSessionTmpDir );
}

- ( void )setSessionTemporaryDirectory: ( NSString * )tmpdir
{
    if ( _rsmPrefsSessionTmpDir != nil ) {
	[ _rsmPrefsSessionTmpDir release ];
    }
    _rsmPrefsSessionTmpDir = [ tmpdir retain ];
}

- ( NSMutableArray * )allowedUsers
{
    return( _rsmPrefsAllowedUsers );
}

- ( void )setAllowedUsers: ( NSMutableArray * )users
{
    if ( _rsmPrefsAllowedUsers != nil ) {
	[ _rsmPrefsAllowedUsers release ];
    }
    _rsmPrefsAllowedUsers = [ users retain ];
}
/* end accessor methods */

/* error display */
- ( void )preferencesError: ( NSString * )fmt, ...
{
    NSString            *errorString = nil;
    va_list             val;
    
    va_start( val, fmt );
    
    errorString = [[ NSString alloc ] initWithFormat: fmt
                                        arguments: val ];
                                        
    va_end( val );
    
    NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                    errorString, NSLocalizedString( @"OK", @"OK" ),
                    @"", @"" );
                    
    [ errorString release ];
}

/* toolbar setup and delegate methods */
- ( void )toolbarSetup
{
    NSToolbar		*tbar = [[[ NSToolbar alloc ]
				    initWithIdentifier: @"RSMPrefToolbar" ]
				    autorelease ];
    
    [ tbar setAllowsUserCustomization: YES ];
    [ tbar setAutosavesConfiguration: YES ];
    [ tbar setDisplayMode: NSToolbarDisplayModeIconAndLabel ];
    
    [ tbar setDelegate: self ];
    [ rsmPrefWindow setToolbar: tbar ];
}

- ( NSToolbarItem * )toolbar: ( NSToolbar * )toolbar
	itemForItemIdentifier: ( NSString * )itemIdent
	willBeInsertedIntoToolbar: ( BOOL )flag
{
    NSToolbarItem *rsmPrefTbarItem = [[[ NSToolbarItem alloc ]
					initWithItemIdentifier: itemIdent ]
					autorelease ];
					
    if ( [ itemIdent isEqualToString: RSMPrefServerIdentifier ] ) {
        [ rsmPrefTbarItem setLabel:
                NSLocalizedString( @"Server", @"Server" ) ];
        [ rsmPrefTbarItem setPaletteLabel:
                NSLocalizedString( @"Server", @"Server" ) ];
        [ rsmPrefTbarItem setToolTip:
                NSLocalizedString( @"Server Preferences", @"Server Preferences" ) ];
        [ rsmPrefTbarItem setImage: [ NSImage imageNamed: @"serverprefs.png" ]];
        [ rsmPrefTbarItem setAction: @selector( showServerPreferences: ) ];
        [ rsmPrefTbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMPrefGeneralIdentifier ] ) {
        [ rsmPrefTbarItem setLabel:
                NSLocalizedString( @"General", @"General" ) ];
        [ rsmPrefTbarItem setPaletteLabel:
                NSLocalizedString( @"General", @"General" ) ];
        [ rsmPrefTbarItem setToolTip:
                NSLocalizedString( @"General Preferences", @"General Preferences" ) ];
        [ rsmPrefTbarItem setImage: [ NSImage imageNamed: @"generalprefs.png" ]];
        [ rsmPrefTbarItem setAction: @selector( showGeneralPreferences: ) ];
        [ rsmPrefTbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMPrefAdvancedIdentifier ] ) {
	[ rsmPrefTbarItem setLabel:
                NSLocalizedString( @"Advanced", @"Advanced" ) ];
        [ rsmPrefTbarItem setPaletteLabel:
                NSLocalizedString( @"Advanced", @"Advanced" ) ];
        [ rsmPrefTbarItem setToolTip:
                NSLocalizedString( @"Advanced Preferences", @"Advanced Preferences" ) ];
        [ rsmPrefTbarItem setImage: [ NSImage imageNamed: @"advancedprefs.png" ]];
        [ rsmPrefTbarItem setAction: @selector( showAdvancedPreferences: ) ];
        [ rsmPrefTbarItem setTarget: self ];
    }
            
    return( rsmPrefTbarItem );
}

- ( NSArray * )toolbarDefaultItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
			    RSMPrefServerIdentifier,
			    RSMPrefGeneralIdentifier, 
			    RSMPrefAdvancedIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarAllowedItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
			    RSMPrefServerIdentifier,
			    RSMPrefGeneralIdentifier, 
			    RSMPrefAdvancedIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarSelectableItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
			    RSMPrefServerIdentifier,
			    RSMPrefGeneralIdentifier, 
			    RSMPrefAdvancedIdentifier, nil ];
                            
    return( tmp );
}
/* end toolbar setup and delegate methods */

/* process monitoring */
- ( void )refreshServerStatusDisplay
{
    char                procinfo[ MAXPATHLEN ] = { 0 };
    int                 rc;
    pid_t               pid = -1;
    
    [ rsmPrefServerProcessInfo setStringValue: @"" ];
    
    rc = get_srv_procinfo( procinfo, MAXPATHLEN, &pid );
    
    [ self setServerPID: pid ];
    
    switch ( rc ) {
    case -1:
        [ self preferencesError:
                NSLocalizedString( @"Unable to get server information: %s",
                                    @"Unable to get server information: %s" ),
                                    strerror( errno ) ];
        break;
        
    case 0:
        [ self setServerIsRunning: NO ];
        break;
        
    case 1:
        [ self setServerIsRunning: YES ];
        [ rsmPrefServerProcessInfo setStringValue:
                [ NSString stringWithUTF8String: procinfo ]];
        break;
    }
        
    if ( [ self serverIsRunning ] ) {
        [ rsmPrefServerStatusField setStringValue:
                NSLocalizedString( @"Running", @"Running" ) ];
        [ rsmPrefServerStartStopButton setTitle:
                NSLocalizedString( @"Stop", @"Stop" ) ];
        [ rsmPrefServerStartStopButton setAction: @selector( stopRadmindServer: ) ];
    } else {
        [ rsmPrefServerStatusField setStringValue:
                NSLocalizedString( @"Stopped", @"Stopped" ) ];
        [ rsmPrefServerStartStopButton setTitle:
                NSLocalizedString( @"Start", @"Start" ) ];
        [ rsmPrefServerStartStopButton setAction: @selector( startRadmindServer: ) ];
    }
}

/*
 * generateServerArgumentList: take options from fields in
 * preference pane and create an array of CLI options
 */
- ( NSArray * )generateServerArgumentList
{
    int                 ssllevel = [ rsmPrefSSLLevel indexOfSelectedItem ];
    int			zlevel = [ rsmPrefCompressionLevelPopUp indexOfSelectedItem ];
    NSArray             *arglist = nil;
    NSString            *directory = [ rsmPrefRadmindDirectory stringValue ],
                        *serverUmask = [ rsmPrefRadmindUmask stringValue ],
                        *port = [ rsmPrefRadmindPort stringValue ],
                        *maxConnections = [ rsmPrefMaxConnections stringValue ];
                            
    arglist = [ NSArray arrayWithObjects: @"/usr/local/sbin/radmind", @"-w",
                        [ NSString stringWithFormat: @"%d", ssllevel ], nil ];
    
    if ( [ directory length ] ) {
        arglist = [ arglist arrayByAddingObject: @"-D" ];
        arglist = [ arglist arrayByAddingObject: directory ];
    }
    if ( [ serverUmask length ] ) {
        arglist = [ arglist arrayByAddingObject: @"-u" ];
        arglist = [ arglist arrayByAddingObject: serverUmask ];
    }
    if ( [ port length ] ) {
        arglist = [ arglist arrayByAddingObject: @"-p" ];
        arglist = [ arglist arrayByAddingObject: port ];
    }
    if ( [ maxConnections length ] ) {
        arglist = [ arglist arrayByAddingObject: @"-m" ];
        arglist = [ arglist arrayByAddingObject: maxConnections ];
    }
    if ( [ rsmPrefRendezvousSwitch state ] == NSOnState ) {
        arglist = [ arglist arrayByAddingObject: @"-R" ];
    }
    if ( [ rsmPrefUserAuthenticationSwitch state ] == NSOnState ) {
        arglist = [ arglist arrayByAddingObject: @"-U" ];
    }
    if ( zlevel > 0 && zlevel < 10 ) {
	arglist = [ arglist arrayByAddingObjectsFromArray:
		    [ NSArray arrayWithObjects: @"-Z",
		    [ NSString stringWithFormat: @"%d", zlevel ], nil ]];
    }
    
    return( arglist );
}

- ( IBAction )startRadmindServer: ( id )sender
{
    NSArray             *arguments = [ NSArray arrayWithObjects:
					@"-A", @"ExecuteCommand",
					@"-d", @"/private/tmp", @"--", nil ];
    
    arguments = [ arguments arrayByAddingObjectsFromArray:
                            [ self generateServerArgumentList ]];

    [ rsmPrefServerStartStopButton setTitle: NSLocalizedString( @"Starting...",
                                                                @"Starting..." ) ];
    [ rsmPrefServerStartStopButton setAction: @selector( stopRadmindServer: ) ];

    [ self doAuthorizedTask: arguments
            rightName: RADMIND_RIGHT_START ];
    
    [ self refreshServerStatusDisplay ];
}

- ( IBAction )stopRadmindServer: ( id )sender
{
    pid_t               pid = [ self serverPID ];
    NSArray             *arguments = nil;
    
    if ( pid == -1 ) {
        [ self preferencesError: NSLocalizedString(
		@"The Radmind server process is not running.",
		@"The Radmind server process is not running." ) ];
        return;
    }
    
    arguments = [ NSArray arrayWithObjects: @"-A", @"KillProcess",
			@"-d", @"/tmp", @"--",
                        [ NSString stringWithFormat: @"%d", pid ], nil ];
    
    [ rsmPrefServerStartStopButton setTitle: NSLocalizedString( @"Stopping...",
                                                                @"Stopping..." ) ];
    [ rsmPrefServerStartStopButton setAction: @selector( startRadmindServer: ) ];
    
    [ self doAuthorizedTask: arguments
            rightName: RADMIND_RIGHT_STOP ];
    
    [ self refreshServerStatusDisplay ];
}

- ( IBAction )toggleUserAuthentication: ( id )sender
{
    unsigned int    state = [ sender state ];
    BOOL	    userauth = NO;
    
    if ( state == NSOnState ) {
	[ rsmPrefAllowedUsersButton setEnabled: YES ];
	userauth = YES;
    } else {
	[ rsmPrefAllowedUsersButton setEnabled: NO ];
    }
            
    [[ NSUserDefaults standardUserDefaults ]
            setBool: userauth
            forKey: @"RadmindServerUserAuthentication" ];
}

- ( IBAction )editAllowedUsers: ( id )sender
{
    NSArray	    *array;
    NSMutableArray  *allowedUsers;
    NSString	    *tmpdir = [ self sessionTemporaryDirectory ];
    NSString	    *users;
    
    users = [ NSString stringWithContentsOfFile:
		[ tmpdir stringByAppendingPathComponent: @"allow" ]];
    
    /* handle mutable object issues */
    array = [ users componentsSeparatedByString: @"\n" ];
    allowedUsers = [ NSMutableArray arrayWithArray: array ];
    
    /* don't display empty strings */
    if ( [ allowedUsers count ] == 1 &&
	    [[ allowedUsers objectAtIndex: 0 ] length ] == 0 ) {
	[ allowedUsers removeObjectAtIndex: 0 ];
    }
    
    if ( allowedUsers != nil ) {
	[ self setAllowedUsers: allowedUsers ];
    }
    
    [ rsmPrefAllowedUsersTable reloadData ];
    
    [ NSApp beginSheet: rsmPrefAllowedUsersPanel
	    modalForWindow: rsmPrefWindow
	    modalDelegate: self
	    didEndSelector: NULL
	    contextInfo: NULL ];
}

- ( IBAction )saveAllowedUsers: ( id )sender
{
    NSArray	    *taskParameters;
    NSString	    *users, *radmindDirPath;
    NSString	    *allowedUsersTmpPath;
    NSString	    *allowedUsersPath;
    
    users = [[ self allowedUsers ] componentsJoinedByString: @"\n" ];
    
    allowedUsersTmpPath = [[ self sessionTemporaryDirectory ]
			    stringByAppendingPathComponent: @"allow" ];
			  
    
    radmindDirPath = [[ NSUserDefaults standardUserDefaults ]
			objectForKey: @"RadmindServerDirectoryPath" ];
    if ( radmindDirPath == nil ) {
	radmindDirPath = @"/var/radmind";
    }
    allowedUsersPath = [ radmindDirPath stringByAppendingPathComponent: @"allow" ];
			    
    if ( [ users writeToFile: allowedUsersTmpPath atomically: YES ] == NO ) {
	[ self preferencesError: @"Failed to write to %@.", allowedUsersTmpPath ];
	return;
    }
    
    taskParameters = [ NSArray arrayWithObjects: @"-A", @"CopyFile",
			@"--", allowedUsersTmpPath, allowedUsersPath, nil ];
    
    [ self doAuthorizedTask: taskParameters
	    rightName: RADMIND_RIGHT_GENERIC ];
}

- ( IBAction )addAllowedUser: ( id )sender
{
    int		row = [ rsmPrefAllowedUsersTable selectedRow ];
    
    if ( row < 0 || row >= [[ self allowedUsers ] count ] ) {
	row = 0;
    }
    
    [[ self allowedUsers ] insertObject: @"newuser" atIndex: row ];

    [ rsmPrefAllowedUsersTable reloadData ];
    [ rsmPrefAllowedUsersTable selectRow: row
	    byExtendingSelection: NO ];
    [ rsmPrefAllowedUsersTable editColumn: 0
	    row: row withEvent: nil select: YES ];
}

- ( IBAction )removeAllowedUser: ( id )sender
{
    int		row = [ rsmPrefAllowedUsersTable selectedRow ];
    
    if ( row < 0 || row >= [[ self allowedUsers ] count ] ) {
	return;
    }
    
    [[ self allowedUsers ] removeObjectAtIndex: row ];
    
    [ rsmPrefAllowedUsersTable reloadData ];
}

- ( IBAction )dismissAllowedUsersPanel: ( id )sender
{
    [ rsmPrefAllowedUsersPanel orderOut: nil ];
    [ NSApp endSheet: rsmPrefAllowedUsersPanel ];
    [ rsmPrefWindow makeKeyAndOrderFront: nil ];
}

- ( IBAction )setSSLLevel: ( id )sender
{
    int         level = [ rsmPrefSSLLevel indexOfSelectedItem ];
    
    [ rsmPrefUserAuthenticationSwitch setEnabled: NO ];
    
    if ( level == 2 ) {
        [ rsmPrefUserAuthenticationSwitch setEnabled: YES ];
    }
    
    [[ NSUserDefaults standardUserDefaults ]
            setObject: [ NSNumber numberWithInt: level ]
            forKey: @"RadmindServerSSLLevel" ];
}

- ( IBAction )setRadmindDirectoryPath: ( id )sender
{
    NSString    *path = [ rsmPrefRadmindDirectory stringValue ];
    
    if ( ! path || [ path length ] == 0 ) {
        [[ NSUserDefaults standardUserDefaults ]
            removeObjectForKey: @"RadmindServerDirectoryPath" ];
        return;
    }
    
    [[ NSUserDefaults standardUserDefaults ]
            setObject: path
            forKey: @"RadmindServerDirectoryPath" ];
}

- ( IBAction )setRadmindPort: ( id )sender
{
    NSString    *port = [ rsmPrefRadmindPort stringValue ];
    
    if ( ! port || [ port length ] == 0 ) {
        [[ NSUserDefaults standardUserDefaults ]
            removeObjectForKey: @"RadmindServerPort" ];
        return;
    }
    
    [[ NSUserDefaults standardUserDefaults ]
            setObject: port
            forKey: @"RadmindServerPort" ];
}

- ( IBAction )setMaxConnections: ( id )sender
{
    NSString    *max = [ rsmPrefMaxConnections stringValue ];
    
    if ( ! max || [ max length ] == 0 ) {
        [[ NSUserDefaults standardUserDefaults ]
            removeObjectForKey: @"RadmindServerMaxConnections" ];
        return;
    }
    
    [[ NSUserDefaults standardUserDefaults ]
            setObject: max
            forKey: @"RadmindServerMaxConnections" ];
}

- ( IBAction )setUmask: ( id )sender
{
    NSString    *u = [ rsmPrefRadmindUmask stringValue ];
    
    if ( ! u || [ u length ] == 0 ) {
        [[ NSUserDefaults standardUserDefaults ]
            removeObjectForKey: @"RadmindServerUmask" ];
        return;
    }
    
    [[ NSUserDefaults standardUserDefaults ]
            setObject: u
            forKey: @"RadmindServerUmask" ];
}

- ( IBAction )setCompressionLevel: ( id )sender
{
    [[ NSUserDefaults standardUserDefaults ]
		setInteger: [ sender indexOfSelectedItem ]
		forKey: @"RadmindServerCompressionLevel" ];
}

- ( IBAction )setCaseSensitivity: ( id )sender
{
    int		cs = [ sender selectedRow ];

    if ( cs == 1 ) {
	[[ NSUserDefaults standardUserDefaults ] setBool: YES
		forKey: @"RadmindServerCaseInsensitive" ];
    } else {
	[[ NSUserDefaults standardUserDefaults ] setBool: NO
		forKey: @"RadmindServerCaseInsensitive" ];
    }
    
    [[ NSUserDefaults standardUserDefaults ] synchronize ];
}

- ( IBAction )toggleSessionRights: ( id )sender
{
    if ( [ sender state ] == NSOffState ) {
	if ( ! removeright( RADMIND_RIGHT_GENERIC )) {
	    NSLog( @"Failed to remove right" );
	    return;
	}
	[ rsmPrefTimeoutStepper setEnabled: YES ];
    } else if ( [ sender state ] == NSOnState ) {
	int	    timeout = [ rsmPrefTimeoutField intValue ];
	
	if ( timeout == INT_MAX || timeout == INT_MIN ) {
	    NSBeep();
	    [ sender setState: NSOffState ];
	    NSLog( @"Invalid timeout value %@",
		    [ rsmPrefTimeoutField stringValue ] );
	    return;
	}
	
	/* timeout value in auth db is in seconds */
	timeout *= 60;
	
	if ( AuthorizationRightGet( RADMIND_RIGHT_GENERIC, NULL )
		== errAuthorizationSuccess ) {
	    NSLog( @"%s is already in auth db", RADMIND_RIGHT_GENERIC );
	    return;
	}
	
	if ( ! setright( RADMIND_RIGHT_GENERIC, timeout )) {
	    NSLog( @"Failed to set right" );
	    return;
	}
	
	[ rsmPrefTimeoutStepper setEnabled: NO ];
    }
}

- ( IBAction )setAuthorizationTimeout: ( id )sender
{
    if ( ! [ sender isKindOfClass: [ NSStepper class ]] ) {
	NSBeep();
	return;
    }
    
    [ rsmPrefTimeoutField setIntValue: [ sender intValue ]];
}

/* misc methods to assist display of prefs */
- ( void )showPreferencesPanel
{
    if ( ! [ rsmPrefWindow isVisible ] ) {
        [ rsmPrefWindow center ];
    }
    
    [ rsmPrefWindow makeKeyAndOrderFront: nil ];
}

- ( void )showServerPreferences: ( id )sender
{
    NSTabViewItem		*item;
    int				index;
    
    index = [ rsmPrefTabView indexOfTabViewItemWithIdentifier: @"Server" ];
    item = [ rsmPrefTabView tabViewItemAtIndex: index ];
    
    [ self refreshServerStatusDisplay ];
    [ rsmPrefTabView selectTabViewItemWithIdentifier: @"DummyTab" ];
    [ rsmPrefWindow resizeForContentView: rsmPrefServerPaneBox ];
    [ item setView: rsmPrefServerPaneBox ];
    [ rsmPrefTabView selectTabViewItemWithIdentifier: @"Server" ];
    [ rsmPrefWindow setTitle:
	NSLocalizedString( @"Radmind Server Manager: Server Preferences",
			    @"Radmind Server Manager: Server Preferences" ) ];
}

- ( void )showGeneralPreferences: ( id )sender
{
    NSTabViewItem		*item;
    int				index;
    
    index = [ rsmPrefTabView indexOfTabViewItemWithIdentifier: @"General" ];
    item = [ rsmPrefTabView tabViewItemAtIndex: index ];
    
    [ rsmPrefTabView selectTabViewItemWithIdentifier: @"DummyTab" ];
    [ rsmPrefWindow resizeForContentView: rsmPrefGeneralPaneBox ];
    [ item setView: rsmPrefGeneralPaneBox ];
    [ rsmPrefTabView selectTabViewItemWithIdentifier: @"General" ];
    
    [ rsmPrefWindow setTitle:
	NSLocalizedString( @"Radmind Server Manager: General Preferences",
			    @"Radmind Server Manager: General Preferences" ) ];
}

- ( void )showAdvancedPreferences: ( id )sender
{
    CFDictionaryRef		dict = NULL;
    NSTabViewItem		*item;
    int				index;
    
    index = [ rsmPrefTabView indexOfTabViewItemWithIdentifier: @"Advanced" ];
    item = [ rsmPrefTabView tabViewItemAtIndex: index ];
    
    [ rsmPrefDefineRightsSwitch setState: NSOffState ];
    [ rsmPrefTimeoutStepper setEnabled: YES ];
    
    if ( getright( RADMIND_RIGHT_GENERIC, &dict )) {
	if ( [ ( NSDictionary * )dict count ] ) {
	    int			timeout;
	    
	    [ rsmPrefDefineRightsSwitch setState: NSOnState ];
	    
	    timeout = [[ ( NSDictionary * )dict objectForKey:
				@"timeout" ] intValue ];
	    timeout /= 60;
	    
	    [ rsmPrefTimeoutField setIntValue: timeout ];
	    [ rsmPrefTimeoutStepper setIntValue: timeout ];
	    [ rsmPrefTimeoutStepper setEnabled: NO ];
	}
    }
    if ( dict != NULL ) {
	CFRelease( dict );
    }
    
    [ rsmPrefTabView selectTabViewItemWithIdentifier: @"DummyTab" ];
    [ rsmPrefWindow resizeForContentView: rsmPrefAdvancedPaneBox ];
    [ item setView: rsmPrefAdvancedPaneBox ];
    [ rsmPrefTabView selectTabViewItemWithIdentifier: @"Advanced" ];
    [ rsmPrefWindow setTitle:
	NSLocalizedString( @"Radmind Server Manager: Advanced Preferences",
			    @"Radmind Server Manager: Advanced Preferences" ) ];
}

- ( void )reloadDefaults
{
    NSUserDefaults              *d = [ NSUserDefaults standardUserDefaults ];
    NSString                    *path, *port, *u, *max;
    BOOL                        userauth, caseInsensitive;
    int                         level = 0, zlevel = 0;
    
    if (( path = [ d objectForKey: @"RadmindServerDirectoryPath" ] ) != nil ) {
        [ rsmPrefRadmindDirectory setStringValue: path ];
    }
    if (( port = [ d objectForKey: @"RadmindServerPort" ] ) != nil ) {
        [ rsmPrefRadmindPort setStringValue: port ];
    }
    if (( max = [ d objectForKey: @"RadmindServerMaxConnections" ] ) != nil ) {
        [ rsmPrefMaxConnections setStringValue: max ];
    }
    if (( u = [ d objectForKey: @"RadmindServerUmask" ] ) != nil ) {
        [ rsmPrefRadmindUmask setStringValue: u ];
    }
    
    caseInsensitive = [ d boolForKey: @"RadmindServerCaseInsensitive" ];
    [ rsmPrefCaseSensitivityMatrix
	    selectCellAtRow: (int)caseInsensitive
	    column: 0 ];
    
    zlevel = [ d integerForKey: @"RadmindServerCompressionLevel" ];
    [ rsmPrefCompressionLevelPopUp selectItemAtIndex: zlevel ];
    
    userauth = [ d boolForKey: @"RadmindServerUserAuthentication" ];
    [ rsmPrefUserAuthenticationSwitch setState: userauth ];
    
    level = [[ d objectForKey: @"RadmindServerSSLLevel" ] intValue ];
    [ rsmPrefSSLLevel selectItemAtIndex: level ];
    if ( level == 2 ) {
	[ rsmPrefUserAuthenticationSwitch setEnabled: YES ];
	[ rsmPrefAllowedUsersButton setEnabled: YES ];
    }
}

/* methods requiring authorization */
- ( void )doAuthorizedTask: ( NSArray * )authorizedTaskParameters
            rightName: ( char * )rightName
{
    NSArray                     *args = nil;
    pid_t                       pid;
    fd_set                      readmask;
    FILE                        *errors = NULL;
    int                         status;
    int                         wfd[ 2 ], efd[ 2 ];
    char                        authtool[ MAXPATHLEN ] = { 0 };
    char                        **execargs = NULL, buf[ MAXPATHLEN ];
    AuthorizationRef		authRef;
    AuthorizationExternalForm   extAuth;
    AuthorizationItem		right = { rightName, 0, NULL, 0 };
    AuthorizationRights 	rights = { 1, &right };
    OSStatus			err;
    AuthorizationFlags		flags = kAuthorizationFlagDefaults |
                                        kAuthorizationFlagInteractionAllowed |
                                        kAuthorizationFlagExtendRights;
                                        
    err = AuthorizationCreate( NULL, kAuthorizationEmptyEnvironment,
                                    kAuthorizationFlagDefaults, &authRef );
                                    
    if ( err != errAuthorizationSuccess ) {
        [ self preferencesError: @"Failed to create authorization reference" ];
        NSLog( @"AuthorizationCreate failed!" );
        return;
    }
    
    err = AuthorizationCopyRights( authRef, &rights,
                    kAuthorizationEmptyEnvironment, flags, NULL );
    
    if ( err != errAuthorizationSuccess ) {
        [ self preferencesError: @"You must supply a valid administrator"
                                @"name and password." ];
        NSLog( @"AuthorizationCopyRights failed!" );
        return;
    }
    
    if (( err = AuthorizationMakeExternalForm( authRef, &extAuth )) != 0 ) {
	[ self preferencesError: @"AuthorizationMakeExternalForm failed "
		@"with error %d", ( int )err ];
        return;
    }
    
    if ( pathfortool( "rsmauthtool", authtool ) != 0 ) {
	[ self preferencesError: @"Failed to locate authorized tool." ];
        return;
    }
    
    args = [ NSArray arrayWithObject:
                    [ NSString stringWithUTF8String: authtool ]];
    args = [ args arrayByAddingObjectsFromArray: authorizedTaskParameters ];

    if ( pipe( wfd ) < 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return;
    }
    if ( pipe( efd ) < 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return;
    }
    
    [ args argv: &execargs ];
    
    switch ( fork()) {
    case 0:
        /* set up descriptors for IPC */
        ( void )close( wfd[ 1 ] );
        ( void )close( efd[ 0 ] );
        ( void )dup2( wfd[ 0 ], 0 );
        ( void )close( wfd[ 0 ] );
	( void )dup2( efd[ 1 ], 1 );
        ( void )dup2( efd[ 1 ], 2 );
        ( void )close( efd[ 1 ] );
        
        execve( *execargs, execargs, NULL );
        [ self preferencesError: @"execve failed: %s", strerror( errno ) ];
        fflush( stderr );
        _exit( 2 );
        
    case -1:
        [ self preferencesError: @"fork failed: %s", strerror( errno ) ];
        return;
        
    default:
        break;
    }
    
    signal( SIGPIPE, SIG_IGN );
    
    /* write external form authref to child's stdin */
    if ( write( wfd[ 1 ], &extAuth, sizeof( extAuth )) != sizeof( extAuth )) {
        [ self preferencesError: @"write failed: %s", strerror( errno ) ];
        return;
    }
    
    if (( errors = fdopen( efd[ 0 ], "r" )) == NULL ) {
        [ self preferencesError: @"fdopen: %s", strerror( errno ) ];
        return;
    }
    
    setvbuf( errors, NULL, _IONBF, 0 );
    
    FD_ZERO( &readmask );
    
    for ( ;; ) {
        /* read any errors from daemon */
        struct timeval          tv;
        
        tv.tv_sec = 0;
        tv.tv_usec = 1;
        
	FD_ZERO( &readmask );
        FD_SET( efd[ 0 ], &readmask );

        switch ( select(( efd[ 0 ] + 1 ), &readmask, NULL, NULL, &tv )) {
	case -1:
	    NSLog( @"select: %s", strerror( errno ));
	    goto DONE;
	    
	default:
	    break;
        }
        
        if ( ! FD_ISSET( efd[ 0 ], &readmask )) {
            break;
        }
        
        if ( fgets( buf, sizeof( buf ), errors ) == NULL ) {
            break;
        }
            
        [ self preferencesError: @"%s", buf ];
	break;
    }

    pid = wait( &status );

DONE:
    AuthorizationFree( authRef, kAuthorizationFlagDefaults );
}

/* tableview data source methods */
- ( int )numberOfRowsInTableView: ( NSTableView * )tableview
{
    return( [[ self allowedUsers ] count ] );
}

- ( id )tableView: ( NSTableView * )tableview
        objectValueForTableColumn: ( NSTableColumn * )tablecolumn
        row: ( int )row
{
    NSString		*user = @"";
    
    user = [[ self allowedUsers ] objectAtIndex: row ];
    
    return( user );
}

- ( void )tableView: ( NSTableView * )tableview
	setObjectValue: ( id )object
	forTableColumn: ( NSTableColumn * )tablecolumn
	row: ( int )row
{
    /* editing handled by handleChangedText:forTable: */
    return;
}

- ( void )handleChangedText: ( NSString * )text forTable: ( id )table
{
    NSString		*newUser;
    int			row = [ rsmPrefAllowedUsersTable editedRow ];

    if ( row < 0 ) {
	return;
    }
    
    newUser = [[ text copy ] autorelease ];
    
    [[ self allowedUsers ] replaceObjectAtIndex: row withObject: newUser ];
}

@end
