#import "RSMLoadsetManager.h"
#import "RSMTranscriptTitleCell.h"
#import "RAServerAuth.h"
#import "RASTableView.h"
#import "RAOutlineView.h"
#import "RAEditorLauncher.h"
#import "RAAuthKiller.h"
#import "RAServerController.h"
#import "RSMLoadsetEditor.h"
#import "NSTextView(LogMessage).h"

#include <sys/types.h>
#include <sys/param.h>
#include <fcntl.h>
#include <errno.h>
#include <libgen.h>
#include <unistd.h>

#include "argcargv.h"
#include "authtools.h"

/* toolbar identifiers */
#define RSMLMToolbarVerifyTranscriptIdentifier      @"rsmlmverify"
#define RSMLMToolbarUpdateTranscriptIdentifier      @"rsmlmupdate"
#define RSMLMToolbarDeleteTranscriptIdentifier      @"rsmlmdelete"
#define RSMLMToolbarTranscriptRefreshIdentifier     @"rsmlmrefresh"
#define RSMLMToolbarMergeTranscriptIdentifier       @"rsmlmmerge"
#define RSMLMToolbarLocalBackupIdentifier           @"RSMLMLocalBackup"
#define RSMLMToolbarLogIdentifier                   @"RSMLMLog"
#define RSMLMToolbarNewFolderIdentifier             @"RSMLMNewFolder"
#define RSMLMToolbarEditIdentifier		    @"RSMLMEdit"

/* custom pasteboard types */
#define RSMMergeItemsPboardType                     @"RSMMergeItemsPboardType"
#define RSMTranscriptInfoPboardType             @"RSMTranscriptInfoPboardType"

static BOOL		firstNegativePromptDone = NO;
static BOOL		firstPositivePromptDone = NO;
static BOOL		assistFirstNegativeCheckIn = NO;
static BOOL		assistFirstPositiveCheckIn = NO;
static BOOL		completeCheckInOfLoadset = YES;
static BOOL		currentProcessKilled = NO;

@implementation RSMLoadsetManager

- ( id )init
{
    NSPort		*recPort;
    NSPort		*sendPort;
    NSArray		*portArray;
    
    /* DO */
    recPort = [ NSPort port ];
    sendPort = [ NSPort port ];
    _connectionToServerAuth = [[ NSConnection alloc ] initWithReceivePort: recPort
                                                sendPort: sendPort ];
    [ _connectionToServerAuth setRootObject: self ];
    _sa = nil;
    _rsmThreadID = -1;
    portArray = [ NSArray arrayWithObjects: sendPort, recPort, nil ];
    
    [ NSThread detachNewThreadSelector: @selector( connectWithPorts: )
                                        toTarget: [ RAServerAuth class ]
                                        withObject: portArray ];
                                        
    _checkedInTranscripts = nil;
    _tmpTranscripts = nil;
    _transcriptName = nil;
    _currentCommand = -1;
    _mergeInfo = nil;
    _loadsetsToMerge = nil;
    dragOriginRow = -1;
    _delegate = nil;
    _rsmNewFolderHelper = nil;
    
    self = [ super init ];
    return( self );
}

- ( int )loadsetManagerHelperThreadID
{
    return( _rsmThreadID );
}

- ( void )setLoadsetManagerHelperThreadID: ( int )ID
{
    _rsmThreadID = ID;
}

- ( void )setDelegate: ( id )delegate
{
    _delegate = nil;
    _delegate = delegate;
}

- ( id )delegate
{
    return( _delegate );
}

- ( void )toolbarSetup
{
    NSToolbar		*toolbar = [[[ NSToolbar alloc ]
                                        initWithIdentifier: @"rsmlmtoolbar" ] autorelease ];
                                        
    [ toolbar setAllowsUserCustomization: YES ];
    [ toolbar setAutosavesConfiguration: YES ];
    [ toolbar setDisplayMode: NSToolbarDisplayModeIconAndLabel ];
    [ toolbar setDelegate: self ];
    
    [ transcriptWindow setToolbar: toolbar ];
}

/* NSToolbar delegate methods */
- ( NSToolbarItem * )toolbar: ( NSToolbar * )toolbar
                    itemForItemIdentifier: ( NSString * )itemIdent
                    willBeInsertedIntoToolbar: ( BOOL )flag
{
    NSToolbarItem *tbarItem = [[[ NSToolbarItem alloc ]
                                    initWithItemIdentifier: itemIdent ] autorelease ];
    
    if ( [ itemIdent isEqualToString: RSMLMToolbarVerifyTranscriptIdentifier ] ) {
        [ tbarItem setLabel: @"Verify" ];
        [ tbarItem setPaletteLabel: @"Verify" ];
        [ tbarItem setToolTip: @"Verify transcript and its associated files" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"verify.png" ]];
        [ tbarItem setAction: @selector( verifyTranscript: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarUpdateTranscriptIdentifier ] ) {
        [ tbarItem setLabel: @"Update" ];
        [ tbarItem setPaletteLabel: @"Update" ];
        [ tbarItem setToolTip: @"Update information in transcript to match files" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"update.png" ]];
        [ tbarItem setAction: @selector( updateTranscript: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarDeleteTranscriptIdentifier ] ) {
        [ tbarItem setLabel: @"Delete" ];
        [ tbarItem setPaletteLabel: @"Delete" ];
        [ tbarItem setToolTip: @"Delete selected transcript and its files" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"deleteloadset.png" ]];
        [ tbarItem setAction: @selector( deleteTranscript: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarTranscriptRefreshIdentifier ] ) {
        [ tbarItem setLabel: @"Refresh" ];
        [ tbarItem setPaletteLabel: @"Refresh" ];
        [ tbarItem setToolTip: @"Refresh transcript lists" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"refresh.png" ]];
        [ tbarItem setAction: @selector( refreshTranscriptListing: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarMergeTranscriptIdentifier ] ) {
        [ tbarItem setLabel: @"Merge" ];
        [ tbarItem setPaletteLabel: @"Merge" ];
        [ tbarItem setToolTip: @"Merge selected transcripts" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"merge.png" ]];
        [ tbarItem setAction: @selector( getMergeSheet: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarLocalBackupIdentifier ] ) {
        [ tbarItem setLabel: NSLocalizedString( @"Local Backup", @"Local Backup" ) ];
        [ tbarItem setPaletteLabel: NSLocalizedString( @"Local Backup", @"Local Backup" ) ];
        [ tbarItem setToolTip: @"Create local backup of selected transcript" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"localbackup.png" ]];
        [ tbarItem setAction: @selector( localBackup: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarLogIdentifier ] ) {
        [ tbarItem setLabel: NSLocalizedString( @"Log", @"Log" ) ];
        [ tbarItem setPaletteLabel: NSLocalizedString( @"Log", @"Log" ) ];
        [ tbarItem setToolTip: NSLocalizedString( @"Display log of current session",
                                                    @"Display log of current session" ) ];
        [ tbarItem setImage: [ NSImage imageNamed: @"brainlog.png" ]];
        [ tbarItem setAction: @selector( showLogWindow: ) ];
        [ tbarItem setTarget: _rsmLogger ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarNewFolderIdentifier ] ) {
        [ tbarItem setLabel: NSLocalizedString( @"New Folder", @"New Folder" ) ];
        [ tbarItem setPaletteLabel: NSLocalizedString( @"New Folder", @"New Folder" ) ];
        [ tbarItem setToolTip: NSLocalizedString( @"Make new subdirectory",
                                                    @"Make new subdirectory" ) ];
        [ tbarItem setImage: [ NSImage imageNamed: @"newfolder.png" ]];
        [ tbarItem setAction: @selector( getNewSubdirectorySheet: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMLMToolbarEditIdentifier ] ) {
	[ tbarItem setLabel: NSLocalizedString( @"Edit", @"Edit" ) ];
        [ tbarItem setPaletteLabel: NSLocalizedString( @"Edit", @"Edit" ) ];
        [ tbarItem setToolTip: NSLocalizedString( @"Edit in Transcript Editor",
                                                    @"Edit in Transcript Editor" ) ];
        [ tbarItem setImage: [ NSImage imageNamed: @"editor.png" ]];
        [ tbarItem setAction: @selector( openTranscriptInEditor: ) ];
        [ tbarItem setTarget: self ];
    }
    
    return( tbarItem );
}

- ( BOOL )validateToolbarItem: ( NSToolbarItem * )tItem
{
    NSArray		*items = nil;
    BOOL		validate = YES;
    int			row = 0;
    
    /* disable everything while renaming a loadset */
    if ( [ transcriptTable editedRow ] >= 0 ) {
	return( NO );
    }
    
    if (( [ transcriptTable selectedRow ] < 0 &&
                    [[ transcriptWindow firstResponder ] isEqual: transcriptTable ] ) ||
            ( [ tmpTranscriptTable selectedRow ] < 0 &&
                    [[ transcriptWindow firstResponder ] isEqual: tmpTranscriptTable ] )) {
        if ( [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarTranscriptRefreshIdentifier ] ) {
            validate = YES;
        } else if ( [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarVerifyTranscriptIdentifier ] ||
                [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarUpdateTranscriptIdentifier ] ||
                [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarDeleteTranscriptIdentifier ] ||
                [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarMergeTranscriptIdentifier ] ||
                [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarLocalBackupIdentifier ] ||
		[[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarEditIdentifier ] ) {
            validate = NO;
        }
    } else {
	if ( [[ transcriptWindow firstResponder ] isEqual: transcriptTable ] ) {
	    row = [ transcriptTable selectedRow ];
	    items = _checkedInTranscripts;
	} else if ( [[ transcriptWindow firstResponder ] isEqual: tmpTranscriptTable ] ) {
	    row = [ tmpTranscriptTable selectedRow ];
	    items = _tmpTranscripts;
	}
	
	if ( row < 0 || ( [[[ items objectAtIndex: row ]
		objectForKey: @"RadmindServerItemType" ] isEqualToString: @"directory" ]
		&& [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarEditIdentifier ] )) {
	    validate = NO;
	} else if (( [ transcriptTable numberOfSelectedRows ] == 1 &&
                    [[ transcriptWindow firstResponder ] isEqual: transcriptTable ] ) ||
                ( [ tmpTranscriptTable numberOfSelectedRows ] == 1 &&
                    [[ transcriptWindow firstResponder ] isEqual: tmpTranscriptTable ] )) {
            if ( [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarMergeTranscriptIdentifier ] ) {
                validate = NO;
            }
        } else {
            if ( [ transcriptTable numberOfSelectedRows ] != 1 &&
                [[ tItem itemIdentifier ] isEqualToString: RSMLMToolbarLocalBackupIdentifier ] ) {
                validate = NO;
            } else {
                validate = YES;
            }
        }
    }
    
    return( validate );
}

- ( NSArray * )toolbarDefaultItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RSMLMToolbarVerifyTranscriptIdentifier,
                            RSMLMToolbarUpdateTranscriptIdentifier,
                            RSMLMToolbarMergeTranscriptIdentifier,
                            RSMLMToolbarLocalBackupIdentifier,
                            RSMLMToolbarLogIdentifier,
			    RSMLMToolbarEditIdentifier,
                            RSMLMToolbarNewFolderIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            RSMLMToolbarDeleteTranscriptIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RSMLMToolbarTranscriptRefreshIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarAllowedItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RSMLMToolbarVerifyTranscriptIdentifier,
                            RSMLMToolbarUpdateTranscriptIdentifier,
                            RSMLMToolbarMergeTranscriptIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            RSMLMToolbarDeleteTranscriptIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RSMLMToolbarTranscriptRefreshIdentifier,
                            RSMLMToolbarLocalBackupIdentifier, 
                            RSMLMToolbarLogIdentifier, 
                            RSMLMToolbarNewFolderIdentifier,
			    RSMLMToolbarEditIdentifier, nil ];
                            
    return( tmp );
}
/* end required toolbar delegate methods */

- ( void )awakeFromNib
{
    NSRect		detailsBoxRect = [ toolProgressDetailsBox frame ];
    NSRect		progressSheetRect = [ progSheet frame ];
    float		detailsBoxHeight = NSHeight( detailsBoxRect );
    
    NSRect		mergedNameBoxRect = [ mergedNameBox frame ];
    NSRect		mergeSheetRect = [ mergeSheet frame ];
    float		mergedNameBoxHeight = NSHeight( mergedNameBoxRect );
    
    /* hide details box */
    [ toolProgressDetailsBox setFrame: NSMakeRect( detailsBoxRect.origin.x, detailsBoxRect.origin.y,
                                                    NSWidth( detailsBoxRect ), 0.0 ) ];
    [ progSheet setFrame: NSMakeRect( 0.0, 0.0,
                    NSWidth( progressSheetRect ), ( NSHeight( progressSheetRect ) - detailsBoxHeight )) 
                display: NO ];
                
    /* hide name of new merged loadset field */
    [ mergedNameBox setFrame: NSMakeRect( mergedNameBoxRect.origin.x, mergedNameBoxRect.origin.y,
                                                    NSWidth( mergedNameBoxRect ), 0.0 ) ];
    [ mergeSheet setFrame: NSMakeRect( 0.0, 0.0,
                    NSWidth( mergeSheetRect ), ( NSHeight( mergeSheetRect ) - mergedNameBoxHeight )) 
                display: NO ];
                
    [ transcriptTable setTarget: self ];
    [ transcriptTable setDelegate: self ];
    [ transcriptTable setDataSource: self ];
    [ transcriptTable setAction: @selector( transcriptTableClick: ) ];
    [ transcriptTable setDoubleAction: @selector( transcriptTableDoubleClick: ) ];
    [ transcriptTable registerForDraggedTypes: [ NSArray arrayWithObjects:
                                RSMTranscriptInfoPboardType, NSStringPboardType, nil ]];
    [ loadsetsToMergeTable registerForDraggedTypes:
                    [ NSArray arrayWithObject: RSMMergeItemsPboardType ]];
                    
    _rsmLogger = [[ RSMLogger alloc ] init ];
    [ NSBundle loadNibNamed: @"RSMLogger" owner: _rsmLogger ];
    
    [ self toolbarSetup ];
    [ self setupTranscriptTableCells ];
    
    [ transcriptWindow setFrameUsingName: @"RSMTranscriptWindow" ];
    [ transcriptWindow setFrameAutosaveName: @"RSMTranscriptWindow" ];
}

- ( void )setServer: ( id )serverObject andThreadID: ( int )ID
{
    [ self setLoadsetManagerHelperThreadID: ID ];
    [ serverObject setProtocolForProxy: @protocol( RAServerAuthInterface ) ];
    [ serverObject retain ];
    _sa = ( RAServerAuth <RAServerAuthInterface> * )serverObject;
}

- ( int )currentCommand
{
    return( _currentCommand );
}

- ( void )setCurrentCommand: ( int )command
{
    _currentCommand = command;
}

- ( pid_t )currentCommandPID
{
    return( _currentCommandPID );
}

- ( oneway void )setCurrentCommandPID: ( pid_t )pid threadID: ( int )ID
{
    if ( ID != [ self loadsetManagerHelperThreadID ] ) {
        return;
    }
    
    _currentCommandPID = pid;
}

- ( void )setLoadsetsToMerge: ( NSArray * )merge
{
    if ( _loadsetsToMerge != nil ) {
        [ _loadsetsToMerge release ];
        _loadsetsToMerge = nil;
    }
    _loadsetsToMerge = [[ NSMutableArray alloc ] initWithArray: merge ];
}

- ( NSArray * )loadsetsToMerge
{
    return( _loadsetsToMerge );
}

- ( NSArray * )availableTranscripts
{
    return( _checkedInTranscripts );
}

- ( NSString * )transcript
{
    return( _transcriptName );
}

- ( void )setTranscript: ( NSString * )transcript
{
    if ( _transcriptName != nil ) {
        [ _transcriptName release ];
    }
    _transcriptName = [[ NSString alloc ] initWithString: transcript ];
}

- ( NSDictionary * )mergeInfo
{
    return( _mergeInfo );
}

- ( void )setMergeInfo: ( NSDictionary * )merge
{
    if ( _mergeInfo != nil ) {
        [ _mergeInfo release ];
        _mergeInfo = nil;
    }
    _mergeInfo = [[ NSDictionary alloc ] initWithDictionary: merge ];
}

- ( void )transcriptTableClick: ( id )sender
{
    int			row, last;
    
    if ( ! [ sender respondsToSelector: @selector( lastSelectedRow ) ] ) {
	return;
    }
    
    if (( row = [ sender clickedRow ] ) < 0 ) {
	[ sender setLastSelectedRow: -1 ];
	return;
    }
    
    if ((( last = [ sender lastSelectedRow ] ) >= 0 ) && ( row == last )) {
	[ sender restartClickTimer ];
    } else {
	[ sender setLastSelectedRow: row ];
    }
}

- ( void )transcriptTableDoubleClick: ( id )sender
{
    int			row;
    id			item;
    
    /* clear click tracking timer */
    [ sender clearClickTimer ];
    
    if (( row = [ sender clickedRow ] ) < 0 ) {
	return;
    }
    
    item = [ sender itemAtRow: row ];
    if ( [ sender isExpandable: item ] ) {
	if ( [ sender isItemExpanded: item ] ) {
	    [ sender collapseItem: item ];
	} else {
	    [ sender expandItem: item ];
	}
    }
}

- ( void )setupTranscriptTableCells
{
    id 			protoCell = [[ RSMTranscriptTitleCell alloc ] init ];
    NSTableColumn 	*kColumn = [[ transcriptTable tableColumns ] objectAtIndex: 0 ];
    NSTableColumn	*lColumn = [[ tmpTranscriptTable tableColumns ] objectAtIndex: 0 ];
            
    [ protoCell setImage: [ NSImage imageNamed: @"transcript-small.tiff" ]];
    [ protoCell setEditable: YES ];
    [ kColumn setDataCell: protoCell ];
    [ lColumn setDataCell: protoCell ];
    [ kColumn setEditable: NO ];
    [ lColumn setEditable: NO ];
    [ protoCell release ];
}

- ( void )moveLoadsetAtPath: ( NSString * )originalPath
            toPath: ( NSString * )newPath
{
    NSArray             *args = [ NSArray arrayWithObjects:
					    @"-A", @"MoveLoadset",
					    @"-d", [[ self delegate ] sessionTmpDirectory ],
					    @"--", originalPath, newPath, nil ];
    
    [ self setCurrentCommand: RA_LOADSET_RENAME ];
    [ _sa executeCommand: RA_LOADSET_RENAME
	    withArguments: args
            controller: self ];
}

- ( void )modifyTranscriptWithCommand: ( int )command
{
    int			row;
    NSString		*transcript = nil;
    NSString		*message = nil;
    NSArray		*args = nil, *items = nil;
    NSMutableArray	*objectsToDelete = nil;
    NSEnumerator	*en;
    BOOL		caseInsensitive = NO;
    char		toolpath[ MAXPATHLEN ];
    id			activeTable = [ transcriptWindow firstResponder ];
    id                  item, rowNumber;
    
    if ( ! [ activeTable isKindOfClass: [ RAOutlineView class ]] ) return;
    if (( row = [ activeTable selectedRow ] ) < 0 ) return;
    
    if ( [ activeTable isEqual: transcriptTable ] ) {
        items = _checkedInTranscripts;
    } else if ( [ activeTable isEqual: tmpTranscriptTable ] ) {
        items = _tmpTranscripts;
    }

    item = [ activeTable itemAtRow: row ];
    transcript = [ item objectForKey: @"RadmindServerItemPath" ];
    [ self setTranscript: transcript ];
    
    caseInsensitive = [[ NSUserDefaults standardUserDefaults ]
			boolForKey: @"RadmindServerCaseInsensitive" ];
    
    [ toolProgressDetailsTextView setString: @"" ];
    
    switch ( command ) {
    case RA_TRANSCRIPT_VERIFY:
        message = [ NSString stringWithFormat: @"Verifying %@...",
                        [ transcript lastPathComponent ]];
        if ( pathfortool( "lcksum", toolpath ) != 0 ) {
            NSLog( @"access lcksum: %s", strerror( errno ));
            return;
        }
        args = [ NSArray arrayWithObjects: @"-A", @"ExecuteCommand", 
		    @"-d", [[ self delegate ] sessionTmpDirectory ], @"--",
                    [ NSString stringWithUTF8String: toolpath ], nil ];
	if ( caseInsensitive ) {
	    args = [ args arrayByAddingObject: @"-I" ];
	}
	args = [ args arrayByAddingObjectsFromArray: 
		    [ NSArray arrayWithObjects:
		    @"-n", @"-i", @"-%", @"-csha1", transcript, nil ]];
        
        [ toolProgressDetailsTextView logMessage: @"# %s -n -i -%% -c sha1 %@\n",
                                        toolpath, transcript ];
        
        break;
        
    case RA_TRANSCRIPT_UPDATE:
        if ( pathfortool( "lcksum", toolpath ) != 0 ) {
            NSLog( @"access lcksum: %s", strerror( errno ));
            return;
        }
        message = [ NSString stringWithFormat: @"Updating %@...",
                                [ transcript lastPathComponent ]];
        args = [ NSArray arrayWithObjects: @"-A", @"ExecuteCommand", 
		    @"-d", [[ self delegate ] sessionTmpDirectory ], @"--",
                    [ NSString stringWithUTF8String: toolpath ], nil ];
	if ( caseInsensitive ) {
	    args = [ args arrayByAddingObject: @"-I" ];
	}
	args = [ args arrayByAddingObjectsFromArray: 
		    [ NSArray arrayWithObjects:
                    @"-i", @"-%", @"-csha1", transcript, nil ]];
		    
        [ toolProgressDetailsTextView logMessage: @"# %s -i -%% -c sha1 %@\n",
                                        toolpath, transcript ];
        
        break;
        
    case RA_TRANSCRIPT_LOCAL_BACKUP:
        if ( pathfortool( "lmerge", toolpath ) != 0 ) {
            NSBeep();
            NSLog( @"access lmerge: %s", strerror( errno ));
            return;
        }
        
        message = [ NSString stringWithFormat: @"Making local backup of %@...",
                                [ transcript lastPathComponent ]];
        args = [ NSArray arrayWithObjects: @"-A", @"ExecuteCommand",
		@"-d", [[ self delegate ] sessionTmpDirectory ], @"--",
                [ NSString stringWithUTF8String: toolpath ], nil ];
	if ( caseInsensitive ) {
	    args = [ args arrayByAddingObject: @"-I" ];
	}
	args = [ args arrayByAddingObjectsFromArray: 
		    [ NSArray arrayWithObjects:
                @"-u077", transcript, transcript,
                [ NSString stringWithFormat: @"%@-%@.backup", transcript, 
                [[ NSDate date ] descriptionWithCalendarFormat: @"%Y%m%d-%H%M%S"
                                                    timeZone: nil
                                                    locale: nil ]], nil ]];
        [ toolProgressDetailsTextView logMessage:
                                @"# %s -u 077 %@ %@ %@.backup\n",
                                toolpath, transcript, transcript, transcript ];
        break;
        
    case RA_TRANSCRIPT_DELETE:
        en = [ activeTable selectedRowEnumerator ];
        objectsToDelete = [[[ NSMutableArray alloc ] init ] autorelease ];
        while (( rowNumber = [ en nextObject ] ) != nil ) {
            item = [ activeTable itemAtRow: [ rowNumber intValue ]];
            [ objectsToDelete addObject: [ item objectForKey: @"RadmindServerItemPath" ]];
        }

        args = [ NSArray arrayWithObjects: @"-A", @"DeleteLoadset", 
		    @"-d", [[ self delegate ] sessionTmpDirectory ], @"--", nil ];
        args = [ args arrayByAddingObjectsFromArray: objectsToDelete ];
        
        message = [ NSString stringWithFormat: @"Deleting..." ];
        break;
    }

    [ progBar setUsesThreadedAnimation: YES ];
    [ progBar startAnimation: nil ];
    [ progMessage setStringValue: message ];
    [ NSApp beginSheet: progSheet
            modalForWindow: transcriptWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
            
    [ self setCurrentCommand: command ];
    [ _sa executeCommand: command
	    withArguments: args
            controller: self ];
}

- ( void )checkInLoadset: ( id )loadset
{
    NSString		*loadsetPath = loadset;
    NSString            *newpath = nil;
    
    newpath = [ NSString stringWithFormat: @"%@/%@", @"/var/radmind/transcript",
                                            [ loadsetPath lastPathComponent ]];

    [ self setCurrentCommand: RA_TRANSCRIPT_CHECKIN ];
    [ _sa executeCommand: RA_TRANSCRIPT_CHECKIN
                withArguments: [ NSArray arrayWithObjects: @"-A",
		    @"MoveLoadset",
		    @"-d", [[ self delegate ] sessionTmpDirectory ],
		    @"--", loadsetPath, newpath, nil ]
                controller: self ];
}

- ( void )verifyTranscript: ( id )sender
{
    [ self modifyTranscriptWithCommand: RA_TRANSCRIPT_VERIFY ];
}

- ( void )updateTranscript: ( id )sender
{
    [ self modifyTranscriptWithCommand: RA_TRANSCRIPT_UPDATE ];
}

- ( void )deleteTranscript: ( id )sender
{
    id			activeTable = [ transcriptWindow firstResponder ];
    id                  item = nil;
    int			row;
    NSMutableArray	*array = nil;
    NSString		*queryObject = nil;
    
    if ( ! [ activeTable isKindOfClass: [ RAOutlineView class ]] ) {
        return;
    }
    if (( row = [ activeTable selectedRow ] ) < 0 ) {
        return;
    }
    
    if ( [ activeTable isEqual: transcriptTable ] ) {
        array = _checkedInTranscripts;
    } else if ( [ activeTable isEqual: tmpTranscriptTable ] ) {
        array = _tmpTranscripts;
    }
    
    if ( [ activeTable numberOfSelectedRows ] > 1 ) {
        queryObject = @"the selected items and their";
    } else {
        item = [ activeTable itemAtRow: row ];
        queryObject = [ NSString stringWithFormat: @"%@ and its",
                            [ item objectForKey: @"RadmindServerItemName" ]];
    }
        
    NSBeginAlertSheet( [ NSString stringWithFormat:
            @"Delete %@ associated files?", queryObject ],
            @"Delete", @"Cancel", @"", transcriptWindow, self,
            @selector( deleteSheetDidEnd:returnCode:contextInfo: ), NULL, nil,
            @"You will not be able to undo this deletion." );
}

- ( void )deleteSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
        
    case NSAlertAlternateReturn:
    default:
        return;
    }
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    [ self modifyTranscriptWithCommand: RA_TRANSCRIPT_DELETE ];
}

- ( void )getNewSubdirectorySheet: ( id )sender
{
    NSString            *path = nil;
    id                  item = nil;
    int                 row = [ transcriptTable selectedRow ];
    
    if ( row < 0 || ( item = [ transcriptTable itemAtRow: row ] ) == nil ) {
        path = [ NSString stringWithFormat: @"%@", @"/var/radmind/transcript" ];
    } else {
        path = [ item objectForKey: @"RadmindServerItemPath" ];
        if ( [[ item objectForKey: @"RadmindServerItemType" ]
		isEqualToString: @"file" ] ) {
            path = [ path stringByDeletingLastPathComponent ];
        }
    }
    
    if ( _rsmNewFolderHelper == nil ) {
	_rsmNewFolderHelper = [[ RSMNewFolderHelper alloc ] init ];
	[ NSBundle loadNibNamed: @"NewFolder" owner: _rsmNewFolderHelper ];
    }
    [ _rsmNewFolderHelper displayNewFolderSheetForWindow: transcriptWindow
	    basePath: path delegate: self ];
}

- ( void )folderHelperCreateFolderAtPath: ( NSString * )path
{
    NSArray             *args = nil;

    args = [ NSArray arrayWithObjects: @"-A", @"NewDirectory",
		@"-d", [[ self delegate ] sessionTmpDirectory ],
		@"--", path, nil ];

    [ self setCurrentCommand: RA_SUBDIRECTORY_CREATE ];
    [ _sa executeCommand: RA_SUBDIRECTORY_CREATE
            withArguments: args
            controller: self ];
}

- ( void )checkInOffer
{
    NSString		*transcriptName = nil;
    
    if ( [ _tmpTranscripts count ] < 1 ) {
        return;
    }
    
    transcriptName = [[ _tmpTranscripts objectAtIndex: 0 ]
			objectForKey: @"RadmindServerItemPath" ];
    
    NSBeginAlertSheet( @"A newly-uploaded loadset has been detected.",
            @"Verify and Check In", @"Cancel", @"Update and Check In",
            transcriptWindow, self,
            @selector( checkInOfferSheetDidEnd:returnCode:contextInfo: ),
            NULL, nil, @"Would you like the Server Manager to check %@ in "
            @"for you?", [ transcriptName lastPathComponent ] );
}

- ( void )checkInOfferSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    
    [ transcriptWindow makeFirstResponder: tmpTranscriptTable ];
    [ tmpTranscriptTable selectRow: 0 byExtendingSelection: NO ];
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        completeCheckInOfLoadset = YES;
        [ self verifyTranscript: nil ];
        break;
        
    case NSAlertAlternateReturn:
        completeCheckInOfLoadset = NO;
        return;
        
    case NSAlertOtherReturn:
        completeCheckInOfLoadset = YES;
        [ self updateTranscript: nil ];
        break;
    }
}

- ( void )firstNegativeSetup
{
    NSString		*transcriptName = nil;
    
    if ( [ _tmpTranscripts count ] != 1 || firstNegativePromptDone ) {
        return;
    }
    
    transcriptName = [[ _tmpTranscripts objectAtIndex: 0 ] objectForKey: @"RadmindServerItemPath" ];
    [ transcriptWindow makeKeyAndOrderFront: nil ];
    
    NSBeginAlertSheet( @"A newly-uploaded loadset has been detected.",
            @"OK", @"Cancel", @"", transcriptWindow, self,
            @selector( firstNegativeSheetDidEnd:returnCode:contextInfo: ),
            NULL, nil, @"%@ seems to be the first loadset you've created. It's "
            @"probably the base negative transcript from which you'll create other "
            @"loadsets. Would you like the Server Manager to update it and make it "
            @"available to client machines?", [ transcriptName lastPathComponent ] );
}

- ( void )firstNegativeSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    
    firstNegativePromptDone = YES;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
        
    case NSAlertAlternateReturn:
    default:
        return;
    }
    
    if ( [ _tmpTranscripts count ] != 1 ) {
        NSLog( @"count != 1" );
        return;
    }
    
    assistFirstNegativeCheckIn = YES;
    [ tmpTranscriptTable selectRow: 0 byExtendingSelection: NO ];
    [ transcriptWindow makeFirstResponder: tmpTranscriptTable ];
    [ self updateTranscript: nil ];
}

- ( void )notifyFirstCheckInComplete
{
    id		transcript = nil;
    
    if ( [ _checkedInTranscripts count ] != 1 || ! assistFirstNegativeCheckIn ) {
        return;
    }
    
    transcript = [ _checkedInTranscripts objectAtIndex: 0 ];
    
    NSBeginAlertSheet( @"Loadset updated and checked in.", @"OK", @"Cancel", @"",
            transcriptWindow, self, @selector( notifySheetDidEnd:returnCode:contextInfo: ),
            NULL, nil, @"%@ is ready for use, but first you will need to add it to your "
            @"client's command file. Would you like the Server Manager to do this for you?",
            [ transcript objectForKey: @"RadmindServerItemName" ] );
}

- ( void )notifySheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
        
    case NSAlertAlternateReturn:
    default:
        return;
    }
    
    /* we're done with this */
    assistFirstNegativeCheckIn = NO;
    
    [ loadsetEditor assistAdditionOfLoadsetEntry:
            [ NSDictionary dictionaryWithObjectsAndKeys: @"n", @"type",
                [[ _checkedInTranscripts objectAtIndex: 0 ]
		    objectForKey: @"RadmindServerItemName" ], @"tname", nil ]
                    isNegative: YES ];
}

- ( void )firstPositiveSetup
{
    id		transcript = nil;
    
    if ( [ _tmpTranscripts count ] != 1 || firstPositivePromptDone ) {
        return;
    }
    
    transcript = [ _tmpTranscripts objectAtIndex: 0 ];
    [ transcriptWindow makeKeyAndOrderFront: nil ];
    
    NSBeginAlertSheet( @"A newly-uploaded loadset has been detected.",
            @"Verify and Check In", @"Cancel", @"Update and Check In", transcriptWindow, self,
            @selector( firstPositiveSheetDidEnd:returnCode:contextInfo: ),
            NULL, nil, @"%@ is probably the base loadset you've created. "
            @"Would you like the Server Manager to verify it and make it "
            @"available to client machines?",
	    [ transcript objectForKey: @"RadmindServerItemName" ] );
}

- ( void )firstPositiveSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    int			action = 0;
    	
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    
    firstPositivePromptDone = YES;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        action = 0;
        break;
        
    case NSAlertOtherReturn:
        action = 1;
        break;
        
    case NSAlertAlternateReturn:
    default:
        return;
    }
    
    if ( [ _tmpTranscripts count ] != 1 ) {
        NSLog( @"count != 1" );
        return;
    }
    
    assistFirstPositiveCheckIn = YES;
    [ tmpTranscriptTable selectRow: 0 byExtendingSelection: NO ];
    [ transcriptWindow makeFirstResponder: tmpTranscriptTable ];
    
    if ( action == 1 ) {
        [ self updateTranscript: nil ];
    } else {
        [ self verifyTranscript: nil ];
    }
}

- ( void )notifyFirstPositiveCheckInComplete
{
    NSString		*transcriptName = nil;
    
    if ( [ _checkedInTranscripts count ] != 2 || ! assistFirstPositiveCheckIn ) {
        return;
    }
    
    transcriptName = [ self transcript ];
    
    NSBeginAlertSheet( @"Loadset updated and checked in.", @"OK", @"Cancel", @"",
            transcriptWindow, self, @selector( notifyPositiveSheetDidEnd:returnCode:contextInfo: ),
            NULL, nil, @"%@ is ready for use, but first you will need to add it to your "
            @"client's command file. Would you like the Server Manager to do this for you?",
            [ transcriptName lastPathComponent ] );
}

- ( void )notifyPositiveSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    NSString		*transcriptName = nil;
    
    /* we're done with this */
    assistFirstPositiveCheckIn = NO;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
        
    case NSAlertAlternateReturn:
    default:
        return;
    }
    
    if (( transcriptName = [ self transcript ] ) == nil ) {
        NSRunAlertPanel( @"Couldn't get transcript name!",
                @"No transcript name to add to command file.",
                @"OK", @"", @"" );
        return;
    }
    
    [ loadsetEditor assistAdditionOfLoadsetEntry:
        [ NSDictionary dictionaryWithObjectsAndKeys: @"p", @"type",
        [ transcriptName lastPathComponent ], @"tname", nil ]
                    isNegative: NO ];
}

- ( void )localBackup: ( id )sender
{
    [ self modifyTranscriptWithCommand: RA_TRANSCRIPT_LOCAL_BACKUP ];
}

- ( void )getMergeSheet: ( id )sender
{
    id			fp = [ transcriptWindow firstResponder ];
    
    if ( ! [ fp isKindOfClass: [ RAOutlineView class ]] ) {
        return;
    }
    
    if ( ! [ sender isEqual: self ] && [ fp numberOfSelectedRows ] > 1 ) {
        NSEnumerator		*en = nil;
        NSNumber		*row = nil;
        NSMutableArray		*items = [[ NSMutableArray alloc ] init ];
        
        en = [ fp selectedRowEnumerator ];
        
        if ( en == nil ) {
            NSLog( @"NIL" );
        }
        
        while (( row = [ en nextObject ] ) != nil ) {
            int			r = [ row intValue ];
    
            [ items addObject: [ _checkedInTranscripts objectAtIndex: r ]];
        }
        
        [ self setMergeInfo:
                [ NSDictionary dictionaryWithObjectsAndKeys:
                                [ items objectAtIndex: 0 ], @"mergeresult",
                                items, @"tomerge", nil ]];
        [ self setLoadsetsToMerge: items ];
        
        [ mergeMatrix selectCellAtRow: 1 column: 0 ];
        [ self mergeTypeToggle: nil ];
        
        /* XXXX MODIFY TO HANDLE MULTIPLE FILES */
        [[ mergeMatrix cellAtRow: 0 column: 0 ] setTitle:
                [ NSString stringWithFormat: @"Merge %@ into %@",
                [[ items objectAtIndex: 1 ] objectForKey: @"RadmindServerItemName" ],
                [[ items objectAtIndex: 0 ] objectForKey: @"RadmindServerItemName" ]]];

        if ( items != nil ) {
            [ items release ];
        }
        [ loadsetsToMergeTable reloadData ];
    }
            
    [ NSApp beginSheet: mergeSheet
                modalForWindow: transcriptWindow
                modalDelegate: self
                didEndSelector: NULL
                contextInfo: nil ];
}

- ( IBAction )mergeTypeToggle: ( id )sender
{
    NSRect		mergedNameBoxRect = [ mergedNameBox frame ];
    NSRect		mergeSheetRect = [ mergeSheet frame ];
    NSRect		mergedNameViewRect = [ mergedNameView frame ];
    float		mergedNameBoxHeight = NSHeight( mergedNameViewRect );

    if ( [ mergeMatrix selectedRow ] == 0 && mergedNameBoxRect.size.height > 0.0 ) {
        [ newMergeNameField setEnabled: NO ];
        [ newMergeNameField setEditable: NO ];
        [ mergedNameBox setContentView: nil ];
        [ mergedNameBox setFrame: NSMakeRect( mergedNameBoxRect.origin.x, mergedNameBoxRect.origin.y,
                                                    NSWidth( mergedNameBoxRect ), 0.0 ) ];
        [ mergeSheet setFrame: NSMakeRect( mergeSheetRect.origin.x,
                    ( mergeSheetRect.origin.y + mergedNameBoxHeight ),
                    NSWidth( mergeSheetRect ), ( NSHeight( mergeSheetRect ) - mergedNameBoxHeight )) 
                    display: YES animate: YES ];
    } else if ( [ mergeMatrix selectedRow ] == 1 && mergedNameBoxRect.size.height == 0.0 ) {
        [ mergedNameBox setFrame: NSMakeRect( mergedNameBoxRect.origin.x, mergedNameBoxRect.origin.y,
                                                    NSWidth( mergedNameBoxRect ), mergedNameBoxHeight ) ];
        [ mergeSheet setFrame: NSMakeRect( mergeSheetRect.origin.x,
                    ( mergeSheetRect.origin.y - mergedNameBoxHeight ),
                    NSWidth( mergeSheetRect ), ( NSHeight( mergeSheetRect ) + mergedNameBoxHeight )) 
                    display: YES animate: YES ];
        [ mergedNameBox setContentView: mergedNameView ];
        [ newMergeNameField setEnabled: YES ];
        [ newMergeNameField setEditable: YES ];
    }
}

- ( IBAction )mergeLoadsets: ( id )sender
{
    int			mergetype = [ mergeMatrix selectedRow ];
    NSString		*newLoadsetName = nil;
    NSArray		*loadsetsToMerge = nil;
    NSArray		*args = nil;
    NSDictionary	*mergeInfo = nil;
    char		lmerge[ MAXPATHLEN ] = { 0 };
    NSString		*lmergePath = nil;
    BOOL		caseInsensitive = NO;
    
    [ mergeSheet orderOut: nil ];
    [ NSApp endSheet: mergeSheet ];
    
    if (( mergeInfo = [ self mergeInfo ] ) == nil ) {
        NSLog( @"mergeInfo returned nil!" );
	return;
    }
    
    caseInsensitive = [[ NSUserDefaults standardUserDefaults ]
			boolForKey: @"RadmindServerCaseInsensitive" ];

    loadsetsToMerge = [ mergeInfo objectForKey: @"tomerge" ];
    
    switch ( mergetype ) {
    case 1:      
        newLoadsetName = [ NSString stringWithFormat: @"/var/radmind/transcript/%@",
						    [ newMergeNameField stringValue ]];
        [ self setCurrentCommand: RA_LOADSET_MERGE ];
        
        [ progMessage setStringValue:
                [ NSString stringWithFormat: @"Merging to create %@...",
                                            [ newLoadsetName lastPathComponent ]]];
                                            
        break;
        
    case 0:
        newLoadsetName = [ mergeInfo objectForKey: @"mergeresult" ];
        [ self setCurrentCommand: RA_LOADSET_MERGEINTO ];
        
        [ progMessage setStringValue:
		[ NSString stringWithFormat: @"Merging %@ into %@....",
		[[ loadsetsToMerge objectAtIndex: 0 ] objectForKey: @"RadmindServerItemName" ],
		[ newLoadsetName lastPathComponent ]]];
        
        break;
        
    default:
        return;
    }
    
    if ( pathfortool( "lmerge", lmerge ) != 0 ) {
        NSLog( @"access lmerge: %s", strerror( errno ));
        return;
    }
    lmergePath = [ NSString stringWithUTF8String: lmerge ];
                                            
    [ progBar setUsesThreadedAnimation: YES ];
    [ progBar startAnimation: nil ];
    [ NSApp beginSheet: progSheet
            modalForWindow: transcriptWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
	    
    args = [ NSArray arrayWithObjects: @"-A", @"ExecuteCommand",
		@"-d", [[ self delegate ] sessionTmpDirectory ], @"--",
		lmergePath, nil ];
    if ( caseInsensitive ) {
	args = [ args arrayByAddingObject: @"-I" ];
    }

    if ( mergetype == 0 ) {
	args = [ args arrayByAddingObjectsFromArray:
		    [ NSArray arrayWithObjects: @"-f", @"-v",
		    [[ loadsetsToMerge objectAtIndex: 0 ] objectForKey: @"RadmindServerItemPath" ],
		    newLoadsetName, nil ]];

	[ _sa executeCommand: [ self currentCommand ]
                        withArguments: args
                        controller: self ];
    } else if ( mergetype == 1 ) {
        int                     i;
        
        for ( i = 0; i < [[ self loadsetsToMerge ] count ]; i++ ) {
            args = [ args arrayByAddingObject: [[[ self loadsetsToMerge ] objectAtIndex: i ]
						    objectForKey: @"RadmindServerItemPath" ]];
        }
        
        args = [ args arrayByAddingObject: newLoadsetName ];
        
	[ _sa executeCommand: [ self currentCommand ]
                        withArguments: args
                        controller: self ];
    }
}

- ( IBAction )cancelMerge: ( id )sender
{
    [ mergeSheet orderOut: nil ];
    [ NSApp endSheet: mergeSheet ];
}

- ( void )loadIfNeeded
{
    if ( _tmpTranscripts == nil ) {
        _tmpTranscripts = [[ NSMutableArray alloc ] init ];
    }
    if ( _checkedInTranscripts == nil ) {
        _checkedInTranscripts = [[ NSMutableArray alloc ] init ];
        [ self readAvailableCheckedInTranscripts: nil ];
    }
}

- ( void )readAvailableCheckedInTranscripts: ( id )sender
{
    NSString		*tmpdir = [[ self delegate ] sessionTmpDirectory ];
    NSString		*plistPath = [ tmpdir stringByAppendingPathComponent:
					@"transcripts.plist" ];
    
    if ( _checkedInTranscripts != nil ) {
	[ _checkedInTranscripts release ];
	_checkedInTranscripts = nil;
    }
    _checkedInTranscripts = [ NSArray arrayWithContentsOfFile: plistPath ];
    if ( _checkedInTranscripts != nil ) {
	[ _checkedInTranscripts retain ];
    }
    
    [ transcriptTable reloadData ];
}

- ( void )readAvailableTmpTranscripts: ( id )sender
{
    NSString		*tmpdir = [[ self delegate ] sessionTmpDirectory ];
    NSString		*plistPath = [ tmpdir stringByAppendingPathComponent:
					@"tmp_transcripts.plist" ];
    
    if ( _tmpTranscripts != nil ) {
	[ _tmpTranscripts release ];
	_tmpTranscripts = nil;
    }
    _tmpTranscripts = [ NSArray arrayWithContentsOfFile: plistPath ];
    if ( _tmpTranscripts != nil ) {
	[ _tmpTranscripts retain ];
    }

    [ tmpTranscriptTable reloadData ];

    if ( [ _checkedInTranscripts count ] == 0 && [ _tmpTranscripts count ] == 1 ) {
	/* we assume this is the initial negative. offer a walk-through. */
	assistFirstNegativeCheckIn = YES;
	assistFirstPositiveCheckIn = YES;
	[ self firstNegativeSetup ];
    } else if ( [ _checkedInTranscripts count ] == 1 && [ _tmpTranscripts count ] == 0
					    && assistFirstNegativeCheckIn ) {
	[ self notifyFirstCheckInComplete ];
    } else if ( [ _checkedInTranscripts count ] == 1 && [ _tmpTranscripts count ] == 1 ) {
	assistFirstNegativeCheckIn = YES;
	assistFirstPositiveCheckIn = YES;
	[ self firstPositiveSetup ];
    } else if ( [ _checkedInTranscripts count ] == 2 && [ _tmpTranscripts count ] == 0 ) {
	[ self notifyFirstPositiveCheckInComplete ];
    } else if ( [ _tmpTranscripts count ] >= 1 ) {
	[ self checkInOffer ];
    }
}

- ( void )refreshTranscriptListing: ( id )sender
{
    [ self setCurrentCommand: RSM_REFRESH_TRANSCRIPTS ];
    [ _sa executeCommand: RSM_REFRESH_TRANSCRIPTS
                withArguments: [ NSArray arrayWithObjects: @"-A",
		    @"RefreshTranscriptPlist",
		    @"-d", [[ self delegate ] sessionTmpDirectory ],
		    @"--", @"/var/radmind", nil ]
                controller: self ];
}

- ( oneway void )updateDisplayWithString: ( NSString * )string
                    threadID: ( int )ID
{
    NSLock                      *lock = nil;
    
    if ( ID != [ self loadsetManagerHelperThreadID ] ) {
        return;
    }
    
    lock = [[[ NSLock alloc ] init ] autorelease ];
    if ( ! [ lock tryLock ] ) {
        NSLog( @"Crash imminent" );
    }
    
    switch ( [ self currentCommand ] ) {
    case RA_TRANSCRIPT_VERIFY:
    case RA_TRANSCRIPT_UPDATE:
        [ _rsmLogger addText: string isError: NO ];
        [ _rsmLogger addText: @"\n" isError: NO ];
        
        break;
    }
    
    if ( lock ) {
        [ lock unlock ];
    }
}

- ( oneway void )updateToolProgressWithString: ( NSString * )string threadID: ( int )ID
{
    if ( ID != [ self loadsetManagerHelperThreadID ] ) {
        return;
    }
    
    /* check to see if it's a progress string */
    if ( [ string characterAtIndex: 0 ] == '%' && [ string length ] > 1 ) {
        NSString            *progressString = [ string substringFromIndex: 1 ];
        NSArray             *progressComponents = [ progressString componentsSeparatedByString: @" " ];
        double              value = 0.0;
        
        if ( [[ progressComponents objectAtIndex: 0 ] intValue ] == 0 ) {
            [ progBar setIndeterminate: NO ];
            [ progBar setMinValue: 0.0 ];
            [ progBar setMaxValue: 100.0 ];
        }
        
        value = [[ progressComponents objectAtIndex: 0 ] doubleValue ];

        [ progBar setDoubleValue: value ];
    }
}

- ( void )authorizationFailedInThreadWithID: ( int )threadID
{
    if ( threadID != [ self loadsetManagerHelperThreadID ] ) {
        return;
    }
    
    if ( [ progSheet isVisible ] ) {
        [ progSheet orderOut: nil ];
        [ NSApp endSheet: progSheet ];
    }
    NSBeginAlertSheet( @"Authorization failed.",
                          @"OK", @"", @"", transcriptWindow,
                          self,
                          NULL,
                          nil, NULL,
            @"You must authenticate as an administrator to proceed." );
}

- ( oneway void )command: ( int )command finishedWithStatus: ( int )status
            inThread: ( int )ID
{
    BOOL                refreshLoadsetListing = NO;
    
    if ( ID != [ self loadsetManagerHelperThreadID ] ) {
        return;
    }
    
    /*
     * always do this when we're not the active application,
     * which ensures we won't have an unresponsive sheet
     * left over when the user returns to the application.
     */
    if ( [ progSheet isVisible ] || ![ NSApp isActive ] ) {
        [ progBar setIndeterminate: YES ];
        [ progBar stopAnimation: nil ];
        [ progSheet orderOut: nil ];
        [ NSApp endSheet: progSheet ];
    }
    
    [ self setCurrentCommandPID: -1 threadID: ID ];
    
    if ( currentProcessKilled ) {
        NSBeginAlertSheet( NSLocalizedString( @"Task cancelled", @"Task cancelled" ),
            NSLocalizedString( @"OK", @"OK" ), @"", @"", transcriptWindow, self,
            NULL, NULL, NULL, NSLocalizedString( @"Task with process id number %d cancelled.",
                                                @"Task with process id number %d cancelled." ),
                                                [ self currentCommandPID ] );
        currentProcessKilled = NO;
        return;
    }
    
    switch ( [ self currentCommand ] ) {
    case RSM_REFRESH_TRANSCRIPTS:
	refreshLoadsetListing = YES;
	break;
	
    case RA_SUBDIRECTORY_CREATE:
	if ( status != 0 ) {
	    goto TASK_ERROR;
	}
        refreshLoadsetListing = YES;
        break;
        
    case RA_TRANSCRIPT_VERIFY:
        if ( status != 0 && ( ! assistFirstPositiveCheckIn || ! assistFirstNegativeCheckIn )) {
            NSBeginAlertSheet( [ NSString stringWithFormat: @"%@: incorrect",
                            [[ self transcript ] lastPathComponent ]],
                            @"OK", @"", @"", transcriptWindow, self, NULL, NULL, nil,
                            @"" );
        } else if ( ! assistFirstPositiveCheckIn ) {
            NSBeginAlertSheet( [ NSString stringWithFormat: @"%@: verified",
                                        [[ self transcript ] lastPathComponent ]],
                            @"OK", @"", @"", transcriptWindow, self, NULL, NULL, nil,
                            @"" );
        }
        
        if ( [ _checkedInTranscripts count ] == 1 && [ _tmpTranscripts count ] == 1
                    && assistFirstPositiveCheckIn ) {
	    if ( status != 0 ) {
		NSBeginAlertSheet( [ NSString stringWithFormat: @"%@: incorrect",
                                        [[ self transcript ] lastPathComponent ]],
                            @"OK", @"", @"", transcriptWindow, self, NULL, NULL, nil,
                            @"%@ contained some errors that need correcting. Please verify "
			    @"that the loadset was uploaded correctly. If you created the loadset "
			    @"without checksums, select the loadset, and click the Update button. "
			    @"When the loadset has been updated, drag the loadset to the "
			    @"Production Loadsets pane to check it in. From there you can drag "
			    @"and drop it to the command file of your choice.", [ self transcript ] );
		return;
	    }
            [ self checkInLoadset: [ _tmpTranscripts objectAtIndex: 0 ]];
        } else if ( [ _tmpTranscripts count ] >= 1 && completeCheckInOfLoadset ) {
            [ self checkInLoadset: [ self transcript ]];
        }
        
        break;
        
    case RA_TRANSCRIPT_UPDATE:
        if ( status == 1 ) {
            if ( ! assistFirstPositiveCheckIn || ! assistFirstNegativeCheckIn ) {
                NSBeginAlertSheet( [ NSString stringWithFormat: @"%@: updated",
                                [[ self transcript ] lastPathComponent ]],
                                @"OK", @"", @"", transcriptWindow, self, NULL, NULL, nil, @"" );
            }
        } else if ( status == 0 ) {
            if ( ! assistFirstPositiveCheckIn || ! assistFirstNegativeCheckIn ) {
                NSBeginAlertSheet( [ NSString stringWithFormat: @"%@: verified (no updates necessary)",
                                [[ self transcript ] lastPathComponent ]],
                                @"OK", @"", @"", transcriptWindow,
                                self, NULL, NULL, nil, @"" );
            }
        } else {
	    NSBeginAlertSheet( [ NSString stringWithFormat: @"An error occurred while updating %@.",
			    [[ self transcript ] lastPathComponent ]],
                            @"OK", @"", @"", transcriptWindow, self, NULL, NULL, nil, @"" );
	    return;
	}
        
        if ( [ _checkedInTranscripts count ] == 0 && [ _tmpTranscripts count ] == 1 &&
                                                            assistFirstNegativeCheckIn ) {
            [ self checkInLoadset: [ _tmpTranscripts objectAtIndex: 0 ]];
        } else if ( [ _checkedInTranscripts count ] == 1 && [ _tmpTranscripts count ] == 1 &&
                                                            assistFirstPositiveCheckIn ) {
            [ self checkInLoadset: [ _tmpTranscripts objectAtIndex: 0 ]];
        } else if ( [ _tmpTranscripts count ] >= 1 && completeCheckInOfLoadset ) {
            [ self checkInLoadset: [ self transcript ]];
        }
        
        break;
        
    case RA_TRANSCRIPT_DELETE:
        if ( status != 0 ) {
            goto TASK_ERROR;
        }
        
        NSLog( @"%@ deleted", [ self transcript ] );
        
        refreshLoadsetListing = YES;
        break;
        
    case RA_TRANSCRIPT_CHECKIN:
        if ( status != 0 ) {
            goto TASK_ERROR;
        }
        
        NSLog( @"checked in %@", [[ self transcript ] lastPathComponent ] );
        refreshLoadsetListing = YES;
        break;
        
    case RA_LOADSET_MERGE:
    case RA_LOADSET_MERGEINTO:
    case RA_LOADSET_RENAME:
        if ( status != 0 ) {
            goto TASK_ERROR;
        }
        
        refreshLoadsetListing = YES;
        break;
        
    case RA_TRANSCRIPT_LOCAL_BACKUP:
        if ( status != 0 ) {
            goto TASK_ERROR;
        }
        
        NSLog( @"%@ backed up.", [[ self transcript ] lastPathComponent ] );
        refreshLoadsetListing = YES;
        break;
    }
    
    if ( refreshLoadsetListing ) {
        [ self readAvailableCheckedInTranscripts: nil ];
        [ self readAvailableTmpTranscripts: nil ];
    }
    
    return;
    
TASK_ERROR:
    NSLog( @"Error: exit status %d", status );
}

- ( void )toolError: ( char * )error fromThreadWithID: ( int )ID
{
    if ( ID != [ self loadsetManagerHelperThreadID ] ) {
        return;
    }
    
    [ _rsmLogger addText: [ NSString stringWithUTF8String: error ]
                    isError: YES ];
    
    NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                @"%s", NSLocalizedString( @"OK", @"OK" ), @"", @"", error );
}

- ( IBAction )toggleToolProgressDetails: ( id )sender
{
    NSRect		detailsBoxRect = [ toolProgressDetailsBox frame ];
    NSRect		progressSheetRect = [ progSheet frame ];
    NSRect		detailsViewRect = [ toolProgressDetailsView frame ];
    float		detailsBoxHeight = NSHeight( detailsViewRect );
    NSButton		*button = nil;
    
    if ( [ sender isKindOfClass: [ NSButton class ]] ) {
	button = sender;
    }
    
    /* hide details box */
                
    if ( detailsBoxRect.size.height == 0.0 ) {
        [ toolProgressDetailsBox setFrame: NSMakeRect( detailsBoxRect.origin.x, detailsBoxRect.origin.y,
                                                    NSWidth( detailsBoxRect ), detailsBoxHeight ) ];
        [ progSheet setFrame: NSMakeRect( progressSheetRect.origin.x,
                    ( progressSheetRect.origin.y - detailsBoxHeight ),
                    NSWidth( progressSheetRect ), ( NSHeight( progressSheetRect ) + detailsBoxHeight )) 
                display: YES animate: YES ];
        [ toolProgressDetailsBox setContentView: toolProgressDetailsView ];
	if ( button != nil ) {
	    [ button setImage: [ NSImage imageNamed: @"downtriangle.png" ]];
	}
    } else {
        [ toolProgressDetailsBox setContentView: nil ];
        [ toolProgressDetailsBox setFrame: NSMakeRect( detailsBoxRect.origin.x, detailsBoxRect.origin.y,
                                                    NSWidth( detailsBoxRect ), 0.0 ) ];
        [ progSheet setFrame: NSMakeRect( progressSheetRect.origin.x,
                    ( progressSheetRect.origin.y + detailsBoxHeight ),
                    NSWidth( progressSheetRect ), ( NSHeight( progressSheetRect ) - detailsBoxHeight )) 
                display: YES animate: YES ];
	if ( button != nil ) {
	    [ button setImage: [ NSImage imageNamed: @"righttriangle.png" ]];
	}
    }
}

- ( IBAction )openTranscriptInEditor: ( id )sender
{
    NSString			*editor;
    NSString			*tpath;
    NSString			*transcriptLocation = @"";
    RAEditorLauncher		*re = [[ RAEditorLauncher alloc ] init ];
    NSMutableArray		*t = nil;
    
    if ( [[ transcriptWindow firstResponder ] isEqual: transcriptTable ] ) {
        transcriptLocation = @"/var/radmind/transcript";
        t = _checkedInTranscripts;
    } else if ( [[ transcriptWindow firstResponder ] isEqual: tmpTranscriptTable ] ) {
        transcriptLocation = @"/var/radmind/tmp/transcript";
        t = _tmpTranscripts;
    }
    
    editor = [[ NSUserDefaults standardUserDefaults ]
                objectForKey: @"Editor" ];
                
    if ( editor == nil ) {
	editor = @"Radmind Transcript Editor";
    }
    
    tpath = [[ t objectAtIndex: [ transcriptTable selectedRow ]]
		    objectForKey: @"RadmindServerItemPath" ];
    
    if ( ! [ re viewTrans: tpath withTextEditor: editor ] ) {
        NSRunAlertPanel( @"Error launching editor.",
            @"Check the console for error messages", @"OK", @"", @"" );
    }
    
    [ re release ];
}

- ( void )killFinishedWithStatus: ( int )status
{
    NSLog( @"kill exited with status %d", status );
    if ( status != 0 ) {
        NSRunAlertPanel( @"Couldn't cancel current task!",
                @"", @"OK", @"", @"" );
    }
}

- ( IBAction )killCurrentTask: ( id )sender
{
    int			rc;
    pid_t		pid;
    RAAuthKiller	*akill = nil;
    
    rc = NSRunAlertPanel( @"Are you sure you want to cancel?",
            @"", @"Cancel", @"Don't Cancel", @"" );
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
    case NSAlertAlternateReturn:
    default:
        return;
    }
    
    if (( pid = [ self currentCommandPID ] ) <= 1 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"Will not kill process with pid %d",
                                @"Will not kill process with pid %d" ),
                NSLocalizedString( @"OK", @"OK" ), @"", @"", pid );
        return;
    }

    currentProcessKilled = YES;
    akill = [[ RAAuthKiller alloc ] init ];
    [ akill killTool: pid forController: self ];
    [ akill release ];
}

- ( BOOL )handleKeyEvent: ( NSEvent * )theEvent fromTable: ( id )table
{
    int			row;
    unichar 		key = [[ theEvent charactersIgnoringModifiers ]
				    characterAtIndex: 0 ];
    
    if ( ![ table isEqual: transcriptTable ]
	    && ![ table isEqual: tmpTranscriptTable ] ) {
	return( NO );
    }
    
    if (( row = [ table selectedRow ] ) < 0 ) {
	return( NO );
    }

    switch ( key ) {
    case NSDeleteCharacter:
	if ( [[ transcriptWindow firstResponder ] isEqual: table ] ) {
            [ self deleteTranscript: nil ];
        }
        return( YES );
	
    case NSEnterCharacter:
	[[ table selectedCell ] setEditable: YES ];
	[[ table selectedCell ] setScrollable: YES ];
	[ table editColumn: 0 row: row withEvent: nil select: YES ];
	return( YES );

    default:
	break;
    }
    
    return( NO );
}

- ( void )handleChangedText: ( NSString * )text forTable: ( id )table
{
    NSArray		*items;
    NSString		*newPath, *oldPath;
    NSRange		range;
    id			item;
    int			row = [ table editedRow ];
    
    if ( row < 0 ) {
	return;
    }
    
    if ( ! [ table isKindOfClass: [ NSOutlineView class ]] ) {
	return;
    }
    
    /* if there's a path separator in the new string, fail */
    range = [ text rangeOfString: @"/" ];
    if ( range.location != NSNotFound ) {
	NSBeep();
	return;
    }
    /* spaces aren't kosher either */
    range = [ text rangeOfString: @" " ];
    if ( range.location != NSNotFound ) {
	NSBeep();
	return;
    }
    
    if ( [ table isEqual: transcriptTable ] ) {
	items = _checkedInTranscripts;
    } else if ( [ table isEqual: tmpTranscriptTable ] ) {
	items = _tmpTranscripts;
    } else {
	return;
    }
    
    if ( row >= [ items count ] ) {
	return;
    }
    
    item = [ table itemAtRow: row ];
    if ( ! [ item isKindOfClass: [ NSDictionary class ]] ) {
	return;
    }
    
    oldPath = [ item objectForKey: @"RadmindServerItemPath" ];
    newPath = [ NSString stringWithFormat: @"%@/%@",
		[ oldPath stringByDeletingLastPathComponent ], text ];
		
    if ( [ newPath isEqualToString: oldPath ] ) {
	/* don't rename if name hasn't changed */
	return;
    }
    
    [ self moveLoadsetAtPath: oldPath toPath: newPath ];
}

- ( NSMenu * )menuForTable: ( id )table
                tableColumn: ( int )column row: ( int )row
{
    if ( [ table isEqual: transcriptTable ] ) {
        return( transcriptTableMenu );
    } else if ( [ table isEqual: tmpTranscriptTable ] ) {
        return( tmpTableMenu );
    }
    return( nil );
}

- ( int )numberOfRowsInTableView: ( NSTableView * )tableview
{
    unsigned int		rows = 0;
    
    if ( [ tableview isEqual: loadsetsToMergeTable ] ) {
        rows = [ _loadsetsToMerge count ];
    }
    
    return( rows );
}

- ( id )tableView: ( NSTableView * )tableview
        objectValueForTableColumn: ( NSTableColumn * )tablecolumn
        row: ( int )row
{
    id              value = @"";
    
    if ( [ tableview isEqual: loadsetsToMergeTable ] ) {
        if ( [ self loadsetsToMerge ] != nil ) {
            value = [[[ self loadsetsToMerge ] objectAtIndex: row ]
			objectForKey: @"RadmindServerItemName" ];
        }
    }
    
    return( value );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        writeRows: ( NSArray * )rows
        toPasteboard: ( NSPasteboard * )pboard
{
    NSArray		*anArray = [ NSArray array ];
    
    if ( [ tableView isEqual: loadsetsToMergeTable ] ) {
        int		row = [[ rows objectAtIndex: 0 ] intValue ];
        
        [ self setDragOriginRow: row ];
        anArray = [ anArray arrayByAddingObject:
                        [[ self loadsetsToMerge ] objectAtIndex: row ]];
        [ pboard declareTypes: [ NSArray arrayWithObject: RSMMergeItemsPboardType ]
                    owner: self ];
        [ pboard setPropertyList: anArray forType: RSMMergeItemsPboardType ];
        return( YES );
    }
    
    return( NO );
}

- ( NSDragOperation )tableView: ( NSTableView * )tableView
        validateDrop: ( id <NSDraggingInfo> )info
        proposedRow: ( int )row
        proposedDropOperation: ( NSTableViewDropOperation )operation
{
    if ( operation == NSTableViewDropOn ) {
        if ( [ tableView isEqual: loadsetsToMergeTable ] ) {
            [ tableView setDropRow: row dropOperation: NSTableViewDropAbove ];
            return( NSDragOperationMove );
        }
    }

    return( NSDragOperationNone );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        acceptDrop: ( id <NSDraggingInfo> )info
        row: ( int )row
        dropOperation: ( NSTableViewDropOperation )operation
{
    NSPasteboard		*pb;
    id				dragData;

    pb = [ info draggingPasteboard ];
    
    if ( [ tableView isEqual: loadsetsToMergeTable ] ) {
        NSMutableArray			*modMergeOrder = nil;
        
        dragData = [ pb propertyListForType: [ pb availableTypeFromArray:
                        [ NSArray arrayWithObject: RSMMergeItemsPboardType ]]];
        if ( dragData == nil ) {
            return( NO );
        }
        
        if (( modMergeOrder = [[ self loadsetsToMerge ] mutableCopy ] ) == nil ) {
            NSLog( @"nothing to merge!" );
            return( NO );
        }
        
        [ modMergeOrder insertObject:
                    [ modMergeOrder objectAtIndex: [ self dragOriginRow ]]
                        atIndex: row ];
        [ modMergeOrder removeObjectAtIndex: 
		( [ self dragOriginRow ] > row ?
		( [ self dragOriginRow ] + 1 ) : [ self dragOriginRow ] ) ];
                
        [ self setLoadsetsToMerge: modMergeOrder ];
        [ modMergeOrder release ];
                
        [ loadsetsToMergeTable reloadData ];
    }
    
    return( YES );
}

- ( id )draggingSource
{
    return( _draggingSource );
}

- ( void )setDraggingSource: ( id )source
{
    _draggingSource = nil;
    _draggingSource = source;
}

- ( int )dragOriginRow
{
    return( dragOriginRow );
}

- ( void )setDragOriginRow: ( int )row
{
    dragOriginRow = row;
}

/* outline view data source methods */
- ( int )outlineView: ( NSOutlineView * )ov numberOfChildrenOfItem: ( id )item
{
    NSMutableArray	*items = nil;
    NSString		*tDir = nil;
    id			realItem = item;
    int			i = 0;
    
    if ( [ ov isEqual: transcriptTable ] ) {
	items = _checkedInTranscripts;
    } else if ( [ ov isEqual: tmpTranscriptTable ] ) {
	items = _tmpTranscripts;
    }
    
    if ( realItem == nil ) {
	if ( [ ov isEqual: transcriptTable ] ) {
	    tDir = [ NSString stringWithFormat: @"%@/transcript",
			@"/var/radmind" ];
	} else if ( [ ov isEqual: tmpTranscriptTable ] ) {
	    tDir = [ NSString stringWithFormat: @"%@/tmp/transcript",
			@"/var/radmind" ];
	}
	i = 0;
	realItem = [ NSDictionary dictionaryWithObjectsAndKeys:
		    tDir, @"RadmindServerItemPath", nil ];
    } else {
	i = [ items indexOfObject: realItem ];
    }
    
    GET_NODE_CHILDREN( i, realItem, items );
    
    return( [ children count ] );
}

- ( BOOL )outlineView: ( NSOutlineView * )ov isItemExpandable: ( id )item
{
    return ( [[ item objectForKey: @"RadmindServerItemType" ]
		isEqualToString: @"directory" ] );
}

- ( id )outlineView: ( NSOutlineView * )ov child: ( int )index ofItem: ( id )item
{
    NSMutableArray	*items = nil;
    NSString		*tDir = nil;
    id			realItem = item;
    int			i = index;
    
    if ( [ ov isEqual: transcriptTable ] ) {
	items = _checkedInTranscripts;
    } else if ( [ ov isEqual: tmpTranscriptTable ] ) {
	items = _tmpTranscripts;
    }
    
    if ( realItem == nil ) {
	if ( [ ov isEqual: transcriptTable ] ) {
	    tDir = [ NSString stringWithFormat: @"%@/transcript",
			@"/var/radmind" ];
	} else if ( [ ov isEqual: tmpTranscriptTable ] ) {
	    tDir = [ NSString stringWithFormat: @"%@/tmp/transcript",
			@"/var/radmind" ];
	}
	i = 0;
	realItem = [ NSDictionary dictionaryWithObjectsAndKeys:
		    tDir, @"RadmindServerItemPath", nil ];
    } else {
	i = [ items indexOfObject: realItem ];
    }
    
    GET_NODE_CHILDREN( i, realItem, items );

    if ( children == nil || [ children count ] <= index ) {
        return( nil );
    }
   
    return( [ children objectAtIndex: index ] );
}

- ( id )outlineView: ( NSOutlineView * )ov objectValueForTableColumn: ( NSTableColumn * )column
                    byItem: ( id )item
{
    NSString            *key = [ column identifier ];
    
    if ( item == nil ) {
        return( @"" );
    }
    
    return( [ item objectForKey: key ] );
}

- ( void )outlineView:( NSOutlineView * )ov willDisplayCell: ( id )cell
            forTableColumn: ( NSTableColumn * )column item: ( id )item
{
    NSImage             *image = nil;

    if ( ! item || ! cell ) {
        return;
    }
    
    if ( ! [[ column identifier ] isEqualToString: @"RadmindServerItemName" ] ) {
        [ cell setEditable: NO ];
        return;
    }

    image = [ NSImage imageNamed: @"transcript-small.tiff" ];
    
    if ( [[ item objectForKey: @"RadmindServerItemType" ]
	    isEqualToString: @"directory" ] ) {
        image = [ NSImage imageNamed: @"folder.png" ];
    } else if ( [[[ item objectForKey: @"RadmindServerItemName" ] pathExtension ]
                    isEqualToString: @"backup" ] ) {
        image = [ NSImage imageNamed: @"bumbershoot.png" ];
    }
    
    [ cell setEditable: YES ];
    [ cell setImage: image ];
}

/* outline view drag and drop methods */
- ( BOOL )outlineView: ( NSOutlineView * )ov writeItems: ( NSArray * )items
            toPasteboard: ( NSPasteboard * )pboard
{
    [ pboard declareTypes: [ NSArray arrayWithObjects:
                    RSMTranscriptInfoPboardType, NSStringPboardType, nil ]
                    owner: self ];
    [ pboard setPropertyList: items forType: RSMTranscriptInfoPboardType ];
    [ pboard setString: @"test" forType: NSStringPboardType ];
    
    //[ self setDraggingSource: ov ];
        
    return( YES );
}

- ( NSDragOperation )outlineView: ( NSOutlineView * )ov
            validateDrop: ( id <NSDraggingInfo> )info
            proposedItem: ( id )item
            proposedChildIndex: ( int )index
{
    if ( item == nil ) {
        return( NSDragOperationCopy );
    }
    
    [ ov setDropItem: item dropChildIndex: NSOutlineViewDropOnItemIndex ];

    return( NSDragOperationAll );
}

- ( BOOL )outlineView: ( NSOutlineView * )ov
            acceptDrop: ( id <NSDraggingInfo> )info
            item: ( id )item childIndex: ( int )index
{
    NSPasteboard        *pb = [ info draggingPasteboard ];
    id                  draggedObject = nil;
    id                  draggedItem = nil;
    
    if ( pb == nil ) {
        return( NO );
    }
    
    draggedObject = [ pb propertyListForType: RSMTranscriptInfoPboardType ];
    if ( draggedObject == nil ) {
        return( NO );
    }
    
    if (( draggedItem = [ draggedObject objectAtIndex: 0 ] ) == nil ) {
        return( NO );
    }
    
    if ( item == nil ) {
        if ( [ draggedItem objectForKey: @"RadmindServerItemPath" ] == nil ) {
            return( NO );
        }
        
        [ self moveLoadsetAtPath: [ draggedItem objectForKey: @"RadmindServerItemPath" ]
                            toPath: @"/var/radmind/transcript" ];
        return( YES );
    } else if ( [[ item objectForKey: @"RadmindServerItemType" ]
		    isEqualToString: @"directory" ] ) {
        [ self moveLoadsetAtPath: [ draggedItem objectForKey: @"RadmindServerItemPath" ]
                            toPath: [ item objectForKey: @"RadmindServerItemPath" ]];
        return( YES );
    } else {
        NSArray             *mergeItems = [ NSArray arrayWithObject: [ draggedObject objectAtIndex: 0 ]];
        int                 i;

        for ( i = 1; i < [ draggedObject count ]; i++ ) {
            mergeItems = [ mergeItems arrayByAddingObject: [ draggedObject objectAtIndex: i ]];
        }

        [ self setMergeInfo: [ NSDictionary dictionaryWithObjectsAndKeys:
                                [ item objectForKey: @"RadmindServerItemPath" ], @"mergeresult",
                                                    mergeItems, @"tomerge", nil ]];
                                                    
        [[ mergeMatrix cellAtRow: 0 column: 0 ] setTitle:
                [ NSString stringWithFormat: @"Merge %@ into %@",
                    [[ draggedObject objectAtIndex: 0 ] objectForKey: @"RadmindServerItemName" ],
                    [ item objectForKey: @"RadmindServerItemName" ]]];
        
        mergeItems = [ mergeItems arrayByAddingObject: item ];
        
        [ self setLoadsetsToMerge: mergeItems ];
        [ loadsetsToMergeTable reloadData ];
        
        if ( [ draggedObject count ] > 1 ) {
            [ mergeMatrix selectCellAtRow: 1 column: 0 ];
        } else {
            [ mergeMatrix selectCellAtRow: 0 column: 0 ];
        }
        [ self mergeTypeToggle: nil ];
                    
        [ self getMergeSheet: self ];
    }
        
    return( YES );
}

#ifdef notdef
- ( BOOL )outlineView: ( NSOutlineView * )ov
	    shouldEditTableColumn: ( NSTableColumn * )tc
	    item: ( id )item
{
    if ( [ ov editedRow ] >= 0 ) {
NSLog( @"yes" );
	return( YES );
    }
    
    return( YES );
}
#endif notdef
- ( void )dealloc
{
    NSConnection        *conn = [[ ( id )_sa connectionForProxy ] retain ];
    
    if ( [ conn isValid ] ) {
        [ _sa disconnect ];
        [ conn release ];
    }
    
    if ( _rsmNewFolderHelper != nil ) {
	[ _rsmNewFolderHelper release ];
    }
    
    [ super dealloc ];
}

@end
