#import "RSMLoadsetEditor.h"
#import "RAServerAuth.h"
#import "RAServerController.h"
#import "RSMLoadsetManager.h"
#import "RSMTranscriptTitleCell.h"

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

#include "argcargv.h"
#include "authtools.h"

#define RSMLEToolbarRefreshIdentifier		@"rsmlerefresh"
#define RSMLEToolbarSaveCommandFileIdentifier	@"rsmlesavekfile"
#define RSMLEToolbarNewCommandFileIdentifier	@"rsmlenewkfile"
#define RSMLEToolbarDeleteCommandFileIdentifier @"rsmledeletekfile"
#define RSMLEToolbarNewSpecialFileEntryIdentifier @"rsmlenewspecialfile"
#define RSMLEToolbarDeleteCommandFileEntryIdentifier @"rsmledeleteentry"
#define RSMLEToolbarNewFolderIdentifier         @"RSMLENewFolder"

#define RSMLECommandFileLinePboardType          @"RSMLECommandFileLine"
#define RSMTranscriptInfoPboardType             @"RSMTranscriptInfoPboardType"

static BOOL		assistAdditionOfFirstNegative = NO;
static BOOL		assistAdditionOfFirstPositive = NO;

@implementation RSMLoadsetEditor

- ( id )init
{
    NSPort		*recPort;
    NSPort		*sendPort;
    NSArray		*portArray;
    
    /* DO */
    recPort = [ NSPort port ];
    sendPort = [ NSPort port ];
    _connectionToServerAuth = [[ NSConnection alloc ]
					initWithReceivePort: recPort
                                                sendPort: sendPort ];
    [ _connectionToServerAuth setRootObject: self ];
    _sa = nil;
    portArray = [ NSArray arrayWithObjects: sendPort, recPort, nil ];
    _rsmThreadID = -1;
    
    [ NSThread detachNewThreadSelector: @selector( connectWithPorts: )
                                        toTarget: [ RAServerAuth class ]
                                        withObject: portArray ];
                                    
    _currentCommandFileName = nil;
    _currentCommandFile = [[ NSMutableArray alloc ] init ];
    _allCommandFiles = [[ NSMutableArray alloc ] init ];
    _currentCommand = -1;
    _isCommandFileEdited = NO;
    _dragOriginRow = -1;
    _rsmSavingCommandFile = NO;
    
    self = [ super init ];
    return( self );
}

- ( void )setServer: ( id )serverObject andThreadID: ( int )ID
{
    [ serverObject setProtocolForProxy: @protocol( RAServerAuthInterface ) ];
    [ serverObject retain ];
    _sa = ( RAServerAuth <RAServerAuthInterface> * )serverObject;
    [ self setLoadsetEditorHelperThreadID: ID ];
}

- ( int )loadsetEditorHelperThreadID
{
    return( _rsmThreadID );
}

- ( void )setLoadsetEditorHelperThreadID: ( int )ID
{
    _rsmThreadID = ID;
}

- ( void )setDelegate: ( id )delegate
{
    _delegate = nil;
    _delegate = delegate;
}

- ( id )delegate
{
    return( _delegate );
}

- ( int )dragOriginRow
{
    return( _dragOriginRow );
}

- ( void )setDragOriginRow: ( int )row
{
    if ( _dragOriginRow != row ) {
        _dragOriginRow = row;
    }
}

- ( void )toolbarSetup
{
    NSToolbar		*toolbar = [[[ NSToolbar alloc ]
                                        initWithIdentifier: @"rsmletoolbar" ]
					autorelease ];
                                        
    [ toolbar setAllowsUserCustomization: YES ];
    [ toolbar setAutosavesConfiguration: YES ];
    [ toolbar setDisplayMode: NSToolbarDisplayModeIconAndLabel ];
    [ toolbar setDelegate: self ];
    
    [ loadsetWindow setToolbar: toolbar ];
}

- ( void )awakeFromNib
{
    [ self toolbarSetup ];
    [ allCommandFilesTable setTarget: self ];
    [ allCommandFilesTable setAction: @selector( commandFileListClick: ) ];
    [ allCommandFilesTable setDoubleAction:
	    @selector( commandFileListDoubleClick: ) ];
    [ allCommandFilesTable registerForDraggedTypes: [ NSArray arrayWithObjects:
                                RSMKFilePboardType, nil ]];
    [ self setDelegate: serverController ];
    if ( ! [ loadsetWindow setFrameUsingName: @"RSMLoadsetWindow" ] ) {
        NSLog( @"Couldn't restore window size from defaults." );
    }
    [ loadsetWindow setFrameAutosaveName: @"RSMLoadsetWindow" ];
    
    [ self setupKFileEntryTypePopups ];
    [ self setupCommandFileNameCells ];
    [ currentKFileTable setDrawsStripes: YES ];
    [ currentKFileTable setRowHeight: 22.0 ];
    [ currentKFileTable registerForDraggedTypes:
            [ NSArray arrayWithObjects: RSMTranscriptInfoPboardType,
					RSMKFilePboardType,
                                        RSMLECommandFileLinePboardType, nil ]];
}

- ( NSString * )currentCommandFileName
{
    return( _currentCommandFileName );
}

- ( void )setCurrentCommandFileName: ( NSString * )name
{
    if ( _currentCommandFileName != nil ) {
        [ _currentCommandFileName release ];
    }
    _currentCommandFileName = [[ NSString alloc ] initWithString: name ];
}

- ( void )setCurrentCommand: ( int )command
{
    _currentCommand = command;
}

- ( int )currentCommand
{
    return( _currentCommand );
}

- ( BOOL )isCommandFileEdited
{
    return( _isCommandFileEdited );
}

- ( void )setCommandFileEdited: ( BOOL )edit
{
    if ( edit == NO ) {
        [ loadsetWindow setDocumentEdited: NO ];
    } else {
        [ loadsetWindow setDocumentEdited: YES ];
    }
    _isCommandFileEdited = edit;
}

- ( BOOL )savingCommandFile
{
    return( _rsmSavingCommandFile );
}

- ( void )setSavingCommandFile: ( BOOL )saving
{
    _rsmSavingCommandFile = saving;
}

/* NSToolbar delegate methods */
- ( NSToolbarItem * )toolbar: ( NSToolbar * )toolbar
                    itemForItemIdentifier: ( NSString * )itemIdent
                    willBeInsertedIntoToolbar: ( BOOL )flag
{
    NSToolbarItem *tbarItem = [[[ NSToolbarItem alloc ]
                                    initWithItemIdentifier: itemIdent ]
				    autorelease ];
    
    if ( [ itemIdent isEqualToString: RSMLEToolbarRefreshIdentifier ] ) {
        [ tbarItem setLabel: @"Refresh" ];
        [ tbarItem setPaletteLabel: @"Refresh" ];
        [ tbarItem setToolTip: @"Refresh server configuration information" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"refresh.png" ]];
        [ tbarItem setAction: @selector( refreshCommandFileListing: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString:
		    RSMLEToolbarSaveCommandFileIdentifier ] ) {
        [ tbarItem setLabel: @"Save" ];
        [ tbarItem setPaletteLabel: @"Save" ];
        [ tbarItem setToolTip: @"Save open command file" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"savekfile.png" ]];
        [ tbarItem setAction: @selector( saveCommandFile: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString:
		    RSMLEToolbarNewCommandFileIdentifier ] ) {
        [ tbarItem setLabel: @"New Command File" ];
        [ tbarItem setPaletteLabel: @"New Command File" ];
        [ tbarItem setToolTip: @"Create new empty command file" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"newkfile.png" ]];
        [ tbarItem setAction: @selector( newCommandFile: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString:
		    RSMLEToolbarDeleteCommandFileIdentifier ] ) {
        [ tbarItem setLabel: NSLocalizedString( @"Delete", @"Delete" ) ];
        [ tbarItem setPaletteLabel: NSLocalizedString( @"Delete", @"Delete" ) ];
        [ tbarItem setToolTip: @"Delete selected item" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"deleteloadset.png" ]];
        [ tbarItem setAction: @selector( deleteCommandFile: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString:
		    RSMLEToolbarNewSpecialFileEntryIdentifier ] ) {
        [ tbarItem setLabel: @"Add Special Entry" ];
        [ tbarItem setPaletteLabel: @"Add Special Entry" ];
        [ tbarItem setToolTip: @"Add special file to command file" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"addspecial.png" ]];
        [ tbarItem setAction: @selector( addSpecialFileToCurrentCommandFile: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString:
		    RSMLEToolbarDeleteCommandFileEntryIdentifier ] ) {
        [ tbarItem setLabel: @"Delete Entry" ];
        [ tbarItem setPaletteLabel: @"Delete Entry" ];
        [ tbarItem setToolTip: @"Delete entry from command file" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"deletekfile.png" ]];
        [ tbarItem setAction: @selector( deleteCommandFileEntry: ) ];
        [ tbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString:
		    RSMLEToolbarNewFolderIdentifier ] ) {
        [ tbarItem setLabel: @"New Folder" ];
        [ tbarItem setPaletteLabel: @"New Folder" ];
        [ tbarItem setToolTip: @"New Folder" ];
        [ tbarItem setImage: [ NSImage imageNamed: @"newfolder.png" ]];
        [ tbarItem setAction: @selector( getNewSubdirectorySheet: ) ];
        [ tbarItem setTarget: self ];
    }
    
    return( tbarItem );
}

- ( BOOL )validateToolbarItem: ( NSToolbarItem * )tItem
{
    BOOL		validate = YES;
    
    if ( [ self savingCommandFile ] ) {
	return( NO );
    }
    
    if ( [ allCommandFilesTable selectedRow ] < 0 ) {
        if ( [[ tItem itemIdentifier ] isEqualToString:
		    RSMLEToolbarSaveCommandFileIdentifier ] ||
            [[ tItem itemIdentifier ] isEqualToString:
		    RSMLEToolbarNewSpecialFileEntryIdentifier ] ||
            [[ tItem itemIdentifier ] isEqualToString:
		    RSMLEToolbarDeleteCommandFileIdentifier ] ||
            [[ tItem itemIdentifier ] isEqualToString:
		    RSMLEToolbarDeleteCommandFileEntryIdentifier ] ) {
            validate = NO;
        }
    } else {
        if ( [[ tItem itemIdentifier ] isEqualToString:
		    RSMLEToolbarDeleteCommandFileEntryIdentifier ] ) {
            if ( [ currentKFileTable selectedRow ] < 0 ) {
                validate = NO;
            }
        } else if ( [[ tItem itemIdentifier ] isEqualToString:
		    RSMLEToolbarSaveCommandFileIdentifier ] ) {
            if ( ! [ self isCommandFileEdited ] ||
		    [ allCommandFilesTable selectedRow ] < 0 ) {
                validate = NO;
            }
        }
    }
    
    return( validate );
}

- ( NSArray * )toolbarDefaultItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RSMLEToolbarNewCommandFileIdentifier,
                            RSMLEToolbarDeleteCommandFileIdentifier,
                            RSMLEToolbarNewFolderIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            RSMLEToolbarSaveCommandFileIdentifier,
                            RSMLEToolbarNewSpecialFileEntryIdentifier,
                            RSMLEToolbarDeleteCommandFileEntryIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RSMLEToolbarRefreshIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarAllowedItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RSMLEToolbarSaveCommandFileIdentifier,
                            RSMLEToolbarNewCommandFileIdentifier,
                            RSMLEToolbarDeleteCommandFileIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RSMLEToolbarRefreshIdentifier,
                            RSMLEToolbarNewSpecialFileEntryIdentifier,
                            RSMLEToolbarDeleteCommandFileEntryIdentifier, 
                            RSMLEToolbarNewFolderIdentifier, nil ];
                            
    return( tmp );
}
/* end required toolbar delegate methods */

- ( void )refreshCommandFileListing: ( id )sender
{
    [ _sa executeCommand: RSM_REFRESH_KFILES
	    withArguments: [ NSArray arrayWithObjects:
			    @"-A", @"RefreshCommandFilePlist",
			    @"-d", [[ self delegate ] sessionTmpDirectory ],
			    @"/var/radmind", nil ]
	    controller: self ];
}

- ( void )reloadCommandFiles
{
    NSString		*tmpdir = [[ self delegate ] sessionTmpDirectory ];
    NSString		*plistPath = [ tmpdir stringByAppendingPathComponent:
					@"command_files.plist" ];
    
    if ( _allCommandFiles != nil ) {
	[ _allCommandFiles release ];
	_allCommandFiles = nil;
    }
    _allCommandFiles = [ NSArray arrayWithContentsOfFile: plistPath ];
    if ( _allCommandFiles != nil ) {
	[ _allCommandFiles retain ];
    }
    
    [ allCommandFilesTable reloadData ];
    
    [[ self delegate ] setAvailableCommandFiles: _allCommandFiles ];
}

- ( void )commandFileListClick: ( id )sender
{
    int			row = [ sender clickedRow ];
    int			last;
    
    if ( [ sender respondsToSelector: @selector( lastSelectedRow ) ] ) {
	if ( row >= 0 ) {
	    if ((( last = [ sender lastSelectedRow ] ) >= 0 )
		    && ( row == last )) {
		/* click on the same row again, start editing timer */
		[ sender restartClickTimer ];
		return;
	    }
	}
    }
    [ sender setLastSelectedRow: row ];
    
    [ self readSelectedCommandFile: sender ];
}

- ( void )commandFileListDoubleClick: ( id )sender
{
    int			row;
    id			item;
    
    /* clear click tracking timer */
    [ sender clearClickTimer ];
    
    if (( row = [ sender clickedRow ] ) < 0 ) {
	return;
    }
    
    item = [ sender itemAtRow: row ];
    if ( [ sender isExpandable: item ] ) {
	if ( [ sender isItemExpanded: item ] ) {
	    [ sender collapseItem: item ];
	} else {
	    [ sender expandItem: item ];
	}
    }
}

- ( IBAction )readSelectedCommandFile: ( id )sender
{
    int			row = [ allCommandFilesTable selectedRow ];
    id                  item = nil;
    
    if ( [ self savingCommandFile ] ) {
	[ allCommandFilesTable selectRow:
                [ _allCommandFiles indexOfObject: [ self currentCommandFileName ]]
				    byExtendingSelection: NO ];
        return;
    }
    
    if ( [ self isCommandFileEdited ] ) {
        NSBeginAlertSheet( [ NSString stringWithFormat:
				@"Save %@ before opening another file?",
				[[ self currentCommandFileName ]
				lastPathComponent ]],
                @"Save", @"Don't Save", @"Cancel",
                loadsetWindow, self,
		@selector( saveChangesSheetDidEnd:returnCode:contextInfo: ),
                NULL, nil, @"If you do not save, your changes will be lost." );
        return;
    }
    
    if ( row < 0 || ( item = [ allCommandFilesTable itemAtRow: row ] ) == nil ) {
        [ _currentCommandFile removeAllObjects ];
        [ self setCurrentCommandFileName: @"" ];
        [ currentKFileTable reloadData ];
        [ loadsetWindow setTitle: @"Radmind Command File Editor" ];
        return;
    }
    
    if ( [[ item objectForKey: @"RadmindServerItemType" ]
	    isEqualToString: @"directory" ] ) {
        [ _currentCommandFile removeAllObjects ];
        [ self setCurrentCommandFileName: @"" ];
        [ currentKFileTable reloadData ];
        [ loadsetWindow setTitle: @"Radmind Command File Editor" ];
        return;
    }
    
    if ( [[ item objectForKey: @"RadmindServerItemPath" ]
                isEqualToString: [ self currentCommandFileName ]] ) {
        return;
    }
    
    [ self setCurrentCommandFileName: [ item objectForKey: @"RadmindServerItemPath" ]];
    [ self setCurrentCommand: RA_CURRENTKFILE_READ ];
    [ _currentCommandFile removeAllObjects ];
    
    [ _sa executeCommand: RA_CURRENTKFILE_READ
	    withArguments: [ NSArray arrayWithObjects:
			    @"-A", @"CopyFile",
			    @"-d", [[ self delegate ] sessionTmpDirectory ],
                            [ item objectForKey: @"RadmindServerItemPath" ],
			    [ NSString stringWithFormat: @"%@/%@",
				    [[ self delegate ] sessionTmpDirectory ],
				    [ item objectForKey: @"RadmindServerItemName" ]], nil ]
            controller: self ];
}

- ( void )saveChangesSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    switch ( rc ) {
    case NSAlertDefaultReturn:
        /* save */
        [ self saveCommandFile: nil ];
        [ self setCommandFileEdited: NO ];
        break;
    case NSAlertAlternateReturn:
        /* don't save */
        [ self setCommandFileEdited: NO ];
        break;
    case NSAlertOtherReturn:
        /* cancel */
        [ allCommandFilesTable selectRow:
                [ _allCommandFiles indexOfObject: [ self currentCommandFileName ]]
				    byExtendingSelection: NO ];
        return;
    }
    
    [ self readSelectedCommandFile: nil ];
}

- ( oneway void )setCurrentCommandPID: ( pid_t )pid threadID: ( int )ID
{
    /* no-op */
}

- ( void )toolError: ( char * )error fromThreadWithID: ( int )ID
{
    if ( ID != [ self loadsetEditorHelperThreadID ] ) {
        return;
    }
    
    NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
        @"%s", NSLocalizedString( @"OK", @"OK" ), @"", @"", error );
}

- ( oneway void )command: ( int )cmd finishedWithStatus: ( int )status
            inThread: ( int )ID
{
    [ self setSavingCommandFile: NO ];
    
    if ( ID != [ self loadsetEditorHelperThreadID ] ) {
        return;
    }
    
    switch ( cmd ) {
    case RA_CURRENTKFILE_READ:
	[ self readFromTmpCommandFile ];
	break;
	
    case RA_FILE_WRITE:
	if ( status == 0 ) {
	    [ self setCommandFileEdited: NO ];
	}
	break;
	
    case RA_NEWKFILE_CREATE:
    case RSM_REFRESH_KFILES:
    case RA_KFILE_DELETE:
    case RA_SUBDIRECTORY_CREATE:
    case RA_LOADSET_RENAME:
	[ self reloadCommandFiles ];
	break;
	
    default:
	break;
    }
}

- ( void )setupCommandFileNameCells
{
    id 			protoCell = [[ RSMTranscriptTitleCell alloc ] init ];
    NSTableColumn 	*kColumn = [[ allCommandFilesTable tableColumns ] objectAtIndex: 0 ];
            
    [ protoCell setImage: [ NSImage imageNamed: @"transcript-small.tiff" ]];
    [ kColumn setDataCell: protoCell ];
    [ protoCell release];
    
    /* we override the default name editing behaviour */
    [ kColumn setEditable: NO ];
}

- ( void )setupKFileEntryTypePopups
{
    id 			protoCell = [[ NSPopUpButtonCell alloc ]
                                        initTextCell: @"" pullsDown: NO ];
    NSTableColumn 	*kColumn = [ currentKFileTable tableColumnWithIdentifier: @"type" ];
            
    [ protoCell insertItemWithTitle: @"p" atIndex: 0 ];
    [ protoCell insertItemWithTitle: @"n" atIndex: 1 ];
    [ protoCell insertItemWithTitle: @"s" atIndex: 2 ];
    [ protoCell insertItemWithTitle: @"k" atIndex: 3 ];

    [ protoCell setEditable: YES ];
    [ protoCell setBordered: NO ];
    [ kColumn setDataCell: protoCell];
    [ protoCell release];
}

- ( void )moveCommandFileAtPath: ( NSString * )src toPath: ( NSString * )dest
{
    NSArray             *args = [ NSArray arrayWithObjects:
					    @"-A", @"MoveFile",
					    @"-d", [[ self delegate ] sessionTmpDirectory ],
					    @"--", src, dest, nil ];
    
    [ self setCurrentCommand: RA_LOADSET_RENAME ];
    [ _sa executeCommand: RA_LOADSET_RENAME
	    withArguments: args
            controller: self ];
}

- ( void )getNewSubdirectorySheet: ( id )sender
{
    NSString            *path = nil;
    id                  item = nil;
    int                 row = [ allCommandFilesTable selectedRow ];
    
    if ( row < 0 || ( item = [ allCommandFilesTable itemAtRow: row ] ) == nil ) {
        path = [ NSString stringWithFormat: @"%@", @"/var/radmind/command" ];
    } else {
        path = [ item objectForKey: @"RadmindServerItemPath" ];
        if ( [[ item objectForKey: @"RadmindServerItemType" ]
		isEqualToString: @"file" ] ) {
            path = [ path stringByDeletingLastPathComponent ];
        }
    }
    
    if ( _rsmNewFolderHelper == nil ) {
	_rsmNewFolderHelper = [[ RSMNewFolderHelper alloc ] init ];
	[ NSBundle loadNibNamed: @"NewFolder" owner: _rsmNewFolderHelper ];
    }
    [ _rsmNewFolderHelper displayNewFolderSheetForWindow: loadsetWindow
	    basePath: path delegate: self ];
}

- ( void )folderHelperCreateFolderAtPath: ( NSString * )path
{
    NSArray             *args = nil;

    args = [ NSArray arrayWithObjects: @"-A", @"NewDirectory",
		@"-d", [[ self delegate ] sessionTmpDirectory ],
		@"--", path, nil ];

    [ self setCurrentCommand: RA_SUBDIRECTORY_CREATE ];
    [ _sa executeCommand: RA_SUBDIRECTORY_CREATE
            withArguments: args
            controller: self ];
}

- ( IBAction )deleteCommandFileEntry: ( id )sender
{
    int			row = [ currentKFileTable selectedRow ];
    
    if ( row < 0 ) return;

    [ _currentCommandFile removeObjectAtIndex: row ];
    [ currentKFileTable reloadData ];
    [ self setCommandFileEdited: YES ];
}

- ( void )addSpecialFileToCurrentCommandFile: ( id )sender
{
    NSArray		*clients = nil, *loadsets = nil;
    unsigned int	i;
    
    if ( _currentCommandFile == nil ) return;

    [ specialTranscriptPopUp removeAllItems ];
    if (( loadsets = [ transcriptManager availableTranscripts ] ) == nil ) {
        return;
    }
    for ( i = 0; i < [ loadsets count ]; i++ ) {
        [ specialTranscriptPopUp addItemWithTitle: [[ loadsets objectAtIndex: i ]
		objectForKey: @"RadmindServerItemRelativePath" ]];
    }
    
    [ specialClientPopUp removeAllItems ];
    if (( clients = [[ self delegate ] validClients ] ) == nil ) {
        return;
    }
    [ specialClientPopUp addItemsWithTitles: clients ];
    
    [ NSApp beginSheet: specialSheet
            modalForWindow: loadsetWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
}

- ( IBAction )addSpecialFile: ( id )sender
{
    NSString		*specialTranscript = [ specialTranscriptPopUp titleOfSelectedItem ];
    NSString		*clientName = [ specialClientPopUp titleOfSelectedItem ];
    
    [ specialSheet orderOut: nil ];
    [ NSApp endSheet: specialSheet ];
    
    [ self setCurrentCommand: RA_ADD_SPECIAL ];
    [ _sa executeCommand: RA_ADD_SPECIAL
	    withArguments: [ NSArray arrayWithObjects: @"-A", @"AddSpecial",
	    @"-d", [[ self delegate ] sessionTmpDirectory ], @"--",
	    [ self currentCommandFileName ], clientName,
	    specialTranscript, nil ] controller: self ];
}

- ( IBAction )cancelAddSpecialFile: ( id )sender
{
    [ specialSheet orderOut: nil ];
    [ NSApp endSheet: specialSheet ];
}

- ( IBAction )saveCommandFile: ( id )sender
{
    int			fd, i;
    FILE		*tmpkfile;
    char		tmpkfilepath[ MAXPATHLEN ];
    char		kfilepath[ MAXPATHLEN ];

    if ( [[ self currentCommandFileName ] length ] >= MAXPATHLEN ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                @"%@ path too long.", NSLocalizedString( @"OK", @"OK" ), @"", @"",
                [ self currentCommandFileName ] );
        return;
    }
    strcpy( kfilepath, [[ self currentCommandFileName ] UTF8String ] );

    if ( snprintf( tmpkfilepath, MAXPATHLEN, "%s/rsm-kfile.XXXXXX.K",
	    [[[ self delegate ] sessionTmpDirectory ] UTF8String ] )
	    >= MAXPATHLEN ) {
	NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                @"%@/rsm-kfile.XXXXXX.K: %s", NSLocalizedString( @"OK", @"OK" ),
                @"", @"", [[ self delegate ] sessionTmpDirectory ], strerror( errno ));
        return;
    }
    if (( fd = mkstemps( tmpkfilepath, strlen( ".K" ))) < 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                @"mkstemp %s: %s", NSLocalizedString( @"OK", @"OK" ),
                @"", @"", tmpkfilepath, strerror( errno ));
        return;
    }
    if (( tmpkfile = fdopen( fd, "w+" )) == NULL ) {
        NSRunAlertPanel( @"Error",
                @"Fdopen %s: %s", @"OK", @"", @"", tmpkfilepath, strerror( errno ));
        return;
    }
    
    for ( i = 0; i < [ _currentCommandFile count ]; i++ ) {
        NSDictionary	*dict = [ _currentCommandFile objectAtIndex: i ];
	
	if ( [[ dict objectForKey: @"type" ] characterAtIndex: 0 ] == '#' ) {
	    fprintf( tmpkfile, "%s", [[ dict objectForKey: @"tname" ] UTF8String ] );
	    continue;
	}
	
        fprintf( tmpkfile, "%s %s\n",
                ( char * )[[ dict objectForKey: @"type" ] UTF8String ],
                ( char * )[[ dict objectForKey: @"tname" ] UTF8String ] );
    }
    ( void )fclose( tmpkfile );
    
    [ self setSavingCommandFile: YES ];
    [ self setCurrentCommand: RA_FILE_WRITE ];
    [ _sa executeCommand: RA_FILE_WRITE
	    withArguments: [ NSArray arrayWithObjects:
		    @"-A", @"MoveFile", @"-d", [[ self delegate ] sessionTmpDirectory ],
		    @"--", [ NSString stringWithUTF8String: tmpkfilepath ],
		    [ NSString stringWithUTF8String: kfilepath ], nil ]
            controller: self ];
}

- ( void )newCommandFile: ( id )sender
{
    unsigned int		i;
    
    [ newCommandFileContents removeAllItems ];
    [ newCommandFileContents addItemWithTitle: @"Empty" ];
    for ( i = 0; i < [ _allCommandFiles count ]; i++ ) {
	if ( [[[ _allCommandFiles objectAtIndex: i ]
		objectForKey: @"RadmindServerItemType" ]
		isEqualToString: @"directory" ] ) {
	    continue;
	}
	
        [ newCommandFileContents addItemWithTitle:
                    [[ _allCommandFiles objectAtIndex: i ]
		    objectForKey: @"RadmindServerItemRelativePath" ]];
    }
    [ newCommandFileContents selectItemWithTitle: @"Empty" ];
    
    [ NSApp beginSheet: newCommandFileSheet
            modalForWindow: loadsetWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
}

- ( IBAction )createNewKFile: ( id )sender
{
    NSString		*src, *dst;
    NSArray		*fileArgs, *cmdArgs;
    
    [ self dismissNewKFileSheet: nil ];
    
    dst = [ NSString stringWithFormat: @"%@/%@", @"/var/radmind/command",
		    [ newCommandFileNameField stringValue ]];
	
    cmdArgs = [ NSArray arrayWithObjects:
		    @"-A", @"NewCommandFile",
		    @"-d", [[ self delegate ] sessionTmpDirectory ],
		    @"--", nil ];
    
    if ( [[ newCommandFileContents titleOfSelectedItem ]
	    isEqualToString: NSLocalizedString( @"Empty", @"Empty" ) ] ) {
	fileArgs = [ NSArray arrayWithObject: dst ];
    } else {
	src = [ NSString stringWithFormat: @"%@/%@", @"/var/radmind/command",
		    [ newCommandFileContents titleOfSelectedItem ]];
	fileArgs = [ NSArray arrayWithObjects: src, dst, nil ];
    }
    
    cmdArgs = [ cmdArgs arrayByAddingObjectsFromArray: fileArgs ];

    [ _sa executeCommand: RA_NEWKFILE_CREATE
	    withArguments: cmdArgs
            controller: self ];
}

- ( IBAction )dismissNewKFileSheet: ( id )sender
{
    [ newCommandFileSheet orderOut: nil ];
    [ NSApp endSheet: newCommandFileSheet ];
}

- ( IBAction )deleteCommandFile: ( id )sender
{
    int		    row = [ allCommandFilesTable selectedRow ];
    id		    item;
    
    if ( row < 0 ) {
	return;
    }
    item = [ _allCommandFiles objectAtIndex: row ];
    
    NSBeginAlertSheet( [ NSString stringWithFormat: @"Delete %@?",
                                [ item objectForKey: @"RadmindServerItemName" ]],
                        @"Delete", @"Cancel", @"", loadsetWindow, self,
                        @selector( deleteKFileSheetDidEnd:returnCode:contextInfo: ),
                        NULL, nil, @"You will not be able to undo this deletion." );
}

- ( void )deleteKFileSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    int			row = [ allCommandFilesTable selectedRow ];
    id			item;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
    default:
    case NSAlertAlternateReturn:
        return;
    }
    
    if ( row < 0 ) {
	return;
    }
    item = [ _allCommandFiles objectAtIndex: row ];
    
    [ self setCurrentCommand: RA_KFILE_DELETE ];
    [ _sa executeCommand: RA_KFILE_DELETE
	    withArguments: [ NSArray arrayWithObjects: @"-A", @"Delete",
		    @"-d", [[ self delegate ] sessionTmpDirectory ],
		    @"--", [ item objectForKey: @"RadmindServerItemPath" ], nil ]
            controller: self ];
}

- ( void )readFromTmpCommandFile
{
    NSMutableDictionary *kLine = nil;
    NSString		*kfile;
    FILE		*f;
    int			tac;
    char		tmp[ MAXPATHLEN ], buf[ MAXPATHLEN ];
    char		**targv;
    
    kfile = [ NSString stringWithFormat: @"%@/%@",
		[[ self delegate ] sessionTmpDirectory ],
		[[ self currentCommandFileName ] lastPathComponent ]];
    
    if (( f = fopen( [ kfile UTF8String ], "r" )) == NULL ) {
	NSLog( @"fopen %@: %s", kfile, strerror( errno ));
	return;
    }
    
    while ( fgets( buf, MAXPATHLEN, f ) != NULL ) {
	strcpy( tmp, buf );
	tac = argcargv( buf, &targv );
	
	if ( tac == 0 ) {
	    kLine = [ NSMutableDictionary dictionaryWithObjectsAndKeys:
			@"#", @"type", @" ", @"tname", nil ];
	    [ _currentCommandFile addObject: kLine ];
	    continue;
	}
	
	switch ( *targv[ 0 ] ) {
	case 'n':
	case 'p':
	case 's':
	case 'k':
	    kLine = [ NSMutableDictionary dictionaryWithObjectsAndKeys:
			[ NSString stringWithUTF8String: targv[ 0 ]], @"type",
			[ NSString stringWithUTF8String: targv[ 1 ]], @"tname", nil ];
			
	    break;
	    
	case '#':
	    kLine = [ NSMutableDictionary dictionaryWithObjectsAndKeys:
			@"#", @"type",
			[ NSString stringWithUTF8String: tmp ], @"tname", nil ];
			
	    break;
	    
	default:
	    NSLog( @"Invalid line in command file: %s", buf );
	    return;
	}
	
	[ _currentCommandFile addObject: kLine ];
    }
    
    if ( fclose( f ) != 0 ) {
	NSLog( @"fclose: %s", strerror( errno ));
    }
    
    [ currentKFileTable reloadData ];
}

- ( BOOL )assistAdditionOfLoadsetEntry: ( NSDictionary * )entry isNegative: ( BOOL )negative
{
    NSMutableDictionary		*newEntry = nil;
    
    if ( entry == nil ) {
        return( NO );
    }
    
    newEntry = [ entry mutableCopy ];
    
    if ( [ _allCommandFiles count ] != 1 ) {
        return( NO );
    }

    [ allCommandFilesTable selectRow: 0 byExtendingSelection: NO ];
    [ self setCurrentCommandFileName: [[ _allCommandFiles objectAtIndex: 0 ]
				    objectForKey: @"RadmindServerItemPath" ]];
    
    if ( negative ) {
        assistAdditionOfFirstNegative = YES;
        [ _currentCommandFile removeAllObjects ];
    } else {
        assistAdditionOfFirstPositive = YES;
    }
    
    if ( ! [ _currentCommandFile containsObject: newEntry ] ) {
        [ _currentCommandFile insertObject: newEntry atIndex: 0 ];
    }
    
    [ newEntry release ];
    [ currentKFileTable reloadData ];
    [ self saveCommandFile: nil ];
    
    return( YES );
}

- ( void )addLoadsetEntry: ( NSDictionary * )entry toCommandFile: ( NSString * )commandFile
{
}

/* RASTableView delegate methods */

- ( BOOL )handleKeyEvent: ( NSEvent * )theEvent fromTable: ( RASTableView * )table
{
    int			row;
    unichar 		key = [[ theEvent charactersIgnoringModifiers ]
				    characterAtIndex: 0 ];
    
    if (( row = [ table selectedRow ] ) < 0 ) {
	return( NO );
    }
    
    switch ( key ) {
    case NSDeleteCharacter:
        if ( [[ loadsetWindow firstResponder ] isEqual: allCommandFilesTable ] ) {
            [ self deleteCommandFile: nil ];
        } else if ( [[ loadsetWindow firstResponder ] isEqual: currentKFileTable ] ) {
            [ self deleteCommandFileEntry: nil ];
        }
        return( YES );
	
    case NSEnterCharacter:
	[[ table selectedCell ] setEditable: YES ];
	[[ table selectedCell ] setScrollable: YES ];
	[ table editColumn: 0 row: row withEvent: nil select: YES ];
	return( YES );
    }
    
    return( NO );
}

- ( void )handleChangedText: ( NSString * )text forTable: ( id )table
{
    NSArray		*items;
    NSString		*newPath, *oldPath;
    NSRange		range;
    id			item;
    int			row = [ table editedRow ];
    
    if ( row < 0 ) {
	return;
    }
    
    if ( ! [ table isKindOfClass: [ NSOutlineView class ]] ) {
	return;
    }
    
    /* if there's a path separator in the new string, fail */
    range = [ text rangeOfString: @"/" ];
    if ( range.location != NSNotFound ) {
	NSBeep();
	return;
    }
    
    items = _allCommandFiles;
    
    if ( row >= [ items count ] ) {
	return;
    }
    
    item = [ table itemAtRow: row ];
    if ( ! [ item isKindOfClass: [ NSDictionary class ]] ) {
	return;
    }
    
    oldPath = [ item objectForKey: @"RadmindServerItemPath" ];
    newPath = [ NSString stringWithFormat: @"%@/%@",
		[ oldPath stringByDeletingLastPathComponent ], text ];
		
    [ self moveCommandFileAtPath: oldPath toPath: newPath ];
}

- ( int )numberOfRowsInTableView: ( NSTableView * )tableView
{
    return( [ _currentCommandFile count ] );
}

- ( void )tableView: ( NSTableView * )tableView willDisplayCell: ( id )aCell
        forTableColumn: ( NSTableColumn * )column row: ( int )row
{
    if ( row < 0 ) return;
 
    if ( [[ column identifier ] isEqualToString: @"tname" ] ) {
	[ aCell setTextColor: [ NSColor blackColor ]];
	
	if ( [[[ _currentCommandFile objectAtIndex: row ] objectForKey: @"tname" ]
		    characterAtIndex: 0 ] == '#' ) {
	    [ aCell setTextColor: [ NSColor blueColor ]];
	} else if ( [[[ _currentCommandFile objectAtIndex: row ] objectForKey: @"type" ]
		    characterAtIndex: 0 ] == 'n' ) {
	    [ aCell setTextColor: [ NSColor redColor ]];
	}
	
	[ aCell setStringValue: [[ _currentCommandFile objectAtIndex: row ] objectForKey: @"tname" ]];
	[ aCell setEditable: NO ];
    } else if ( [[ column identifier ] isEqualToString: @"type" ] ) {                      
	[ aCell setEnabled: YES ];
	
	if ( [[[ _currentCommandFile objectAtIndex: row ] objectForKey: @"type" ]
		    characterAtIndex: 0 ] == '#' ) {
	    [ aCell setEnabled: NO ];
	}
	[ aCell selectItemWithTitle:
		[[ _currentCommandFile objectAtIndex: row ] objectForKey: @"type" ]];
    }
}

- ( id )tableView: ( NSTableView * )tableView
            objectValueForTableColumn: ( NSTableColumn * )column
            row: ( int )row
{
    /* set in willDisplayCell... */
    return( nil );
}

- ( void )tableView: ( NSTableView * )tableView setObjectValue: ( id )object
        forTableColumn: ( NSTableColumn * )tableColumn row: ( int )row
{
    if ( row < 0 ) return;

    if ( [[ tableColumn identifier ] isEqualToString: @"type" ] ) {
        int		index = [ object intValue ];
        NSString	*oldtype = [[ _currentCommandFile objectAtIndex: row ] objectForKey: @"type" ];
        NSString	*newtype;
        
        if ( [ oldtype characterAtIndex: 0 ] == 's' ) {
            NSBeep();
            return;
        }
        
        switch ( index ) {
        default:
        case 0:
            newtype = @"p";
            break;
        case 1:
            newtype = @"n";
            break;
        case 2:
            /* don't let users set type to special. make them use the button */
            return;
	case 3:
	    newtype = @"k";
	    break;
        }
        
        if ( ! [ newtype isEqualToString: oldtype ] ) {
            [[ _currentCommandFile objectAtIndex: row ]
                    setObject: newtype forKey: @"type" ];
            [ self setCommandFileEdited: YES ];
        }
    }
}

- ( BOOL )tableView: ( NSTableView * )aTableView
        shouldEditTableColumn: ( NSTableColumn * )aTableColumn
        row: ( int )rowIndex
{
    if ( [ aTableView editedRow ] >= 0 ) return( YES );
    return( NO );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        writeRows: ( NSArray * )rows
        toPasteboard: ( NSPasteboard * )pboard
{
    id			dsrc;
    NSArray		*anArray = nil;
    
    if ( [ tableView isEqual: currentKFileTable ] ) {
        dsrc = [ _currentCommandFile copy ];
        
        anArray = [ NSArray arrayWithObject:
                        [ dsrc objectAtIndex: [[ rows objectAtIndex: 0 ] intValue ]]];
        [ self setDragOriginRow: [[ rows objectAtIndex: 0 ] intValue ]];
        [ pboard declareTypes: [ NSArray arrayWithObjects: NSStringPboardType,
                                    RSMLECommandFileLinePboardType, nil ]
                    owner: self ];
        [ pboard setPropertyList: anArray forType: RSMLECommandFileLinePboardType ];
        [ pboard setPropertyList: nil forType: NSStringPboardType ];
        [ dsrc release ];
        return( YES );
    }
    
    return( NO );
}

- ( NSDragOperation )tableView: ( NSTableView * )tableView
        validateDrop: ( id <NSDraggingInfo> )info
        proposedRow: ( int )row
        proposedDropOperation: ( NSTableViewDropOperation )operation
{
    if ( row < 0 || [ allCommandFilesTable selectedRow ] < 0 ) return( NSDragOperationNone );

    if ( operation == NSTableViewDropAbove
            && tableView == currentKFileTable ) {
        return( NSDragOperationCopy );
    } else if ( operation == NSTableViewDropOn
            && tableView == currentKFileTable ) {
        [ tableView setDropRow: row dropOperation: NSTableViewDropAbove ];
    }

    return( NSDragOperationNone );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        acceptDrop: ( id <NSDraggingInfo> )info
        row: ( int )row
        dropOperation: ( NSTableViewDropOperation )operation
{
    NSPasteboard		*pb = nil;
    NSArray                     *types = nil;
    NSString                    *availableType = nil;
    id				dragData = nil;

    if ( row < 0 ) {
        return( NO );
    }
    
    pb = [ info draggingPasteboard ];
    types = [ NSArray arrayWithObjects: RSMTranscriptInfoPboardType,
                                        RSMLECommandFileLinePboardType,
					RSMKFilePboardType,
                                        NSStringPboardType, nil ];
                                        
    availableType = [ pb availableTypeFromArray: types ];
    dragData = [ pb propertyListForType: availableType ];
            
    if ( dragData == nil ) {
        return( NO );
    }

    if ( [ tableView isEqual: currentKFileTable ] ) {
        if ( [ availableType isEqualToString: RSMLECommandFileLinePboardType ] ) {
            [ _currentCommandFile insertObject: [ _currentCommandFile objectAtIndex: [ self dragOriginRow ]]
                            atIndex: row ];
            [ _currentCommandFile removeObjectAtIndex: 
                        ( [ self dragOriginRow ] > row ?
                        ( [ self dragOriginRow ] + 1 ) : [ self dragOriginRow ] ) ];
            [ currentKFileTable reloadData ];
        } else if ( [ availableType isEqualToString: RSMTranscriptInfoPboardType ] ||
		    [ availableType isEqualToString: RSMKFilePboardType ] ) {
            int			i;

            for ( i = 0; i < [ dragData count ]; i++ ) {
                id                      item = [ dragData objectAtIndex: i ];
                NSString                *type = @"p";
                NSString                *name = [ item objectForKey: @"RadmindServerItemRelativePath" ];
		NSString		*ext = [ name pathExtension ];
                NSRange                 range;
                
                range = [ name rangeOfString: @"negative"
                                        options: NSCaseInsensitiveSearch ];
                
                if ( range.location != NSNotFound && [ ext isEqualToString: @"T" ] ) {
                    type = @"n";
                } else if ( [ ext isEqualToString: @"K" ] ) {
		    type = @"k";
		}
                
                [ _currentCommandFile insertObject:
                    [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                            type, @"type", name, @"tname", nil ]
                    atIndex: ( row + i ) ];
                [ currentKFileTable reloadData ];
            }
        }
    }

    [ self setCommandFileEdited: YES ];
    return( YES );
}

/* outline view data source methods */
- ( int )outlineView: ( NSOutlineView * )ov numberOfChildrenOfItem: ( id )item
{
    NSMutableArray	*items = nil;
    NSString		*kfileDir = nil;
    id			realItem = item;
    int			i = 0;
    
    items = _allCommandFiles;
    
    if ( realItem == nil ) {
	kfileDir = [ NSString stringWithFormat:
			@"%@/command", @"/var/radmind" ];
		
	realItem = [ NSDictionary dictionaryWithObjectsAndKeys:
		    kfileDir, @"RadmindServerItemPath", nil ];
    } else {
	i = [ _allCommandFiles indexOfObject: realItem ];
    }
    
    GET_NODE_CHILDREN( i, realItem, items );

    return( [ children count ] );
}

- ( BOOL )outlineView: ( NSOutlineView * )ov isItemExpandable: ( id )item
{
    return ( [[ item objectForKey: @"RadmindServerItemType" ]
		isEqualToString: @"directory" ] );
}

- ( id )outlineView: ( NSOutlineView * )ov child: ( int )index ofItem: ( id )item
{
    NSMutableArray	*items = nil;
    NSString		*kfileDir = nil;
    id			realItem = item;
    int			i = index;
    
    items = _allCommandFiles;
    
    if ( realItem == nil ) {
	kfileDir = [ NSString stringWithFormat:
			@"%@/command", @"/var/radmind" ];
		
	realItem = [ NSDictionary dictionaryWithObjectsAndKeys:
		    kfileDir, @"RadmindServerItemPath", nil ];
	i = 0;
    }
    
    GET_NODE_CHILDREN( i, realItem, items );

    if ( children == nil || [ children count ] <= index ) {
        return( nil );
    }
    
    return( [ children objectAtIndex: index ] );
}

- ( id )outlineView: ( NSOutlineView * )ov objectValueForTableColumn: ( NSTableColumn * )column
                    byItem: ( id )item
{
    NSString            *key = [ column identifier ];
    
    if ( item == nil ) {
        return( @"" );
    }
    
    return( [ item objectForKey: key ] );
}

- ( void )outlineView: ( NSOutlineView * )ov setObjectValue: ( id )object
            forTableColumn: ( NSTableColumn * )column byItem: ( id )item
{
    NSString            *key = [ column identifier ];
    NSString            *path = nil;
    
    if ( ! [ key isEqualToString: @"RadmindServerItemName" ] ) {
        NSBeep();
        return;
    }
    
    if ( ! [ item isKindOfClass: [ NSMutableDictionary class ]] ) {
        return;
    }
    
    path = [ NSString stringWithFormat: @"%@/%@",
	    [ item objectForKey: @"RadmindServerItemParent" ],
	    [ item objectForKey: @"RadmindServerItemName" ]];

    [ item setObject: path forKey: @"RadmindServerItemPath" ];
    [ item setObject: object forKey: key ];
    [ ov reloadData ];
}

- ( void )outlineView:( NSOutlineView * )ov willDisplayCell: ( id )cell
            forTableColumn: ( NSTableColumn * )column item: ( id )item
{
    NSImage             *image = [ NSImage imageNamed: @"transcript-small.tiff" ];

    if ( ! [[ column identifier ] isEqualToString: @"RadmindServerItemName" ] ) {
        [ cell setEditable: NO ];
        return;
    }
    
    if ( [[ item objectForKey: @"RadmindServerItemType" ]
	    isEqualToString: @"directory" ] ) {
        image = [ NSImage imageNamed: @"folder.png" ];
    } else if ( [[[ item objectForKey: @"RadmindServerItemName" ] pathExtension ]
                    isEqualToString: @"backup" ] ) {
        image = [ NSImage imageNamed: @"bumbershoot.png" ];
    }
    
    [ cell setEditable: YES ];
    [ cell setImage: image ];
}

/* outline view drag and drop methods */
- ( BOOL )outlineView: ( NSOutlineView * )ov writeItems: ( NSArray * )items
            toPasteboard: ( NSPasteboard * )pboard
{
    [ pboard declareTypes: [ NSArray arrayWithObjects:
                    RSMKFilePboardType, NSStringPboardType, nil ]
                    owner: self ];
    [ pboard setPropertyList: items forType: RSMKFilePboardType ];
    [ pboard setString: @"test" forType: NSStringPboardType ];
    
    //[ self setDraggingSource: ov ];
        
    return( YES );
}

- ( NSDragOperation )outlineView: ( NSOutlineView * )ov
            validateDrop: ( id <NSDraggingInfo> )info
            proposedItem: ( id )item
            proposedChildIndex: ( int )index
{
    if ( item == nil ) {
        return( NSDragOperationCopy );
    } else if ( ! [[ item objectForKey: @"RadmindServerItemType" ]
	    isEqualToString: @"directory" ] ) {
	return( NSDragOperationNone );
    }
    
    [ ov setDropItem: item dropChildIndex: NSOutlineViewDropOnItemIndex ];

    return( NSDragOperationAll );
}

- ( BOOL )outlineView: ( NSOutlineView * )ov
            acceptDrop: ( id <NSDraggingInfo> )info
            item: ( id )item childIndex: ( int )index
{
    NSPasteboard        *pb = [ info draggingPasteboard ];
    id                  draggedObject = nil;
    id                  draggedItem = nil;
    
    if ( pb == nil ) {
        return( NO );
    }
    
    draggedObject = [ pb propertyListForType: RSMKFilePboardType ];
    if ( draggedObject == nil ) {
        return( NO );
    }
    
    if (( draggedItem = [ draggedObject objectAtIndex: 0 ] ) == nil ) {
        return( NO );
    }
    
    if ( item == nil ) {
        if ( [ draggedItem objectForKey: @"RadmindServerItemPath" ] == nil ) {
            return( NO );
        }
        
        [ self moveCommandFileAtPath: [ draggedItem objectForKey: @"RadmindServerItemPath" ]
                            toPath: [ NSString stringWithFormat: @"%@/command/%@",
					@"/var/radmind",
					[ draggedItem objectForKey: @"RadmindServerItemName" ]]];
        return( YES );
    } else if ( [[ item objectForKey: @"RadmindServerItemType" ]
		    isEqualToString: @"directory" ] ) {
        [ self moveCommandFileAtPath: [ draggedItem objectForKey: @"RadmindServerItemPath" ]
                            toPath: [ NSString stringWithFormat: @"%@/%@",
			    [ item objectForKey: @"RadmindServerItemPath" ],
			    [ draggedItem objectForKey: @"RadmindServerItemName" ]]];
        return( YES );
    }        
    
    return( NO );
}

@end
