/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RAServerController.h"
#import "RAServerAuth.h"
#import "RAEditorLauncher.h"
#import "RAAuthKiller.h"
#import "RSMLoadsetEditor.h"
#import "RSMLoadsetManager.h"
#import "RSMPreferences.h"

#import "NSFileManager(mktemp).h"

#include <sys/types.h>
#include <sys/param.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

#include "argcargv.h"
#include "authtools.h"

extern int		errno;
static int		CURRENTCMD = -1;

@implementation RAServerController

- ( id )init
{
    NSPort		*recPort;
    NSPort		*sendPort;
    NSArray		*portArray;
    
    self = [ super init ];
    [ NSApp setDelegate: self ];
    
    /* DO */
    recPort = [ NSPort port ];
    sendPort = [ NSPort port ];
    connectionToServerAuth = [[ NSConnection alloc ]
				initWithReceivePort: recPort
				sendPort: sendPort ];
    [ connectionToServerAuth setRootObject: self ];
    sa = nil;
    _rsmThreadID = -1;
    portArray = [ NSArray arrayWithObjects: sendPort, recPort, nil ];
    
    [ NSThread detachNewThreadSelector: @selector( connectWithPorts: )
                                        toTarget: [ RAServerAuth class ]
                                        withObject: portArray ];
                                        
    serverConfig = nil;
    _allCommandFiles = nil;
    _validClients = nil;
    _rsmTmpDirectory = nil;
    prefs = nil;
    
    _isConfigFileEdited = NO;
    
    return( self );
}

- ( int )configHelperThreadID
{
    return( _rsmThreadID );
}

- ( void )setConfigHelperThreadID: ( int )ID
{
    _rsmThreadID = ID;
}

- ( void )setServer: ( id )serverObject andThreadID: ( int )threadID
{
    [ serverObject setProtocolForProxy: @protocol( RAServerAuthInterface ) ];
    [ serverObject retain ];
    [ self setConfigHelperThreadID: threadID ];
    sa = ( RAServerAuth <RAServerAuthInterface> * )serverObject;
}

- ( BOOL )isConfigFileEdited
{
    return( _isConfigFileEdited );
}

- ( void )setConfigFileEdited: ( BOOL )edit
{
    if ( edit == NO ) {
        [ serverWindow setDocumentEdited: NO ];
    } else {
        [ serverWindow setDocumentEdited: YES ];
    }
    _isConfigFileEdited = edit;
}

- ( void )setCommandFileList: ( NSMutableArray * )commandFiles
{
    if ( _allCommandFiles != nil ) {
        [ _allCommandFiles release ];
    }
    _allCommandFiles = [[ NSArray alloc ] initWithArray: commandFiles ];
    
    [ self setupCommandFilePopupButtons ];
}

- ( int )dragOriginRow
{
    return( dragOriginRow );
}

- ( void )setDragOriginRow: ( int )row
{
    dragOriginRow = row;
}

- ( NSArray * )validClients
{
    return( _validClients );
}

- ( void )addToValidClients: ( NSString * )client
{
    if ( _validClients == nil ) {
        _validClients = [[ NSMutableArray alloc ] init ];
    }
    
    if ( ! [ _validClients containsObject: client ] ) {
        [ _validClients addObject: client ];
    }
}

- ( void )removeFromValidClients: ( NSString * )client
{
    if ( [ _validClients containsObject: client ] ) {
        [ _validClients removeObject: client ];
    }
}

- ( void )setSessionTmpDirectory: ( NSString * )tmpdir
{
    if ( _rsmTmpDirectory != nil ) {
	[ _rsmTmpDirectory release ];
	_rsmTmpDirectory = nil;
    }
    _rsmTmpDirectory = [ tmpdir retain ];
}

- ( NSString * )sessionTmpDirectory
{
    return( _rsmTmpDirectory );
}

- ( NSArray * )availableCommandFiles
{
    return( _allCommandFiles );
}

/* preferences */
- ( IBAction )showPreferences: ( id )sender
{
    if ( prefs == nil ) {
        prefs = [[ RSMPreferences alloc ] init ];
        [ NSBundle loadNibNamed: @"Preferences" owner: prefs ];
	[ prefs setSessionTemporaryDirectory: [ self sessionTmpDirectory ]];
    }
    
    [ prefs showPreferencesPanel ];
}

- ( void )awakeFromNib
{
    if ( access( "/var/radmind/config", F_OK ) < 0 ) {
        NSRunCriticalAlertPanel( @"No config file found.",
            @"You don't seem to have set up this machine to be a radmind server. "
            @"To set it up as a server, run the Radmind Assistant, and choose "
            @"\"Radmind Setup\" from the Session menu.",
            @"Quit", @"", @"" );
        exit( 1 );
    }
    
    [ NSApp setDelegate: self ];
    [ loadsetManager setDelegate: self ];
    
    [ hostsTable setAllowsMultipleSelection: YES ];
    [ hostsTable setDataSource: self ];
    [ hostsTable setDelegate: self ];
    [ hostsTable setRowHeight: 22.0 ];
    [ hostsTable setTarget: self ];
    [ hostsTable setAction: @selector( hostClick: ) ];
    [ hostsTable setDoubleAction: @selector( beginEditingClientName: ) ];
    [ hostsTable setDrawsStripes: YES ];
    [ hostsTable registerForDraggedTypes:
            [ NSArray arrayWithObject: NSFileContentsPboardType ]];

    [ self toolbarSetup ];
    [ self setupCommandFilePopupButtons ];
    [ self setDragOriginRow: -1 ];
    
    if ( ! [ serverWindow setFrameUsingName: @"RSMServerWindow" ] ) {
        NSLog( @"Couldn't restore window size from defaults." );
    }
    [ serverWindow setFrameAutosaveName: @"RSMServerWindow" ];
    
    [ serverWindow makeKeyAndOrderFront: nil ];
}

- ( void )toolbarSetup
{
    NSToolbar *rastbar = [[[ NSToolbar alloc ] initWithIdentifier: 
                                @"rastoolbar" ] autorelease ];
    
    [ rastbar setAllowsUserCustomization: YES ];
    [ rastbar setAutosavesConfiguration: YES ];
    [ rastbar setDisplayMode: NSToolbarDisplayModeIconAndLabel ];
    
    [ rastbar setDelegate: self ];
    [ serverWindow setToolbar: rastbar ];
}

/**/
/* required toolbar delegate methods */
/**/

- ( NSToolbarItem * )toolbar: ( NSToolbar * )toolbar
                    itemForItemIdentifier: ( NSString * )itemIdent
                    willBeInsertedIntoToolbar: ( BOOL )flag
{
    NSToolbarItem *rastbarItem = [[[ NSToolbarItem alloc ]
                                    initWithItemIdentifier: itemIdent ] autorelease ];
    
    if ( [ itemIdent isEqualToString: RASToolbarRefreshIdentifier ] ) {
        [ rastbarItem setLabel: @"Refresh" ];
        [ rastbarItem setPaletteLabel: @"Refresh" ];
        [ rastbarItem setToolTip: @"Refresh server configuration information" ];
        [ rastbarItem setImage: [ NSImage imageNamed: @"refresh.png" ]];
        [ rastbarItem setAction: @selector( reloadConfigFile: ) ];
        [ rastbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RASToolbarSaveConfigIdentifier ] ) {
        [ rastbarItem setLabel: NSLocalizedString( @"Save", @"Save" ) ];
        [ rastbarItem setPaletteLabel: NSLocalizedString( @"Save", @"Save" ) ];
        [ rastbarItem setToolTip: NSLocalizedString( @"Save server configuration file",
                                                    @"Save server configuration file" ) ];
        [ rastbarItem setImage: [ NSImage imageNamed: @"saveconfig.png" ]];
        [ rastbarItem setAction: @selector( saveConfigFile: ) ];
        [ rastbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMToolbarNewClientIdentifier ] ) {
        [ rastbarItem setLabel: NSLocalizedString( @"New Client", @"New Client" ) ];
        [ rastbarItem setPaletteLabel: NSLocalizedString( @"New Client", @"New Client" ) ];
        [ rastbarItem setToolTip: NSLocalizedString( @"Add new client to config file",
                                                    @"Add new client to config file" ) ];
        [ rastbarItem setImage: [ NSImage imageNamed: @"newclient.png" ]];
        [ rastbarItem setAction: @selector( addNewClientToConfigFile: ) ];
        [ rastbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMToolbarDeleteClientIdentifier ] ) {
        [ rastbarItem setLabel: NSLocalizedString( @"Delete", @"Delete" ) ];
        [ rastbarItem setPaletteLabel: NSLocalizedString( @"Delete", @"Delete" ) ];
        [ rastbarItem setToolTip: NSLocalizedString( @"Delete line from config file",
                                                    @"Delete line from config file" ) ];
        [ rastbarItem setImage: [ NSImage imageNamed: @"deleteclient.png" ]];
        [ rastbarItem setAction: @selector( deleteClientFromConfigFile: ) ];
        [ rastbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMToolbarToggleCommentIdentifier ] ) {
        [ rastbarItem setLabel: NSLocalizedString( @"Comment", @"Comment" ) ];
        [ rastbarItem setPaletteLabel: NSLocalizedString( @"Comment", @"Comment" ) ];
        [ rastbarItem setToolTip: NSLocalizedString( @"Comment out selected line",
                                                        @"Comment out selected line" ) ];
        [ rastbarItem setImage: [ NSImage imageNamed: @"comment.png" ]];
        [ rastbarItem setAction: @selector( toggleCommentOnSelectedLine: ) ];
        [ rastbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RSMToolbarNewCommentIdentifier ] ) {
        [ rastbarItem setLabel: NSLocalizedString( @"New Comment", @"New Comment" ) ];
        [ rastbarItem setPaletteLabel: NSLocalizedString( @"New Comment", @"New Comment" ) ];
        [ rastbarItem setToolTip: NSLocalizedString( @"Add a comment to the Radmind Configuration",
                                                    @"Add a comment to the Radmind Configuration" ) ];
        [ rastbarItem setImage: [ NSImage imageNamed: @"newcomment.png" ]];
        [ rastbarItem setAction: @selector( newComment: ) ];
        [ rastbarItem setTarget: self ];
    }
    
    return( rastbarItem );
}

- ( BOOL )validateToolbarItem: ( NSToolbarItem * )tItem
{
    if ( [[ tItem itemIdentifier ] isEqualToString: RASToolbarSaveConfigIdentifier ] ) {
        if ( ! [ self isConfigFileEdited ] ) {
            return( NO );
        }
    } else if ( [[ tItem itemIdentifier ] isEqualToString: RSMToolbarDeleteClientIdentifier ] ) {
        if ( [ hostsTable selectedRow ] < 0 ) {
            return( NO );
        }
    } else if ( [[ tItem itemIdentifier ] isEqualToString: RSMToolbarToggleCommentIdentifier ] ) {
        if ( [ hostsTable selectedRow ] < 0 ) {
            return( NO );
        }
        
        if ( [[[ serverConfig objectAtIndex: [ hostsTable selectedRow ]]
                    objectForKey: @"host" ] characterAtIndex: 0 ] == '#' ) {
            [ tItem setLabel: NSLocalizedString( @"Uncomment", @"Uncomment" ) ];
            [ tItem setPaletteLabel: NSLocalizedString( @"Uncomment", @"Uncomment" ) ];
            [ tItem setToolTip: NSLocalizedString( @"Uncomment selected line",
                                                    @"Uncomment selected line" ) ];
        } else {
            [ tItem setLabel: NSLocalizedString( @"Comment", @"Comment" ) ];
            [ tItem setPaletteLabel: NSLocalizedString( @"Comment", @"Comment" ) ];
            [ tItem setToolTip: NSLocalizedString( @"Comment out selected line",
                                                    @"Comment out selected line" ) ];
        }
    }
    
    return( YES );
}

- ( NSArray * )toolbarDefaultItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RSMToolbarNewClientIdentifier,
                            RSMToolbarDeleteClientIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            RASToolbarSaveConfigIdentifier,
                            RSMToolbarToggleCommentIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RASToolbarRefreshIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarAllowedItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            NSToolbarSeparatorItemIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RSMToolbarNewClientIdentifier,
                            RSMToolbarDeleteClientIdentifier,
                            RASToolbarSaveConfigIdentifier,
                            RSMToolbarToggleCommentIdentifier,
                            RASToolbarRefreshIdentifier, 
                            RSMToolbarNewCommentIdentifier, nil ];
                            
    return( tmp );
}
/* end required toolbar delegate methods */

- ( BOOL )validateMenuItem: ( NSMenuItem * )menuItem
{
    return( YES );
}

- ( IBAction )addNewClientToConfigFile: ( id )sender
{
    int			row;
    NSString		*newHostKFile = @"";
    
    if ( [ _allCommandFiles count ] > 0 ) {
        newHostKFile = [[ _allCommandFiles objectAtIndex: 0 ]
			objectForKey: @"RadmindServerItemRelativePath" ];
    }
    
    [ serverConfig addObject:
            [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                @"newhost.radmind.edu", @"host",
                newHostKFile, @"kfile", nil ]];
    
    [ hostsTable reloadData ];
    [ hostsTable setDrawsStripes: YES ];
    row = ( [ serverConfig count ] - 1 );
    [ hostsTable selectRow: row byExtendingSelection: NO ];
    [ hostsTable editColumn: 0 row: row withEvent: nil select: YES ];
    [ self setConfigFileEdited: YES ];
}

- ( IBAction )deleteClientFromConfigFile: ( id )sender
{
    NSEnumerator        *lineEnumerator = nil;
    NSNumber            *lineNumber = nil;
    int                 count = [ hostsTable numberOfSelectedRows ];
    unsigned int        *lines = NULL, i = 0;
    
    if ( count == 0 ) {
        NSBeep();
        return;
    }
    
    if (( lines = calloc( count, sizeof( int ))) == NULL ) {
        NSLog( @"malloc: %s", strerror( errno ));
        exit( 2 );
    }
    
    lineEnumerator = [ hostsTable selectedRowEnumerator ];
    
    while (( lineNumber = [ lineEnumerator nextObject ] ) != nil ) {
        lines[ i ] = [ lineNumber intValue ];
        [ self removeFromValidClients:
                [[ serverConfig objectAtIndex: i ] objectForKey: @"host" ]];
        i++;
    }
    
    [ hostsTable selectRow: *lines byExtendingSelection: NO ];
    [ serverConfig removeObjectsFromIndices: lines numIndices: count ];
    free( lines );
    
    [ hostsTable reloadData ];
    [ hostsTable setDrawsStripes: YES ];
    [ self setConfigFileEdited: YES ];
}

- ( IBAction )saveConfigFile: ( id )sender
{
    NSString		*tmpdir = [ self sessionTmpDirectory ];
    FILE		*tmpconf;
    char		tmpconfpath[ MAXPATHLEN ];
    int			fd, i;
    
    if ( snprintf( tmpconfpath, MAXPATHLEN, "%s/.radmind.config.XXXXXX",
		[ tmpdir UTF8String ] ) >= MAXPATHLEN ) {
	NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
		NSLocalizedString( @"mkstemp %s: %s", @"mkstemp %s: %s" ),
		NSLocalizedString( @"OK", @"OK" ), @"", @"",
		tmpconfpath, strerror( errno ));
	return;
    }
    
    if (( fd = mkstemp( tmpconfpath )) < 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"mkstemp %s: %s", @"mkstemp %s: %s" ),
                NSLocalizedString( @"OK", @"OK" ),
                    @"", @"", tmpconfpath, strerror( errno ));
        return;
    }
    if (( tmpconf = fdopen( fd, "w+" )) == NULL ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"fdopen %s: %s", @"fdopen %s: %s" ),
                NSLocalizedString( @"OK", @"OK" ),
                    @"", @"", tmpconfpath, strerror( errno ));
        return;
    }
    
    for ( i = 0; i < [ serverConfig count ]; i++ ) {
        NSDictionary	*dict = [ serverConfig objectAtIndex: i ];
        if ( [[ dict objectForKey: @"host" ] characterAtIndex: 0 ] == '#' ) {
            fprintf( tmpconf, "%s\n",
                ( char * )[[ dict objectForKey: @"host" ] UTF8String ] );
        } else {
            fprintf( tmpconf, "%-37s\t%s\n",
                    ( char * )[[ dict objectForKey: @"host" ] UTF8String ],
                    ( char * )[[ dict objectForKey: @"kfile" ] UTF8String ] );
        }
    }
    ( void )fclose( tmpconf );
    
    [ self loadServerInfoWithCommand: RA_FILE_WRITE
	    arguments: [ NSArray arrayWithObjects: @"-A", @"MoveFile",
		    @"-d", [ self sessionTmpDirectory ], @"--", 
		    [ NSString stringWithUTF8String: tmpconfpath ],
		    [ NSString stringWithFormat: @"%@/config", @"/var/radmind" ], nil ]];
}

- ( IBAction )toggleCommentOnSelectedLine: ( id )sender
{
    char			c, **tav;
    int                         row, tac;
    NSMutableDictionary		*dict, *newdict;
    NSString			*line;
    
    if ( [ hostsTable numberOfSelectedRows ] > 1 ) return;
    if (( row = [ hostsTable selectedRow ] ) < 0 ) return;
    
    dict = [ serverConfig objectAtIndex: row ];
    c = [[ dict objectForKey: @"host" ] characterAtIndex: 0 ];
                
    if ( c == '#' ) {   /* uncomment line */
        line = [ dict objectForKey: @"host" ];
        
        if ( [ line length ] < 2 ) {
            NSBeep();
            return;
        }
        
        /* get the string in the range after the # */
        line = [ line substringFromIndex: 1 ];
        tac = argcargv(( char * )[ line UTF8String ], &tav );
        
        if ( tac != 2 ) {
            NSBeep();
            return;
        }
        
        newdict = [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                            [ NSString stringWithUTF8String: tav[ 0 ]], @"host",
                            [ NSString stringWithUTF8String: tav[ 1 ]], @"kfile", nil ];
    } else {            /* comment line */
        line = [ NSString stringWithFormat: @"#%@\t\t%@",
                                    [ dict objectForKey: @"host" ],
                                    [ dict objectForKey: @"kfile" ]];
        newdict = [ NSMutableDictionary dictionaryWithObjectsAndKeys: 
                                line, @"host",
                                @"", @"kfile", nil ];
    }
    
    [ serverConfig replaceObjectAtIndex: row withObject: newdict ];
    [ self setConfigFileEdited: YES ];
    [ hostsTable reloadData ];
}

- ( IBAction )newComment: ( id )sender
{
    int                 row = [ hostsTable selectedRow ];
    
    if ( row < 0 ) {
        row = 0;
    }
    
    [ serverConfig insertObject: [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                                @"# New Comment", @"host", @"", @"kfile", nil ]
                atIndex: row ];
    [ hostsTable reloadData ];
    [ hostsTable selectRow: row byExtendingSelection: NO ];
    [ hostsTable editColumn: 0 row: row withEvent: nil select: YES ];
    [ self setConfigFileEdited: YES ];
}

- ( oneway void )firstServerInfoLoad
{
    [ self reloadServerInformation: self ];
}

- ( IBAction )reloadServerInformation: ( id )sender
{
    NSString		*tmpdir = nil;
    
    if ( [ self isConfigFileEdited ] ) {
        int		rc;
        
        rc = NSRunAlertPanel( @"Save changes before reloading?",
                    @"You will lose your changes if you do not choose to save.",
                    @"Save", @"Don't Save", @"Cancel" );
        switch ( rc ) {
        case NSAlertDefaultReturn:
            [ self saveConfigFile: nil ];
            break;
            
        case NSAlertAlternateReturn:
            break;
            
        case NSAlertOtherReturn:
            return;
        }
    }
    
    [ gatheringProgBar setUsesThreadedAnimation: YES ];
    [ gatheringProgBar startAnimation: nil ];
    [ NSApp beginSheet: gatheringSheet
            modalForWindow: serverWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
    [ serverConfig removeAllObjects ];
    [ hostsTable reloadData ];
    [ hostsTable setDrawsStripes: YES ];
    
    /* create tmpdir for session */
    if ( [ self sessionTmpDirectory ] == nil ) {
	tmpdir = [[ NSFileManager defaultManager ]
		    makeTemporaryDirectory: @"/tmp/.rsm.XXXXXX"
		    withMode: (mode_t)0700 ];
	if ( tmpdir == nil ) {
	    exit( 2 );
	}
	[ self setSessionTmpDirectory: tmpdir ];
    }
    
    [ self loadServerInfoWithCommand: RSM_REFRESHALL
	    arguments: [ NSArray arrayWithObjects:
			    @"-A", @"RefreshAllPlists",
			    @"-d", [ self sessionTmpDirectory ],
			    @"/var/radmind", nil ]];
}

- ( IBAction )reloadConfigFile: ( id )sender
{
    NSString		*tmpdir = nil;
    
    if ( [ self isConfigFileEdited ] ) {
        int		rc;
        
        rc = NSRunAlertPanel( @"Save changes before reloading?",
                    @"You will lose your changes if you do not choose to save.",
                    @"Save", @"Don't Save", @"Cancel" );
        switch ( rc ) {
        case NSAlertDefaultReturn:
            [ self saveConfigFile: nil ];
            break;
            
        case NSAlertAlternateReturn:
            break;
            
        case NSAlertOtherReturn:
            return;
        }
    }
    
    [ gatheringProgBar setUsesThreadedAnimation: YES ];
    [ gatheringProgBar startAnimation: nil ];
    [ NSApp beginSheet: gatheringSheet
            modalForWindow: serverWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
    [ serverConfig removeAllObjects ];
    [ hostsTable reloadData ];
    [ hostsTable setDrawsStripes: YES ];
    
    /* create tmpdir for session */
    if ( [ self sessionTmpDirectory ] == nil ) {
	tmpdir = [[ NSFileManager defaultManager ]
		    makeTemporaryDirectory: @"/tmp/.rsm.XXXXXX"
		    withMode: (mode_t)0700 ];
	if ( tmpdir == nil ) {
	    exit( 2 );
	}
	[ self setSessionTmpDirectory: tmpdir ];
    }
    
    [ self loadServerInfoWithCommand: RSM_REFRESH_CONFIG
	    arguments: [ NSArray arrayWithObjects:
			    @"-A", @"RefreshConfig",
			    @"-d", [ self sessionTmpDirectory ],
			    @"/var/radmind", nil ]];
}

- ( void )loadServerInfoWithCommand: ( int )cmd
            arguments: ( NSArray * )args
{
    CURRENTCMD = cmd;
    [ sa executeCommand: CURRENTCMD withArguments: args controller: self ];
}

- ( void )addServerLine: ( char * )line toTable: ( NSTableView * )table
{
    int			tac;
    char		**targv, tmp[ MAXPATHLEN ];
    
    if ( line == NULL ) {
	return;
    }
    
    if ( strlen( line ) >= MAXPATHLEN ) {
	NSLog( @"%s: too long\n", line );
	return;
    }
    strcpy( tmp, line );
    
    tac = argcargv( tmp, &targv );
    
    if ( tac == 2 && *line != '#' ) {
        [ serverConfig addObject:
                    [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                        [ NSString stringWithUTF8String: targv[ 0 ]], @"host",
                        [ NSString stringWithUTF8String: targv[ 1 ]], @"kfile", nil ]];
        [ self addToValidClients: [ NSString stringWithUTF8String: targv[ 0 ]]];
    } else if ( *line == '#' ) {
        /* display comments */
        [ serverConfig addObject:
                    [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                        [ NSString stringWithUTF8String: line ], @"host",
                        @"", @"kfile", nil ]];
    }
            
    [ hostsTable setDrawsStripes: YES ];
}

- ( void )setupCommandFilePopupButtons
{
    int			i;
    id 			protoCell = [[ NSPopUpButtonCell alloc ]
                                        initTextCell: @"" pullsDown: NO ];
    NSTableColumn 	*kColumn = [ hostsTable tableColumnWithIdentifier: @"kfile" ];
    NSArray		*kfiles = [ self availableCommandFiles ];

    for ( i = 0; i < [ kfiles count ]; i++ ) {
        [ protoCell insertItemWithTitle:
		    [[ kfiles objectAtIndex: i ]
			objectForKey: @"RadmindServerItemRelativePath" ]
		    atIndex: i ];
    }

    [ protoCell setEditable: YES ];
    [ protoCell setBordered: NO ];
    [ kColumn setDataCell: protoCell];
    [ protoCell release];
}

- ( oneway void )setCurrentCommandPID: ( pid_t )pid threadID: ( int )ID
{
    /* no-op */
}

- ( void )readRadmindServerConfigFile
{
    NSString		*radmindPath = [ self sessionTmpDirectory ];
    FILE		*f;
    char		cfg[ MAXPATHLEN ];
    char		line[ MAXPATHLEN ];
    
    if ( serverConfig == nil ) {
	serverConfig = [[ NSMutableArray alloc ] init ];
    }
    
    if ( snprintf( cfg, MAXPATHLEN, "%s/config",
	    [ radmindPath fileSystemRepresentation ] ) >= MAXPATHLEN ) {
	NSLog( @"%s/config: path too long\n", radmindPath );
	return;
    }
    
    if (( f = fopen( cfg, "r" )) == NULL ) {
	NSLog( @"fopen %s: %s", cfg, strerror( errno ));
	return;
    }
    
    while ( fgets( line, MAXPATHLEN, f ) != NULL ) {
	[ self addServerLine: line toTable: hostsTable ];
    }
    
    if ( ferror( f )) {
	fprintf( stderr, "fgets: %s\n", strerror( errno ));
    }
    if ( fclose( f ) != 0 ) {
	fprintf( stderr, "fclose: %s\n", strerror( errno ));
    }
    
    [ hostsTable reloadData ];
}

- ( void )authorizationFailedInThreadWithID: ( int )ID
{
    if ( ID != [ self configHelperThreadID ] ) {
        return;
    }
    
    if ( [ gatheringSheet isVisible ] ) {
        [ gatheringSheet orderOut: nil ];
        [ NSApp endSheet: gatheringSheet ];
    }
    NSBeginAlertSheet( @"Authorization failed.",
                          @"Try Again", @"Cancel", @"", serverWindow,
                          self,
                          @selector( authFailedSheetDidEnd:returnCode:contextInfo: ),
                          nil, NULL,
            @"You must authenticate as an administrator to proceed." );
}

- ( void )authFailedSheetDidEnd:( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    [ sheet orderOut: nil ];
    [ NSApp endSheet: sheet ];
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
    default:
    case NSAlertAlternateReturn:
        return;
    }
    [ self reloadServerInformation: nil ];
}

- ( void )toolError: ( char * )error fromThreadWithID: ( int )ID
{
    if ( ID != [ self configHelperThreadID ] ) {
        return;
    }
    
    NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
        @"%s", NSLocalizedString( @"OK", @"OK" ), @"", @"", error );
}

- ( oneway void )command: ( int )cmd finishedWithStatus: ( int )status
            inThread: ( int )ID
{
    if ( ID != [ self configHelperThreadID ] ) {
        return;
    }
    
    switch ( cmd ) {
    case RSM_REFRESHALL:
	[ self readRadmindServerConfigFile ];
	[ self setupCommandFilePopupButtons ];
	[ hostsTable reloadData ];
	[ loadsetEditor reloadCommandFiles ];
	[ loadsetManager readAvailableCheckedInTranscripts: self ];
	[ loadsetManager readAvailableTmpTranscripts: self ];
	break;
	
    case RSM_REFRESH_CONFIG:
	[ self readRadmindServerConfigFile ];
	[ self setupCommandFilePopupButtons ];
	[ hostsTable reloadData ];
	break;
        
    case RA_FILE_WRITE:
        if ( status != 0 ) {
            NSRunAlertPanel( @"Error saving.", @"", @"OK", @"", @"" );
        }
        [ self setConfigFileEdited: NO ];
        break;
        
    default:
        break;
    }
    if ( [ gatheringSheet isVisible ] ) {
        [ gatheringSheet orderOut: nil ];
        [ NSApp endSheet: gatheringSheet ];
    }
}

- ( void )beginEditingClientName: ( id )sender
{
    int			row = [ hostsTable selectedRow ];
    int			column = [ hostsTable selectedColumn ];

    if ( row >= 0 && column == 0 ) {
        [[ hostsTable selectedCell ] setEditable: YES ];
        [ hostsTable editColumn: 0 row: row withEvent: nil select: YES ];
    }
}

- ( IBAction )hostClick: ( id )sender
{
}

/* table view data source/delegate methods */
- ( void )handleChangedText: ( NSString * )text forTable: ( RASTableView * )tv
{
    int			row;
    NSMutableDictionary	*dict;
    NSString		*newstring = [ text copy ];
    
    if (( row = [ tv editedRow ] ) < 0 ) {
        return;
    }
    if (( dict = [ serverConfig objectAtIndex: row ] ) == nil ) {
        return;
    }

    if ( ! [[ dict objectForKey: @"host" ] isEqualToString: newstring ] ) {
        [ dict setObject: newstring forKey: @"host" ];
        [ serverConfig replaceObjectAtIndex: row withObject: dict ];
	[ self addToValidClients: newstring ];
        [ self setConfigFileEdited: YES ];
    }
    [ newstring release ];
}

- ( BOOL )handleKeyEvent: ( NSEvent * )theEvent fromTable: ( RASTableView * )table
{
    unichar 		key = [[ theEvent charactersIgnoringModifiers ] characterAtIndex: 0 ];
    
    if ( key == NSDeleteCharacter ) {
        [ self deleteClientFromConfigFile: nil ];
        return( YES );
    }
        
    return( NO );
}

- ( int )numberOfRowsInTableView: ( NSTableView * )aTableView
{
    if ( aTableView == hostsTable ) {
        return( [ serverConfig count ] );
    }

    return( 0 );
}

- ( void )tableView: ( NSTableView * )aTableView willDisplayCell: ( id )aCell
        forTableColumn: ( NSTableColumn * )aTableColumn row: ( int )rowIndex
{
    if ( rowIndex < 0 || rowIndex > [ serverConfig count ] ) {
        return;
    }
    if ( aTableView == hostsTable ) {
        if ( [[ aTableColumn identifier ] isEqualToString: @"host" ] ) {
            [ aCell setStringValue: 
                    [[ serverConfig objectAtIndex: rowIndex ]
                        objectForKey: @"host" ]];
            [ aCell setTextColor: [ NSColor blackColor ]];
            if ( [[ aCell stringValue ] characterAtIndex: 0 ] == '#' ) {
                [ aCell setTextColor: [ NSColor blueColor ]];
            }
            [ aCell setAction: nil ];
        } else if ( [[ aTableColumn identifier ] isEqualToString: @"kfile" ] ) {
            if ( [[[ serverConfig objectAtIndex: rowIndex ] objectForKey: @"host" ]
                                characterAtIndex: 0 ] == '#' ) {
                [ aCell setEnabled: NO ];
            } else {
                [ aCell setEnabled: YES ];
                [ aCell selectItemWithTitle: [[ serverConfig objectAtIndex: rowIndex ]
                        objectForKey: @"kfile" ]];
            }
        }
    }
}

- ( id )tableView: ( NSTableView * )aTableView
        objectValueForTableColumn: ( NSTableColumn * )aTableColumn
        row: ( int )rowIndex
{
    /* display is handled by tableView:willDisplayCell... */
    return( nil );
}

- ( void )tableView: ( NSTableView * )theTableView setObjectValue: ( id )theObject
        forTableColumn: ( NSTableColumn * )theColumn
        row: ( int )rowIndex
{
    if ( rowIndex < 0 || rowIndex > [ serverConfig count ] ) {
        return;
    }
    
    if ( [[ theColumn identifier ] isEqualToString: @"host" ] ) {
        NSString		*old = [[ serverConfig objectAtIndex: rowIndex ] objectForKey: @"host" ];
        
        if ( theObject == nil || [ theObject isEqualToString: old ] ||
                    [ theObject isEqualToString: @"" ] ||
                    [ ( NSString * )theObject length ] == 0 ) {
            return;
        }
        
        [ self removeFromValidClients: old ];
        [ self addToValidClients: theObject ];
        [[ serverConfig objectAtIndex: rowIndex ]
                setObject: theObject forKey: @"host" ];
    } else if ( [[ theColumn identifier ] isEqualToString: @"kfile" ] ) {
	if ( [ theObject isKindOfClass: [ NSNumber class ]] ) {
            NSArray		*kfiles = [ self availableCommandFiles ];
            NSString		*old = nil;
            
            if ( kfiles == nil || rowIndex < 0 ) {
                return;
            }
            
            old = [[ serverConfig objectAtIndex: rowIndex ] objectForKey: @"kfile" ];
            
            if ( [[[ kfiles objectAtIndex: [ theObject intValue ]]
		    objectForKey: @"RadmindServerItemRelativePath" ]
		    isEqualToString: old ] ) {
                return;
            }
            
            [[ serverConfig objectAtIndex: rowIndex ]
                    setObject: [[ kfiles objectAtIndex: [ theObject intValue ]]
				    objectForKey: @"RadmindServerItemRelativePath" ]
                    forKey: @"kfile" ];
        }
    }
    
    [ self setConfigFileEdited: YES ];
}

- ( BOOL )tableView: ( NSTableView * )tableView
        writeRows: ( NSArray * )rows
        toPasteboard: ( NSPasteboard * )pboard
{
    id			dsrc;
    NSArray		*anArray = nil;
    
    if ( [ rows count ] > 1 ) {
	return( NO );
    }
    
    dsrc = [ serverConfig copy ];
        
    anArray = [ NSArray arrayWithObject:
		    [ dsrc objectAtIndex: [[ rows objectAtIndex: 0 ] intValue ]]];
    [ self setDragOriginRow: [[ rows objectAtIndex: 0 ] intValue ]];
    [ pboard declareTypes: [ NSArray arrayWithObject: NSFileContentsPboardType ]
                    owner: self ];
    [ pboard setPropertyList: anArray forType: NSFileContentsPboardType ];
    [ dsrc release ];
    return( YES );
}

- ( NSDragOperation )tableView: ( NSTableView * )tableView
        validateDrop: ( id <NSDraggingInfo> )info
        proposedRow: ( int )row
        proposedDropOperation: ( NSTableViewDropOperation )operation
{
    if ( row < 0 ) {
	return( NSDragOperationNone );
    }

    if ( operation == NSTableViewDropAbove ) {
        return( NSDragOperationMove );
    } else if ( operation == NSTableViewDropOn ) {
        [ tableView setDropRow: row dropOperation: NSTableViewDropAbove ];
    }

    return( NSDragOperationNone );
}

- ( BOOL )tableView: ( NSTableView * )tableView
        acceptDrop: ( id <NSDraggingInfo> )info
        row: ( int )row
        dropOperation: ( NSTableViewDropOperation )operation
{
    NSPasteboard		*pb;
    id				dragData;

    if ( row < 0 || [ self dragOriginRow ] < 0 ) {
	return( NO );
    }
    
    pb = [ info draggingPasteboard ];
    
    dragData = [ pb propertyListForType:
	[ pb availableTypeFromArray:
	    [ NSArray arrayWithObject: NSFileContentsPboardType ]]];
	    
    if ( dragData == nil ) {
	return( NO );
    }
    
    [ serverConfig
	    insertObject: [ serverConfig objectAtIndex: [ self dragOriginRow ]]
	    atIndex: row ];
    [ serverConfig removeObjectAtIndex: 
		( [ self dragOriginRow ] > row ?
		( [ self dragOriginRow ] + 1 ) : [ self dragOriginRow ] ) ];
    [ hostsTable reloadData ];
    
    [ self setConfigFileEdited: YES ];
    return( YES );
}

/* RSMLoadsetEditor delegate methods */
- ( void )setAvailableCommandFiles: ( NSMutableArray * )commandFiles
{
    NSMutableArray	    *kfiles = [ NSMutableArray array ];
    int			    i;
    
    for ( i = 0; i < [ commandFiles count ]; i++ ) {
	if ( [[[ commandFiles objectAtIndex: i ]
		objectForKey: @"RadmindServerItemType" ]
		isEqualToString: @"directory" ] ) {
	    continue;
	}
	[ kfiles addObject: [ commandFiles objectAtIndex: i ]];
    }
    
    /* make a local copy */
    [ self setCommandFileList: kfiles ];
    if ( serverConfig == nil ) {
        serverConfig = [[ NSMutableArray alloc ] init ];
        [ self reloadServerInformation: nil ];
    }
}

- ( void )availableCommandFilesDidChange: ( NSMutableArray * )commandFiles
{
    /* make a local copy */
    [ self setAvailableCommandFiles: commandFiles ];
}

- ( NSApplicationTerminateReply )applicationShouldTerminate: ( NSApplication * )application
{
    NSApplicationTerminateReply		reply = NSTerminateNow;
    
    if ( [ self isConfigFileEdited ] ) {
        int			rc;
        
        [ serverWindow makeKeyAndOrderFront: nil ];
        rc = NSRunAlertPanel( @"Save changes to Radmind Server Configuration?",
                @"If you choose not to save, your changes will be lost.",
                @"Save", @"Don't Save", @"Cancel" );
                
        switch ( rc ) {
        case NSAlertDefaultReturn:
            [ self saveConfigFile: nil ];
            reply = NSTerminateCancel;
            break;
        case NSAlertAlternateReturn:
            break;
        default:
        case NSAlertOtherReturn:
            reply = NSTerminateCancel;
            break;
        }
    }
    
    return( reply );
}

- ( void )applicationWillTerminate: ( NSNotification * )notification
{
    NSString		*resolvedTmpPath = [ self sessionTmpDirectory ];
    
    resolvedTmpPath = [ resolvedTmpPath stringByResolvingSymlinksInPath ];

    [[ NSFileManager defaultManager ]
	    removeFileAtPath: resolvedTmpPath
	    handler: self ];
}

- ( void )dealloc
{
    NSConnection        *conn = [[ ( id )sa connectionForProxy ] retain ];
    
    if ( [ conn isValid ] ) {
        [ sa disconnect ];
        [ conn release ];
    }
    
    [ serverConfig release ];
    
    [ super dealloc ];
}

@end
