/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RCMStepController.h"
#import "RAEditorLauncher.h"
#import "RAServerCreator.h"
#import "RAVersionCompare.h"
#import "NSString(ComparisonExtensions).h"
#import "NSArray(SystemVersion).h"
#import "NSWorkspace(QuitOtherApplications).h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pwd.h>
#include <stdlib.h>
#include <unistd.h>

#include "authtools.h"
#include "argcargv.h"

#define RADMIND_DIR	"/var/radmind"
		
#define NEWSERVER	0
#define NEWCLIENT	1
#define OLDHAND		2

@implementation RCMStepController

static BOOL		firstTimeWalkThrough = NO;
static char		*tools[] = { "/usr/local/bin/fsdiff",
				    "/usr/local/bin/ktcheck",
				    "/usr/local/bin/lapply",
				    "/usr/local/bin/lcksum",
				    "/usr/local/bin/lcreate",
				    "/usr/local/bin/lfdiff",
				    "/usr/local/bin/lmerge",
				    "/usr/local/bin/lsort",
				    "/usr/local/bin/twhich",
				    "/usr/local/sbin/radmind", NULL };

- ( id )init
{
    NSPort		*recPort;
    NSPort		*sendPort;
    NSArray		*portArray;
    
    /* DO */
    recPort = [ NSPort port ];
    sendPort = [ NSPort port ];
    connectionToAuthExec = [[ NSConnection alloc ] initWithReceivePort: recPort
                                                sendPort: sendPort ];
    [ connectionToAuthExec setRootObject: self ];
    aexec = nil;
    portArray = [ NSArray arrayWithObjects: sendPort, recPort, nil ];
    
    [ NSThread detachNewThreadSelector: @selector( connectWithPorts: )
                                        toTarget: [ RAAuthExec class ]
                                        withObject: portArray ];
    
    _raSessionTmpDir = nil;
    _creatableTranscriptName = nil;
    _applicableTranscriptName = nil;
    
    KILLED = 0;
    svcBrowser = [[ NSNetServiceBrowser alloc ] init ];
    zcServerAddresses = nil;
    _validNegativeTranscripts = nil;
    [ negativeTranscriptTable setDelegate: self ];
    [ negativeTranscriptTable setDataSource: self ];
    needsToolUpdate = NO;
    
    self = [ super init ];
    return( self );
}

- ( void )preferencesDidChange: ( NSNotification * )notification
{
    NSUserDefaults	*defaults = [ notification object ];
    NSString		*servername = [ defaults objectForKey: @"radmindhost" ];
    
    [ cLcreateServerField setStringValue: servername ];
    [ aKtcheckHostField setStringValue: servername ];
    [ aLapplyServerField setStringValue: servername ];
    [ setupDefaultsServerField setStringValue: servername ];
}

- ( void )setServer: ( id )serverObject
{
    [ serverObject setProtocolForProxy: @protocol( RAAuthExecInterface ) ];
    [ serverObject retain ];
    aexec = ( RAAuthExec <RAAuthExecInterface> * )serverObject;
}

- ( void )setValidNegativeTranscripts: ( NSArray * )validTranscripts
{
    if ( _validNegativeTranscripts != nil ) {
        [ _validNegativeTranscripts release ];
        _validNegativeTranscripts = nil;
    }
    
    if ( validTranscripts != nil ) {
        _validNegativeTranscripts = [[ NSArray alloc ] initWithArray: validTranscripts ];
    }
}

- ( NSArray * )validNegativeTranscripts
{
    return( _validNegativeTranscripts );
}

- ( void )setCreatableTranscriptName: ( NSString * )name
{
    if ( _creatableTranscriptName != nil ) {
        [ _creatableTranscriptName release ];
        _creatableTranscriptName = nil;
    }
    
    if ( name != nil ) {
        _creatableTranscriptName = [[ NSString alloc ] initWithString: name ];
    }
}

- ( NSString * )creatableTranscriptName
{
    return( _creatableTranscriptName );
}

- ( void )setApplicableTranscriptName: ( NSString * )name
{
    if ( _applicableTranscriptName != nil ) {
        [ _applicableTranscriptName release ];
        _applicableTranscriptName = nil;
    }
    
    if ( name != nil ) {
        _applicableTranscriptName = [[ NSString alloc ] initWithString: name ];
    }
}

- ( NSString * )applicableTranscriptName
{
    return( _applicableTranscriptName );
}

- ( void )setSessionTempDirectory: ( NSString * )tmpdir
{
    if ( _raSessionTmpDir ) {
        [ _raSessionTmpDir release ];
        _raSessionTmpDir = nil;
    }
    _raSessionTmpDir = [ tmpdir retain ];
}

- ( NSString * )sessionTempDirectory
{
    return( _raSessionTmpDir );
}

- ( void )setNeedsToolUpdate: ( BOOL )update
{
    needsToolUpdate = update;
}

- ( BOOL )needsToolUpdate
{
    return( needsToolUpdate );
}

- ( void )setPreApply: ( BOOL )preapply
{
    _raPreApply = preapply;
}

- ( BOOL )preApply
{
    return( _raPreApply );
}

- ( void )setPostApply: ( BOOL )postapply
{
    _raPostApply = postapply;
}

- ( BOOL )postApply
{
    return( _raPostApply );
}

- ( void )scanForRadmindServers: ( id )sender
{
    /* find bonjour-enabled radmind servers, populate list with them */
    [ svcBrowser setDelegate: self ];
    [ svcBrowser searchForServicesOfType: @"_radmind._tcp" inDomain: @"" ];
}

- ( void )versionCheck
{
    if ( [[ RAVersionCompare sharedComparison ]
		compareInstalledVersion ] > 0 ) {
	[ self setNeedsToolUpdate: YES ];
    }
}

/* get version info for all installed tools */
- ( IBAction )showAboutPanel: ( id )sender
{
    FILE		*fp;
    char		buf[ MAXPATHLEN ], cmd[ MAXPATHLEN ], **p;
    
    [ aboutToolsVersionView setEditable: YES ];
    [ aboutToolsVersionView setString: @"" ];
    
    for ( p = tools; *p != NULL; p++ ) {
	char		*toolpath = *p;
	
	if ( access( toolpath, F_OK | X_OK ) < 0 ) {
	    if ( errno == ENOENT ) {
		[ aboutToolsVersionView insertText:
			[ NSString stringWithFormat: 
			@"%s not installed.\n\n", toolpath ]];
	    } else {
		NSLog( @"Couldn't access %s: %s\n", *p, strerror( errno ));
		[ aboutToolsVersionView setEditable: NO ];
		return;
	    }
	}
	
	if ( snprintf( cmd, MAXPATHLEN, "%s -V", toolpath ) >= MAXPATHLEN ) {
	    NSLog( @"%s -V: too long", toolpath );
	    [ aboutToolsVersionView setEditable: NO ];
	    return;
	}
	
	if (( fp = popen( cmd, "r" )) == NULL ) {
	    NSLog( @"popen failed: %s", strerror( errno ));
	    [ aboutToolsVersionView insertText:
		    [ NSString stringWithFormat:
		    @"Couldn't get version for %s: %s.\n\n",
		    toolpath, strerror( errno ) ]];
	    continue;
	}
	
	[ aboutToolsVersionView insertText:
		[ NSString stringWithFormat: @"%s:\n\t", toolpath ]];
	while ( fgets( buf, MAXPATHLEN, fp ) != NULL ) {
	    [ aboutToolsVersionView insertText:
		    [ NSString stringWithUTF8String: buf ]];
	    [ aboutToolsVersionView insertText: @"\t" ];
	}
	
	( void )pclose( fp );
    
	[ aboutToolsVersionView insertText: @"\n\n" ];
    }
    
    [ aboutToolsVersionView setEditable: NO ];
    [ aboutToolsVersionView scrollRangeToVisible: NSMakeRange( 0, 0 ) ];
    
    [ aboutPanel makeKeyAndOrderFront: nil ];
}

- ( void )awakeFromNib
{
    [ backButton setEnabled: NO ];
    [ logDrawer setContentSize: NSMakeSize( 527.0, 169.0 ) ];
    [ logDrawer setLeadingOffset: 15.0 ];
    [ viewUpdatesButton setEnabled: NO ];
    
    [ stepperWindow setFrameUsingName: @"RAStepperWindow" ];
    [ stepperWindow setFrameAutosaveName: @"RAStepperWindow" ];
    
    /* check local fsdiff version */
    [ self versionCheck ];
    
    [[ NSNotificationCenter defaultCenter ] addObserver: self
                                            selector: @selector( preferencesDidChange: )
                                            name: NSUserDefaultsDidChangeNotification
                                            object: nil ];
}

- ( void )setupSessionTempDirectory
{
    NSString        *tmproot = nil;
    char            template[ MAXPATHLEN ];
    
    tmproot = [[ NSUserDefaults standardUserDefaults ]
                    objectForKey: @"RATmpDirectory" ];
    if ( tmproot == nil ) {
        tmproot = @"/tmp";
    }
    
    if ( snprintf( template, MAXPATHLEN, "%s/.radassist-XXXXXX",
                [ tmproot UTF8String ] ) >= MAXPATHLEN ) {
        NSLog( @"template %@/.radassist-XXXXXX too long", tmproot );
        exit( 2 );
    }
    
    /* create a tmpdir for the session */
    if ( mkdtemp( template ) == NULL ) {
        /* XXXX we need to let the user know why */
        NSLog( @"mkdtemp %s: %s", template, strerror( errno ));
        exit( 2 );
    }
    
    [ self setSessionTempDirectory:
            [ NSString stringWithUTF8String: template ]];
}

- ( void )prePostQuery: ( NSString * )type
{
    NSAlert		*alert;
    NSString		*ppath;
    NSString		*prefix = @"Pre-";
    NSString		*key = @"RAAlwaysPreProcess";
    BOOL		isDir = NO;
    int			rc;

    
    ppath = [ NSString stringWithFormat: @"%s/%@", RADMIND_DIR, type ];
    ppath = [ ppath stringByResolvingSymlinksInPath ];
    
    if ( ![[ NSFileManager defaultManager ] fileExistsAtPath: ppath
	    isDirectory: &isDir ] || !isDir ) {
	return;
    }
    
    if ( [ type isEqualToString: @"postapply" ] ) {
	prefix = @"Post-";
	key = @"RAAlwaysPostProcess";
    }
    
    /* prompt user if the preferences don't tell us to prepost automatically */
    if ( ! [[ NSUserDefaults standardUserDefaults ]
	    boolForKey: key ] ) {
	alert = [ NSAlert alertWithMessageText: [ NSString stringWithFormat:
				NSLocalizedString( @"%@process difference transcript with scripts?",
						@"%@process difference transcript with scripts?" ),
						prefix ]
			    defaultButton: NSLocalizedString( @"Process", @"Process" )
			    alternateButton: NSLocalizedString( @"Cancel", @"Cancel" )
			    otherButton: @""
			    informativeTextWithFormat: NSLocalizedString(
					    @"%@processing will run all executable files in\n\n %@ "
						@"\n\non the difference transcript.",
					    @"%@processing will run all executable files in\n\n %@ "
						"\n\non the difference transcript." ), prefix, ppath ];

	rc = [ alert runModal ];
	switch ( rc ) {
	case NSAlertDefaultReturn:
	case NSAlertFirstButtonReturn:
	    break;
	    
	default:
	    return;
	}
    }
    
    if ( [ type isEqualToString: @"preapply" ] ) {
	[ self setPreApply: YES ];
    } else if ( [ type isEqualToString: @"postapply" ] ) {
	[ self setPostApply: YES ];
    }
}

- ( void )disableButtons
{
    [ continueButton setEnabled: NO ];
    [ backButton setEnabled: NO ];
    [ skipButton setEnabled: NO ];
}

- ( void )setTitleForStep: ( int )s
{
    NSString            *title = nil;
    
    if ( sessiontype == CREATE ) {
        switch ( s ) {
        case 0:
            title = @"Welcome to the Radmind Loadset Creator";
            break;
        case 1:
            title = @"Prepare for New Loadset Creation";
            break;
        case 2:
            title = @"Create New Loadset Transcript";
            break;
        case 3:
            title = @"Upload New Loadset to Radmind Server";
            break;
        case 4:
            title = @"New Loadset Successfully Created";
            break;
        default:
            title = @"";
            break;
        }
    } else if ( sessiontype == UPDATE ) {
        switch ( s ) {
        case 0:
            title = @"Welcome to the Radmind Updater";
            break;
        case 1:
            title = @"Check with Radmind Server for Updates";
            break;
        case 2:
            title = @"Check Filesystem for Changes to Make";
            break;
        case 3:
            title = @"Apply Necessary Changes to This Machine";
            break;
        case 4:
            title = @"Machine Successfully Updated";
            break;
        default:
            title = @"";
            break;
        }
    }
    
    [ stepTitleField setStringValue: title ];
}

- ( void )enableButtonsForStep: ( int )st
{
    switch ( st ) {
    case 0:
        [ continueButton setTitle: @"Continue" ];
        [ continueButton setEnabled: YES ];
        [ backButton setEnabled: YES ];
        [ skipButton setEnabled: NO ];
        break;
        
    case 1:
        [ skipButton setEnabled: YES ];
        [ backButton setEnabled: YES ];
        [ continueButton setEnabled: YES ];
        break;
        
    case 2:
        [ skipButton setEnabled: (( sessiontype == CREATE ) ? YES : NO ) ];
        [ backButton setEnabled: YES ];
        [ continueButton setEnabled: YES ];
        break;
        
    case 3:
        [ skipButton setEnabled: NO ];
        [ backButton setEnabled: YES ];
        [ continueButton setEnabled: YES ];
        break;
        
    case 4:
        if ( ! firstTimeWalkThrough ) {
	    [ continueButton setTitle: @"Finish" ];
	}
        [ continueButton setEnabled: YES ];
        [ skipButton setEnabled: NO ];
        break;
        
    default:
        return;
    }
}

- ( NSView * )viewForStep: ( int )st
{
    NSView		*view = nil;
    
    switch ( st ) {
    case 4:
        view = (( sessiontype == CREATE ) ? cCreationDone : aSysUpToDateView );
        break;
        
    case 3:
        view = (( sessiontype == CREATE ) ? cLcreateView : aDiffsFoundView );
        break;
        
    case 2:
        view = (( sessiontype == CREATE ) ? cFsdiffView : aFsdiffView );
        break;
    
    case 1:
        view = (( sessiontype == CREATE ) ? cCheckUpView : aCkForUpdatesView );
        break;
        
    default:
    case 0:
        view = (( sessiontype == CREATE ) ? cWelcomeView : aWelcomeView );
        break;
    }
    
    return( view );
}

- ( IBAction )installTools: ( id )sender
{
    NSString		*pkgName;
    
    pkgName = [[ NSBundle mainBundle ]
		objectForInfoDictionaryKey:
		@"RABundledRadmindToolsPackageName" ];
    if ( pkgName == nil ) {
	goto INSTALL_FAILED;
    }
    
    if ( ! [[ NSWorkspace sharedWorkspace ] openFile:
            [[ NSBundle mainBundle ] pathForResource: pkgName
                                        ofType: @"pkg" ]] ) {
        goto INSTALL_FAILED;
    }
    
    [ self setNeedsToolUpdate: NO ];
    
    return;
    
INSTALL_FAILED:
    NSBeginAlertSheet( NSLocalizedString( @"Error", @"Error" ),
	    NSLocalizedString( @"OK", @"OK" ), @"", @"", stepperWindow, self,
	    NULL, NULL, NULL,
	    NSLocalizedString(
		@"Couldn't open Radmind Tool Installer. You may need to "
		    @"reinstall the Radmind Assistant before continuing.",
		@"Couldn't open Radmind Tool Installer. You may need to "
		    @"reinstall the Radmind Assistant before continuing." ));
}

- ( void )installRadmindTools: ( BOOL )firstInstall
{
    int			rc;
    NSString		*msg = @"";
    
    if ( firstInstall == YES ) {
	msg = @"You need to install the radmind tools before continuing. "
		@"Click OK to open the tool installer.";
    } else if ( firstInstall == NO ) {
	msg = @"Click OK to upgrade the radmind tools on this system.";
    }
    
    rc = NSRunAlertPanel( @"Install Radmind Tools",
		msg, @"OK", @"Cancel", @"" );
		
    switch ( rc ) {
    case NSAlertDefaultReturn:
	break;
	
    case NSAlertAlternateReturn:
	if ( firstInstall ) {
	    rc = NSRunAlertPanel( @"Warning",
		    @"You cannot use the Radmind Assistant without first installing the "
		    @"radmind tools. Are you sure you want to cancel?",
		    @"Don't Cancel", @"Cancel", @"" );
	    
	    switch ( rc ) {
	    case NSAlertDefaultReturn:
		break;
		
	    case NSAlertAlternateReturn:
		[ NSApp terminate: nil ];
	    }
	}
	if ( firstInstall ) {
	    break;
	} else {
	    return;
	}
    }
	
    [ self installTools: nil ];
}

- ( IBAction )firstTimeRun: ( id )sender
{
    int			choice = [ ftChoiceMatrix selectedRow ], i;
    NSString		*tmpTranscript = nil;
    NSString		*negativeTranscriptPath = nil;
    NSArray		*negativeTranscripts = nil;
    
    [ stepperWindow makeKeyAndOrderFront: nil ];
    
    switch ( choice ) {
    case OLDHAND:
        step = 0;
        firstTimeWalkThrough = NO;
        [ continueButton setAction: @selector( next: ) ];
        [ backButton setAction: @selector( goBackToFirstRunChoice: ) ];
        [ self enableButtonsForStep: step ];
        [ self setTitleForStep: step ];
        [ viewBox setContentView: aWelcomeView ];
        [ toolsLogField setString: @"" ];
        break;
        
    case NEWSERVER:
        [ continueButton setAction: @selector( setupServer: ) ];
        [ backButton setAction: @selector( goBackToFirstRunChoice: ) ];
        [ viewBox setContentView: setupServerView ];
        break;
        
    case NEWCLIENT:
        negativeTranscripts = [ NSArray negativeTranscriptsForCurrentSystem ];
        if ( negativeTranscripts == nil ) {
            negativeTranscripts = [ NSArray arrayWithObjects:
                                        @"10.3-desktop-negative",
                                        @"10.3-lab-negative", nil ];
        }
        
        [ self setValidNegativeTranscripts: negativeTranscripts ];
        [ negativeTranscriptTable reloadData ];
        
        for ( i = 0; i < [[ self validNegativeTranscripts ] count ]; i++ ) {
            negativeTranscriptPath = [[ NSBundle mainBundle ]
                                    pathForResource: [ negativeTranscripts objectAtIndex: i ]
                                    ofType: @"T" ];
            tmpTranscript = [ NSString stringWithFormat: @"%@/%@",
                                [ self sessionTempDirectory ],
                                [ negativeTranscriptPath lastPathComponent ]];
            /* copy negative transcript to tmp space first */
            if ( ! [[ NSFileManager defaultManager ] copyPath: negativeTranscriptPath
                                                    toPath: tmpTranscript
                                                    handler: nil ] ) {
                if ( access(( char * )[ tmpTranscript UTF8String ], F_OK ) < 0 ) {
                    NSLog( @"Failed to copy %@ to %@.", negativeTranscriptPath,
                                                            tmpTranscript );
                    return;
                }
            }
            tmpTranscript = nil;
            negativeTranscriptPath = nil;
        }
        
        [ stepTitleField setStringValue: @"Set Up a New Radmind Client" ];
        [ continueButton setAction: @selector( setSystemWideDefaults: ) ];
        [ backButton setAction: @selector( goBackToFirstRunChoice: ) ];
        [ self enableButtonsForStep: 3 ];
        
#define RADMIND_CERT_DIR	"/var/radmind/cert"
        [[ setupDefaultsTLSLevel menu ] setAutoenablesItems: NO ];
        if ( access( RADMIND_CERT_DIR, F_OK ) < 0 ) {
            [[[ setupDefaultsTLSLevel menu ] itemAtIndex: 1 ] setEnabled: NO ];
            [[[ setupDefaultsTLSLevel menu ] itemAtIndex: 2 ] setEnabled: NO ];
        } else {
            [[[ setupDefaultsTLSLevel menu ] itemAtIndex: 1 ] setEnabled: YES ];
            [[[ setupDefaultsTLSLevel menu ] itemAtIndex: 2 ] setEnabled: YES ];
        }
        
        [ viewBox setContentView: setupDefaultsView ];
        [ stepperWindow makeFirstResponder: setupDefaultsServerField ];
	
        break;
    }
}

- ( IBAction )goBackToFirstRunChoice: ( id )sender
{
    step = 0;
    [ self enableButtonsForStep: step ];
    [ continueButton setAction: @selector( firstTimeRun: ) ];
    [ viewBox setContentView: firstTimeRunView ];
    [ stepperWindow setTitle: @"Radmind Assistant: First Run" ];
    [ stepperWindow makeKeyAndOrderFront: nil ];
    [ stepTitleField setStringValue: @"First-Time Run" ];
}

- ( IBAction )firstRunDetected: ( id )sender
{
    firstTimeWalkThrough = YES;
    [ stepTitleField setStringValue: @"First-Time Run" ];
    step = 0;
    [ self scanForRadmindServers: nil ];
    [ self enableButtonsForStep: 2 ];
    [ continueButton setAction: @selector( firstTimeRun: ) ];
    [ viewBox setContentView: firstTimeRunView ];
    [ stepperWindow setTitle: @"Radmind Assistant: First Run" ];
    
    /* if there's a radmind server on the subnet, assume we're on a client */
    if ( [ zcServerAddresses count ] > 0 ) {
        [ ftChoiceMatrix selectCellAtRow: 1 column: 0 ];
    }
    
    [ stepperWindow makeKeyAndOrderFront: nil ];
}

- ( IBAction )beginUpdate: ( id )sender
{
    BOOL		firstRun = NO;
    
    sessiontype = UPDATE;

    [ self setupSessionTempDirectory ];
    
    [ stepperWindow makeKeyAndOrderFront: nil ];
    
    if ( access( RADMIND_DIR, F_OK ) == 0 ) {
        step = 0;
        [ continueButton setAction: @selector( next: ) ];
        [ backButton setAction: @selector( back: ) ];
        [ self enableButtonsForStep: step ];
        [ self setTitleForStep: step ];
        [ viewBox setContentView: aWelcomeView ];
        [ stepperWindow setTitle: @"Radmind Assistant: Update System" ];
    } else {
        [ self firstRunDetected: nil ];
	firstRun = YES;
    }
    
    [ self setApplicableTranscriptName:
                [ NSString stringWithFormat: @"%@/radassist-apply-%d.T",
                [ self sessionTempDirectory ], getpid() ]];
    
    if ( [ self needsToolUpdate ] ) {
	[ self installRadmindTools: firstRun ];
    }
}

- ( IBAction )beginCreation: ( id )sender
{
    sessiontype = CREATE;
    [ stepperWindow setTitle: @"Radmind Assistant: Create Loadset" ];
    
    step = 0;
    firstTimeWalkThrough = NO;
    [ self enableButtonsForStep: step ];
    [ self setTitleForStep: step ];
    [ backButton setAction: @selector( back: ) ];
    [ continueButton setAction: @selector( next: ) ];
    [ cLcreateZeroSwitch setState: NSOffState ];
    [ viewBox setContentView: cWelcomeView ];
    [ toolsLogField setString: @"" ];
    [ stepperWindow makeKeyAndOrderFront: nil ];
}

- ( IBAction )setupServer: ( id )sender
{
    NSString		*error = nil;
    RAServerCreator	*rsc = [[ RAServerCreator alloc ] init ];
    const char		*serverpath = "/usr/local/sbin/radmind";
    
    error = [ rsc setupServer ];
    
    if ( error != nil ) {
        [ errorMessageField setStringValue: error ];
        [ viewBox setContentView: errorView ];
        [ backButton setAction: @selector( firstTimeRun: ) ];
    } else {
        NSArray		*args = nil;
        int		exec = 0;
        
        switch ( [ radmindServerOptionMatrix selectedRow ] ) {
        case 0:
            args = [ NSArray arrayWithObjects: @"-R", @"-u077", nil ];
            exec = 1;
            break;
            
        case 1:
            args = [ NSArray arrayWithObject: @"-u077" ];
            exec = 1;
            break;
            
        case 2:
        default:
            [ cFinishedMessageField setStringValue: @"Installation was successful." ];
            break;
        }
    
        if ( exec ) {
            if ( access( serverpath, X_OK | F_OK ) < 0 ) {
                NSRunAlertPanel( @"Couldn't locate the radmind daemon.",
                        @"The Radmind Assistant couldn't locate the radmind daemon. "
                        @"Please install the radmind tools and try again.",
                        @"OK", @"", @"" );
                return;
            }
            [ self execTool: RADMIND arguments: args ];
        }
        
        [ cFinishedHeaderField setStringValue:
                @"Radmind Server Setup Complete." ];
        [ continueButton setTitle: @"Finish" ];
        step = 4;
        [ continueButton setAction: @selector( finishServerSetup: ) ];
        [ viewBox setContentView: cCreationDone ];
    }
}

- ( IBAction )finishServerSetup: ( id )sender
{
    [ self openServerManager: nil ];
    [ NSApp terminate: nil ];
}

/* set defaults used in automated scripts */
- ( IBAction )setSystemWideDefaults: ( id )sender
{
    NSString			*servername = [ setupDefaultsServerField stringValue ];
    NSString			*fsdiffpath = [ setupDefaultsFsdiffPath titleOfSelectedItem ];
    int				tlslevel = [ setupDefaultsTLSLevel indexOfSelectedItem ];
    int				globalcksums = [ setupDefaultsChecksums selectedRow ];
    int				rc = 0;
    NSUserDefaults		*defaults = [ NSUserDefaults standardUserDefaults ];
    RAAuthExec			*rae = nil;
                                           
    [ defaults setObject: servername forKey: @"radmindhost" ];
    [ defaults setObject: fsdiffpath forKey: @"runfrompath" ];
    [ defaults setInteger: tlslevel forKey: @"tlslevel" ];
    [ defaults setInteger: globalcksums forKey: @"cksum" ];
    [ defaults synchronize ];
    
    rae = [[ RAAuthExec alloc ] init ];
    rc = [ rae executeTool: -1
		withArgs: [ NSArray arrayWithObjects: @"-A",
			    @"SetGlobalDefaults", @"--", @".", nil ]
		controller: nil ];
    [ rae release ];

    switch ( rc ) {
    case 0:
        break;
        
    case -2:
        [ self authorizationFailed ];
        return;
    
    default:
        /* stopgap, obviously */
        NSRunAlertPanel( @"Failed to set system-wide preferences.", @"", 
                NSLocalizedString( @"OK", @"OK" ), @"", @"" );
        return;
    }
    
    [ continueButton setAction: @selector( automationConfigured: ) ];
    [ viewBox setContentView: setupAutomationView ];
    [ backButton setAction: @selector( goBackToFirstRunChoice: ) ];
}

- ( IBAction )configureRadmindAutomation: ( id )sender
{
    [[ NSApp delegate ] showAutomationPreferencesPane ];
}

- ( IBAction )automationConfigured: ( id )sender
{
    [ continueButton setAction: @selector( negativePrepared: ) ];
    [ viewBox setContentView: negativePreparationView ];
    [ stepperWindow makeFirstResponder: negativeTranscriptTable ];
    [ backButton setAction: @selector( goBackToFirstRunChoice: ) ];
}

- ( IBAction )chooseBaseNegativeTranscript: ( id )sender
{
    NSOpenPanel			*op = [ NSOpenPanel openPanel ];
    NSString			*defaultDir = nil;
    
    defaultDir = [[ NSUserDefaults standardUserDefaults ]
                        objectForKey: @"NSDefaultOpenDirectory" ];
                        
    if ( defaultDir == nil ) {
        defaultDir = NSHomeDirectory();
    }
    
    [ op setCanChooseDirectories: NO ];
    [ op setResolvesAliases: YES ];
    [ op setCanChooseFiles: YES ];
    
    [ op beginSheetForDirectory: defaultDir file: nil types: nil
            modalForWindow: stepperWindow modalDelegate: self
            didEndSelector: @selector( chooseBaseNegativePanelDidEnd:returnCode:contextInfo: )
            contextInfo: NULL ];
}

- ( void )chooseBaseNegativePanelDidEnd: ( NSOpenPanel * )op returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    NSString			*negativePath = nil;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
        
    case NSAlertAlternateReturn:
    default:
        return;
    }
    
    negativePath = [[ op filenames ] objectAtIndex: 0 ];
    
    [ baseNegativeTranscriptField setStringValue: [ negativePath lastPathComponent ]];
    [ self setCreatableTranscriptName: negativePath ];
    [ clearCustomNegativeButton setEnabled: YES ];
}

- ( IBAction )clearCustomNegativeTranscriptChoice: ( id )sender
{
    [ baseNegativeTranscriptField setStringValue: @"" ];
    [ self setCreatableTranscriptName: nil ];
    [ clearCustomNegativeButton setEnabled: NO ];
}

- ( IBAction )negativePrepared: ( id )sender
{
    NSString			*tmpTranscript = nil;
    NSString			*negativeTranscript = nil;
    int				row = [ negativeTranscriptTable selectedRow ];
    
    if ( row < 0 && [ self creatableTranscriptName ] == nil ) {
        NSBeginAlertSheet( @"You must choose a negative transcript to continue.",
                @"OK", @"", @"", stepperWindow, self, NULL, NULL, nil, @"" );
        return;
    }
    
    if ( [ self creatableTranscriptName ] == nil ) {
        negativeTranscript = [[ self validNegativeTranscripts ] objectAtIndex: row ];
        
        tmpTranscript = [ NSString stringWithFormat: @"%@/%@.T",
                            [ self sessionTempDirectory ], negativeTranscript ];
        [ self setCreatableTranscriptName: tmpTranscript ];
    }
    
    [ backButton setAction: @selector( firstTimeRun: ) ];
    
    step = 3;
    sessiontype = CREATE;
    
    [ continueButton setAction: @selector( uploadNegativeTranscript: ) ];
    [ cLoadsetNameField setStringValue:
                    [[ self creatableTranscriptName ] lastPathComponent ]];
    [ cLcreateZeroSwitch setState: NSOnState ];
    [ stepTitleField setStringValue: @"Upload Negative Loadset to Radmind Server" ];
    [ viewBox setContentView: cLcreateView ];
    
    if ( [ zcServerAddresses count ] > 0 ) {
        NSBeginAlertSheet( @"Radmind Server Detected", @"Set Default",
                @"Don't Set Default", @"", stepperWindow, self,
                @selector( setDefaultServerSheetDidEnd:returnCode:contextInfo: ),
                NULL, NULL, @"The Radmind Assistant detected a radmind server "
                @"running on %@. Would you like to make this the default server "
                @"for this machine?",
                [ zcServerAddresses objectAtIndex: 0 ] );
    }
}

- ( void )setDefaultServerSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    if ( [ zcServerAddresses count ] <= 0 ) return;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        [[ NSUserDefaults standardUserDefaults ]
                setObject: [ zcServerAddresses objectAtIndex: 0 ]
                forKey: @"radmindhost" ];
        break;
    
    case NSAlertAlternateReturn:
    default:
        break;
    }
}

- ( IBAction )examineNegativeTranscript: ( id )sender
{
    NSString		*tmpTranscript;
    NSString		*editor;
    RAEditorLauncher	*re;
    int			row = [ negativeTranscriptTable selectedRow ];
    
    if ( row < 0 && [ self creatableTranscriptName ] == nil ) {
        NSBeginAlertSheet( @"No transcript selected.", @"OK", @"", @"",
                stepperWindow, self, NULL, NULL, nil,
                @"Select a transcript from the list, and try again." );
        return;
    }

    if ( [ self creatableTranscriptName ] == nil ) {
        tmpTranscript = [ NSString stringWithFormat: @"%@/%@.T",
                            [ self sessionTempDirectory ],
                            [[ self validNegativeTranscripts ] objectAtIndex: row ]];
    } else {
        tmpTranscript = [ self creatableTranscriptName ];
    }
    
    re = [[ RAEditorLauncher alloc ] init ];
    editor = [[ NSUserDefaults standardUserDefaults ]
                objectForKey: @"transeditor" ];
                
    if ( editor == nil ) editor = @"Radmind Transcript Editor";
    
    if ( ! [ re viewTrans: tmpTranscript withTextEditor: editor ] ) {
        NSRunAlertPanel( @"Error launching text editor.",
            @"Check the console for error messages", @"OK", @"", @"" );
    }
    
    [ re release ];
}

- ( IBAction )uploadNegativeTranscript: ( id )sender
{
    [ self actionForStep: step ];
    [ backButton setAction: @selector( negativePrepared: ) ];
    
    if ( step == 3 && sessiontype == CREATE 
            && [[ NSUserDefaults standardUserDefaults ] boolForKey: @"userauth" ] ) {
        if ( [[ cLoadsetNameField stringValue ] length ] ) {
            [ self getPassword ];
        }
    }
}

- ( NSString * )retrievePassword
{
    NSString            *pass = [ userAuthPassField stringValue ];
    
    if ( [ pass length ] == 0 ) {
        pass = nil;
    }
    
    /* clear the password field when we're done */
    [ userAuthPassField setStringValue: @"" ];
    
    return( pass );
}

- ( void )getPassword
{
    NSString		*user = @"";

    user = [[ NSUserDefaults standardUserDefaults ]
                stringForKey: @"username" ];
    if ( user == nil || [[ userAuthNameField stringValue ] length ] ) {
        struct passwd	*pwd = getpwuid( getuid());
        
        if ( pwd == NULL ) {
            [  self toolError: "Couldn't get a username for authentication!" ];
            return;
        }
        [ userAuthNameField setStringValue:
                    [ NSString stringWithUTF8String: pwd->pw_name ]];
    } else {
        [ userAuthNameField setStringValue: user ];
    }
    [ userAuthNameField setEditable: YES ];
    
    [ NSApp beginSheet: passwordSheet
            modalForWindow: stepperWindow
            modalDelegate: self
            didEndSelector: NULL
            contextInfo: nil ];
}

- ( IBAction )storePassword: ( id )sender
{
    NSArray		*argv = nil;
    int			tlslevel = 0, port;
    NSNumber		*tL = [[ NSUserDefaults standardUserDefaults ]
                            objectForKey: @"tlslevel" ];
    BOOL		caseInsensitive = [[ NSUserDefaults standardUserDefaults ]
					    boolForKey: @"RACaseInsensitive" ];

    if ( ! [[ userAuthNameField stringValue ] length ]
            || ! [[ userAuthPassField stringValue ] length ] ) {
        NSRunAlertPanel( @"You must provide a username and password to continue.",
                @"", @"OK", @"", @"" );
        return;
    }
    
    if ( tL != nil ) {
        tlslevel = [ tL intValue ];
    } else {
        tlslevel = 0;
    }
    
    argv = [ NSArray arrayWithObjects: @"-i", @"-%", nil ];
                
    if ( [ cLcreateCksumSwitch selectedRow ] == 1 ) {
        argv = [ argv arrayByAddingObject: @"-csha1" ];
    }
    
    if ( [ cLcreateZeroSwitch state ] == NSOnState ) {
        argv = [ argv arrayByAddingObject: @"-N" ];
    }
    
    if ( [[ NSUserDefaults standardUserDefaults ]
            boolForKey: @"DieOnErrorsDuringUpload" ] == NO ) {
        argv = [ argv arrayByAddingObject: @"-F" ];
    }
    
    if ( caseInsensitive ) {
	argv = [ argv arrayByAddingObject: @"-I" ];
    }
    
    if (( port = [[ NSUserDefaults standardUserDefaults ]
	    integerForKey: @"RadmindServerPort" ] ) > 0 ) {
	argv = [ argv arrayByAddingObjectsFromArray:
		    [ NSArray arrayWithObjects: @"-p",
		    [ NSString stringWithFormat: @"%d", port ], nil ]];
    }
    
    argv = [ argv arrayByAddingObjectsFromArray:
                    [ NSArray arrayWithObjects:
                        @"-l",
                        [ NSString stringWithFormat: @"-U%@",
                                [ userAuthNameField stringValue ]],
                        [ NSString stringWithFormat: @"-w%d", tlslevel ],
                        [ NSString stringWithFormat: @"-h%@",
                                [ cLcreateServerField stringValue ]],
                        [ self creatableTranscriptName ], nil ]];
    
    [ argv retain ];
    
    [ self endPasswordSheet: nil ];
    [ self execTool: CURRENTTOOL arguments: argv ];
    [ argv release ];
}

- ( IBAction )endPasswordSheet: ( id )sender
{
    [ passwordSheet orderOut: nil ];
    [ NSApp endSheet: passwordSheet ];
}

- ( IBAction )chooseTranscript: ( id )sender
{
    NSOpenPanel		*op = [ NSOpenPanel openPanel ];
    NSString		*lastDir =
                        [[ NSUserDefaults standardUserDefaults ]
                            objectForKey: @"NSDefaultOpenDirectory" ];
                                
    if ( lastDir == nil ) {
        lastDir = @"/tmp";
    }
    
    [ op setTitle: @"Choose a Transcript for Your Loadset" ];
    [ op setPrompt: @"Choose" ];
    [ op setCanChooseDirectories: NO ];
    [ op setAllowsMultipleSelection: NO ];
    
    [ op beginSheetForDirectory: lastDir file: nil types: nil
            modalForWindow: stepperWindow modalDelegate: self
            didEndSelector:
                @selector( chooseLoadsetSheetDidEnd:returnCode:contextInfo: )
            contextInfo: nil ];
}

- ( void )chooseLoadsetSheetDidEnd: ( NSOpenPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    NSString		*t = nil;
    
    switch ( rc ) {
    case NSOKButton:
        t = [[ sheet filenames ] objectAtIndex: 0 ];
        break;
        
    default:
    case NSCancelButton:
        return;
    }
    
    if ( t == nil ) {
        return;
    }
    
    [ self setCreatableTranscriptName: t ];
    [ cLoadsetNameField setStringValue:
                        [[ self creatableTranscriptName ] lastPathComponent ]];
}

- ( void )actionForStep: ( int )st
{
    int             toolforstep = 0, uauth = 0, rc;
    int             tlslevel = 0, port;
    int		    zlevel = 0;

    NSNumber        *tL = [[ NSUserDefaults standardUserDefaults ]
                            objectForKey: @"tlslevel" ];
    NSString        *fsdiffPath, *serverAddress;
    NSArray         *argv = nil;
    BOOL	    caseInsensitive = [[ NSUserDefaults standardUserDefaults ]
					    boolForKey: @"RACaseInsensitive" ];
    
    if ( tL != nil ) {
        tlslevel = [ tL intValue ];
    }
    
    switch ( st ) {
    case 1:
        if ( sessiontype == CREATE ) {
            [ self nextStepUI ];
            return;
        }
        toolforstep = KTCHECK;
        
        serverAddress = [ aKtcheckHostField stringValue ];
        
        if ( [ serverAddress length ] == 0 ) {
            serverAddress = @"radmind";
        }
        
        argv = [ NSArray arrayWithObjects: @"-csha1", @"-i",
                        [ NSString stringWithFormat: @"-w%d", tlslevel ],
                        [ NSString stringWithFormat: @"-h%@",
                                                    [ aKtcheckHostField stringValue ]],
                        nil ];
	
	if (( port = [[ NSUserDefaults standardUserDefaults ]
		integerForKey: @"RadmindServerPort" ] ) > 0 ) {
	    argv = [ argv arrayByAddingObjectsFromArray:
			[ NSArray arrayWithObjects: @"-p",
			[ NSString stringWithFormat: @"%d", port ], nil ]];
	}
	zlevel = [[ NSUserDefaults standardUserDefaults ]
			integerForKey: @"RadmindServerCompressionLevel" ];
	if ( zlevel > 1 && zlevel < 10 ) {
	    argv = [ argv arrayByAddingObjectsFromArray:
			[ NSArray arrayWithObjects: @"-Z",
			[ NSString stringWithFormat: @"%d", zlevel ], nil ]];
	}
        [ argv retain ];
        break;
        
    case 2:
        toolforstep = (( sessiontype == CREATE ? FSDIFFC : FSDIFFA ));
        
        fsdiffPath = [[ NSUserDefaults standardUserDefaults ]
                                objectForKey: @"runfrompath" ];
        
        if ( fsdiffPath == nil ) {
            fsdiffPath = @"/";
        }
                                
        if ( sessiontype == CREATE ) {
            NSString		*loadsetName = [ cFsdiffLoadsetName stringValue ];
            NSString		*type = @"-C";
            
            if ( [ loadsetName length ] == 0 ) {
                loadsetName = @"New-Loadset.T";
            }
            
            [ self setCreatableTranscriptName:
                    [ NSString stringWithFormat: @"%@/%@",
                                [ self sessionTempDirectory ], loadsetName ]];
                    
            [ cLoadsetNameField setStringValue: [ cFsdiffLoadsetName stringValue ]];
            
            argv = [ NSArray arrayWithObject: type ];
            
            if ( [ cFsdiffCksumSwitch selectedRow ] == 1 ) {
                argv = [ argv arrayByAddingObject: @"-csha1" ];
            }
	    if ( caseInsensitive ) {
		argv = [ argv arrayByAddingObject: @"-I" ];
            }
	    
            argv = [ argv arrayByAddingObjectsFromArray:
                            [ NSArray arrayWithObjects: @"-%",
                                [ NSString stringWithFormat: @"-o%@",
                                            [ self creatableTranscriptName ]],
                                fsdiffPath, nil ]];
        } else if ( sessiontype == UPDATE ) {
            NSString		*type = @"-A";
            
            argv = [ NSArray arrayWithObject: type ];
            
            if ( [ aFsdiffCksumSwitch selectedRow ] == 1 ) {
                argv = [ argv arrayByAddingObject: @"-csha1" ];
            }
	    if ( caseInsensitive ) {
		argv = [ argv arrayByAddingObject: @"-I" ];
            }
            
            argv = [ argv arrayByAddingObjectsFromArray:
                            [ NSArray arrayWithObjects: @"-%",
                                [ NSString stringWithFormat: @"-o%@",
                                            [ self applicableTranscriptName ]],
                                fsdiffPath, nil ]];
        }

        [ argv retain ];
        break;
        
    case 3:
        if ( sessiontype == CREATE ) {
            if ( ! [[ cLoadsetNameField stringValue ] length ] ) {
                NSBeginAlertSheet( @"You must choose a loadset transcript"
                        @" before continuing.", @"OK", @"", @"",
                        stepperWindow, nil, NULL, NULL, NULL,
                        @"Click the \'Choose Loadset...\' button"
                        @" to select your transcript." );
                return;
            }
            
            if ( [[ NSUserDefaults standardUserDefaults ]
                        boolForKey: @"RADontPromptToQuitAppsBeforeUpload" ] == NO ) {
                rc = NSRunAlertPanel( NSLocalizedString( @"Quit other open applications?",
                                                        @"Quit other open applications?" ),
                        NSLocalizedString( @"It's a good idea to quit other open applications "
                                                @"before proceeding, as they can change filesystem "
                                                @"information and cause the radmind tools to generate "
                                                @"errors. Would you like to quit the other open "
                                                @"applications?",
                                             @"It's a good idea to quit other open applications "
                                                @"before proceeding, as they can change filesystem "
                                                @"information and cause the radmind tools to generate "
                                                @"errors. Would you like to quit the other open "
                                                @"applications?" ),
                        NSLocalizedString( @"Quit Other Applications", @"Quit Other Applications" ),
                        NSLocalizedString( @"Don't Quit", @"Don't Quit" ),
                        NSLocalizedString( @"Don't Quit and Don't Ask Again", @"Don't Quit and Don't Ask Again" ));
                
                switch ( rc ) {
                case NSAlertDefaultReturn:
                    [[ NSWorkspace sharedWorkspace ] quitOtherOpenApplications ];
                    break;
                
                case NSAlertAlternateReturn:
                    break;
                    
                case NSAlertOtherReturn:
                    [[ NSUserDefaults standardUserDefaults ] setBool: YES
                                        forKey: @"RADontPromptToQuitAppsBeforeUpload" ];
                    break;
                }
            }
            
            /* if user authentication is on, get password sheet */
            if ( [[ NSUserDefaults standardUserDefaults ]
                                    boolForKey: @"userauth" ] ) {
                toolforstep = LCREATE;
                uauth = 1;
            } else {
                toolforstep = LCREATE;
                
                argv = [ NSArray arrayWithObject: @"-i" ];
                
                if ( [ cLcreateCksumSwitch selectedRow ] == 1 ) {
                    argv = [ argv arrayByAddingObject: @"-csha1" ];
                }
                
                if ( [ cLcreateZeroSwitch state ] == NSOnState ) {
                    argv = [ argv arrayByAddingObject: @"-N" ];
                }
                
                if ( [[ NSUserDefaults standardUserDefaults ]
                        boolForKey: @"DieOnErrorsDuringUpload" ] == NO ) {
                    argv = [ argv arrayByAddingObject: @"-F" ];
                }
		if ( caseInsensitive ) {
		    argv = [ argv arrayByAddingObject: @"-I" ];
		}
		
		if (( port =  [[ NSUserDefaults standardUserDefaults ]
			integerForKey: @"RadmindServerPort" ] ) > 0 ) {
		    argv = [ argv arrayByAddingObjectsFromArray:
				[ NSArray arrayWithObjects: @"-p",
				[ NSString stringWithFormat: @"%d", port ], nil ]];
		}
		
		zlevel = [[ NSUserDefaults standardUserDefaults ]
				integerForKey: @"RadmindServerCompressionLevel" ];
		if ( zlevel > 1 && zlevel < 10 ) {
		    argv = [ argv arrayByAddingObjectsFromArray:
				[ NSArray arrayWithObjects: @"-Z",
				[ NSString stringWithFormat: @"%d", zlevel ], nil ]];
		}
                
                argv = [ argv arrayByAddingObjectsFromArray:
                                [ NSArray arrayWithObjects: @"-%",
                                    [ NSString stringWithFormat: @"-w%d", tlslevel ],
                                    [ NSString stringWithFormat: @"-h%@",
                                            [ cLcreateServerField stringValue ]],
                                    [ self creatableTranscriptName ], nil ]];
                
                [ argv retain ];
            }
        } else {
            toolforstep = LAPPLY;
            
	    /* pre- and post-apply checks */
	    [ self setPreApply: NO ];
	    [ self setPostApply: NO ];
	    [ self prePostQuery: @"preapply" ];
	    [ self prePostQuery: @"postapply" ];
	    
            argv = [ NSArray arrayWithObjects: @"-i", @"-%", nil ];
            
            if ( [ aLapplyCksumSwitch selectedRow ] == 1 ) {
                argv = [ argv arrayByAddingObject: @"-csha1" ];
            }
	    if ( caseInsensitive ) {
		argv = [ argv arrayByAddingObject: @"-I" ];
            }
            
            if ( [[ NSUserDefaults standardUserDefaults ]
                                    boolForKey: @"DontRemoveFileLocks" ] == NO ) {
                argv = [ argv arrayByAddingObject: @"-F" ];
            }
	    
	    if (( port = [[ NSUserDefaults standardUserDefaults ]
		    integerForKey: @"RadmindServerPort" ] ) > 0 ) {
		argv = [ argv arrayByAddingObjectsFromArray:
			    [ NSArray arrayWithObjects: @"-p",
			    [ NSString stringWithFormat: @"%d", port ], nil ]];
	    }
	    
	    zlevel = [[ NSUserDefaults standardUserDefaults ]
			    integerForKey: @"RadmindServerCompressionLevel" ];
	    if ( zlevel > 1 && zlevel < 10 ) {
		argv = [ argv arrayByAddingObjectsFromArray:
			    [ NSArray arrayWithObjects: @"-Z",
			    [ NSString stringWithFormat: @"%d", zlevel ], nil ]];
	    }
            
            argv = [ argv arrayByAddingObjectsFromArray: [ NSArray arrayWithObjects:
                            [ NSString stringWithFormat: @"-w%d", tlslevel ],
                            [ NSString stringWithFormat: @"-h%@",
                                        [ aKtcheckHostField stringValue ]],
                            [ self applicableTranscriptName ], nil ]];
            [ argv retain ];
        }
        break;

    default:
        break;
    }
    
    CURRENTTOOL = toolforstep;
    
    if ( st > 0 && ! uauth ) {
        [ self execTool: toolforstep arguments: argv ];
        [ argv release ];
    }
}

- ( void )execTool: ( int )tool arguments: ( NSArray * )args
{
    [ aexec onewayExecuteTool: tool
                    withArgs: args
                    controller: self ];
}

- ( oneway void )showProgUIForTool: ( int )tool
{
    NSString		*toolheader = @"";
    
    switch ( tool ) {
    case KTCHECK:
        toolheader = @"Checking for updates on the server....";
        break;
    
    case FSDIFFA:
        toolheader = @"Checking filesystem for differences....";
        break;
        
    case FSDIFFC:
        toolheader = @"Creating transcript for new loadset...";
        break;
        
    case LAPPLY:
        toolheader = @"Applying changes to the system....";
        break;
    
    case LCREATE:
        toolheader = @"Uploading new loadset to server....";
        break;
        
    case REBOOT:
        return;
    
    default:
        break;
    }
    [ toolProgressView addSubview: toolProgCancelButton ];
    [ toolProgressView addSubview: toolProgBar ];
    [ toolHeaderField setStringValue: toolheader ];
    [ toolMsgField setStringValue: @"" ];
    [ toolProgBar setUsesThreadedAnimation: YES ];
    [ toolProgBar setAnimationDelay: ( 30.0 / 60.0 ) ];
    [ toolProgBar setIndeterminate: YES ];
    [ toolProgBar startAnimation: nil ];
    [ self disableButtons ];
    [ viewBox setContentView: toolProgressView ];
    [ toolProgBar setNeedsDisplay: YES ]; /* make sure animation starts */
}

- ( oneway void )updateProgress: ( NSString * )progressString
{
    NSLock              *lock = nil;
    char                progress[ LINE_MAX ];
    char                **targv;
    int                 tac;
    double              pct;
    
    if ( progressString == nil || ! [ progressString length ] ) {
        return;
    }
    
    lock = [[ NSLock alloc ] init ];
    if ( ! [ lock tryLock ] ) {
        if ( lock ) {
            [ lock release ];
        }
        NSLog( @"Couldn't set lock!" );
        return;
    }
    
    ( void )strlcpy( progress, [ progressString UTF8String ],
                                sizeof( progress ));
    
    if (( tac = argcargv( progress, &targv )) < 1 ) {
        NSLog( @"%@: not a progress string", progressString );
        [ lock release ];
        return;
    }
    
    ++targv[ 0 ];
    
    pct = strtod( targv[ 0 ], NULL );
    if ( errno == ERANGE ) {
        NSLog( @"strtod %s: %s", targv[ 0 ], strerror( errno ));
        [ lock release ];
        return;
    }
    
    if ( tac >= 2 ) {
        [ progressMessageField setStringValue:
                    [ NSString stringWithUTF8String: targv[ 1 ]]];
    }
    
    [ toolProgBar setIndeterminate: NO ];
    [ toolProgBar setMinValue: 0.0 ];
    [ toolProgBar setMaxValue: 100.0 ];
    [ toolProgBar setDoubleValue: pct ];
    
    if ( lock ) {
        [ lock release ];
    }
}

- ( IBAction )cancel: ( id )sender
{
    if ( ! [[ viewBox contentView ] isEqual: toolProgressView ] ) {
        return;
    }
    
    NSBeginAlertSheet( @"Are you sure you want to cancel this step?",
            @"Cancel", @"Don't Cancel", @"", stepperWindow, self,
            @selector( cancelStepSheetDidEnd:returnCode:contextInfo: ), NULL,
            NULL, @"" );
}

- ( void )cancelStepSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    RAAuthKiller	*akill;
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
	break;
	
    default:
	return;
    }
    
    akill = [[ RAAuthKiller alloc ] init ];
    [ akill killTool: CURRENTTOOL forController: self ];
    [ akill release ];

    KILLED = 1;
}

- ( void )killFinishedWithStatus: ( int )status
{
}

- ( void )authorizationFailed
{
    [ toolProgBar stopAnimation: nil ];
    [ toolProgBar retain ];
    [ toolProgBar removeFromSuperview ];
    [ toolProgCancelButton retain ];
    [ toolProgCancelButton removeFromSuperview ];
    [ toolHeaderField setStringValue: @"" ];
    [ errorMessageField setStringValue: AUTH_FAILED_MESSAGE ];
    [ viewBox setContentView: errorView ];
    [ backButton setAction: @selector( errorBack: ) ];
    [ backButton setEnabled: YES ];
}

- ( oneway void )tool: ( int )tool finishedWithStatus: ( int )status
{
    int		err = 0;
    
    [ progressMessageField setStringValue: @"" ];
    
    if ( status < 0 ) {
        [ toolsLogField insertText:
        @"There was an error locating one of the components of this application.\n" ];
        [ self toolError: "Failed to locate a component of this application." ];
        err++;
    }
    
    [ toolMsgField setStringValue: @"" ];
    
    if ( tool == KTCHECK ) {
        if ( firstTimeWalkThrough && status == 1 ) {
            sessiontype = CREATE;
            [ viewBox setContentView: cFsdiffView ];
            step = 2;
            [ stepperWindow makeFirstResponder: cFsdiffLoadsetName ];
            [ cFsdiffLoadsetName setStringValue: @"base-loadset.T" ];
	    [ cFsdiffCksumSwitch selectCellAtRow: 1 column: 0 ];
            [ continueButton setEnabled: YES ];
	    [ cLcreateZeroSwitch setState: NSOffState ];
	    [ cLcreateCksumSwitch selectCellAtRow: 1 column: 0 ];
            return;
        }
        
        if ( status == 1 ) {
            [ updatesInfoField setTextColor: [ NSColor redColor ]];
            [ updatesInfoField setStringValue: @"Updates found on the server." ];
            [ viewUpdatesButton setEnabled: YES ];
        } else if ( status == 0 ) {
            [ updatesInfoField setTextColor: [ NSColor blueColor ]];
            [ updatesInfoField setStringValue: @"No updates needed from server." ];
            [ viewUpdatesButton setEnabled: NO ];
        } else {
            err++;
        }
    } else if ( status != 0 ) {
        err++;
    }

    if ( err || KILLED ) {
        char		*toolname = nameforcmd( tool );
        
        [ toolsLogField insertText:
            [ NSString stringWithFormat:
                @"%s encountered an error. Abnormal exit status %d.\n",
                toolname, status ]];
                
        KILLED = 0; 
        return;
    }
    
    /* we started the radmind daemon for a first time user */
    if ( tool == RADMIND ) {
        NSString		*message = @"";
        
        switch ( [ radmindServerOptionMatrix selectedRow ] ) {
        case 0:
            message = @"The radmind server is now running on port "
                        @"6662 with Bonjour enabled.";
            break;
            
        case 1:
            message = @"The radmind server is now running on port "
                        @"6662 with Bonjour disabled.";
            break;
            
        default:	/* shouldn't happen */
            break;
        }
        
        [ cFinishedMessageField setStringValue: [ NSString stringWithFormat:
                @"Server installation was successful. %@ "
                @"At this point, you should install the Radmind Assistant "
                @"and the radmind tools on your first client machine, "
                @"and begin creating your loadsets.", message ]];
        return;
    }
    
    if ( sessiontype == UPDATE && step == 2 ) {
        char	*transpath = ( char * )[[ self applicableTranscriptName ] UTF8String ];
        struct stat		st;
                    
        if ( stat( transpath, &st ) < 0 ) {
            NSLog( @"couldn't stat %s: %s", transpath, strerror( errno ));
            /* possibly show an error view here */
        } else {
            /* if the transcript is empty, skip to "System is up to date." */
            if ( st.st_size == 0 ) {
                step++;
                [ aFinalUpdateMsg setStringValue:
                    @"This machine is up to date. No updates are necessary." ];
            } else {
                [ aFinalUpdateMsg setStringValue: @"You have successfully updated this "
			@"machine to match the information from the Radmind server." ];
            }
        }
    }
    
    if ( CURRENTTOOL == LCREATE ) {
        if ( firstTimeWalkThrough ) {
            if ( ! [[ cFinishedMessageField stringValue ] containsString: @"base loadset" ] ) {
                [ cFinishedMessageField setStringValue: [ NSString stringWithFormat:
                    @"Upload of %@ was successful. When you have checked the loadset in "
                    @"on the server, return to this client and click Continue to "
                    @"move on to the creation of the base loadset.",
                    [[ self creatableTranscriptName ] lastPathComponent ]]];
		    [ continueButton setTitle: @"Continue" ];
            } else {
                [ cFinishedMessageField setStringValue: [ NSString stringWithFormat:
                    @"Upload of %@ was successful. At this point, you should move to the "
                    @"server to verify and check in the loadset.",
                    [[ self creatableTranscriptName ] lastPathComponent ]]];
                firstTimeWalkThrough = NO;
		[ viewBox setContentView: cCreationDone ];
                step = 4;
		[ continueButton setAction: @selector( next: ) ];
		[ continueButton setEnabled: YES ];
		[ continueButton setTitle: @"Finish" ];
		return;
            }
        } else {
            [ cFinishedMessageField setStringValue: [ NSString stringWithFormat:
                    @"Upload of %@ was successful. At this point, you should move to the "
                    @"server to verify and check in the loadset.",
                    [[ self creatableTranscriptName ] lastPathComponent ]]];
        }
        
        [ self setCreatableTranscriptName: nil ];
    }
    
    [ continueButton setAction: @selector( next: ) ];
    [ self nextStepUI ];
}

- ( NSWindow * )assistantWindow
{
    return( stepperWindow );
}

- ( IBAction )viewUpdatesSheet: ( id )sender
{
    if ( [ logDrawer state ] == NSDrawerOpenState ||
                [ logDrawer state ] == NSDrawerOpeningState ) {
        [ logDrawer close ];
        [ viewUpdatesButton setTitle: @"View Updates" ];
    } else {
        [ logDrawer openOnEdge: NSMinYEdge ];
        [ viewUpdatesButton setTitle: @"Hide Updates" ];
    }
}

- ( void )openDrawer
{
    if ( [ stepperWindow isVisible ] ) {
        [ logDrawer openOnEdge: NSMinYEdge ];
    }
}

- ( BOOL )drawerShouldOpen: ( NSDrawer * )sender
{
    BOOL                        open = YES;
    
    if ( ! [ stepperWindow isVisible ] ) {
        NSBeep();
        open = NO;
    }
    
    return( YES );
}

- ( oneway void )addTextToLog: ( NSString * )text color: ( NSColor * )color
            displayInPane: ( BOOL )display
{
    NSLock			*lock = nil;
    NSMutableAttributedString	*attr = nil;
    
    lock = [[ NSLock alloc ] init ];
    
    if ( ! [ lock tryLock ] ) {
        NSLog( @"Couldn't set lock" );
        if ( lock != nil ) {
            [ lock release ];
        }
    }
    
    attr = [[ NSMutableAttributedString alloc ] initWithString: text ];
                                                    
    if ( display ) {
        [ toolMsgField setStringValue: text ];
    }
    
    [ toolsLogField setEditable: YES ];
    [ toolsLogField setSelectedRange:
                NSMakeRange( [[ toolsLogField textStorage ] length ], 0 ) ];
                
    [ attr beginEditing ];
    [ attr addAttribute: NSForegroundColorAttributeName
            value: color
            range: [ text rangeOfString: text ]];
    [ attr endEditing ];
    
    [ toolsLogField insertText: attr ];
    [ attr release ];
    [ toolsLogField setEditable: NO ];
    
    if ( lock != nil ) {
        [ lock release ];
    }
}

- ( oneway void )toolError: ( char * )errmsg
{
    [ toolProgBar stopAnimation: nil ];
    [ toolProgBar retain ];
    [ toolProgBar removeFromSuperview ];
    [ toolProgCancelButton retain ];
    [ toolProgCancelButton removeFromSuperview ];
    [ toolHeaderField setStringValue: @"" ];
    [ errorMessageField setStringValue: [ NSString stringWithUTF8String: errmsg ]];
    [ viewBox setContentView: errorView ];
    [ backButton setAction: @selector( errorBack: ) ];
    [ backButton setEnabled: YES ];
}

- ( IBAction )errorBack: ( id )sender
{
    [ viewBox setContentView: [ self viewForStep: step ]];
    [ self enableButtonsForStep: step ];
    [ self setTitleForStep: step ];
    [ backButton setAction: @selector( back: ) ];
}

- ( IBAction )back: ( id )sender
{
    if ( step > 0 ) step--;
    [ viewBox setContentView: [ self viewForStep: step ]];
    [ self enableButtonsForStep: step ];
    [ self setTitleForStep: step ];
}

/* NSNetService delegate method */
- ( void )netServiceDidResolveAddress: ( NSNetService * )sender
{
    NSData		*address;
    NSString		*serviceAddress = nil;
    struct sockaddr_in  *ip;
    struct hostent	*he;
    char                *service_address;
    unsigned int        port;

    if ( [[ sender addresses ] count ] <= 0 ) return;
    
    address = [[ sender addresses ] objectAtIndex: 0 ];
    
    if ((( struct sockaddr * )[ address bytes ] )->sa_family != AF_INET ) {
        fprintf( stderr, "%d: unknown address family.\n",
                        (( struct sockaddr * )[ address bytes ] )->sa_family );
        return;
    }

    ip = ( struct sockaddr_in * )[ address bytes ];
    if (( service_address = strdup( inet_ntoa( ip->sin_addr ))) == NULL ) {
        perror( "strdup" );
        exit( 2 );
    }

    port = ntohs( ip->sin_port );

    if (( he = gethostbyaddr(( char * )&ip->sin_addr,
                                sizeof( struct in_addr ), AF_INET )) == NULL ) {
        serviceAddress = [ NSString stringWithUTF8String: service_address ];
    } else {
        serviceAddress = [ NSString stringWithUTF8String: he->h_name ];
    }
        
    NSLog( @"Radmind available at %@ on port %d\n", serviceAddress, port );
    
    if ( ! [[ aKtcheckHostField objectValues ] containsObject: serviceAddress ] ) {
        [ aKtcheckHostField addItemWithObjectValue: serviceAddress ];
        if ( [[ aKtcheckHostField stringValue ] length ] == 0 ) {
            [ aKtcheckHostField selectItemWithObjectValue: serviceAddress ];
        }
    }
    if ( ! [[ aLapplyServerField objectValues ] containsObject: serviceAddress ] ) {
        [ aLapplyServerField addItemWithObjectValue: serviceAddress ];
        if ( [[ aLapplyServerField stringValue ] length ] == 0 ) {
            [ aLapplyServerField selectItemWithObjectValue: serviceAddress ];
        }
    }
    if ( ! [[ cLcreateServerField objectValues ] containsObject: serviceAddress ] ) {
        [ cLcreateServerField addItemWithObjectValue: serviceAddress ];
        if ( [[ cLcreateServerField stringValue ] length ] == 0 ) {
            [ cLcreateServerField selectItemWithObjectValue: serviceAddress ];
        }
    }
    if ( ! [[ setupDefaultsServerField objectValues ]
                        containsObject: serviceAddress ] ) {
        [ setupDefaultsServerField addItemWithObjectValue: serviceAddress ];
        if ( [[ setupDefaultsServerField stringValue ] length ] == 0 ) {
            [ setupDefaultsServerField selectItemWithObjectValue: serviceAddress ];
        }
    }

    /* if we find a radmind server, assume they want to set up a client */
    if ( [[ viewBox contentView ] isEqual: firstTimeRunView ] ) {
        [ ftChoiceMatrix selectCellAtRow: 1 column: 0 ];
    }
    
    if ( zcServerAddresses == nil ) {
        zcServerAddresses = [[ NSMutableArray alloc ] init ];
    }
    if ( ! [ zcServerAddresses containsObject: serviceAddress ] ) {
        [ zcServerAddresses addObject: serviceAddress ];
    }

    free( service_address );
}

- ( void )netServiceBrowser: ( NSNetServiceBrowser * )aNetServiceBrowser
            didFindService: ( NSNetService * )netService
            moreComing: ( BOOL )moreComing
{
    if ( services == nil ) {
        services = [[ NSMutableArray alloc ] init ];
    }
    [ services addObject: netService ];
    
    if ( ! moreComing ) {
        int		i;
        
        for ( i = 0; i < [ services count ]; i++ ) {
            [[ services objectAtIndex: i ] setDelegate: self ];
            if ( [[ services objectAtIndex: i ]
			    respondsToSelector: @selector( resolveWithTimeout: ) ] ) {
		    [[ services objectAtIndex: i ] resolveWithTimeout: 5 ];
	    } else {
		    [[ services objectAtIndex: i ] resolve ];
	    }
        }
    }
}

- ( void )netServiceBrowser: ( NSNetServiceBrowser * )aNetServiceBrowser
        didRemoveService: ( NSNetService * )netService
        moreComing: ( BOOL )moreComing
{
    int			i;
    
    if ( services == nil ) return;
    
    for ( i = 0; i < [ services count ]; i++ ) {
        if ( [[ services objectAtIndex: i ] isEqual: netService ] ) {
            [ services removeObjectAtIndex: i ];
            break;
        }
    }
    
    if ( ! moreComing ) {
        int		i;
        NSString	*defaultHost = [[ NSUserDefaults standardUserDefaults ]							objectForKey: @"radmindhost" ];
        
        [ aKtcheckHostField removeAllItems ];
        [ cLcreateServerField removeAllItems ];
        if ( defaultHost != nil ) {
            [ aKtcheckHostField addItemWithObjectValue: defaultHost ];
            [ cLcreateServerField addItemWithObjectValue: defaultHost ];
        }

        for ( i = 0; i < [ services count ]; i++ ) {
            [[ services objectAtIndex: i ] setDelegate: self ];
            [[ services objectAtIndex: i ] resolve ];
        }
    }
}

- ( void )finishUpdate
{
    NSBeginAlertSheet( @"Reboot now?", @"Reboot", @"Don't Reboot", @"",
            stepperWindow, self, @selector( rebootSheetDidEnd:returnCode:contextInfo:),
            NULL, NULL, @"It's a good idea to reboot after updating, so automatically "
            @"generated system files can be updated. If you choose to reboot, the machine "
            @"will restart immediately. You will not have a chance to review other open "
            @"applications." );
}

- ( void )rebootSheetDidEnd: ( NSPanel * )sheet returnCode: ( int )rc
            contextInfo: ( void * )contextInfo
{
    char	*rebootpath = "/sbin/shutdown";
    
    switch ( rc ) {
    case NSAlertDefaultReturn:
        break;
    default:
    case NSAlertAlternateReturn:
        [ stepperWindow close ];
        return;
    }
    
    if ( access( rebootpath, X_OK | F_OK ) < 0 ) {
        NSRunAlertPanel( @"Couldn't access reboot command!",
                @"access %s: %s", @"OK", @"", @"", rebootpath, strerror( errno ));
        return;
    }
    
    [ aexec executeTool: REBOOT
                    withArgs: [ NSArray arrayWithObjects: @"-r", @"now", nil ]
                    controller: self ];
}

- ( void )nextStepUI
{
    if ( step < 0 ) {
        step = 0;
    } else {
        step++;
    }

    [ viewBox setContentView: [ self viewForStep: step ]];
    if ( step == 2 && sessiontype == CREATE ) {
	NSText		*fieldEditor;
	NSString	*loadsetName = @"";
	NSRange		range = { 0, 0 };
	
        [[ cFsdiffLoadsetName window ] makeFirstResponder: cFsdiffLoadsetName ];
	loadsetName = [ cFsdiffLoadsetName stringValue ];
	loadsetName = [ loadsetName stringByDeletingPathExtension ];
	if ( [ loadsetName length ] > 0 ) {
	    fieldEditor = [ cFsdiffLoadsetName currentEditor ];
	    range = [ loadsetName rangeOfString: loadsetName ];
	    if ( range.location != NSNotFound ) {
		[ fieldEditor setSelectedRange: range ];
	    }
	}
    }
    
    [ self enableButtonsForStep: step ];
    [ self setTitleForStep: step ];
}

- ( IBAction )next: ( id )sender
{
    if ( step >= 4 ) {
        step = 4;
        if ( sessiontype == UPDATE
                    && ! [[ aFinalUpdateMsg stringValue ] containsString:
                            @"No updates are necessary." ] ) {
            [ self finishUpdate ];
        } else if ( sessiontype == CREATE && firstTimeWalkThrough ) {
            [ viewBox setContentView: aCkForUpdatesView ];
            step = 1;
            sessiontype = UPDATE;
            [ continueButton setTitle: @"Continue" ];
            [ stepTitleField setStringValue: @"Check for Updates on the Radmind Server" ];
            [ continueButton setAction: @selector( next: ) ];
        } else {
            [ stepperWindow close ];
        }
        return;
    }
    
    if ( step == 2 && sessiontype == CREATE ) {
        if ( ![[ cFsdiffLoadsetName stringValue ] length ] ) {
            NSBeginAlertSheet( @"Your loadset must have a name.",
                @"OK", @"", @"", stepperWindow, nil, NULL, NULL, NULL,
                @"Give your loadset a name and try again." );
            return;
        }
    } 

    [ self actionForStep: step ];
    
    if ( step == 3 && sessiontype == CREATE 
            && [[ NSUserDefaults standardUserDefaults ] boolForKey: @"userauth" ] ) {
        if ( [[ cLoadsetNameField stringValue ] length ] ) {
            [ self getPassword ];
        }
    }
    
    if ( step == 0 ) {
        NSUserDefaults	*defaults = [ NSUserDefaults standardUserDefaults ];
        NSString	*hostname = [ defaults objectForKey: @"radmindhost" ];
        BOOL		cksum = [ defaults boolForKey: @"cksum" ];
        
        switch ( sessiontype ) {
        case CREATE:
            if ( hostname != nil ) {
                [ cLcreateServerField setStringValue: hostname ];
            }
            [ cFsdiffLoadsetName setStringValue: @"New-Loadset.T" ];
            break;
            
        case UPDATE:
            if ( hostname != nil ) {
                [ aKtcheckHostField setStringValue: hostname ];
            }
            [ aFsdiffCksumSwitch selectCellAtRow: ( cksum ? 1 : 0 ) column: 0 ];
            [ aLapplyCksumSwitch selectCellAtRow: ( cksum ? 1 : 0 ) column: 0 ];
            break;
        
        default:
            NSLog( @"Unknown sessiontype." );
            return;
        }
        
        [ self scanForRadmindServers: nil ];
        [ self nextStepUI ];
    }
}

- ( IBAction )skip: ( id )sender
{
    [ self nextStepUI ];
}

- ( IBAction )openServerManager: ( id )sender
{
    NSString		*smpath;
    NSString		*bundleID = @"edu.umich.radmindservermanager";
    
    smpath = [[ NSWorkspace sharedWorkspace ]
		absolutePathForAppBundleWithIdentifier: bundleID ];
    if ( smpath == nil ) {
	/* check in resource dir */
	smpath = [[ NSBundle mainBundle ] pathForResource:
					@"Radmind Server Manager"
					ofType: @"app" ];
	if ( smpath == nil ) {
	    NSRunAlertPanel( @"Couldn't locate Radmind Server Manager!",
		    @"The Radmind Assistant couldn't open the Radmind Server Manager "
		    @"because the Server Manager could not be found. Please reinstall "
		    @"the Radmind Assistant.", @"OK", @"", @"" );
	    exit( 2 );
	}
    }
    
    if ( ! [[ NSWorkspace sharedWorkspace ] openFile: smpath ] ) {
	NSRunAlertPanel( @"Failed to open the Radmind Server Manager.",
		@"Please make sure that the Radmind Assistant was properly installed, "
		@"and try again.", @"OK", @"", @"" );
	return;
    }
}

- ( IBAction )openTranscriptEditor: ( id )sender
{
    RAEditorLauncher	*re = [[ RAEditorLauncher alloc ] init ];
    
    [ re viewTrans: nil withTextEditor: @"Radmind Transcript Editor" ];
    [ re release ];
}

- ( IBAction )viewTranscript: ( id )sender
{
    NSString		*editor;
    RAEditorLauncher	*re = [[ RAEditorLauncher alloc ] init ];
    
    editor = [[ NSUserDefaults standardUserDefaults ]
                objectForKey: @"transeditor" ];
                
    if ( editor == nil ) editor = @"Radmind Transcript Editor";
    
    if ( ![ re viewTrans: ( sessiontype == CREATE ?
            [ self creatableTranscriptName ] : [ self applicableTranscriptName ] )
            withTextEditor: editor ] ) {
        NSRunAlertPanel( @"Error launching text editor.",
            @"Check the console for error messages", @"OK", @"", @"" );
    }
    
    [ re release ];
}

- ( IBAction )emailAuthors: ( id )sender
{
    [[ NSWorkspace sharedWorkspace ] openURL: [ NSURL URLWithString:
                                        @"mailto:radmind@umich.edu" ]];
}

- ( IBAction )visitWebsite: ( id )sender
{
    [[ NSWorkspace sharedWorkspace ] openURL: [ NSURL URLWithString:
                                        @"http://www.radmind.org" ]];
}

- ( int )numberOfRowsInTableView: ( NSTableView * )tableview
{
    return( [ _validNegativeTranscripts count ] );
}

- ( id )tableView: ( NSTableView * )tv objectValueForTableColumn: ( NSTableColumn * )tc
        row: ( int )row
{
    if ( row < 0 ) {
        return( nil );
    }
    
    return( [ _validNegativeTranscripts objectAtIndex: row ] );
}

- ( void )cleanUpTemporaryDirectory
{
    NSString                *tmpdir = [ self sessionTempDirectory ];
    if ( tmpdir ) {
        if ( ! [[ NSFileManager defaultManager ]
                                        removeFileAtPath: tmpdir
                                        handler: nil ] ) {
            /* XXXX callback for handling failed dir removal */
            NSLog( @"failed to delete %@", tmpdir );
        }
    }
}

- ( void )dealloc
{
    [ self setApplicableTranscriptName: nil ];
    [ self setCreatableTranscriptName: nil ];
    if ( _raSessionTmpDir ) {
        [ _raSessionTmpDir release ];
    }
    [ svcBrowser release ];
    [ services release ];
    [ super dealloc ];
}

@end
