/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RAVersionCompare.h"

#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>

extern int	errno;

@implementation RAVersionCompare

+ ( RAVersionCompare * )sharedComparison
{
    RAVersionCompare	*cmp = [[ RAVersionCompare alloc ] init ];
    
    return( [ cmp autorelease ] );
}

/*
 * Since users can install "internal" versions,
 * check for that first. If they've got an internal
 * version, assume they're up-to-date. They can
 * choose to check for updates elsewhere.
 *
 * Turn the x.x.x version number into a three digit
 * integer, compare it with the version bundled
 * with the Assistant.
 */
- ( int )radmindVersion
{
    NSString	*fsdiffPath = nil;
    NSString	*radmindToolDirectory = [[ NSUserDefaults standardUserDefaults ]
					    stringForKey: @"RadmindToolDirectory" ];
    FILE	*f;
    const char	*path = NULL;
    char	cmd[ MAXPATHLEN ], buf[ MAXPATHLEN ] = { 0 };
    char	vbuf[ MAXPATHLEN ] = { 0 };
    int		len, i, j, version;
					    
    if ( radmindToolDirectory == nil ) {
	radmindToolDirectory = @"/usr/local/bin";
    }
    
    fsdiffPath = [ radmindToolDirectory stringByAppendingPathComponent: @"fsdiff" ];
    path = [ fsdiffPath fileSystemRepresentation ];
    
    /* an install is required if we don't find fsdiff */
    if ( ![[ NSFileManager defaultManager ] fileExistsAtPath: fsdiffPath ] ) {
	return( 1 );
    }
    
    if ( snprintf( cmd, MAXPATHLEN, "%s -V", path ) >= MAXPATHLEN ) {
	NSLog( @"%s -V: too long", path );
	return( -1 );
    }
    
    if (( f = popen( cmd, "r" )) == NULL ) {
	NSLog( @"popen %s: %s", cmd, strerror( errno ));
	return( -1 );
    }
    
    fgets( buf, MAXPATHLEN, f );
    if ( pclose( f ) != 0 ) {
	NSLog( @"pclose: %s", strerror( errno ));
	return( -1 );
    }
    if ( ferror( f )) {
	NSLog( @"fgets: %s", strerror( errno ));
	return( -1 );
    }
    
    if (( len = strlen( buf )) == 0 ) {
	NSLog( @"No output" );
	return( -1 );
    }
    buf[ len - 1 ] = '\0';
    
    if ( strncmp( buf, "internal",
		MIN( strlen( buf ), strlen( "internal" ))) == 0 ) {
	/*
	 * internal is always considered newest.
	 * could check datestamps of installed
	 * tools and compare with package version.
	 */
	return( INT_MAX );
    }
    
    for ( i = 0, j = 0; i < MIN( strlen( buf ), 5 ); i++ ) {
	if ( isdigit( buf[ i ] )) {
	    vbuf[ j ] = buf[ i ];
	    j++;
	}
    }
    vbuf[ j ] = '\0';
    
    if (( version = strtol( vbuf, NULL, 10 )) == 0 ) {
	if ( errno ) {
	    NSLog( @"strtol %s: %s", vbuf, strerror( errno ));
	    ( void )pclose( f );
	    return( -1 );
	}
    }
    
    return( version );
}

- ( int )compareInstalledVersion
{
    NSDictionary    *dict;
    NSString	    *bundledVersionString;
    int		    version = [ self radmindVersion ];
    int		    bvers;
    
    if ( version < 0 ) {
	return( 0 );
    }
    
    dict = [[ NSBundle mainBundle ] infoDictionary ];
    bundledVersionString = [ dict objectForKey: @"RABundledRadmindVersion" ];
    
    if ( bundledVersionString == nil ) {
	NSLog( @"No bundled version string in Info.plist" );
	return( 0 );
    }
    bvers = [ bundledVersionString intValue ];
    if (( bvers == 0 ) || ( bvers == INT_MIN ) || ( bvers == INT_MAX )) {
	NSLog( @"%@ intValue returned %d", bundledVersionString, bvers );
	return( 0 );
    }
    
    if ( version < bvers ) {
	NSLog( @"Update needed. Installed version: %d. Bundled version: %d",
				    version, bvers );
	return( 1 );
    }
    
    return( 0 );
}

@end
