/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import <Security/Authorization.h>
#import "RAServerCreator.h"

#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>
#include "authtools.h"

#define RADMIND_SERVER_CONF	"/var/radmind/config"

static NSString			*mkdirError;

@implementation RAServerCreator

    static int
installserver( AuthorizationRef authRef )
{
    NSString			*sscriptpath, *splistpath;
    char			authpath[ MAXPATHLEN ];
    char			buf[ MAXPATHLEN ];
    char			*argv[ 7 ] = { NULL, NULL, NULL, NULL,
						NULL, NULL, NULL };
    int				wfd[ 2 ], efd[ 2 ], rr, wr;
    int				status;
    pid_t			pid;
    AuthorizationExternalForm	extAuth;
    
    /* get path to setuid tool */
    if ( pathfortool( "rmauthexec", authpath ) != 0 ) {
        mkdirError = [[ NSString stringWithFormat: @"Couldn't find authorized tool." ]
                                                        retain ];
        return( 2 );
    }
    
    if (( sscriptpath = [[ NSBundle mainBundle ] pathForResource: @"RadmindServer"
						ofType: @"" ] ) == nil ) {
	return( -1 );
    }
    if (( splistpath = [[ NSBundle mainBundle ] pathForResource: @"StartupParameters"
						ofType: @"plist" ] ) == nil ) {
	return( -1 );
    }
    
    /* create argv for dir creation */
    argv[ 0 ] = authpath;
    argv[ 1 ] = "-A";
    argv[ 2 ] = "ServerSetup";
    argv[ 3 ] = "--";
    argv[ 4 ] = ( char * )[ sscriptpath UTF8String ];
    argv[ 5 ] = ( char * )[ splistpath UTF8String ];
    argv[ 6 ] = NULL;
    
    if ( AuthorizationMakeExternalForm( authRef, &extAuth ) != 0 ) {
        syslog( LOG_ERR, "Failed to make external auth form" );
        return( -1 );
    }
    
    /* set up pipes for interprocess communication */
    if ( pipe( wfd ) != 0 ) {
        mkdirError = [[ NSString stringWithFormat: @"pipe failed: %s", strerror( errno ) ]
                                                        retain ];
        return( errno );
    }
    if ( pipe( efd ) != 0 ) {
        mkdirError = [[ NSString stringWithFormat: @"pipe failed: %s", strerror( errno ) ]
                                                        retain ];
        return( errno );
    }
    
    if ( chdir( "/" ) != 0 ) {
        mkdirError = [[ NSString stringWithFormat: @"chdir to / failed: %s",
                                                            strerror( errno ) ] retain ];
        return( errno );
    }
        
    switch ( fork()) {
    case 0:
        ( void )close( wfd[ 1 ] );
        if ( dup2( wfd[ 0 ], 0 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            _exit( 2 );
        }

        ( void )close( efd[ 0 ] );
        if ( dup2( efd[ 1 ], 2 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            _exit( 2 );
        }
        ( void )close( efd[ 1 ] );
        ( void )close( wfd[ 0 ] );
        
        execve( argv[ 0 ], argv, NULL );
        syslog( LOG_ERR, "execve: %s: %s", argv[ 0 ], strerror( errno ));
        _exit( 2 );
        
    case -1:
        ( void )close( wfd[ 0 ] );
        ( void )close( wfd[ 1 ] );
        ( void )close( efd[ 0 ] );
        ( void )close( efd[ 1 ] );
        
        mkdirError = [[ NSString stringWithFormat: @"fork failed: %s", strerror( errno ) ]
                                                            retain ];
        
    default:
        break;
    }
    
    signal( SIGPIPE, SIG_IGN );
    
    ( void )close( wfd[ 0 ] );
    ( void )close( efd[ 1 ] );
    
    /* write authorization reference to setuid tool, to verify user */
    if (( wr = write( wfd[ 1 ], &extAuth, sizeof( extAuth ))) != sizeof( extAuth )) {
        mkdirError = [[ NSString stringWithFormat: @"write failed: %s", strerror( errno ) ]
                                                            retain ];
        ( void )close( wfd[ 1 ] );
        ( void )close( efd[ 0 ] );
        return( errno );
    }

    /* read errors, if any */
    for ( ;; ) {
        if (( rr = read( efd[ 0 ], buf, MAXPATHLEN )) <= 0 ) break;
        buf[ rr ] = '\0';
        mkdirError = [[ NSString stringWithUTF8String: buf ] retain ];
        memset( buf, '\0', strlen( buf ));
    }
    if ( rr < 0 ) {
        NSLog( @"read returned < 0 (%d): %s\n", rr, strerror( errno ));
        exit( 2 );
    }
    ( void )close( wfd[ 1 ] );
    ( void )close( efd[ 0 ] );
    
    pid = wait( &status );
    
    /* remove tmp radmind file and startup item here */
    
    return( WEXITSTATUS( status ));
}

    static int
authorize( AuthorizationItem right )
{
    int				err;
    AuthorizationRef		authRef;
    AuthorizationRights		rights = { 1, &right };
    OSStatus			status;
    AuthorizationFlags		flags =
                                    kAuthorizationFlagDefaults |
                                    kAuthorizationFlagPreAuthorize |
                                    kAuthorizationFlagInteractionAllowed |
                                    kAuthorizationFlagExtendRights;
                                
    status = AuthorizationCreate( 
                NULL,
                kAuthorizationEmptyEnvironment,
                kAuthorizationFlagDefaults,
                &authRef );
                
    if ( status != errAuthorizationSuccess ) {
        syslog( LOG_ERR, "AuthorizationCreate failed." );
        return( -1 );
    }
    
    /* make user authenticate as an administrator */
    status = AuthorizationCopyRights(
                authRef,
                &rights,
                kAuthorizationEmptyEnvironment,
                flags,
                NULL );
                
    if ( status != errAuthorizationSuccess ) {
        syslog( LOG_ERR, "AuthorizationCopyRights failed." );
        return( -2 );
    }
    
    err = installserver( authRef );
    return( err );
}

- ( NSString * )setupServer;
{
    NSString			*error = nil;
    AuthorizationItem		right = { "edu.umich.rmauthexec", 0, 
                                           NULL, 0 };
                                           
    switch ( authorize( right )) {
    case -1:
        error = @"You must authenticate as an administrator to perform this action.";
        break;
    case -2:
        error = @"Authorization failed.";
        break;
    case 0:
        break;
    default:
        error = mkdirError;
        break;
    }
    
    return( error );
}

@end
