# SPDX-FileCopyrightText: Florian Bruhin (The Compiler) <mail@qutebrowser.org>
#
# SPDX-License-Identifier: GPL-3.0-or-later

import re
import dataclasses

import pytest
webview = pytest.importorskip('qutebrowser.browser.webengine.webview')

from qutebrowser.qt.webenginecore import QWebEnginePage

from helpers import testutils


@dataclasses.dataclass
class Naming:

    prefix: str = ""
    suffix: str = ""


def camel_to_snake(naming, name):
    if naming.prefix:
        assert name.startswith(naming.prefix)
        name = name[len(naming.prefix):]
    if naming.suffix:
        assert name.endswith(naming.suffix)
        name = name[:-len(naming.suffix)]
    # https://stackoverflow.com/a/1176023
    return re.sub(r'(?<!^)(?=[A-Z])', '_', name).lower()


@pytest.mark.parametrize("naming, name, expected", [
    (Naming(prefix="NavigationType"), "NavigationTypeLinkClicked", "link_clicked"),
    (Naming(prefix="NavigationType"), "NavigationTypeTyped", "typed"),
    (Naming(prefix="NavigationType"), "NavigationTypeBackForward", "back_forward"),
    (Naming(suffix="MessageLevel"), "InfoMessageLevel", "info"),
])
def test_camel_to_snake(naming, name, expected):
    assert camel_to_snake(naming, name) == expected


@pytest.mark.parametrize("enum_type, naming, mapping", [
    (
        QWebEnginePage.JavaScriptConsoleMessageLevel,
        Naming(suffix="MessageLevel"),
        webview.WebEnginePage._JS_LOG_LEVEL_MAPPING,
    ),
    (
        QWebEnginePage.NavigationType,
        Naming(prefix="NavigationType"),
        webview.WebEnginePage._NAVIGATION_TYPE_MAPPING,
    )
])
def test_enum_mappings(enum_type, naming, mapping):
    members = testutils.enum_members(QWebEnginePage, enum_type).items()
    for name, val in members:
        mapped = mapping[val]
        assert camel_to_snake(naming, name) == mapped.name
