/*

Copyright 2002 the QuickASCII team
http://quickascii.sourceforge.net

Portions  Copyright 2002 Apple Computer, Inc.

See the LICENSE.txt file for licensing information

*/

#include <stdio.h>
#include <QuickTime/QuickTime.h>

#include "commands.h"

#define USE_COLOR

#define ESC	27

#define BLACK   30
#define RED     31
#define GREEN   32
#define YELLOW  33
#define BLUE    34
#define MAGENTA 35
#define CYAN    36
#define WHITE   37

// define pframe(y,x) prevframe[y + x * intwidth]
// define pframerow(x) prevframe[x * intwidth]

static short *prevframe, *pframepos;

// ASCII pixel value array
static char convert[256];

static char line[100000];
static char *linepos;

static int lastcolor = -1, lastbold = -1;

// boolean whether or not to use color
static int use_color;

// boolean whether inverted
static int inverted;

// dimensions
static float height,width;
static int intheight,intwidth;

// DrawCompleteProc - After the frame has been drawn QuickTime calls us to do some work
static pascal OSErr DrawCompleteProc(Movie theMovie, long refCon)
{
	int		y, x, cury = 0, curx = 0;
	GWorldPtr	offWorld = (GWorldPtr)refCon;
	Rect		bounds;
	Ptr			baseAddr;
	float		yfact;
	float		xfact;
	long		rowBytes;
	char		ch;
        
#ifdef USE_COLOR
  //                 R  G  B 
  static char colors[4][4][4] =
  //    000   001  002 003     010  011 012   013    020   021  022 023      030   031  032  033
    {{{BLACK,BLUE,BLUE,BLUE},{GREEN,CYAN,CYAN,BLUE},{GREEN,CYAN,CYAN,CYAN},{GREEN,GREEN,CYAN,CYAN}},
  //    100  101    102    103     110    111   112  113    120   121   122  123    130    131  132  133
     {{RED,MAGENTA,MAGENTA,BLUE},{YELLOW,GREEN,CYAN,BLUE},{YELLOW,GREEN,CYAN,CYAN},{GREEN,GREEN,CYAN,CYAN}},
  //   200 201     202     203       210    211     212     213       220     221   222   223    230     231   232   233
     {{RED,MAGENTA,MAGENTA,MAGENTA},{YELLOW,MAGENTA,MAGENTA,MAGENTA},{YELLOW,YELLOW,WHITE,CYAN},{YELLOW,YELLOW,WHITE,CYAN}},
  //   300 301  302    303       310 311 312     313       320     321   322   323       330     331   332   333
     {{RED,RED,MAGENTA,MAGENTA},{RED,RED,MAGENTA,MAGENTA},{YELLOW,YELLOW,WHITE,MAGENTA},{YELLOW,YELLOW,YELLOW,WHITE}}};
#endif //USE_COLOR
	
	// get the information we need from the GWorld
	GetPixBounds(GetGWorldPixMap(offWorld), &bounds);
	baseAddr = GetPixBaseAddr(GetGWorldPixMap(offWorld));
	rowBytes = GetPixRowBytes(GetGWorldPixMap(offWorld));
	yfact = (double)(bounds.bottom - bounds.top) / height;
	xfact = (double)(bounds.right - bounds.left) / width;
	
	// goto home
	printf("%c[0;0H", ESC);
	pframepos = prevframe;
	
	// for each row
	for (y = 0; y < intheight; ++y) {
		long	*p;
		linepos = line;
		
		// for each pixel
		p = (long*)(baseAddr + rowBytes * (long)(y * yfact));
                
		for (x = 0; x < intwidth; ++x) {
			UInt32			color;
			long			Y;
			long			R;
			long			G;
			long			B;

			color = *(long *)((long)p + 4 * (long)(x*xfact));
			R = (color & 0x00FF0000) >> 16;
			G = (color & 0x0000FF00) >> 8;
			B = (color & 0x000000FF) >> 0;

			// convert to YUV for comparison
			// 5 * R + 9 * G + 2 * B
			Y = (R + (R << 2) + G + (G << 3) + (B + B)) >> 4;
			ch = convert[Y];
			
			{
				char	attr;

#ifdef USE_COLOR
				int bold;
				if (use_color) {
					bold = (Y % 5) > 4;

					color = colors[R>>6][G>>6][B>>6];
					attr = (bold << 7) | color;
				} else {
#else //USE_COLOR
					if (inverted != 0) {
						color = BLACK;
					} else {
						color = WHITE;
					}
                                        attr = color;
#endif //USE_COLOR
#ifdef USE_COLOR
				}
#endif //USE_COLOR

        // if the color, boldness and character didn't change from the last frame, don't draw again
				if (*pframepos != ((attr << 8) | ch)) {

#ifdef USE_COLOR
          // change both the color and the boldness
					static char allseq[7] = "\x1B[X;3Xm";
          // change the color
					static char colorseq[5] = "\x1B[3Xm";
          // change the boldness
					static char boldseq[4] = "\x1B[Xm";
#endif //USE_COLOR

          // move 1-9 positions to the right
					static char xmoveseq1[4] = "\x1B[XC";
          // move 10-99 positions to the right
					static char xmoveseq2[5] = "\x1B[XXC";
          // move 100-999 positions to the right
					static char xmoveseq3[6] = "\x1B[XXXC";

          // calculate the difference between the position of the current character and the
          // the position right after the previously drawn character (that's where the cursor still is)
					int xdiff = x - curx;

          // remember the character we're going to draw
					*pframepos = ((attr << 8) | ch);

          // is the cursor in the right position?
					if (xdiff != 0) {
              // no, check which sequence is optimal to move the cursor
						if (xdiff < 10) {
							xmoveseq1[2] = xdiff + '0';
							memcpy(linepos,xmoveseq1,sizeof(xmoveseq1));
							linepos+=sizeof(xmoveseq1);
						} else if (xdiff < 100) {
							xmoveseq2[2] = (xdiff / 10) + '0';
							xmoveseq2[3] = (xdiff % 10) + '0';
							memcpy(linepos,xmoveseq2,sizeof(xmoveseq2));
							linepos+=sizeof(xmoveseq2);
						} else {
															int xdiffdiv100 = xdiff / 100;
							xmoveseq3[2] = (xdiffdiv100) + '0';
							xmoveseq3[3] = (xdiff / 10 - xdiffdiv100 * 10) + '0';
							xmoveseq3[4] = (xdiff % 10) + '0';
							memcpy(linepos,xmoveseq3,sizeof(xmoveseq3));
							linepos+=sizeof(xmoveseq3);
						}
					}

#ifdef USE_COLOR
					if (use_color) {
          // now that we're in the right position, check whether currently set color and
          // boldness are correct and if not, adjust them
						if (color != lastcolor) {
							if (bold != lastbold) {
								allseq[2] = bold + '0';
								allseq[5] = (color - 30) + '0';
								memcpy(linepos,allseq,sizeof(allseq));
								linepos+=sizeof(allseq);
								lastcolor = color;
								lastbold = bold;
							} else {
								colorseq[3] = (color - 30) + '0';
								memcpy(linepos,colorseq,sizeof(colorseq));
								linepos+=sizeof(colorseq);
								lastcolor = color;
							}
						} else if (bold != lastbold) {
							boldseq[2] = bold + '0';
							memcpy(linepos,boldseq,sizeof(boldseq));
							linepos+=sizeof(boldseq);
							lastbold = bold;
						}
					}
#endif //USE_COLOR

					// draw the character itself
					*linepos++ = ch;
					curx = x+1;
					cury = y;
				}
			}
			pframepos++;
		}
    // we're at the end of the line -> terminate the string and draw it
		*linepos = '\0';
                
    // puts() adds a '\n', we want to avoid this behavior on the last line                
		if (y != intheight -1) {
			puts(line);

		} else {
			int i=1;
			char c = line[0];
			while (c != '\0') {
				putchar(c);
				c = line[i++];
			}
		}

		curx = 0;
	}
	
        
	return noErr;
}



int main(int argc, char *argv[])
{
	
		MovieController	thePlayer = nil;
		Movie		theMovie = nil;
		GWorldPtr	offWorld;
		Rect		bounds;
		short		resRefNum = -1;
		short		actualResId = DoTheRightThing;
		FSSpec		theFSSpec;
		OSErr		result = 0;
		MovieDrawingCompleteUPP	myDrawCompleteProc = NewMovieDrawingCompleteUPP(DrawCompleteProc);
		int i;
        char*		table;

        
        if (run_init_flags(argc,argv)) {
            exit(0);
        }
        
        set_dimensions(argc,argv,&height,&width,&intheight,&intwidth);
        prevframe = (short *) malloc ((intwidth * intheight) * sizeof(short));
        use_color = resolve_color(argc,argv);
        inverted = resolve_inversion(argc,argv);
        
	/* build the luminance value to ASCII value conversion table
		   Y			  ASCII
		0 - 30 			  space
		31 - 40 	 		.
		41 - 51 	 		,
		52 - 61 	 		:
		62 - 71 	 		!
		72 - 81 	 		-
		82 - 92 	 		+
		93 - 102  			=
		103 - 112 			;
		113 - 122 			i
		123 - 133 			o
		134 - 143 			t
		144 - 153 			7
		154 - 163 			6
		164 - 174			x
		175 - 184			0
		185 - 194			s
		195 - 204			&
		205 - 215			8
		216 - 225			%
		226 - 235			#
		236 - 245			@
		246 - 255			$
	*/
        
        if (inverted) {
            table = "   .,:!-+=;iot76x0s&8%#@$";
        } else {
            table = "$@#%8&s0x67toi;=+-!:,.   ";
        }
        int table_length = strlen(table);

	for (i = 0; i < 256; ++i) {
		convert[i] = table[i * table_length / 256];
	}
		
	EnterMovies();
	// home
	printf("%c[0;0H", ESC);
	// erase to end of display
	printf("%c[0J", ESC);

        char *fname = resolve_file(argc,argv);
        if (fname == 0) {
            printf("Error: No file name specified\n\n");
            exit(0);
        }
	result = NativePathNameToFSSpec(fname, &theFSSpec, 0 /* flags */);
	if (result) {printf("NativePathNameToFSSpec failed %d\n", result); goto bail; }
	result = OpenMovieFile(&theFSSpec, &resRefNum, 0);
	if (result) {printf("OpenMovieFile failed %d\n", result); goto bail; }
		
	result = NewMovieFromFile(&theMovie, resRefNum, &actualResId, (unsigned char *) 0, 0, (Boolean *) 0);
	if (result) {printf("NewMovieFromFile failed %d\n", result); goto bail; }

	if (resRefNum != -1)
		CloseMovieFile(resRefNum);
                
        memset(prevframe,-1,(intwidth * intheight) * sizeof(short));

	GetMovieBox(theMovie, &bounds);
	QTNewGWorld(&offWorld, k32ARGBPixelFormat, &bounds, NULL, NULL, 0);
	LockPixels(GetGWorldPixMap(offWorld));
	SetGWorld(offWorld, NULL);
	
	thePlayer = NewMovieController(theMovie, &bounds, mcTopLeftMovie | mcNotVisible);
	SetMovieGWorld(theMovie, offWorld, NULL);
	SetMovieActive(theMovie, true);
	SetMovieDrawingCompleteProc(theMovie, movieDrawingCallWhenChanged, myDrawCompleteProc, (long)offWorld); 
	MCDoAction(thePlayer, mcActionPrerollAndPlay, (void*)Long2Fix(1));
	
	do {
		MCIdle(thePlayer);
	} while (1);
	
	bail:
		DisposeMovieController( thePlayer );
		DisposeMovie(theMovie);
		DisposeMovieDrawingCompleteUPP(myDrawCompleteProc);
		
		return result;
}
