#!/usr/bin/env python
# Copyright 2016 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Certificates for testing issuer lookup.

               Root
             /|    |
            / |    |
           /  |    |
          /   |    |
         /    |    |
        v     v    v
      I1_1  i1_2   I2
        |    |     |
        |    |     |
        |    |     |
        |    |     |
        v    v     v
       C1    C2    D

I1 (i1_1.pem) and i1 (i1_2.pem) have subjects that are equal after
normalization.

C1 and C2 should (attempt to) chain up through both I1 and i1, since I1 and i1
have the same the name (after normalization).
"""

import os
import sys
sys.path += [os.path.join('..', 'verify_certificate_chain_unittest')]

import common


def write_cert_to_file(cert, filename):
  common.write_string_to_file(
      "Generated by %s.\n"
      "Refer to generator script docstring for details.\n%s" % (
          sys.argv[0], cert.get_cert_pem()),
      filename)


# Self-signed root certificate
root = common.create_self_signed_root_certificate('Root')
write_cert_to_file(root, 'root.pem')


# Intermediate certificates
i1_1 = common.create_intermediate_certificate('I1', root)
write_cert_to_file(i1_1, 'i1_1.pem')

# same name (after normalization), different key
i1_2 = common.create_intermediate_certificate('i1', root)
write_cert_to_file(i1_2, 'i1_2.pem')

# different name
i2 = common.create_intermediate_certificate('I2', root)
write_cert_to_file(i2, 'i2.pem')


# target certs

c1 = common.create_end_entity_certificate('C1', i1_1)
write_cert_to_file(c1, 'c1.pem')

c2 = common.create_end_entity_certificate('C2', i1_2)
write_cert_to_file(c2, 'c2.pem')

d = common.create_end_entity_certificate('D', i2)
write_cert_to_file(d, 'd.pem')


