// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "services/device/power_monitor/power_monitor_message_broadcaster.h"

#include "base/macros.h"
#include "base/run_loop.h"
#include "base/test/power_monitor_test_base.h"
#include "services/device/device_service_test_base.h"
#include "services/device/public/cpp/power_monitor/power_monitor_broadcast_source.h"

namespace device {

class MockPowerMonitorBroadcastSource : public PowerMonitorBroadcastSource {
 public:
  MockPowerMonitorBroadcastSource(service_manager::Connector* connector,
                                  base::Closure service_connected)
      : PowerMonitorBroadcastSource(connector),
        service_connected_(service_connected) {}
  ~MockPowerMonitorBroadcastSource() override = default;

  // Implement device::mojom::PowerMonitorClient
  void PowerStateChange(bool on_battery_power) override {
    power_state_changes_++;
    service_connected_.Run();
  }
  void Suspend() override { suspends_++; }
  void Resume() override { resumes_++; }

  // Test status counts.
  int power_state_changes() { return power_state_changes_; }
  int suspends() { return suspends_; }
  int resumes() { return resumes_; }

 private:
  int power_state_changes_ = 0;  // Count of OnPowerStateChange notifications.
  int suspends_ = 0;             // Count of OnSuspend notifications.
  int resumes_ = 0;              // Count of OnResume notifications.
  base::Closure service_connected_;
};

class PowerMonitorMessageBroadcasterTest : public DeviceServiceTestBase {
 protected:
  PowerMonitorMessageBroadcasterTest() = default;
  ~PowerMonitorMessageBroadcasterTest() override = default;

  void SetUp() override {
    DeviceServiceTestBase::SetUp();

    power_monitor_source_ = new base::PowerMonitorTestSource();
    power_monitor_.reset(new base::PowerMonitor(
        std::unique_ptr<base::PowerMonitorSource>(power_monitor_source_)));
  }

  base::PowerMonitorTestSource* source() { return power_monitor_source_; }

 private:
  base::PowerMonitorTestSource* power_monitor_source_;
  std::unique_ptr<base::PowerMonitor> power_monitor_;

  DISALLOW_COPY_AND_ASSIGN(PowerMonitorMessageBroadcasterTest);
};

TEST_F(PowerMonitorMessageBroadcasterTest, PowerMessageBroadcast) {
  base::RunLoop run_loop;
  MockPowerMonitorBroadcastSource client(connector(), run_loop.QuitClosure());
  run_loop.Run();

  // Above PowerMonitorBroadcastSource ctor will connect to Device Service to
  // bind device::mojom::PowerMonitor interface, on which AddClient() will be
  // called then, this should invoke immediatelly a power state change back to
  // PowerMonitorBroadcastSource.
  base::RunLoop().RunUntilIdle();
  EXPECT_EQ(client.power_state_changes(), 1);

  // Sending resume when not suspended should have no effect.
  source()->GenerateResumeEvent();
  EXPECT_EQ(client.resumes(), 0);

  // Pretend we suspended.
  source()->GenerateSuspendEvent();
  EXPECT_EQ(client.suspends(), 1);

  // Send a second suspend notification.  This should be suppressed.
  source()->GenerateSuspendEvent();
  EXPECT_EQ(client.suspends(), 1);

  // Pretend we were awakened.
  source()->GenerateResumeEvent();
  EXPECT_EQ(client.resumes(), 1);

  // Send a duplicate resume notification.  This should be suppressed.
  source()->GenerateResumeEvent();
  EXPECT_EQ(client.resumes(), 1);

  // Pretend the device has gone on battery power
  source()->GeneratePowerStateEvent(true);
  EXPECT_EQ(client.power_state_changes(), 2);

  // Repeated indications the device is on battery power should be suppressed.
  source()->GeneratePowerStateEvent(true);
  EXPECT_EQ(client.power_state_changes(), 2);

  // Pretend the device has gone off battery power
  source()->GeneratePowerStateEvent(false);
  EXPECT_EQ(client.power_state_changes(), 3);

  // Repeated indications the device is off battery power should be suppressed.
  source()->GeneratePowerStateEvent(false);
  EXPECT_EQ(client.power_state_changes(), 3);
}

}  // namespace device
