// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/browser/media/url_provision_fetcher.h"

#include "base/memory/ptr_util.h"
#include "content/public/browser/provision_fetcher_factory.h"
#include "media/base/bind_to_current_loop.h"
#include "net/base/load_flags.h"
#include "net/traffic_annotation/network_traffic_annotation.h"
#include "net/url_request/url_fetcher.h"

using net::URLFetcher;

namespace content {

// Implementation of URLProvisionFetcher.

URLProvisionFetcher::URLProvisionFetcher(
    net::URLRequestContextGetter* context_getter)
    : context_getter_(context_getter) {}

URLProvisionFetcher::~URLProvisionFetcher() {}

void URLProvisionFetcher::Retrieve(
    const std::string& default_url,
    const std::string& request_data,
    const media::ProvisionFetcher::ResponseCB& response_cb) {
  response_cb_ = response_cb;

  const std::string request_string =
      default_url + "&signedRequest=" + request_data;
  DVLOG(1) << __func__ << ": request:" << request_string;

  DCHECK(!request_);
  net::NetworkTrafficAnnotationTag traffic_annotation =
      net::DefineNetworkTrafficAnnotation("url_prevision_fetcher", R"(
        semantics {
          sender: "Content Decryption Module"
          description:
            "For a Content Decryption Module (CDM) to obtain origin-specific "
            "identifiers from an individualization or provisioning server. See "
            "https://w3c.github.io/encrypted-media/#direct-individualization."
          trigger:
            "During protected content playback, if the CDM hasn’t been "
            "provisioned yet, it may trigger a provision request which will be "
            "sent to a provisioning server."
          data:
            "Opaque provision request generated by the CDM. It may contain "
            "distinctive identifiers (see "
            "https://w3c.github.io/encrypted-media/#distinctive-identifier) "
            "and/or distinctive permanent identifiers (see "
            "https://w3c.github.io/encrypted-media/#distinctive-permanent-"
            "identifier), which must be encrypted. It does NOT contain origin "
            "information, even in encrypted form."
          destination: OTHER
        }
        policy {
          cookies_allowed: false
          setting:
            "On Android, users can disable this feature by disabling Protected "
            "Media Identifier permissions."
          policy_exception_justification: "Not implemented."
        })");
  request_ = URLFetcher::Create(GURL(request_string), URLFetcher::POST, this,
                                traffic_annotation);

  // SetUploadData is mandatory even if we are not uploading anything.
  request_->SetUploadData("", "");
  request_->AddExtraRequestHeader("User-Agent: Widevine CDM v1.0");
  request_->AddExtraRequestHeader("Content-Type: application/json");
  request_->SetLoadFlags(net::LOAD_DO_NOT_SAVE_COOKIES |
                         net::LOAD_DO_NOT_SEND_COOKIES);

  DCHECK(context_getter_);
  request_->SetRequestContext(context_getter_);

  request_->Start();
}

void URLProvisionFetcher::OnURLFetchComplete(const net::URLFetcher* source) {
  DCHECK(source);

  int response_code = source->GetResponseCode();
  DVLOG(1) << __func__ << ": response code:" << source->GetResponseCode();

  std::string response;
  bool success = false;
  if (response_code == 200) {
    success = source->GetResponseAsString(&response);
    DVLOG_IF(1, !success) << __func__ << ": GetResponseAsString() failed";
  } else {
    DVLOG(1) << "CDM provision: server returned error code " << response_code;
  }

  request_.reset();
  // URLFetcher implementation calls OnURLFetchComplete() on the same thread
  // that called Start() and it does this asynchronously.
  response_cb_.Run(success, response);
}

// Implementation of content public method CreateProvisionFetcher().

std::unique_ptr<media::ProvisionFetcher> CreateProvisionFetcher(
    net::URLRequestContextGetter* context_getter) {
  DCHECK(context_getter);
  return base::MakeUnique<URLProvisionFetcher>(context_getter);
}

}  // namespace content
