// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef MEDIA_BLINK_WEBCONTENTDECRYPTIONMODULESESSION_IMPL_H_
#define MEDIA_BLINK_WEBCONTENTDECRYPTIONMODULESESSION_IMPL_H_

#include <stddef.h>
#include <stdint.h>

#include <string>
#include <vector>

#include "base/callback.h"
#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "base/memory/weak_ptr.h"
#include "base/threading/thread_checker.h"
#include "media/base/content_decryption_module.h"
#include "media/blink/new_session_cdm_result_promise.h"
#include "third_party/WebKit/public/platform/WebContentDecryptionModuleSession.h"
#include "third_party/WebKit/public/platform/WebString.h"

namespace media {

class CdmSessionAdapter;

class WebContentDecryptionModuleSessionImpl
    : public blink::WebContentDecryptionModuleSession {
 public:
  WebContentDecryptionModuleSessionImpl(
      const scoped_refptr<CdmSessionAdapter>& adapter);
  ~WebContentDecryptionModuleSessionImpl() override;

  // blink::WebContentDecryptionModuleSession implementation.
  void SetClientInterface(Client* client) override;
  blink::WebString SessionId() const override;

  void InitializeNewSession(
      blink::WebEncryptedMediaInitDataType init_data_type,
      const unsigned char* initData,
      size_t initDataLength,
      blink::WebEncryptedMediaSessionType session_type,
      blink::WebContentDecryptionModuleResult result) override;
  void Load(const blink::WebString& session_id,
            blink::WebContentDecryptionModuleResult result) override;
  void Update(const uint8_t* response,
              size_t response_length,
              blink::WebContentDecryptionModuleResult result) override;
  void Close(blink::WebContentDecryptionModuleResult result) override;
  void Remove(blink::WebContentDecryptionModuleResult result) override;

  // Callbacks.
  void OnSessionMessage(CdmMessageType message_type,
                        const std::vector<uint8_t>& message);
  void OnSessionKeysChange(bool has_additional_usable_key,
                           CdmKeysInfo keys_info);
  void OnSessionExpirationUpdate(base::Time new_expiry_time);
  void OnSessionClosed();

 private:
  // Called when a new session is created or loaded. |status| is set as
  // appropriate, depending on whether the session already exists or not.
  void OnSessionInitialized(const std::string& session_id,
                            SessionInitStatus* status);

  scoped_refptr<CdmSessionAdapter> adapter_;

  // Non-owned pointer.
  Client* client_;

  // Session ID is the app visible ID for this session generated by the CDM.
  // This value is not set until the CDM resolves the initializeNewSession()
  // promise.
  std::string session_id_;

  // Keep track of whether the session has been closed or not. The session
  // may be closed as a result of an application calling close(), or the CDM
  // may close the session at any point.
  // https://w3c.github.io/encrypted-media/#session-closed
  // |has_close_been_called_| is used to keep track of whether close() has
  // been called or not. |is_closed_| is used to keep track of whether the
  // close event has been received or not.
  bool has_close_been_called_;
  bool is_closed_;

  // Keep track of whether this is a persistent session or not.
  bool is_persistent_session_;

  base::ThreadChecker thread_checker_;
  // Since promises will live until they are fired, use a weak reference when
  // creating a promise in case this class disappears before the promise
  // actually fires.
  base::WeakPtrFactory<WebContentDecryptionModuleSessionImpl> weak_ptr_factory_;

  DISALLOW_COPY_AND_ASSIGN(WebContentDecryptionModuleSessionImpl);
};

}  // namespace media

#endif  // MEDIA_BLINK_WEBCONTENTDECRYPTIONMODULESESSION_IMPL_H_
