Qore supports three types of container types (see also \mbox{\hyperlink{basic_data_types}{Basic Data Types}} and \mbox{\hyperlink{code_data_types}{Code Data Types}})\+:
\begin{DoxyItemize}
\item \mbox{\hyperlink{container_data_types_list}{lists}}
\item \mbox{\hyperlink{container_data_types_hash}{hashes}} (associative arrays)
\item \mbox{\hyperlink{container_data_types_object}{objects}} (see \mbox{\hyperlink{qore_classes}{Classes}} for more information)
\end{DoxyItemize}

These container types can be combined to make arbitrarily complex data structures. The data type of any element can be any basic type or another aggregate type. The types do not have to be uniform in one container structure.

\DoxyHorRuler{0}
 \hypertarget{container_data_types_list}{}\doxysection{List}\label{container_data_types_list}
\begin{DoxyParagraph}{Description\+:}
Lists (or arrays) are simply ordered containers of values. A list element can be any Qore type (even another list, \mbox{\hyperlink{container_data_types_hash}{hash}}, or \mbox{\hyperlink{container_data_types_object}{object}}).~\newline
~\newline
 Lists are specified by giving expressions separated by commas as follows\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{list = (expression, expression [, expression ...]);}

\end{DoxyCode}
 ~\newline
 Here is a concrete example\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{list list = (}
\DoxyCodeLine{    1,}
\DoxyCodeLine{    2,}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}three"{}},}
\DoxyCodeLine{    4.0,}
\DoxyCodeLine{    5e20n,}
\DoxyCodeLine{    6,}
\DoxyCodeLine{    2001-\/01-\/15Z,}
\DoxyCodeLine{);}

\end{DoxyCode}
 ~\newline
 Trailing commas can be left on the end of a list (or a \mbox{\hyperlink{container_data_types_hash}{hash}}), making it easier to insert and remove elements at the end of a multi-\/line list.~\newline
~\newline
 List elements are dereferenced using square brackets\+: {\ttfamily \char`\"{}\mbox{[}\char`\"{}} and {\ttfamily \char`\"{}\mbox{]}\char`\"{}}. The first element in a list has index zero.~\newline
~\newline
 {\bfseries{Example\+:}} 
\begin{DoxyCode}{0}
\DoxyCodeLine{element3 = list[2];}

\end{DoxyCode}
 ~\newline
 The following operators perform special processing on lists\+:
\begin{DoxyItemize}
\item \mbox{\hyperlink{operators_elements}{elements}}
\item \mbox{\hyperlink{operators_shift}{shift}}
\item \mbox{\hyperlink{operators_unshift}{unshift}}
\item \mbox{\hyperlink{operators_push}{push}}
\item \mbox{\hyperlink{operators_pop}{pop}}
\item \mbox{\hyperlink{operators_splice}{splice}}
\item \mbox{\hyperlink{operators_list_element_operator}{\mbox{[}\mbox{]}}}
\item \mbox{\hyperlink{operators_plus_operator}{+}}
\item \mbox{\hyperlink{operators_plus_equals_operator}{+=}}
\item \mbox{\hyperlink{operators_map}{map}}
\item \mbox{\hyperlink{operators_foldl}{foldl}}
\item \mbox{\hyperlink{operators_foldr}{foldr}}
\item \mbox{\hyperlink{operators_select}{select}} ~\newline
 
\end{DoxyItemize}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Immediate Value Example\+:}
\begin{DoxyVerb}(1, "two", 3.0)\end{DoxyVerb}
 Gives an empty list (note that {\ttfamily \{\}} gives an empty hash)\+: \begin{DoxyVerb}()\end{DoxyVerb}

\end{DoxyParagraph}
\begin{DoxyParagraph}{Pseudo Class for Type List\+:}
\mbox{\hyperlink{class_qore_1_1zzz8listzzz9}{Qore\+::zzz8listzzz9}}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Type Code\+:}
\mbox{\hyperlink{group__type__code__constants_gada83cf42174092e6f4d28f21f1ce3c2c}{Qore\+::\+NT\+\_\+\+LIST}}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Type Name\+:}
{\ttfamily \char`\"{}list\char`\"{}} 
\end{DoxyParagraph}
The list type supports a complex element type specification as well, however \char`\"{}list\char`\"{} lvalues without a complex element type specification will strip the complex type when assigned as in the following example\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{list l0 = (2, 3);}
\DoxyCodeLine{\textcolor{comment}{\# prints "{}list"{}}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, l0.fullType());}
\DoxyCodeLine{list<auto> l1 = (2, 3);}
\DoxyCodeLine{\textcolor{comment}{\# prints "{}list<int>"{}}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, l1.fullType());}

\end{DoxyCode}


\label{container_data_types_list_slicing}%
\Hypertarget{container_data_types_list_slicing}%
 \begin{DoxyParagraph}{List slicing\+:}
Lists can be \char`\"{}sliced\char`\"{} by dereferencing them using the \mbox{\hyperlink{operators_list_element_operator}{\mbox{[}\mbox{]} operator}} with a range or a list, as in the following examples\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{list<int> l = (1, 2, 3, 4);}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Creates a new list with the first two elements of the original list}}
\DoxyCodeLine{list l1a = l[0..1];}
\DoxyCodeLine{list l1b = l[0,1];}
\DoxyCodeLine{list l1c = l[..1];}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 1a: \%y\(\backslash\)n"{}}, l1a);}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 1b: \%y\(\backslash\)n"{}}, l1b);}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 1c: \%y\(\backslash\)n"{}}, l1c);}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Creates a new list with the last two elements of the original list in reverse order}}
\DoxyCodeLine{list l2a = l[3..2];}
\DoxyCodeLine{list l2b = l[3,2];}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 2a: \%y\(\backslash\)n"{}}, l2a);}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 2b: \%y\(\backslash\)n"{}}, l2b);}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Takes a slice of different parts of the list}}
\DoxyCodeLine{list l3 = l[3,2..1,0];}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 3: \%y\(\backslash\)n"{}}, l3);}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# -\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/}}
\DoxyCodeLine{\textcolor{comment}{\# the output is as follows:}}
\DoxyCodeLine{}
\DoxyCodeLine{slice 1a: [1, 2]}
\DoxyCodeLine{slice 1b: [1, 2]}
\DoxyCodeLine{slice 1c: [1, 2]}
\DoxyCodeLine{slice 2a: [4, 3]}
\DoxyCodeLine{slice 2b: [4, 3]}
\DoxyCodeLine{slice 3: [4, 3, 2, 1];}

\end{DoxyCode}
 ~\newline
 When making a slice of a list with a list of elements and referring to an element index that does not exist, \mbox{\hyperlink{basic_data_types_nothing}{NOTHING}} is returned for that list element in the resulting list as in the following example\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{list<int> l = (1, 2, 3, 4);}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, l[2,4,6]);}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# -\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/}}
\DoxyCodeLine{\textcolor{comment}{\# the output is as follows:}}
\DoxyCodeLine{}
\DoxyCodeLine{[3, null, null]}

\end{DoxyCode}

\end{DoxyParagraph}
\begin{DoxySeeAlso}{See also}

\begin{DoxyItemize}
\item \mbox{\hyperlink{data_type_declarations_list_type}{list}}, \mbox{\hyperlink{data_type_declarations_softlist_type}{softlist}}
\item \mbox{\hyperlink{group__list__functions}{List Functions}}
\end{DoxyItemize}
\end{DoxySeeAlso}
\begin{DoxyParagraph}{Complex Type Support for Lists}
The {\ttfamily \char`\"{}list\char`\"{}} type also supports an optional type argument which allows the value type of list elements to be declared; the following example demonstrates a list declaration with a specific value type\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{list<int> l(2, 3);}

\end{DoxyCode}
 ~\newline
 This type is supported at parse-\/time and at runtime; to convert such values to an untyped list, assign it to a \mbox{\hyperlink{data_type_declarations_list_type}{list}} lvalue, use \mbox{\hyperlink{operators_cast}{cast$<$list$>$(...)}} on the value, or call the \mbox{\hyperlink{group__type__conversion__functions_ga4b52a1cbee7a3aed46c32cb0a938629c}{list()}} function on the value. Each of these options can be used to convert a type-\/safe list to an untyped list.
\end{DoxyParagraph}
A special type argument, {\ttfamily \char`\"{}auto\char`\"{}}, allows for the lvalue to maintain the complex list type as in the following example\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{list l0 = (2, 3);}
\DoxyCodeLine{\textcolor{comment}{\# prints "{}list"{}}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, l0.fullType());}
\DoxyCodeLine{list<auto> l1 = (2, 3);}
\DoxyCodeLine{\textcolor{comment}{\# prints "{}list<int>"{}}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, l1.fullType());}

\end{DoxyCode}


\begin{DoxySeeAlso}{See also}
\mbox{\hyperlink{data_type_declarations_list_complex_type}{List With Declared Value Type}}
\end{DoxySeeAlso}
\begin{DoxyNote}{Note}

\begin{DoxyItemize}
\item Trailing commas can be left on the end of a list (or a \mbox{\hyperlink{container_data_types_hash}{hash}}), making it easier to insert and remove elements at the end of a multi-\/line list.
\item List elements are dereferenced using square brackets\+: {\ttfamily \char`\"{}\mbox{[}\char`\"{}} and {\ttfamily \char`\"{}\mbox{]}\char`\"{}}. The first element in a list has index zero.
\item Dereferencing an invalid element (past the end of the list or with a negative number) will return \mbox{\hyperlink{basic_data_types_nothing}{NOTHING}}
\item Use \mbox{\hyperlink{class_qore_1_1zzz8listzzz9_ac8e243d7957d6136a6c77d32f320bf24}{Qore\+::zzz8listzzz9\+::iterator()}} as an easy way to get a list iterator object for the current list
\end{DoxyItemize}
\end{DoxyNote}
\DoxyHorRuler{0}
 \hypertarget{container_data_types_hash}{}\doxysection{Hash}\label{container_data_types_hash}
\begin{DoxyParagraph}{Description\+:}
Hashes are containers that associate values to a string key and also preserve key order for consistent data serialization/deserialization. If anything else than a string is used as a key, the value is one-\/way converted to a string and the string value is used as a key. This can sometimes have unexpected results and therefore only string keys are recommended. Hash key lookups are case-\/sensitive and use a hashing algorithm that in the worst case should provide {\itshape O(ln(n))} complexity to execute; in the best case finding a particular key in a hash is executed in constant time (ie {\itshape O(1)} complexity).~\newline
~\newline
 Hashes are specified using the following syntax\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{hash h = \{}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    ...,}
\DoxyCodeLine{\};}

\end{DoxyCode}
 ~\newline
 with a \mbox{\hyperlink{hashdecl}{typed hash}}, immediate values can look like\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{\textcolor{comment}{\# the following declarations are equivalent}}
\DoxyCodeLine{\textcolor{comment}{\# note that typed hashes are "{}constructed"{} with default key values; immediate values override the defaults}}
\DoxyCodeLine{\textcolor{comment}{\# with the following format, there can be no spaces between the brackets, the hashdecl identifier, and the open curly brace}}
\DoxyCodeLine{auto th0 = <MyHash>\{}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    ...,}
\DoxyCodeLine{\};}
\DoxyCodeLine{\textcolor{comment}{\# with the following format, there can be no spaces between the "{}hash"{} string, the brackets, the hashdecl identifier, and the open curly brace}}
\DoxyCodeLine{auto th1 = hash<MyHash>\{}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    ...,}
\DoxyCodeLine{\};}
\DoxyCodeLine{auto th2 = cast<hash<MyHash>>(\{}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    ...,}
\DoxyCodeLine{\});}
\DoxyCodeLine{auto th3 = new <hash<MyHash>>(\{}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    ...,}
\DoxyCodeLine{\});}

\end{DoxyCode}
 ~\newline
 or alternatively, a simple non-\/typed immediate hash can be declared with parentheses\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{hash h2 = (}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    key\_expression: value\_expression,}
\DoxyCodeLine{    ...,}
\DoxyCodeLine{);}

\end{DoxyCode}
 ~\newline
 Here is a concrete example\+: ~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{hash hash = \{}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}apple"{}}: 1 + 1,}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}pear"{}}: \textcolor{stringliteral}{"{}string"{}},}
\DoxyCodeLine{\};}

\end{DoxyCode}
 ~\newline
 Trailing commas are ignored (as with \mbox{\hyperlink{container_data_types_list}{lists}}) to allow for easier insertion and deletion of elements in source code.~\newline
~\newline
 Hashes are dereferenced in one of two ways, either using curly brackets\+: {\ttfamily \char`\"{}\{\char`\"{}} and {\ttfamily \char`\"{}\}\char`\"{}}, where any valid Qore expression can be used, or using the dot \char`\"{}.\char`\"{} hash member dereferencing operator, where literal strings can be used.~\newline
~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{element3 = hash\{\textcolor{stringliteral}{"{}pe"{}} + \textcolor{stringliteral}{"{}ar"{}}\};}

\end{DoxyCode}
 ~\newline
 Is equivalent to\+: ~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{element3 = hash.pear;}

\end{DoxyCode}
 ~\newline
 and\+: ~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{element3 = hash.\textcolor{stringliteral}{"{}pear"{}};}

\end{DoxyCode}
 ~\newline
 and\+: ~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{element3 = hash.(\textcolor{stringliteral}{"{}pe"{}} + \textcolor{stringliteral}{"{}ar"{}});}

\end{DoxyCode}
 ~\newline
 Hash members can have the names of keywords or names that are not valid identifiers, but in this case to dereference them you cannot use the dot operator with a literal string, otherwise a parse error will be raised. Use quotes around the member name when dereferencing hash members with the same name as a qore keyword or other name that is not a valid identifier as follows\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{element3 = hash.\textcolor{stringliteral}{"{}keys"{}};}
\DoxyCodeLine{element3 = hash\{\textcolor{stringliteral}{"{}this-\/element-\/1!"{}}\};}

\end{DoxyCode}
 ~\newline
 A literal string after the dot \char`\"{}.\char`\"{} hash member dereferencing operator must be a valid Qore identifier; therefore if you want to use a hash key that\textquotesingle{}s not a valid identifier, enclose the string in quotes.~\newline
~\newline
 If you want to use the result of an expression to dereference the hash, then either the curly bracket syntax must be used or the expression must be enclosed in parentheses.~\newline
~\newline
 In the case of using a literal string with the dot operator, keep in mind that the string is always interpreted as a literal string name of the member, even if there is a constant with the same name. To use the value of a constant to dereference a hash, use curly brackets with the constant\+: ex\+:
\begin{DoxyCode}{0}
\DoxyCodeLine{hash\{MyConstant\} }

\end{DoxyCode}
 Note that hash keys can also be given by constants (as long as the constant resolves to a string) when using curly brackets.~\newline
 
\end{DoxyParagraph}
\begin{DoxyNote}{Note}
using curly-\/bracket delimiters with an immediate hash with {\ttfamily map} results in the \mbox{\hyperlink{operators_hmap}{hash version of the map operator}})
\end{DoxyNote}
\begin{DoxyParagraph}{Hash slicing\+:}
Hashes can be \char`\"{}sliced\char`\"{} by dereferencing them using parenthesis syntax with more than one key, as in the following example\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{hash h = \{}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}a"{}}: 3,}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}b"{}}: 5,}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}x"{}}: \textcolor{stringliteral}{"{}test"{}},}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}y"{}}: 0,}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}text"{}}: \textcolor{stringliteral}{"{}lorem ipsum dolor sit amet"{}},}
\DoxyCodeLine{    \textcolor{stringliteral}{"{}float"{}}: 2.71828}
\DoxyCodeLine{\};}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Creates a new hash with key-\/value pairs "{}a"{} and "{}b"{} from the original hash.}}
\DoxyCodeLine{hash nh = h.(\textcolor{stringliteral}{"{}a"{}}, \textcolor{stringliteral}{"{}b"{}});}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 1: \%y\(\backslash\)n"{}}, nh);}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Creates a new hash with key-\/value pairs "{}x"{}, "{}y"{} and "{}float"{} from the original hash.}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 2: \%y\(\backslash\)n"{}}, h.(\textcolor{stringliteral}{"{}x"{}}, \textcolor{stringliteral}{"{}y"{}}, \textcolor{stringliteral}{"{}float"{}}));}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Creates a new hash with a key-\/value pair "{}x"{} from the original hash.}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 3: \%y\(\backslash\)n"{}}, h.(\textcolor{stringliteral}{"{}x"{}}, \textcolor{stringliteral}{"{}random"{}}));}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# Creates an empty hash because hash h does not contain keys "{}abc"{} or "{}xyz"{}.}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}slice 4: \%y\(\backslash\)n"{}}, h.(\textcolor{stringliteral}{"{}abc"{}}, \textcolor{stringliteral}{"{}xyz"{}}));}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# -\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/-\/}}
\DoxyCodeLine{\textcolor{comment}{\# the output is as follows:}}
\DoxyCodeLine{}
\DoxyCodeLine{slice 1: \{a: 3, b: 5\}}
\DoxyCodeLine{slice 2: \{x: \textcolor{stringliteral}{"{}test"{}}, y: 0, float: 2.71828\}}
\DoxyCodeLine{slice 3: \{x: \textcolor{stringliteral}{"{}test"{}}\}}
\DoxyCodeLine{slice 4: \{\}}

\end{DoxyCode}
 ~\newline
 All keys matching with the original hash\textquotesingle{}s key-\/value pairs will be included in the newly created hash. Keys which are not in the original hash will be silently ignored.
\end{DoxyParagraph}
\begin{DoxyParagraph}{Immediate Value Examples\+:}
\begin{DoxyVerb}{"key1": 1, "key2": "two", get_key_3(): 3.141592653589793238462643383279502884195n}\end{DoxyVerb}
 \begin{DoxyVerb}<Container>{"i": 2}\end{DoxyVerb}
 \begin{DoxyVerb}hash<Container>{}\end{DoxyVerb}
 Hashes can be declared with curly brackets (preferred) or parentheses\+: \begin{DoxyVerb}("key1": 1, "key2": "two", get_key_3(): 3.141592653589793238462643383279502884195n)\end{DoxyVerb}
 Gives an empty hash (note that {\ttfamily ()} gives an empty list)\+: \begin{DoxyVerb}hash h = {};\end{DoxyVerb}

\end{DoxyParagraph}
\begin{DoxyParagraph}{Typed Hash Declarations}
The {\ttfamily \char`\"{}hash\char`\"{}} type supports a variant with pre-\/defined type-\/safe keys which can be specified with a single argument giving a \mbox{\hyperlink{hashdecl}{type-\/safe hashdecl}} identifier in angle brackets after the {\ttfamily \char`\"{}hash\char`\"{}} type. Furthermore, type-\/safe hashes can be declared using the \mbox{\hyperlink{hashdecl}{hashdecl}} keyword; the following example demonstrates a type-\/safe hash declaration and then a variable restricted to this type\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{\textcolor{comment}{\# hashdecl typed hash declaration}}
\DoxyCodeLine{hashdecl Container \{}
\DoxyCodeLine{    int i = 1;}
\DoxyCodeLine{    string str = \textcolor{stringliteral}{"{}default"{}};}
\DoxyCodeLine{\}}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# immediate value with implicit construction: default values are assigned from the declaration}}
\DoxyCodeLine{auto ah1 = hash<Container>\{\};}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# immediate value with implicit construction: default values are assigned from the declaration}}
\DoxyCodeLine{auto ah2 = <Container>\{\};}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# immediate value with implicit construction: overrides the "{}i"{} member's initial value}}
\DoxyCodeLine{auto ah3 = <Container>\{\textcolor{stringliteral}{"{}i"{}}: 2\};}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# declaration and initialization to default key values}}
\DoxyCodeLine{hash<Container> c1();}
\DoxyCodeLine{}
\DoxyCodeLine{\textcolor{comment}{\# declaration and initialization with explicit values}}
\DoxyCodeLine{hash<Container> h2((\textcolor{stringliteral}{"{}i"{}}: 10, \textcolor{stringliteral}{"{}str"{}}: \textcolor{stringliteral}{"{}other"{}}));}

\end{DoxyCode}
 ~\newline
 This type is supported at parse-\/time and at runtime; to convert such values to an untyped hash, assign it to a \mbox{\hyperlink{data_type_declarations_hash_type}{hash}} lvalue, use \mbox{\hyperlink{operators_cast}{cast$<$hash$>$(...)}} on the value, or call the \mbox{\hyperlink{group__type__conversion__functions_ga0231dbb472dd230310af7c7c34f56e92}{hash()}} function on the value. Each of these options can be used to convert a type-\/safe hash to an untyped hash.
\end{DoxyParagraph}
A special single argument, {\ttfamily \char`\"{}auto\char`\"{}}, allows for the lvalue to maintain the complex hash type as in the following example\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{hash h0 = (\textcolor{stringliteral}{"{}a"{}}: 2, \textcolor{stringliteral}{"{}b"{}}: 3);}
\DoxyCodeLine{\textcolor{comment}{\# prints "{}hash"{}}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, h0.fullType());}
\DoxyCodeLine{hash<auto> h1 = (\textcolor{stringliteral}{"{}a"{}}: 2, \textcolor{stringliteral}{"{}b"{}}: 3);}
\DoxyCodeLine{\textcolor{comment}{\# prints "{}hash<string, int>"{}}}
\DoxyCodeLine{\mbox{\hyperlink{group__string__functions_ga32070517a9f8a70a32ef48f83a6f128e}{printf}}(\textcolor{stringliteral}{"{}\%y\(\backslash\)n"{}}, h1.fullType());}

\end{DoxyCode}


\begin{DoxyNote}{Note}
a {\ttfamily hashdecl} may not have the name {\ttfamily \char`\"{}auto\char`\"{}}, this name has a special meaning in complex types
\end{DoxyNote}
\begin{DoxySeeAlso}{See also}

\begin{DoxyItemize}
\item \mbox{\hyperlink{hashdecl}{Hashdecl\+: Type-\/\+Safe Hash Declarations}}
\item \mbox{\hyperlink{data_type_declarations_hash_hashdecl_type}{Type-\/\+Safe Hash Type}}
\end{DoxyItemize}
\end{DoxySeeAlso}
\begin{DoxyParagraph}{Complex Type Support for Hashes}
The {\ttfamily \char`\"{}hash\char`\"{}} type also supports two type arguments which allow the value type of the hash to be declared. The key type is also included in the declaration, but is currently restricted to type \mbox{\hyperlink{data_type_declarations_string_type}{string}}; the following example demonstrates a hash declaration with a specific value type\+:~\newline

\begin{DoxyCode}{0}
\DoxyCodeLine{hash<string, int> h((\textcolor{stringliteral}{"{}key1"{}}: 2, \textcolor{stringliteral}{"{}key2"{}}: 3));}

\end{DoxyCode}
 ~\newline
 This type is supported at parse-\/time and at runtime; to convert such values to an untyped hash, assign it to a \mbox{\hyperlink{data_type_declarations_hash_type}{hash}} lvalue, use \mbox{\hyperlink{operators_cast}{cast$<$hash$>$(...)}} on the value, or call the \mbox{\hyperlink{group__type__conversion__functions_ga0231dbb472dd230310af7c7c34f56e92}{hash()}} function on the value. Each of these options can be used to convert a type-\/safe hash to an untyped hash.
\end{DoxyParagraph}
\begin{DoxySeeAlso}{See also}
\mbox{\hyperlink{data_type_declarations_hash_complex_type}{Hash With Declared Value Type}}
\end{DoxySeeAlso}
\begin{DoxyParagraph}{Pseudo Class for Type Hash\+:}
\mbox{\hyperlink{class_qore_1_1zzz8hashzzz9}{Qore\+::zzz8hashzzz9}}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Type Code\+:}
\mbox{\hyperlink{group__type__code__constants_gaf94c74753aa73379b90ab30d092ffc60}{Qore\+::\+NT\+\_\+\+HASH}}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Type Name\+:}
{\ttfamily \char`\"{}hash\char`\"{}} 
\end{DoxyParagraph}
\begin{DoxySeeAlso}{See also}
\mbox{\hyperlink{data_type_declarations_hash_type}{hash}}
\end{DoxySeeAlso}
\begin{DoxyNote}{Note}

\begin{DoxyItemize}
\item Trailing commas are ignored in immediate hash declarations (as with \mbox{\hyperlink{container_data_types_list}{lists}}) to allow for easier insertion and deletion of elements in source code.
\item Hashes are dereferenced in one of two ways, either using curly brackets\+: {\ttfamily \char`\"{}\{\char`\"{}} and {\ttfamily \char`\"{}\}\char`\"{}}, where any valid Qore expression can be used, or using the dot \char`\"{}.\char`\"{} hash member dereferencing operator, where literal strings can be used. In the case of using a literal string with the dot operator, keep in mind that the string is always interpreted as a literal string name of the member, even if there is a constant with the same name. To use the value of a constant to dereference a hash, use curly brackets with the constant\+: ex\+:
\begin{DoxyCode}{0}
\DoxyCodeLine{hash\{MyConstant\} }

\end{DoxyCode}

\item Use quotes around the member name when dereferencing hash members with the same name as a Qore keyword or other name that is not a valid identifier\+: ex\+:
\begin{DoxyCode}{0}
\DoxyCodeLine{hash.\textcolor{stringliteral}{"{}my-\/tag-\/1!"{}} }

\end{DoxyCode}

\item Qore hashes preserve key insertion order to support consistent serialization and deserialization to strings (such as XML, JSON, or YAML strings) or data encoding formats, therefore the \mbox{\hyperlink{operators_keys}{keys operator}} and the \mbox{\hyperlink{class_qore_1_1zzz8hashzzz9_abea95265a8d43bda81e922865978fa1f}{Qore\+::zzz8hashzzz9\+::keys()}} pseudo-\/method will always return the hash keys in insertion/creation order
\item Dereferencing hash values that are not present in the hash will return \mbox{\hyperlink{basic_data_types_nothing}{NOTHING}}; to catch typos in hash member names when dereferencing a hash, you can use an \mbox{\hyperlink{container_data_types_object}{object}} and declare a list of allowed public members in the \mbox{\hyperlink{qore_classes}{class definition}} (in which case dereferencing a non-\/existant member will cause a parse exception to be thrown, if the object\textquotesingle{}s class is known at parse time, otherwise it will cause a runtime exception), also the \mbox{\hyperlink{class_qore_1_1_list_hash_iterator}{Qore\+::\+List\+Hash\+Iterator}} class allows for hash members to be dereferenced transparently and will throw an exception if a non-\/existant member name is given (to catch typos).
\item There are several pseudo-\/methods implemented in both the \mbox{\hyperlink{class_qore_1_1zzz8hashzzz9}{Qore\+::zzz8hashzzz9}} and the \mbox{\hyperlink{class_qore_1_1zzz8nothingzzz9}{Qore\+::zzz8nothingzzz9}} pseudo-\/classes that provide asy access to special hash iterators\+:
\begin{DoxyItemize}
\item \mbox{\hyperlink{class_qore_1_1zzz8hashzzz9_ad9c31b0d467e2a1496f57e3feff505ee}{Qore\+::zzz8hashzzz9\+::iterator()}} provides easy access to a hash value iterator; it returns a \mbox{\hyperlink{class_qore_1_1_hash_iterator}{Hash\+Iterator}} object where the \mbox{\hyperlink{class_qore_1_1_hash_iterator_ac476d576e747b685d4284fb02a5cf83e}{Hash\+Iterator\+::get\+Value()}} method returns the current key value
\item \mbox{\hyperlink{class_qore_1_1zzz8hashzzz9_a4dde09aa84522bbd942d9339c0315ed5}{Qore\+::zzz8hashzzz9\+::key\+Iterator()}} provides easy access to a hash key iterator; it returns a \mbox{\hyperlink{class_qore_1_1_hash_key_iterator}{Hash\+Key\+Iterator}} object where the \mbox{\hyperlink{class_qore_1_1_hash_key_iterator_a749d9db18178ea1f86aa721b3c37a770}{Hash\+Key\+Iterator\+::get\+Value()}} method returns the current key string; this is very useful when using a hash as a simulation for a set of strings and quickly iterating the hash with the \mbox{\hyperlink{operators_map}{map operator}}, for example
\item \mbox{\hyperlink{class_qore_1_1zzz8hashzzz9_aae2e67b218a51791103c21866df5217b}{Qore\+::zzz8hashzzz9\+::pair\+Iterator()}} provides easy access to a hash key/value pair iterator; it returns a \mbox{\hyperlink{class_qore_1_1_hash_pair_iterator}{Hash\+Pair\+Iterator}} object where the \mbox{\hyperlink{class_qore_1_1_hash_pair_iterator_a7fbbc6dd431b481042d49eeb706b871e}{Hash\+Pair\+Iterator\+::get\+Value()}} method returns a hash with {\ttfamily \char`\"{}key\char`\"{}} and {\ttfamily \char`\"{}value\char`\"{}} keys giving the key-\/value pair for the current iterator position in the hash
\item \mbox{\hyperlink{class_qore_1_1zzz8hashzzz9_aab3d2e6e69d212dbe05de43136f14367}{Qore\+::zzz8hashzzz9\+::context\+Iterator()}} provides easy access to the current element in a hash of lists such as that returned by the \mbox{\hyperlink{class_qore_1_1_s_q_l_1_1_datasource_a05b1d0d126d815b2d7578e5ce8b9a0de}{Datasource\+::select()}} or \mbox{\hyperlink{class_qore_1_1_s_q_l_1_1_datasource_pool_acafe45a0cf883bc51917e7a0bb8e77fd}{Datasource\+Pool\+::select()}} methods; it returns a \mbox{\hyperlink{class_qore_1_1_hash_list_iterator}{Hash\+List\+Iterator}} object where the \mbox{\hyperlink{class_qore_1_1_hash_list_iterator_ac30588382753a49e3bd1eaac0c6f9918}{Hash\+List\+Iterator\+::get\+Value()}} method returns a hash of the current row value (which is very useful when iterating query results with the \mbox{\hyperlink{operators_map}{map operator}}, for example); this class also provides a \mbox{\hyperlink{class_qore_1_1_hash_list_iterator_a216d0595e6098538ff9c48d117f20106}{member\+Gate() method}} that allows the iterator object itself to be dereferenced directly as a hash representing the current row (which is useful in \mbox{\hyperlink{statements_foreach}{foreach statements}} for example).
\item the corresponding pseudo-\/methods in the \mbox{\hyperlink{class_qore_1_1zzz8nothingzzz9}{Qore\+::zzz8nothingzzz9}} pseudo-\/class return empty \mbox{\hyperlink{class_qore_1_1_single_value_iterator}{Single\+Value\+Iterator}} objects so that these pseudo-\/methods can be safely used with \mbox{\hyperlink{data_type_declarations_hash_or_nothing_type}{$\ast$hash}} (ie \char`\"{}hash or nothing\char`\"{}) value types.
\end{DoxyItemize}
\item the \mbox{\hyperlink{operators_hmap}{hash version of the map operator}} is used when a literal single member hash expression with curly brackets is given to the \mbox{\hyperlink{operators_hmap}{map}} operator; this version of the map operator can be used to build a hash dynamically from a list or iterator expression
\end{DoxyItemize}
\end{DoxyNote}
\DoxyHorRuler{0}
 \hypertarget{container_data_types_object}{}\doxysection{Object}\label{container_data_types_object}
\begin{DoxyParagraph}{Description\+:}
Qore objects are instantiations of a \mbox{\hyperlink{qore_classes}{class}}. They have members (like a hash; values associated to string keys), and methods. The \mbox{\hyperlink{qore_classes}{class definition}} specifies the methods that run on objects of that class, public and private members, static methods and variables, etc associated with the class (however note that static methods do not run in the scope of an object). See \mbox{\hyperlink{qore_classes}{Classes}} for information on how to create a class in Qore.
\end{DoxyParagraph}
\begin{DoxyParagraph}{Immediate Value Example\+:}

\begin{DoxyCode}{0}
\DoxyCodeLine{new Mutex();}

\end{DoxyCode}

\end{DoxyParagraph}
\begin{DoxyParagraph}{Pseudo Class for Type Object\+:}
\mbox{\hyperlink{class_qore_1_1zzz8objectzzz9}{Qore\+::zzz8objectzzz9}}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Type Code\+:}
\mbox{\hyperlink{group__type__code__constants_gac9fe4267bd9924d2f99d19304aaf88c4}{Qore\+::\+NT\+\_\+\+OBJECT}}
\end{DoxyParagraph}
\begin{DoxyParagraph}{Type Name\+:}
{\ttfamily \char`\"{}object\char`\"{}} 
\end{DoxyParagraph}
\begin{DoxySeeAlso}{See also}

\begin{DoxyItemize}
\item \mbox{\hyperlink{data_type_declarations_object_type}{object}}
\item \mbox{\hyperlink{group__object__functions}{Object Functions}}
\end{DoxyItemize}
\end{DoxySeeAlso}
\begin{DoxyNote}{Note}
each Qore type has a \char`\"{}pseudo-\/class\char`\"{} associated with it (the default is \mbox{\hyperlink{class_qore_1_1zzz8valuezzz9}{Qore\+::zzz8valuezzz9}}); methods from the data type\textquotesingle{}s \char`\"{}pseudo-\/class\char`\"{} can be run on any value of that type; see \char`\"{}\+Pseudo Class for Type\char`\"{} headings in \mbox{\hyperlink{basic_data_types}{Basic Data Types}} for more information. Pseudo-\/methods can be overridden in classes; if a class implements a method with the same name and signature as an object pseudo-\/method, then the class\textquotesingle{} method will be executed instead of the pseudo-\/method.
\end{DoxyNote}
\hypertarget{container_data_types_object_overview}{}\doxysubsection{Object Overview}\label{container_data_types_object_overview}
The recommended way to instantiate an object is to declare its type and give constructor arguments after the variable name in parentheses as follows\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{class\_name\_or\_path var\_name([argument list])}

\end{DoxyCode}


For example (for a constructor taking no arguments or having only default values for the aguments, the list is empty)\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{\textcolor{comment}{\# simple object declaration and instantiation}}
\DoxyCodeLine{Mutex m();}
\DoxyCodeLine{\textcolor{comment}{\# this is equivalent to the above}}
\DoxyCodeLine{object<Mutex> m1();}

\end{DoxyCode}


Objects can also be instantiated using the \mbox{\hyperlink{operators_new}{new operator}} as follows.


\begin{DoxyCode}{0}
\DoxyCodeLine{new class\_identifier([argument list])}

\end{DoxyCode}


For example\+:


\begin{DoxyCode}{0}
\DoxyCodeLine{Mutex m = new Mutex();}

\end{DoxyCode}


Objects have named data members that are referenced like hash elements, although this behavior can be modified for objects using the member\+Gate() method. Object members are accessed by appending a dot \textquotesingle{}.\textquotesingle{} and the member name to the object reference as follows\+:


\begin{DoxyCode}{0}
\DoxyCodeLine{object\_reference.member\_name}

\end{DoxyCode}


For more information, see \mbox{\hyperlink{qore_classes_class_members}{Class Members}}.

Object methods are called by appending a dot \textquotesingle{}.\textquotesingle{} and a method name to the object reference as follows\+:


\begin{DoxyCode}{0}
\DoxyCodeLine{object\_reference.method\_name([argument\_list])}

\end{DoxyCode}


Or, from within the class code itself to call another method from inside the same class hierarchy\+:


\begin{DoxyCode}{0}
\DoxyCodeLine{method\_name([argument\_list])}

\end{DoxyCode}


For more information, see \mbox{\hyperlink{qore_classes_object_method_calls}{Object Method Calls}}.

The object references above are normally variable references holding an object, but could be any expression that returns an object, such as a \mbox{\hyperlink{operators_new}{new expression}} or even a function call.

\begin{DoxyNote}{Note}
Objects are treated differently than other Qore data types; they are only explicitly copied (see \mbox{\hyperlink{container_data_types_object_references}{Object References}} for more information). Any object instantiated with the \mbox{\hyperlink{operators_new}{new operator}} will remain unique until deleted or explicitly copied. An explicit copy is made with the copy method, and does not always guarantee an exact copy of the source object (it depends on the definition of the copy method for the class in question).
\end{DoxyNote}
Objects exist until they go out of \mbox{\hyperlink{container_data_types_object_scope}{scope}}, are explicitly deleted, or their last thread exits. For detailed information, see \mbox{\hyperlink{qore_classes}{Classes}}.

\begin{DoxySeeAlso}{See also}
\mbox{\hyperlink{data_type_declarations_object_type}{object}}
\end{DoxySeeAlso}
\hypertarget{container_data_types_object_references}{}\doxysubsection{Object References}\label{container_data_types_object_references}
In Qore objects are treated differently from all other data types in that they are by default passed as arguments to functions and methods by passing a copy of a reference to the object (similar to Java\textquotesingle{}s handling of objects). That means that passing an object to a function that modifies the object will by default modify the original object and not a copy, however reassigning a local parameter variable assigned an object passed as an argument (that is only assigned to a local variable in the calling function) will not result in deleting the object, but rather decrement its scope reference count (note that if the object were created as a part of the call and reassigning the variable would cause the object\textquotesingle{}s scope reference count to reach zero, then the object would be deleted in this case).

Assigning an object to a variable has the same effect; a copy of a reference to the object is assigned to the variable. This results in prolonging the object\textquotesingle{}s scope (by owning a new copy of a reference to the object).

An example\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{sub test2(any x) \{}
\DoxyCodeLine{   \textcolor{comment}{\# we can modify the original object like so:}}
\DoxyCodeLine{   x.member = \textcolor{stringliteral}{"{}tree"{}};}
\DoxyCodeLine{}
\DoxyCodeLine{   \textcolor{comment}{\# here we re-\/assign x, but since the object is also assigned}}
\DoxyCodeLine{   \textcolor{comment}{\# to o in the calling function, the object's scope is still}}
\DoxyCodeLine{   \textcolor{comment}{\# valid, and therefore nothing happens to the object}}
\DoxyCodeLine{   x = 1;}
\DoxyCodeLine{\}}
\DoxyCodeLine{}
\DoxyCodeLine{sub test() \{}
\DoxyCodeLine{   TestObject o();}
\DoxyCodeLine{}
\DoxyCodeLine{   \textcolor{comment}{\# here we pass a copy of a reference to the object in o}}
\DoxyCodeLine{   test2(o);}
\DoxyCodeLine{}
\DoxyCodeLine{   \textcolor{comment}{\# this will print out "{}ok\(\backslash\)n"{}, because the object is still}}
\DoxyCodeLine{   \textcolor{comment}{\# valid and the member has been set by test2()}}
\DoxyCodeLine{   \textcolor{keywordflow}{if} (o.member == \textcolor{stringliteral}{"{}tree"{}})}
\DoxyCodeLine{      \mbox{\hyperlink{group__string__functions_ga996c5d1686fbc245f0bc7a02012e6a33}{print}}(\textcolor{stringliteral}{"{}ok\(\backslash\)n"{}});}
\DoxyCodeLine{\}}
\DoxyCodeLine{\textcolor{comment}{\# when test() exits, the object in o will go out of scope}}
\DoxyCodeLine{\textcolor{comment}{\# and be deleted}}

\end{DoxyCode}


If, however, an object is passed by reference, then the local variable of the called function that accepts the object owns the scope reference of the calling functions\textquotesingle{}s variable.

An example\+: 
\begin{DoxyCode}{0}
\DoxyCodeLine{sub test2(any x) \{}
\DoxyCodeLine{   \textcolor{comment}{\# we can modify the original object like so:}}
\DoxyCodeLine{   x.member = \textcolor{stringliteral}{"{}tree"{}};}
\DoxyCodeLine{}
\DoxyCodeLine{   \textcolor{comment}{\# here we re-\/assign x, and since we own the only scope}}
\DoxyCodeLine{   \textcolor{comment}{\# reference to the object, the object will go out of}}
\DoxyCodeLine{   \textcolor{comment}{\# scope here and be deleted}}
\DoxyCodeLine{   x = 1;}
\DoxyCodeLine{\}}
\DoxyCodeLine{}
\DoxyCodeLine{sub test() \{}
\DoxyCodeLine{   TestObject o();}
\DoxyCodeLine{}
\DoxyCodeLine{   \textcolor{comment}{\# here we pass a reference to the object in o}}
\DoxyCodeLine{   test2(\(\backslash\)o);}
\DoxyCodeLine{}
\DoxyCodeLine{   \textcolor{comment}{\# the object has already been deleted in test2() and}}
\DoxyCodeLine{   \textcolor{comment}{\# therefore nothing will be printed out}}
\DoxyCodeLine{   \textcolor{keywordflow}{if} (o.member == \textcolor{stringliteral}{"{}tree"{}})}
\DoxyCodeLine{      \mbox{\hyperlink{group__string__functions_ga996c5d1686fbc245f0bc7a02012e6a33}{print}}(\textcolor{stringliteral}{"{}ok\(\backslash\)n"{}});}
\DoxyCodeLine{\}}

\end{DoxyCode}


\begin{DoxyNote}{Note}
that when parse option \mbox{\hyperlink{parse_directives_allow-bare-refs}{\%allow-\/bare-\/refs}} is set, then variable references as in the above examples are made without the {\ttfamily \char`\"{}\$\char`\"{}} character.
\end{DoxyNote}
\hypertarget{container_data_types_object_scope_and_gc}{}\doxysubsection{Object Scope and Garbage Collection}\label{container_data_types_object_scope_and_gc}
\hypertarget{container_data_types_garbage_collection}{}\doxysubsubsection{Garbage Collection and Resource Management}\label{container_data_types_garbage_collection}
All complex data structures in Qore use atomic references to support copy-\/on-\/write semantics, but objects are unique since they are always passed with a copy of a reference to the object (similar to Java\textquotesingle{}s object handling), therefore it\textquotesingle{}s possible to make recursive references to objects, a fact which complicates object scope management and garbage collection.

The \href{http://en.wikipedia.org/wiki/Resource_Acquisition_Is_Initialization}{\texttt{ RAII idiom}} for resource management is a fundamental feature of \mbox{\hyperlink{namespace_qore}{Qore}}\textquotesingle{}s design; the language should always guarantee that objects are destroyed and therefore their associated resources are managed and memory freed when objects go out of scope. This is tricky when objects have recursive references; other languages use a variety of approaches to deal with this (for example, a mark-\/and-\/sweep algorithm in a dedicated garbage collection thread) which can have various drawbacks (such as non-\/deterministic performance impacts, such as when the garbage-\/collection thread grabs global locks to verify references, etc).

Qore supports deterministic garbage collection of objects even when objects participate in recursive directed graphs. In this case the deterministic aspect of Qore\textquotesingle{}s garbage collection algorithm means that objects are collected immediately when they have no more valid references to them. In case of recursive references, it means that no valid references are pointing to any object in the recursive graph; in other words, the scope references for every object participating in the recursive graph are only due to references related to recursive references. In this case, the affected objects are collected immediately by having their destructors run.

The consequence of this is that destructors are run immediately when the objects are no longer valid, and therefore the \href{http://en.wikipedia.org/wiki/Resource_Acquisition_Is_Initialization}{\texttt{ RAII idiom}} for resource management is supported in Qore even when objects participate in recursive directed graphs.

Some examples of \href{http://en.wikipedia.org/wiki/Resource_Acquisition_Is_Initialization}{\texttt{ RAII}} in builtin Qore classes are (a subset of possible examples)\+:
\begin{DoxyItemize}
\item the \mbox{\hyperlink{class_qore_1_1_thread_1_1_auto_lock}{Autolock}} class releases the \mbox{\hyperlink{class_qore_1_1_thread_1_1_mutex}{Mutex}} in the destructor (this class is designed to be used with scope-\/bound exception-\/safe resource management; see also the \mbox{\hyperlink{class_qore_1_1_thread_1_1_auto_gate}{Auto\+Gate}}, \mbox{\hyperlink{class_qore_1_1_thread_1_1_auto_read_lock}{Auto\+Read\+Lock}}, and \mbox{\hyperlink{class_qore_1_1_thread_1_1_auto_write_lock}{Auto\+Write\+Lock}} classes)
\item the \mbox{\hyperlink{class_qore_1_1_s_q_l_1_1_datasource}{Datasource}} class closes any open connection in the destructor, and, if a transaction is still in progress, the transaction is rolled back automatically and an exception is thrown before the connection is closed
\item the \mbox{\hyperlink{class_qore_1_1_s_q_l_1_1_datasource_pool}{Datasource\+Pool}} class closes all open connections in the destructor, and, if a transaction is still in progress, the transaction is rolled back automatically and an exception is thrown before the connection is closed
\item the \mbox{\hyperlink{class_qore_1_1_read_only_file}{Read\+Only\+File}} and \mbox{\hyperlink{class_qore_1_1_file}{File}} classes close the file descriptor in the destructor if it\textquotesingle{}s open
\item the \mbox{\hyperlink{class_qore_1_1_thread_1_1_r_w_lock}{RWLock}} class throws an exception if destroyed while any thread is still holding the lock; note that in this case the underlying object is only destroyed when all threads holding locks have released their locks; this is handled with Qore\textquotesingle{}s \mbox{\hyperlink{threading_thread_resources}{thread resource handling}} and strong references to the underlying \mbox{\hyperlink{class_qore_1_1_thread_1_1_r_w_lock}{RWLock}} object while \mbox{\hyperlink{threading_thread_resources}{thread resources}} are held
\item the \mbox{\hyperlink{class_qore_1_1_socket}{Socket}} class first shuts down any TLS/\+SSL connection and then closes the connection in the destructor if it\textquotesingle{}s open
\item the \mbox{\hyperlink{class_qore_1_1_thread_1_1_thread_pool}{Thread\+Pool}} class detaches all currently in-\/progress worker threads, cancels pending tasks not yet executed (by calling their cancellation closure, if any), terminates the worker thread and destroys the thread pool
\end{DoxyItemize}

\begin{DoxyNote}{Note}
Qore also supports scope-\/related resource management support in the form of the \mbox{\hyperlink{statements_on_exit}{on\+\_\+exit}}, \mbox{\hyperlink{statements_on_success}{on\+\_\+success}}, and \mbox{\hyperlink{statements_on_error}{on\+\_\+error}} statements
\end{DoxyNote}
\hypertarget{container_data_types_object_scope}{}\doxysubsubsection{Object Scope}\label{container_data_types_object_scope}
As mentioned in the previous section, objects are automatically collected and have their destructors run when their scope-\/relevant reference count reaches zero or when the scope-\/relevant reference count is equal to the number of recursive references for every object in a recursive directed graph (note that objects can be deleted manually at any time by using the \mbox{\hyperlink{operators_delete}{delete operator}}). Whenever an object is collected, the object\textquotesingle{}s destructor method as defined in the object\textquotesingle{}s class is run on the object (multiple destructor methods can be run on objects instantiated from a class hierarchy).

The following affect objects\textquotesingle{} scope\+:
\begin{DoxyItemize}
\item external references to the object\+: an object\textquotesingle{}s automatic scope is prolonged as long as the object has valid external references
\item object method thread launched within the object\+: if a member function thread was launched from within the object using the \mbox{\hyperlink{operators_background}{background operator}}, the object\textquotesingle{}s automatic scope is prolonged to the life of the new thread. Object threads started externally to the object (i.\+e. not directly from an expression with the \mbox{\hyperlink{operators_background}{background operator}} within a method) will not prolong the scope of the object.
\end{DoxyItemize}

\begin{DoxyNote}{Note}
If an object with running threads is explicitly deleted, and this case is not handled in the object\textquotesingle{}s destructor() method (by ensuring that all other running threads terminate gracefully), exceptions will be thrown in other threads at any attempt to access the already-\/deleted object. The fact that object threads and closures can prolong object scope means, for example, that objects assigned to local variables can exist for longer than the scope of their host variable if they have one or more methods running in other threads or if closures created from within the object still exist at the time the local variable goes out of scope. For more information about threading, please see \mbox{\hyperlink{threading}{Threading}}
\end{DoxyNote}
\begin{DoxySince}{Since}
As of Qore 0.\+8.\+10 the existence of a closure created within the object no longer prolongs the scope of the object; the closure encapsulate the object\textquotesingle{}s state (along with any local variables referenced within the closure), but if the object goes out of scope while the closure still exists, then any references to the object after the object has been destroyed will cause {\ttfamily OBJECT-\/\+ALREADY-\/\+DELETED} exceptions to be thrown. This addresses memory and reference leaks caused by recursive references when closures encapsulating an object\textquotesingle{}s scope are assigned to or accessible from members of the object.
\end{DoxySince}
\hypertarget{container_data_types_copying_objects}{}\doxysubsection{Copying Objects}\label{container_data_types_copying_objects}
To explicitly generate a copy of an object, the {\ttfamily copy()} constructor must be called. This is a special method that exists implicitly for every class even if it is not explicitly defined (like {\ttfamily constructor()} and {\ttfamily destructor()} methods). The implicit behavior of the {\ttfamily copy()} constructor is to create a new object with new members that are copies of the original members (except objects are created as copies of references to the object). Then, if any {\ttfamily copy()} method exists, it will be executed in the new object, passing a copy of a reference to the old object as the first argument to the {\ttfamily copy()} method.

\begin{DoxyNote}{Note}
In a class hierarchy copy() methods are called in the same order as constructor() methods.~\newline
~\newline
 Not all built-\/in classes can be copied. Classes not supporting copying will throw an exception when the {\ttfamily copy()} methods are called. See the documentation for each class for more information. 
\end{DoxyNote}
