/*
  BackquoteNode.cpp
 
  Qore Programming Language
 
  Copyright (C) 2003 - 2014 David Nichols
 
  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
  DEALINGS IN THE SOFTWARE.

  Note that the Qore library is released under a choice of three open-source
  licenses: MIT (as above), LGPL 2+, or GPL 2+; see README-LICENSE for more
  information.
*/

#include <qore/Qore.h>

#include <errno.h>

BackquoteNode::BackquoteNode(char *c_str) : ParseNode(NT_BACKQUOTE), str(c_str) {
}

BackquoteNode::~BackquoteNode() {
   if (str)
      free(str);
}

// get string representation (for %n and %N), foff is for multi-line formatting offset, -1 = no line breaks
// the ExceptionSink is only needed for QoreObject where a method may be executed
// use the QoreNodeAsStringHelper class (defined in QoreStringNode.h) instead of using these functions directly
// returns -1 for exception raised, 0 = OK
int BackquoteNode::getAsString(QoreString &qstr, int foff, ExceptionSink *xsink) const {
   qstr.sprintf("backquote '%s' (0x%08p)", str ? str : "<null>", this);
   return 0;
}

// if del is true, then the returned QoreString * should be deleted, if false, then it must not be
QoreString *BackquoteNode::getAsString(bool &del, int foff, ExceptionSink *xsink) const {
   del = true;
   QoreString *rv = new QoreString();
   getAsString(*rv, foff, xsink);
   return rv;
}

// returns the type name as a c string
const char *BackquoteNode::getTypeName() const {
   return "backquote expression";
}

// eval(): return value requires a deref(xsink)
AbstractQoreNode *BackquoteNode::evalImpl(ExceptionSink *xsink) const {
   int rc;
   return backquoteEval(str, rc, xsink);
}

// eval(): return value requires a deref(xsink)
AbstractQoreNode *BackquoteNode::evalImpl(bool &needs_deref, ExceptionSink *xsink) const {
   needs_deref = true;
   int rc;
   return backquoteEval(str, rc, xsink);
}

int64 BackquoteNode::bigIntEvalImpl(ExceptionSink *xsink) const {
   int rc;
   ReferenceHolder<AbstractQoreNode> rv(backquoteEval(str, rc, xsink), xsink);
   return rv ? rv->getAsBigInt() : 0;
}

int BackquoteNode::integerEvalImpl(ExceptionSink *xsink) const {
   int rc;
   ReferenceHolder<AbstractQoreNode> rv(backquoteEval(str, rc, xsink), xsink);
   return rv ? rv->getAsInt() : 0;
}

bool BackquoteNode::boolEvalImpl(ExceptionSink *xsink) const {
   int rc;
   ReferenceHolder<AbstractQoreNode> rv(backquoteEval(str, rc, xsink), xsink);
   return rv ? rv->getAsBool() : 0;
}

double BackquoteNode::floatEvalImpl(ExceptionSink* xsink) const {
   int rc;
   ReferenceHolder<AbstractQoreNode> rv(backquoteEval(str, rc, xsink), xsink);
   return rv ? rv->getAsFloat() : 0;
}

#ifndef READ_BLOCK
#define READ_BLOCK 1024
#endif

QoreStringNode* backquoteEval(const char* cmd, int& rc, ExceptionSink* xsink) {
   rc = 0;
   // execute command in a new process and read stdout in parent
   FILE* p = popen(cmd, "r");
   if (!p) {
      // could not fork or create pipe
      xsink->raiseException("BACKQUOTE-ERROR", q_strerror(errno));
      return 0;
   }

   // allocate buffer for return value
   QoreStringNodeHolder s(new QoreStringNode);

   // read in result string
   while (true) {
      char buf[READ_BLOCK];
      int size = fread(buf, 1, READ_BLOCK, p);

      // break if no data is available or an error occurred
      if (!size || size == -1)
         break;

      s->concat(buf, size);

      // break if there is no more data
      if (size != READ_BLOCK)
         break;
   }

   // wait for child process to terminate and close pipe
   rc = pclose(p);

   return s.release();
}
