/* -*- mode: c++; indent-tabs-mode: nil -*- */
/*
  QoreCounter.h

  Qore Programming Language

  Copyright (C) 2005 - 2014 David Nichols

  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
  DEALINGS IN THE SOFTWARE.

  Note that the Qore library is released under a choice of three open-source
  licenses: MIT (as above), LGPL 2+, or GPL 2+; see README-LICENSE for more
  information.
*/

#ifndef _QORE_QORECOUNTER_H

#define _QORE_QORECOUNTER_H

#include <qore/Qore.h>
#include <qore/QoreCondition.h>

//! a simple thread-safe counter object; objects can block on it until the counter reaches zero
class QoreCounter {
private:
   //! private implementation of the counter
   struct qore_counter_private* priv;

   //! this function is not implemented; it is here as a private function in order to prohibit it from being used
   DLLLOCAL QoreCounter(const QoreCounter&);

   //! this function is not implemented; it is here as a private function in order to prohibit it from being used
   DLLLOCAL QoreCounter& operator=(const QoreCounter&);

public:
   //! creates the counter and initializes the count
   DLLEXPORT QoreCounter(int nc = 0);

   //! destroys the object and frees all memory
   DLLEXPORT ~QoreCounter();

   //! throws a Qore-language exception if there are any waiting threads and wakes them all up
   DLLEXPORT void destructor(ExceptionSink* xsink);

   //! increments the counter
   DLLEXPORT void inc();

   //! decrements the counter and wakes up any threads if the counter reaches 0
   /** a Qore-language exception will be raised here if QoreCounter::destructor() has already been run before calling this function.

       @param xsink any Qore-language exception thrown will be added here
    */
   DLLEXPORT void dec(ExceptionSink* xsink);

   //! blocks the calling thread until the counter reaches 0
   /** a Qore-language exception will be raised here if QoreCounter::destructor() is run while threads are still blocked

       @param xsink any Qore-language exception thrown will be added here
       @param timeout_ms indicates a timeout in milliseconds to wait, 0 means no timeout

       @return non-zero means an exception was thrown
    */
   DLLEXPORT int waitForZero(ExceptionSink* xsink, int timeout_ms = 0);

   //! returns the current count
   DLLEXPORT int getCount() const;

   //! returns the number of threads blocked on this object
   DLLEXPORT int getWaiting() const;

   // internal use only - for internal counters
   DLLLOCAL void waitForZero();
   DLLLOCAL void dec();
};

#endif
