//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
//! The XmlReader class allows XML strings to be iterated and parsed piecewise
/***/
class XmlReader {

public:
//! Returns the number of attributes of the current node
/** @return the number of attributes of the current node

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code int n = xr.attributeCount(); @endcode
 */
int attributeCount();

public:
//! Returns the base URI of the node if known, \c NOTHING if not.
/** @return the base URI of the node if known, \c NOTHING if not.

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.baseUri(); @endcode
 */
__7_ string baseUri();

public:
//! Creates the XmlReader object based on an XmlDoc object
/** @throw XML-READER-ERROR libxml2 returned an error code when creating the XML reader object

    @par Example:
    @code XmlReader xr(xmldoc); @endcode
*/
 constructor(XmlDoc doc);

public:
//! Creates the XmlReader object based on the string passed; the string will be converted to UTF-8 encoding before parsing if necessary
/** @par Example:
    @code XmlReader xr(xmlstr); @endcode

    @throw XML-READER-ERROR libxml2 returned an error code when creating the XML reader object (ex: XML string could not be parsed)
    @throw ENCODING-CONVERSION-ERROR the string could not be converted to UTF-8 (usually due to an encoding error in the source string)
*/
 constructor(string xml);

public:
//! creates a new XmlReader object from the input stream
/** @param is the input stream
    @param opts the following options are accepted:
    - \c encoding: (string) the file's character encoding
    - \c xml_input_io: (AbstractXmlIoInputCallback) an AbstractXmlIoInputCallback object to resolve external XSD schema references
    - \c xsd: (string) an XSD string for schema validation while parsing

    @par Example:
    @code
    XmlReader xr(is);
    @endcode

    @throw XML-READER-OPTION-ERROR error in option hash
    @throw XML-READER-ERROR libxml2 returned an error code when creating the XML reader object

    @note the \c xml_parse_options option is accepted but ignored in this method; XML parsing options can be used in @ref XmlReader::toQore() and @ref XmlReader::toQoreData()

    @since xml 1.4
*/
 constructor(Qore::InputStream is, __7_ hash opts);

public:
//! Creates an independent copy of the XmlReader object
/** @return an independent copy of the XmlReader object

    @par Example:
    @code XmlReader xrcopy = xr.copy(); @endcode
*/
 copy();

public:
//! Returns the depth of the node in the tree
/** @return the depth of the node in the tree

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code int depth = xr.depth(); @endcode
*/
int depth();

public:
//! Returns the encoding string given in the original XML string or \c NOTHING if none is given
/** @return the encoding string given in the original XML string or \c NOTHING if none is given

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.encoding(); @endcode

    @note all XML strings are automatically converted to UTF-8 before parsing
 */
__7_ string encoding();

public:
//! Returns the value of the attribute matching the qualified name passed, or \c NOTHING if no such attribute exists in the current XmlReader
/** @par Code Flags:
    @ref CONSTANT

    @param attr the name of the attribute to retrieve

    @return the value of the attribute matching the qualified name passed, or \c NOTHING if no such attribute exists in the current XmlReader

    @par Example:
    @code *string value = xmlreader.getAttribute(name); @endcode

    @see XmlReader::getAttributeNs()
 */
__7_ string getAttribute(string attr);

public:
//! Returns the value of the given attribute anchored in the given namespace, or \c NOTHING if no such attribute exists in the current XmlReader
/** @par Code Flags:
    @ref CONSTANT

    @param attr the name of the attribute to retrieve
    @param ns the namespace URI of the attribute

    @return the value of the given attribute anchored in the given namespace, or \c NOTHING if no such attribute exists in the current XmlReader

    @par Example:
    @code *string value = xmlreader.getAttributeNs(localname, namespaceuri); @endcode

    @see XmlReader::getAttribute()
 */
__7_ string getAttributeNs(string attr, string ns);

public:
//! Returns the value of the attribute with the specified index relative to the containing element, or \c NOTHING if no such attribute exists in the current XmlReader
/** @par Code Flags:
    @ref CONSTANT

    @param offset the attribute number of the attribute to retrieve relative to the containing element starting from 0

    @return the value of the attribute with the specified index relative to the containing element, or \c NOTHING if no such attribute exists in the current XmlReader

    @par Example:
    @code *string value = xmlreader.getAttributeOffset(offset); @endcode

    @see XmlReader::getAttribute()
 */
__7_ string getAttributeOffset(softint offset = 0);

public:
//! Returns an XML string of the contents of the all current node's child nodes and markup, or \c NOTHING if the current node is neither an element nor an attribute or has no child nodes
/** @return an XML string of the contents of the all current node's child nodes and markup, or \c NOTHING if the current node is neither an element nor an attribute or has no child nodes

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.getInnerXml(); @endcode

    @see XmlReader::getOuterXml()
 */
__7_ string getInnerXml();

public:
//! Return an XML string of the contents of the current node and all child nodes and markup, or \c NOTHING if the current node is neither an element nor an attribute or has no child nodes
/** @return an XML string of the contents of the current node and all child nodes and markup, or \c NOTHING if the current node is neither an element nor an attribute or has no child nodes

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.getOuterXml(); @endcode

    @see XmlReader::getInnerXml()
 */
__7_ string getOuterXml();

public:
//! Returns \c True if the node has attributes or \c False if not
/** @return \c True if the node has attributes or \c False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code bool b = xr.hasAttributes(); @endcode
 */
bool hasAttributes();

public:
//! Returns \c True if the node has a text value or \c False if not
/** @return \c True if the node has a text value or \c False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code bool b = xr.hasValue(); @endcode
 */
bool hasValue();

public:
//! Returns \c True if an attribute node was generated from the default value defined in the DTD or schema, \c False if not
/** @return \c True if an attribute node was generated from the default value defined in the DTD or schema, \c False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code bool b = xr.isDefault(); @endcode
 */
bool isDefault();

public:
//! Returns \c True if the current node is empty or \c False if not
/** @return \c True if the current node is empty or \c False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code bool b = xr.isEmptyElement(); @endcode
 */
bool isEmptyElement();

public:
//! Returns \c True if the current node is a namespace declaration rather than a regular attribute or \c False if not
/** @return \c True if the current node is a namespace declaration rather than a regular attribute or \c False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code bool b = xr.isNamespaceDecl(); @endcode
 */
bool isNamespaceDecl();

public:
//! Returns \c True if the current reader parser context is valid, \c False if not
/** @return \c True if the current reader parser context is valid, \c False if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code bool b = xr.isValid(); @endcode
 */
bool isValid();

public:
//! Returns the local name of the node or \c NOTHING if no name is available
/** @return the local name of the node or \c NOTHING if no name is available

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.localName(); @endcode
 */
__7_ string localName();

public:
//! Returns the default namespace in the scope of the current element, or \c NOTHING if none exists
/** @return the default namespace in the scope of the current element, or \c NOTHING if none exists

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string ns = xr.lookupNamespace(); @endcode
 */
__7_ string lookupNamespace();

public:
//! Returns the namespace corresponding to the given prefix in the scope of the current element, or \c NOTHING if none exists
/** @par Code Flags:
    @ref CONSTANT

    @param prefix the namespace prefix to resolve

    @return the namespace corresponding to the given prefix in the scope of the current element, or \c NOTHING if none exists

    @par Example:
    @code *string value = xmlreader.lookupNamespace(prefix); @endcode
 */
__7_ string lookupNamespace(string prefix);

public:
//! Moves the position of the current instance to the attribute with the specified qualified name
/** If an XML parsing error occurs, an exception is thrown

    @param attr the qualified name of the attribute to move to

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.moveToAttribute(name); @endcode

    @see XmlReader::moveToAttributeNs()
 */
bool moveToAttribute(string attr);

public:
//! Moves the position of the current instance to the attribute with the specified local name and namespace URI
/** If an XML parsing error occurs, an exception is thrown

    @param attr the qualified name of the attribute to move to
    @param ns the namespace URI of the attribute

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.moveToAttributeNs(localname, namespaceuri); @endcode

    @see XmlReader::moveToAttribute()
 */
bool moveToAttributeNs(string attr, string ns);

public:
//! Moves the position of the current instance to the attribute with the specified index relative to the containing element
/** If an XML parsing error occurs, an exception is thrown

    @param offset the index of the attribute relative to the containing element to move to starting with 0

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.moveToAttributeOffset(offset); @endcode

    @see XmlReader::moveToAttribute()
 */
bool moveToAttributeOffset(softint offset);

public:
//! Moves the position of the current instance to the element node containing the current attribute node
/** If an XML parsing error occurs, an exception is thrown

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.moveToElement(); @endcode
 */
bool moveToElement();

public:
//! Moves the position of the current instance to the first attribute of the current node
/** If an XML parsing error occurs, an exception is thrown

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.moveToFirstAttribute(); @endcode
 */
bool moveToFirstAttribute();

public:
//! Moves the position of the current instance to the next attribute of the current node
/** If an XML parsing error occurs, an exception is thrown

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.moveToNextAttribute(); @endcode
 */
bool moveToNextAttribute();

public:
//! Returns the qualified name of the node (\c prefix:LocalName) or \c NOTHING if no name is available
/** @return the qualified name of the node (\c prefix:LocalName) or \c NOTHING if no name is available

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.name(); @endcode

    @see XmlReader::localName()
*/
__7_ string name();

public:
//! Returns the URI defining the namespace associated with the node, or \c NOTHING if not available
/** @return the URI defining the namespace associated with the node, or \c NOTHING if not available

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.namespaceUri(); @endcode
 */
__7_ string namespaceUri();

public:
//! Moves the position of the current instance to the next node in the tree at the same level, skipping any subtree
/** If an XML parsing error occurs, an exception is thrown

    @return \c True in case of success, \c False if not found

    @throw PARSE-XML-EXCEPTION error parsing XML

    @par Example:
    @code bool b = xmlreader.next(); @endcode
 */
bool next();

public:
//! Returns the node type of the current node
/** @return the node type of the current node; for return values, see @ref XmlNodeTypes

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code int nt = xr.nodeType(); @endcode

    @see @ref NodeTypeMap
    @see XmlReader::nodeTypeName()
*/
int nodeType();

public:
//! Returns a string giving the node type of the current node or \c NOTHING if no current node is available
/** @return a string giving the node type of the current node or \c NOTHING if no current node is available; for possible return values, see the values of the @ref NodeTypeMap

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.nodeTypeName(); @endcode

    @see XmlReader::nodeType()
*/
__7_ string nodeTypeName();

public:
//! Returns the shorthand reference to the namespace associated with the node, or \c NOTHING if not available
/** @return the shorthand reference to the namespace associated with the node, or \c NOTHING if not available

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.prefix(); @endcode
 */
__7_ string prefix();

public:
//! Moves the position of the current instance to the next node in the stream
/** If an error occurs parsing the XML string, an exception is raised

    @return \c True if the read was successful, \c False if there are no more nodes to read

    @par Example:
    @code if (!xr.read())
        break; @endcode

    @throw PARSE-XML-EXCEPTION cannot move to next node due to an error parsing the XML string (exception description string contains details about the error)

    @see XmlReader::readSkipWhitespace()
*/
bool read();

public:
//! the position of the current instance to the next node in the stream, skipping any whitespace nodes
/** If an error occurs parsing the XML string, an exception is raised

    @return \c True if the read was successful, \c False if there are no more nodes to read

    @par Example:
    @code if (!xr.readSkipWhitespace())
        break; @endcode

    @throw PARSE-XML-EXCEPTION cannot move to next node due to an error parsing the XML string (exception description string contains details about the error)

    @see XmlReader::read()
*/
bool readSkipWhitespace();

public:
//! Set a RelaxNG schema for schema validation while parsing the XML document
/** This method must be called before the first call to XmlReader::read()

    The availability of this method depends on the presence of libxml2's \c xmlTextReaderRelaxNGValidate() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before calling this method.

    @param relaxng the RelaxNG schema string to use to validate the XML document

    @throw RELAXNG-SYNTAX-ERROR invalid RelaxNG string
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHRELAXNG" before calling this function

    @par Example:
    @code xr.relaxNGValidate(relaxng); @endcode
 */
nothing relaxNGValidate(string relaxng);

public:
//! Set an XSD schema for schema validation while parsing the XML document
/** This method must be called before the first call to XmlReader::read()

    The availability of this method depends on the presence of libxml2's \c xmlTextReaderSetSchema() function when the xml module was compiled; for maximum portability check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before calling this method.

    @param xsd the XSD schema string to use to validate the XML document

    @throw XSD-SYNTAX-ERROR invalid XSD string
    @throw MISSING-FEATURE-ERROR this exception is thrown when the function is not available; for maximum portability, check the constant @ref optionconstants "HAVE_PARSEXMLWITHSCHEMA" before calling this function

    @par Example:
    @code xr.schemaValidate(xsd); @endcode
 */
nothing schemaValidate(string xsd);

public:
//! Returns Qore data corresponding to the XML starting at the current node position, maintains element order by mangling node names
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code any data = xr.toQore(); @endcode

    @param pflags XML parsing flags; see @ref xml_parsing_constants for more information; note that this method assumes @ref XPF_PRESERVE_ORDER for backwards-compatibility

    @return If there are sub elements, a hash of the XML is returned, the sub elements representing the current node's children.  If only text is present at the current element, a string is returned. If no information is available, then \c NOTHING is returned.

    @note This method is functionally similar to parse_xml()

    @see XmlReader::toQoreData()
 */
any toQore(int pflags = XPF_PRESERVE_ORDER);

public:
//! Returns Qore data corresponding to the XML starting at the current node position, collapses duplicate out of order elements to a list
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code any data = xr.toQoreData(); @endcode

    @param pflags XML parsing flags; see @ref xml_parsing_constants for more information

    @return If there are sub elements, a hash of the XML is returned, the sub elements representing the current node's children.  If only text is present at the current element, a string is returned.  If no information is available, then \c NOTHING is returned.

    @note This method is functionally similar to parse_xml()

    @see XmlReader::toQore()
 */
any toQoreData(__7_ int pflags);

public:
//! Returns the text value of the node or \c NOTHING if not available
/** @return the text value of the node or \c NOTHING if not available

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string val = xr.value(); @endcode
 */
__7_ string value();

public:
//! Returns the \c xml:lang scope within which the node resides or \c NOTHING if there is none
/** @return the \c xml:lang scope within which the node resides or \c NOTHING if there is none

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.xmlLang(); @endcode
 */
__7_ string xmlLang();

public:
//! Returns a string giving the XML version of the source document (normally \c "1.0") or \c NOTHING if none is present
/** @return a string giving the XML version of the source document (normally \c "1.0") or \c NOTHING if none is present

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code *string value = xmlreader.xmlVersion(); @endcode
 */
__7_ string xmlVersion();
};
}
}
