//! Qore namespace
namespace Qore {
//! Oracle namespace
namespace Oracle {
//! Implementation of AQ Queue.
/**@par Restrictions:
    @ref Qore::PO_NO_DATABASE

* AQ messages notification with Oracle Client 10gR2 or above
 */
class AQQueue {

public:
//! Commit the transaction.
/** Confirm all posted or fetched messages.
 */
bool commit();

public:
//! Construct a queue object for posting/getting messages from a datasource description string
/** @param queueName a mandatory name of the queue in the database
    @param typeName a mandatory name of the type for this queue in the DB
    @param desc a datasource description string in the format that can be parsed by parse_datasource()

    @throw AQQUEUE-CONSTRUCTOR-ERROR invalid arguments to constructor
    @throw DBI-OPTION-ERROR unknown or unsupported option passed to the Oracle driver
 */
 constructor(string queueName, string typeName, string desc);

public:
//! Construct a queue object for posting/getting messages.
/** @param queueName a mandatory name of the queue in the database
    @param typeName a mandatory name of the type for this queue in the DB
    @param opts a hash giving parameters for the new dedicated connection to the Oracle database server with the following possible keys (the \c "type" key is optional, but if present, must be \c "oracle"):
    - \c type: (@ref string_or_nothing_type "*string") The name of the database driver to use; this key is optional, but if present must be \c "oracle" or an exception will be raised
    - \c user: (@ref string "string") The user name for the new connection
    - \c pass: (@ref string "string") The password for the new connection
    - \c db: (@ref string "string") The database name for the new connection
    - \c charset: (@ref string_or_nothing_type "*string") The Oracle-specific name of the character encoding to use for the new connection. If no value is passed for this parameter, then the Oracle character encoding corresponding to the default character encoding for the %Qore process will be used instead.
    - \c host: (@ref string_or_nothing_type "*string") The host name for the new connection
    - \c port: (@ref softint_type "softint") The port number for the new connection. If this key is present and is 0 then an exception will be raised.
    - \c options: (@ref hash_or_nothing_type "*hash") Any options for the new connection

    @throw AQQUEUE-CONSTRUCTOR-ERROR invalid arguments to constructor
    @throw DBI-OPTION-ERROR unknown or unsupported option passed to the Oracle driver
 */
 constructor(string queueName, string typeName, hash opts);

public:
//! Copying objects of this class is not supported, an exception will be thrown
/**@throw AQQUEUE-COPY-ERROR copying AQQueue objects is not yet supported
 */
 copy();

public:
//! stops any active subscriptions and destroys the object
/***/
 destructor();

public:
//! Get message from queue
/** @param timeout the timeout in seconds; 0 = no timeout

    @retval AQMessage or NOTHING if no message was available in the timeout period
 */
__7_ AQMessage getMessage(softint timeout = 0);

public:
//! Automatically called when a new message is received on the queue when there is an active subscription
/** A method which is called when there is a new message in the queue.

    The default implementation throws an exception; this method must be overridden in subclasses to receive messages;
    use the @ref AQQueue::getMessage() method to receive the message

    @throw AQQUEUE-ONASYNCMESSAGE-ERROR abstract implementation of AQQueue::onAsyncMessage called
 */
nothing onAsyncMessage();

public:
//! Automatically called when the connection is externally disconnected
/** A method which is called when there is a subscription but the periodical check for a live connection fails.

    The default implementation throws an exception; this method must be overridden in subclasses to receive messages;
    use the @ref AQQueue::startSubscription() method to restore the subscription for example

    @throw AQQUEUE-ONSUBSCRIPTIONLOST-ERROR abstract implementation of AQQueue::onSubscriptionLost called
 */
nothing onSubscriptionLost();

public:
//! Post the AQMessage into the queue.
/** @param msg a AQMessage instance
 */
nothing post(AQMessage msg);

public:
//! Post the plain Oracle object hash into the queue.
/** @param msg a qore hash (Oracle object)
 *
 * The hash/object will use default message properties. Use post() with an
 * AQMessage instance if you need to modify the message defaults.
 */
nothing post(hash msg);

public:
//! Rollback the transaction.
/** Rollback all posted or fetched messages.
 */
bool rollback();

public:
//! Subscribe for asynchronous messages notifications
/** @param port an optional non-negative port number (0 = use the default port)
    @param timeout an optional timeout value in seconds; any positive value in seconds is valid. The value 0 is accepted and means that it does not wait for messages and returns immediately if no messages are available.  The value -1 is accepted and means it waits for ever (until a message is available in the queue)

    @throw AQQUEUE-ERROR port or timeout value invalid; subscription already active for this object
 */
nothing startSubscription(int port = 0, int timeout = 0);

public:
//! Unsubscribe for asynchronous messages notifications.
/** If no subscription is active, no action is taken

    @throw AQQUEUE-ERROR oracle error stopping the subscription
 */
nothing stopSubscription();
};
}
}
