//! Qore namespace
namespace Qore {
//! Oracle namespace
namespace Oracle {
//! Implementation of message to enqueue/dequeue from/to queues
/** @par Restrictions:
    @ref Qore::PO_NO_DATABASE

Advanced properties can be set to base hash.
 */
class AQMessage {

public:
//! Create a message object.
/** @param obj \c hash, or \c NOTHING. An Oracle object created with \c bindOracleObject
 */
 constructor(__7_ hash obj);

public:
//! Return the number of attempts that have been made to dequeue the message.
/** @retval int the count of dequeue attempts
 */
int getAttemptCount();

public:
//! Get the correlation identifier of the message.
/** @retval string a correlation identifier
 */
string getCorrelation();

public:
//! Return the number of seconds that a message is delayed for dequeuing.
/** @retval int a delay value
 */
int getEnqueueDelay();

public:
//! Return the duration that the message is available for dequeuing.
/** @param int seconds count
 */
int getExpiration();

public:
//! Get the qore hash from the message (Oracle NTY object)
/** @retval hash or NOTHING
 */
__7_ hash getObject();

public:
//! Return the priority of the message.
/** @retval int a message priority
 */
int getPriority();

public:
//! Return the state of the message at the time of the dequeue.
/** @retval int a state of message
 *
 * These constants are defined to handle this status:
 * - OCI_UNKNOWN : the function has failed to get the message state
 * - OCI_AMS_READY : the message is ready to be processed
 * - OCI_AMS_WAITING : the message delay has not yet completed
 * - OCI_AMS_PROCESSED : the message has been processed
 * - OCI_AMS_EXPIRED : the message has moved to exception queue
 */
int getState();

public:
//! Set the correlation identifier of the message.
/** @param c a message correlation text
 *
 * Special pattern matching characters, such as "%" or "_" can be used.
 * If more than one message satisfies the pattern, the order of dequeuing
 * is undetermined.
 */
nothing setCorrelation(string c);

public:
//! Set the number of seconds to delay the enqueued message.
/** @param delay Delay in seconds
 *
 * The delay represents the number of seconds after which a message is
 * available for dequeuing. When the message is enqueued, its state is
 * set to \c OCI_AMS_WAITING. When the delay expires, its state
 * is set to \c OCI_AMS_READY.
 *
 * If parameter 'value' is set to zero (default value), the message will
 * be immediately available for dequeuing
 */
nothing setEnqueueDelay(int delay);

public:
//! Set the duration that the message is available for dequeuing.
/** @param exp a expiration in seconds or -1
 *
 * This parameter is an offset from the delay. While waiting for
 * expiration, the message state is set to \c OCI_AMS_READY.
 * If the message is not dequeued before it expires, it will be
 * moved to the exception queue with the state \c OCI_AMS_EXPIRED.
 * If parameter 'value' is set to -1 (default value), the message will not expire
 */
nothing setExpiration(int exp);

public:
//! Set \c hash as Oracle object for this message instance
/** @param obj \c hash, or \c NOTHING. An Oracle object created with \c bindOracleObject
 */
nothing setObject(hash h);

public:
//! Set the priority of the message.
/** @param prio a priority value
 *
 * The priority can be any number, including negative numbers.
 * A smaller number indicates higher priority.
 * Default value is zero.
 */
nothing setPriority(int prio);
};
/** @defgroup aqmessage_status_constants AQ Message Status Constants
///@{
*/

//! Expired status
    const OCI_AMS_EXPIRED = 4;
//! Processed status
    const OCI_AMS_PROCESSED = 3;
//! Ready status
    const OCI_AMS_READY = 1;
//! Waiting status
    const OCI_AMS_WAITING = 2;
//! Unknown status
    const OCI_UNKNOWN = 0;
/** @} */
}
}
