/*
  sybase.cc

  Sybase DB layer for QORE
  uses Sybase OpenClient C library

  Qore Programming language

  Copyright (C) 2003 - 2010 Qore Technologies, sro

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "sybase.h"

#include "minitest.hpp"

#include <ctpublic.h>
#include <assert.h>
#include <ctype.h>
#include <memory>
#include <string>
#include <vector>

#include "connection.h"
#include "encoding_helpers.h"

#ifndef QORE_MONOLITHIC
#ifdef SYBASE
DLLEXPORT char qore_module_name[] = "sybase";
DLLEXPORT char qore_module_description[] = "Sybase database driver";
#else
DLLEXPORT char qore_module_name[] = "freetds";
DLLEXPORT char qore_module_description[] = "FreeTDS-based database driver for MS-SQL Server and Sybase";
#endif
DLLEXPORT char qore_module_version[] = PACKAGE_VERSION;
DLLEXPORT char qore_module_author[] = "Qore Technologies";
DLLEXPORT char qore_module_url[] = "http://qore.org";
DLLEXPORT int qore_module_api_major = QORE_MODULE_API_MAJOR;
DLLEXPORT int qore_module_api_minor = QORE_MODULE_API_MINOR;
DLLEXPORT qore_module_init_t qore_module_init = sybase_module_init;
DLLEXPORT qore_module_ns_init_t qore_module_ns_init = sybase_module_ns_init;
DLLEXPORT qore_module_delete_t qore_module_delete = sybase_module_delete;
DLLEXPORT qore_license_t qore_module_license = QL_LGPL;
#endif

static DBIDriver* DBID_SYBASE;

// capabilities of this driver
int DBI_SYBASE_CAPS = DBI_CAP_TRANSACTION_MANAGEMENT 
   | DBI_CAP_CHARSET_SUPPORT
   | DBI_CAP_LOB_SUPPORT
   | DBI_CAP_STORED_PROCEDURES
   | DBI_CAP_BIND_BY_VALUE
   | DBI_CAP_BIND_BY_PLACEHOLDER
#ifdef _QORE_HAS_DBI_EXECRAW
   | DBI_CAP_HAS_EXECRAW 
#endif
   ;

#ifdef DEBUG
// exported
AbstractQoreNode* runSybaseTests(const QoreListNode *params, ExceptionSink *xsink) {
   minitest::result res = minitest::execute_all_tests();
   if (res.all_tests_succeeded) {
      printf("************************************************\n");
      printf("Sybase module: %d tests succeeded\n", res.sucessful_tests_count);
      printf("************************************************\n");
      return 0;
   }

   xsink->raiseException("SYBASE-TEST-FAILURE", "Sybase test in file %s, line %d threw an exception.",
			 res.failed_test_file, res.failed_test_line);
   return 0;
}

AbstractQoreNode* runRecentSybaseTests(const QoreListNode *params, ExceptionSink *xsink) {
   minitest::result res = minitest::test_last_changed_files(1);
   if (res.all_tests_succeeded) {
      printf("************************************************\n");
      printf("Sybase module: %d recent tests succeeded\n", res.sucessful_tests_count);
      printf("************************************************\n");
      return 0;
   }
   
   xsink->raiseException("SYBASE-TEST-FAILURE", "Sybase test in file %s, line %d threw an exception.",
			 res.failed_test_file, res.failed_test_line);
   return 0;
}
#endif

static int sybase_open(Datasource *ds, ExceptionSink *xsink) {
   // username is a required parameter
   if (!ds->getUsername()) {
      xsink->raiseException("DATASOURCE-MISSING-USERNAME", "Datasource has an empty username parameter");
      return -1;
   }

   // DB name is a required parameter
   if (!ds->getDBName()) {
      xsink->raiseException("DATASOURCE-MISSING-DBNAME", "Datasource has an empty dbname parameter");
      return -1;
   }

   // set the encoding for the connection
   if (ds->getDBEncoding()) {
      const QoreEncoding *enc = name_to_QoreEncoding(ds->getDBEncoding());
      ds->setQoreEncoding(enc);
   } 
   else {
      const char *enc = QoreEncoding_to_SybaseName(QCS_DEFAULT);
      // if the encoding cannot be mapped, throw a Qore-language exception and return
      if (!enc) {
	 xsink->raiseException("DBI:SYBASE:UNKNOWN-CHARACTER-SET", "cannot find the Sybase character encoding equivalent for '%s'", QCS_DEFAULT->getCode());
	 return -1;
      }
      ds->setDBEncoding(enc);
      ds->setQoreEncoding(QCS_DEFAULT);
   }
  
   // create the connection object
   std::auto_ptr<connection> sc(new connection(ds, xsink));
   if (*xsink)
      return -1;
  
#ifdef QORE_HAS_DATASOURCE_PORT
   int port = ds->getPort();
#else
   int port = 0;
#endif

   if (port && !ds->getHostName()) {
      xsink->raiseException("DBI:SYBASE:CONNECT-ERROR", "port is set to %d, but no hostname is set; both hostname and port must be set to override the interfaces file", port);
      return -1;
   }

   if (!port && ds->getHostName()) {
      xsink->raiseException("DBI:SYBASE:CONNECT-ERROR", "hostname is set to '%s', but no port is set; both hostname and port must be set to override the interfaces file", ds->getHostName());
      return -1;
   }

   // make the actual connection to the database
   sc->init(ds->getUsername(), ds->getPassword() ? ds->getPassword() : "", ds->getDBName(), ds->getDBEncoding(), ds->getQoreEncoding(), ds->getHostName(), port, xsink);
   // return with an error if it didn't work
   if (*xsink)
      return -1;

   // set the private data
   ds->setPrivateData(sc.release());

   // return 0 for OK
   return 0;
}

static int sybase_close(Datasource *ds) {
   connection* sc = (connection*)ds->getPrivateData();
   ds->setPrivateData(0);
   delete sc;

   return 0;
}

//------------------------------------------------------------------------------
static AbstractQoreNode* sybase_select(Datasource *ds, const QoreString *qstr, const QoreListNode *args, ExceptionSink *xsink) {
   connection *conn = (connection*)ds->getPrivateData();
   return conn->exec(qstr, args, xsink);
}

static AbstractQoreNode* sybase_select_rows(Datasource *ds, const QoreString *qstr, const QoreListNode *args, ExceptionSink *xsink) {
   connection *conn = (connection*)ds->getPrivateData();
   //printd(5, "sybase_select_rows(ds=%08p, qstr='%s', args=%08p)\n", ds, qstr->getBuffer(), args);
   return conn->exec_rows(qstr, args, xsink);
}

static AbstractQoreNode* sybase_exec(Datasource *ds, const QoreString *qstr, const QoreListNode *args, ExceptionSink *xsink) {
   connection *conn = (connection*)ds->getPrivateData();
   return conn->exec(qstr, args, xsink);
}

#ifdef _QORE_HAS_DBI_EXECRAW
static AbstractQoreNode* sybase_execRaw(Datasource *ds, const QoreString *qstr, ExceptionSink *xsink) {
   connection *conn = (connection*)ds->getPrivateData();
   return conn->execRaw(qstr, xsink);
}
#endif

static int sybase_commit(Datasource *ds, ExceptionSink *xsink) {
   connection* conn = (connection*)ds->getPrivateData();
   return conn->commit(xsink);
}

static int sybase_rollback(Datasource *ds, ExceptionSink *xsink) {
   connection* conn = (connection*)ds->getPrivateData();
   return conn->rollback(xsink);
}

static AbstractQoreNode *sybase_get_client_version(const Datasource *ds, ExceptionSink *xsink) {
   context m_context(xsink);
   if (!m_context)
      return 0;

   return m_context.get_client_version(xsink);
}

static AbstractQoreNode *sybase_get_server_version(Datasource *ds, ExceptionSink *xsink) {
   connection* conn = (connection*)ds->getPrivateData();
   return conn->get_server_version(xsink);
}

/*
// constants are not needed now as specifying placeholder buffer types is not necessary
static void add_constants(QoreNamespace* ns) {
}

#ifdef SYBASE
static QoreNamespace sybase_ns("Sybase");
#else
static QoreNamespace sybase_ns("FreeTDS");
#endif
 
static void init_namespace() {
   sybase_ns.addConstant("CS_CHAR_TYPE", new QoreBigIntNode(CS_CHAR_TYPE));
   sybase_ns.addConstant("CS_BINARY_TYPE", new QoreBigIntNode(CS_BINARY_TYPE));
   sybase_ns.addConstant("CS_LONGCHAR_TYPE", new QoreBigIntNode(CS_LONGCHAR_TYPE));
   sybase_ns.addConstant("CS_LONGBINARY_TYPE", new QoreBigIntNode(CS_LONGBINARY_TYPE));
   sybase_ns.addConstant("CS_TEXT_TYPE", new QoreBigIntNode(CS_TEXT_TYPE));
   sybase_ns.addConstant("CS_IMAGE_TYPE", new QoreBigIntNode(CS_IMAGE_TYPE));
   sybase_ns.addConstant("CS_TINYINT_TYPE", new QoreBigIntNode(CS_TINYINT_TYPE));
   sybase_ns.addConstant("CS_SMALLINT_TYPE", new QoreBigIntNode(CS_SMALLINT_TYPE));
   sybase_ns.addConstant("CS_INT_TYPE", new QoreBigIntNode(CS_INT_TYPE));
   sybase_ns.addConstant("CS_REAL_TYPE", new QoreBigIntNode(CS_REAL_TYPE));
   sybase_ns.addConstant("CS_FLOAT_TYPE", new QoreBigIntNode(CS_FLOAT_TYPE));
   sybase_ns.addConstant("CS_BIT_TYPE", new QoreBigIntNode(CS_BIT_TYPE));
   sybase_ns.addConstant("CS_DATETIME_TYPE", new QoreBigIntNode(CS_DATETIME_TYPE));
   sybase_ns.addConstant("CS_DATETIME4_TYPE", new QoreBigIntNode(CS_DATETIME4_TYPE));
   sybase_ns.addConstant("CS_MONEY_TYPE", new QoreBigIntNode(CS_MONEY_TYPE));
   sybase_ns.addConstant("CS_MONEY4_TYPE", new QoreBigIntNode(CS_MONEY4_TYPE));
   sybase_ns.addConstant("CS_NUMERIC_TYPE", new QoreBigIntNode(CS_NUMERIC_TYPE));
   sybase_ns.addConstant("CS_DECIMAL_TYPE", new QoreBigIntNode(CS_DECIMAL_TYPE));
   sybase_ns.addConstant("CS_VARCHAR_TYPE", new QoreBigIntNode(CS_VARCHAR_TYPE));
   sybase_ns.addConstant("CS_VARBINARY_TYPE", new QoreBigIntNode(CS_VARBINARY_TYPE));
   sybase_ns.addConstant("CS_LONG_TYPE", new QoreBigIntNode(CS_LONG_TYPE));
   sybase_ns.addConstant("CS_SENSITIVITY_TYPE", new QoreBigIntNode(CS_SENSITIVITY_TYPE));
   sybase_ns.addConstant("CS_BOUNDARY_TYPE", new QoreBigIntNode(CS_BOUNDARY_TYPE));
   sybase_ns.addConstant("CS_VOID_TYPE", new QoreBigIntNode(CS_VOID_TYPE));
   sybase_ns.addConstant("CS_USHORT_TYPE", new QoreBigIntNode(CS_USHORT_TYPE));
   sybase_ns.addConstant("CS_UNICHAR_TYPE", new QoreBigIntNode(CS_UNICHAR_TYPE));
#ifdef CS_BLOB_TYPE
   sybase_ns.addConstant("CS_BLOB_TYPE", new QoreBigIntNode(CS_BLOB_TYPE));
#endif
#ifdef CS_DATE_TYPE
   sybase_ns.addConstant("CS_DATE_TYPE", new QoreBigIntNode(CS_DATE_TYPE));
#endif
#ifdef CS_TIME_TYPE
   sybase_ns.addConstant("CS_TIME_TYPE", new QoreBigIntNode(CS_TIME_TYPE));
#endif
#ifdef CS_UNITEXT_TYPE
   sybase_ns.addConstant("CS_UNITEXT_TYPE", new QoreBigIntNode(CS_UNITEXT_TYPE));
#endif
#ifdef CS_BIGINT_TYPE
   sybase_ns.addConstant("CS_BIGINT_TYPE", new QoreBigIntNode(CS_BIGINT_TYPE));
#endif
#ifdef CS_USMALLINT_TYPE
   sybase_ns.addConstant("CS_USMALLINT_TYPE", new QoreBigIntNode(CS_USMALLINT_TYPE));
#endif
#ifdef CS_UINT_TYPE
   sybase_ns.addConstant("CS_UINT_TYPE", new QoreBigIntNode(CS_UINT_TYPE));
#endif
#ifdef CS_UBIGINT_TYPE
   sybase_ns.addConstant("CS_UBIGINT_TYPE", new QoreBigIntNode(CS_UBIGINT_TYPE));
#endif
#ifdef CS_XML_TYPE
   sybase_ns.addConstant("CS_XML_TYPE", new QoreBigIntNode(CS_XML_TYPE));
#endif
}
 */

QoreStringNode *sybase_module_init() {
   QORE_TRACE("sybase_module_init()");

   // init_namespace();
   
#ifdef DEBUG
   builtinFunctions.add("runSybaseTests", runSybaseTests, QDOM_DATABASE);
   builtinFunctions.add("runRecentSybaseTests", runRecentSybaseTests, QDOM_DATABASE);
#endif
   
   // register driver with DBI subsystem
   qore_dbi_method_list methods;
   methods.add(QDBI_METHOD_OPEN, sybase_open);
   methods.add(QDBI_METHOD_CLOSE, sybase_close);
   methods.add(QDBI_METHOD_SELECT, sybase_select);
   methods.add(QDBI_METHOD_SELECT_ROWS, sybase_select_rows);
   methods.add(QDBI_METHOD_EXEC, sybase_exec);
#ifdef _QORE_HAS_DBI_EXECRAW
   methods.add(QDBI_METHOD_EXECRAW, sybase_execRaw);
#endif
   methods.add(QDBI_METHOD_COMMIT, sybase_commit);
   methods.add(QDBI_METHOD_ROLLBACK, sybase_rollback);
   methods.add(QDBI_METHOD_GET_CLIENT_VERSION, sybase_get_client_version);
   methods.add(QDBI_METHOD_GET_SERVER_VERSION, sybase_get_server_version);
   
#ifdef SYBASE
   DBID_SYBASE = DBI.registerDriver("sybase", methods, DBI_SYBASE_CAPS);
#else
   DBID_SYBASE = DBI.registerDriver("freetds", methods, DBI_SYBASE_CAPS);
#endif

   return 0;
}

void sybase_module_ns_init(QoreNamespace *rns, QoreNamespace *qns) {
/*
  // this is commented out because the constants are not needed (or documented) at the moment
  // maybe later we can use them
   QORE_TRACE("sybase_module_ns_init()");
   qns->addInitialNamespace(sybase_ns.copy());
*/
}

void sybase_module_delete() {
   QORE_TRACE("sybase_module_delete()");
}
