/***************************************************************************
  qgsrubberband3d.h
  --------------------------------------
  Date                 : June 2021
  Copyright            : (C) 2021 by Martin Dobias
  Email                : wonder dot sk at gmail dot com
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QGSRUBBERBAND3D_H
#define QGSRUBBERBAND3D_H

#include "qgis_3d.h"

#include <QColor>

#define SIP_NO_FILE

/// @cond PRIVATE

//
//  W A R N I N G
//  -------------
//
// This file is not part of the QGIS API.  It exists purely as an
// implementation detail.  This header file may change from version to
// version without notice, or even be removed.
//

#include "qgslinestring.h"

class QgsWindow3DEngine;
class QgsLineMaterial;
class Qgs3DMapSettings;
class QgsBillboardGeometry;
class QgsMarkerSymbol;
class QgsPoint3DBillboardMaterial;

namespace Qt3DCore
{
  class QEntity;
#if QT_VERSION >= QT_VERSION_CHECK( 6, 0, 0 )
  class QBuffer;
  class QGeometry;
  class QAttribute;
#endif
} // namespace Qt3DCore

namespace Qt3DRender
{
#if QT_VERSION < QT_VERSION_CHECK( 6, 0, 0 )
  class QBuffer;
  class QGeometry;
  class QAttribute;
#endif
  class QGeometryRenderer;
} // namespace Qt3DRender

/**
 * \ingroup 3d
 * Rubber band implementation for use in 3D map views.
 *
 * Coordinates are expected to be passed in map CRS and the 3D entity generated by this
 * class will be attached to the parentEntity given in the constructor (normally this
 * should be the root entity, i.e. map scene object).
 *
 * \note Currently only supports multi point and linestring geometry.
 * \since QGIS 3.20
 */
class _3D_EXPORT QgsRubberBand3D
{
  public:
    //! Icons
    enum MarkerType
    {

      /**
       * A box is used to highlight points (□)
       */
      Square,

      /**
       * A circle is used to highlight points (○)
       */
      Circle
    };

    QgsRubberBand3D( Qgs3DMapSettings &map, QgsWindow3DEngine *engine, Qt3DCore::QEntity *parentEntity, Qgis::GeometryType geometryType = Qgis::GeometryType::Line );
    ~QgsRubberBand3D();

    //! Returns the rubber band width in pixels
    float width() const;

    //! Sets the rubber band width in pixels
    void setWidth( float width );

    //! Returns the rubber band color
    QColor color() const;

    //! Sets the rubber band color.
    void setColor( QColor color );

    /**
     * Sets the \a marker type to highlight point geometries and line vertices.
     */
    void setMarkerType( MarkerType marker );

    /**
     * Returns the current marker type to highlight point geometries and line vertices.
     */
    MarkerType markerType() const;

    void reset();

    void addPoint( const QgsPoint &pt );

    void setPoints( const QgsLineString &points );

    void removeLastPoint();

    void moveLastPoint( const QgsPoint &pt );

    //! Sets whether the marker on the last vertex is displayed. We typically do not want it displayed while it is still tracked by the mouse.
    void setHideLastMarker( bool hide ) { mHideLastMarker = hide; }

    bool isEmpty() const { return mLineString.isEmpty(); }

  private:
    void updateGeometry();
    void updateMarkerMaterial();

  private:
    QgsLineString mLineString;
    bool mHideLastMarker = false;

    Qgs3DMapSettings *mMapSettings = nullptr; // not owned
    QgsWindow3DEngine *mEngine = nullptr;
    Qgis::GeometryType mGeometryType = Qgis::GeometryType::Line;

    //! point and vertex marker type
    MarkerType mMarkerType = Circle;
    float mWidth = 3.f;
    QColor mColor = Qt::red;

    Qt3DCore::QEntity *mLineEntity = nullptr;   // owned by parentEntity (from constructor)
    Qt3DCore::QEntity *mMarkerEntity = nullptr; // owned by parentEntity (from constructor)

    // all these are owned by mLineEntity
    Qt3DRender::QGeometryRenderer *mLineGeomRenderer = nullptr;
#if QT_VERSION < QT_VERSION_CHECK( 6, 0, 0 )
    Qt3DRender::QGeometry *mGeometry = nullptr;
    Qt3DRender::QAttribute *mPositionAttribute = nullptr;
    Qt3DRender::QAttribute *mIndexAttribute = nullptr;
#else
    Qt3DCore::QGeometry *mGeometry = nullptr;
    Qt3DCore::QAttribute *mPositionAttribute = nullptr;
    Qt3DCore::QAttribute *mIndexAttribute = nullptr;
#endif
    QgsLineMaterial *mLineMaterial = nullptr;

    // and these are owned by mMarkerEntity
    Qt3DRender::QGeometryRenderer *mMarkerGeometryRenderer = nullptr;
    QgsBillboardGeometry *mMarkerGeometry = nullptr;
    QgsPoint3DBillboardMaterial *mMarkerMaterial = nullptr;

    std::unique_ptr<QgsMarkerSymbol> mMarkerSymbol;

    // Disable copying as we have pointer members.
    QgsRubberBand3D( const QgsRubberBand3D & ) = delete;
    QgsRubberBand3D &operator=( const QgsRubberBand3D & ) = delete;
};

/// @endcond

#endif // QGSRUBBERBAND3D_H
