
/* qmmatch.c: TA matching routines for the Q machine */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qdefs.h"

/* The nextstate() function looks up a transition on a given symbol with a
   given type in the transition table. It returns the transition with
   the type which is the "smallest" supertype of the given type. This is
   implemented by a kind of mergesort algorithm which compares the list
   of types for the symbol in the transition table to the list of supertypes
   of the given type. */

/* Note: If your compiler supports it, make sure that nextstate() and the
   following defaultstate() function are inlined. This considerably improves
   performance of the interpreter. */

inline
static
nextstate(int s, int type, int fno)
{
  int k = statetb[s].trans, left = k, right = k+statetb[s].ntrans;

  if (!statetb[s].ntrans) return 0;
  
  if (statetb[s].ntrans <= 10) {
    /* linear search */
    for (; fno > transtb[k].fno && k < right; k++)
      ;
  } else {
    /* binary search */
    int l = left, r = right-1;
    while (l <= r) {
      k = (l+r)>>1;
      if (fno < transtb[k].fno)
	r = k-1;
      else if (fno > transtb[k].fno)
	l = k+1;
      else
	break;
    }
    while (k > left && transtb[k-1].fno == fno)
      k--;
  }

  if (k >= right || transtb[k].fno != fno)
    return 0;
		
  for (;;)
    if (type > transtb[k].type)
      type = symtb[type].type;
    else if (type < transtb[k].type) {
      if (++k >= right || transtb[k].fno != fno)
	return 0;
    } else
      return transtb[k].next;
}

/* The defaultstate() function looks up the default transition on a given
   type in the transition table. */

inline
static
defaultstate(int s, int type)
{
  int k = statetb[s].trans, left = k, right = k+statetb[s].ntrans;

  if (k >= right || transtb[k].fno != 0) return 0;
	
  for (;;)
    if (type > transtb[k].type)
      type = symtb[type].type;
    else if (type < transtb[k].type) {
      if (++k >= right || transtb[k].fno != 0)
	return 0;
    } else
      return transtb[k].next;
}

/* Check whether there is a non-default transition in the given state. */

inline
static
nondefault(int s)
{
  int k = statetb[s].trans, right = k+statetb[s].ntrans;
  for (; k < right && transtb[k].fno <= 0 && transtb[k].type <= 0; k++) ;
  return k < right;
}

/* Check whether there is a transition for the given type in the given
   state. */

inline
static
typedtrans(int s, int type)
{
  int k = statetb[s].trans, right = k+statetb[s].ntrans;
  for (; k < right && transtb[k].type != type; k++) ;
  if (k < right)
    return transtb[k].next;
  else
    return 0;
}

/* Check whether there is a transition for a virtual constructor of the given
   type in the given state. */

inline
static
virtual(int s, int type)
{
  int k = statetb[s].trans, right = k+statetb[s].ntrans, u = -1, v = -1;
  /* check for a non-var transition on the given type */
  for (; k < right; k++)
    if (transtb[k].type != type || transtb[k].fno <= 0)
      continue;
    else if (transtb[k].fno == APPOP)
      /* application */
      u = k;
    else if (symtb[transtb[k].fno].flags&VIRT)
      /* virtual constructor */
      v = k;
 test:
  if (v >= 0)
    /* bingo, we found a virtual constructor */
    return 1;
  else if (u >= 0) {
    /* it's an application, so go to the next state and check whether we have
       a transition on a virtual constructor there */
    s = transtb[u].next;
    k = statetb[s].trans; right = k+statetb[s].ntrans;
    u = v = -1;
    for (; k < right; k++)
      if (transtb[k].fno <= 0)
	continue;
      else if (transtb[k].fno == APPOP)
	u = k;
      else if (symtb[transtb[k].fno].flags&VIRT)
	v = k;
    goto test;
  }
  return 0;
}

/* Matching mode table. To keep track of special args which must be evaluated
   recursively by the pattern matcher, we distinguish the following different
   matching "modes":

   0 ("spine"): This mode is used while traversing the toplevel spine, where
   we are looking for non-special arguments in which recursive evals might
   have to be performed.

   -1 ("special arg"): This mode is entered from mode 0 when we begin
   traversing a _special_ argument of the toplevel spine, where we just
   perform ordinary matching without any recursive evaluation. This mode is
   always inherited by subterms.

   1 ("nonspecial arg"): This mode is entered from mode 0 when we begin
   traversing a _non-special_ argument of the toplevel spine, where we keep
   looking for embedded special subterms which might have to be evaluated.

   2 ("eval"): This mode is entered from mode 1 if we encounter a special
   argument. If the current matching state has transitions on non-variable
   symbols then the current subterm is evaluated and the resulting normal form
   replaces the original expression. Matching then continues in mode 1.

   The following transition table specifies for each given mode >=0 the
   successor modes for the left and right subterm of a function application
   (for other binary nodes, i.e., [|] and (|), the subterms always inherit the
   mode of their parent nodes). Note that there are no transitions for mode 2
   since after the recursive eval the mode is always reset to mode 1. */

typedef struct { int left, right; } modes_t;
static modes_t nextmodes[2][2] = {
  /* NOTE: argv&1 of the left subterm determines whether we have a non-special
     (N) or special (S) arg. */
  /*                               N      S */
  /* mode == 0 (left,right) */ { {0,1}, {0,-1} },
  /* mode == 1 (left,right) */ { {1,1}, {1,2}  },
};

/* get the argv vector of the given expression */
#define get_argv(x) (((x)->fno==APPOP)?(x)->data.args.argv:symtb[(x)->fno].argv)

/* The matchx() function matches an expression x starting at the given state s
   in mode mode and returns the resulting end state (0 if the match fails). *y
   returns the new expression if x itself or some subterm of x is evaluated
   along the way, otherwise *y == x. */

inline
static
matchx(THREAD *thr, int s, int mode, int vmode, EXPR *x, EXPR **y)
{
  int s1, fno = x->fno, type = (x->argc)?0:x->type, virt = x->virt;
#ifdef DEBUG
  char fsym[MAXSTRLEN], tsym[MAXSTRLEN];
  printf("state %d, mode %d, symbol %s, type %s\n", s, mode,
	 fno>=RESERVED?utf8_to_sys(pname(fsym,fno)):"_",
	 type?utf8_to_sys(pname(tsym,type)):"<none>");
#endif
  if (y) *y = x;
  if (!nondefault(s)) goto skip;
  if (mode == 2) {
    if (x->red) {
      /* reducible special subterm in non-special arg, evaluate recursively */
      EXPR *u, *v;
#ifdef DEBUG
      printf("state %d, mode %d, entering recursive eval: ", s, mode); printx(x);
      printf("\n");
#endif
      if (!eval(thr, x)) return 0;
      u = *--thr->xsp; u->refc--;
#ifdef DEBUG
      printf("state %d, mode %d, exiting recursive eval: ", s, mode); printx(u);
      printf("\n");
#endif
      s = matchx(thr, s, 1, 0, u, &v);
      if (u != v) qmfree(thr, qmnew(u));
      if (y) *y = v;
      return s;
    }
    mode = 1;
  }
  if (!vmode && mode &&
      type && (symtb[type].flags&VIRT) && virtual(s, type)) {
    /* this type has a view and we got transitions on virtual constructors, so
       generate the view and match against it in place of the original
       expression */
    int _mode = thr->mode, res;
    EXPR *f, *u, *v;
#ifdef DEBUG
    printf("state %d, mode %d, entering recursive eval: ", s, mode); printx(x);
    printf("\n");
#endif
    thr->mode = 1;
    f = funexpr(thr, UNPARSEOP);
    if (!f) {
      thr->mode = _mode;
      return 0;
    } else if (!(u = qmnew(consexpr(thr, APPOP, f, x)))) {
      qmfree(thr, qmnew(f));
      thr->mode = _mode;
      return 0;
    }
    thr->mode = _mode;
    res = eval(thr, u);
    x->refc++;
    qmfree(thr, u);
    x->refc--;
    if (!res) return 0;
    u = *--thr->xsp;
#ifdef DEBUG
    printf("state %d, mode %d, exiting recursive eval: ", s, mode); printx(u);
    printf("\n");
#endif
    if (u->fno == APPOP && u->data.args.x1->fno == QUOTEOP) {
      v = qmnew(u->data.args.x2);
      qmfree(thr, u);
      u = v;
      s = matchx(thr, s, 1, 1, u, &v);
      if (u != v)
	qmfree(thr, u);
      else
	u->refc--;
      if (y) *y = v;
      return s;
    } else {
      qmfree(thr, u);
    }
  }
 skip:
  switch (fno) {
  case CONSOP: case PAIROP:
    if ((s1 = nextstate(s, type, fno))) {
      EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2, *y1 = x1, *y2 = x2;
      if (mode==0) mode = 1;
      (s = matchx(thr, s1, mode, 0, x1, &y1)) &&
	(s = matchx(thr, s, mode, 0, x2, &y2));
      if (s) {
	if ((x1 != y1 || x2 != y2) && !(x = consexpr(thr, fno, y1, y2))) {
	  if (x1 != y1) qmfree(thr, qmnew(y1));
	  if (x2 != y2) qmfree(thr, qmnew(y2));
	  s = 0;
	}
      } else {
	if (x1 != y1) qmfree(thr, qmnew(y1));
	if (x2 != y2) qmfree(thr, qmnew(y2));
      }
    } else
      s = defaultstate(s, type);
    break;
  case APPOP:
    if ((s1 = nextstate(s, type, fno))) {
      EXPR *x1 = x->data.args.x1, *x2 = x->data.args.x2, *y1 = x1, *y2 = x2;
      modes_t modes;
      modes.left = mode; modes.right = mode;
      if (mode >= 0) modes = nextmodes[mode][get_argv(x1)&1];
      (s = matchx(thr, s1, modes.left, 0, x1, &y1)) &&
	(s = matchx(thr, s, modes.right, 0, x2, &y2));
      if (s) {
	if ((x1 != y1 || x2 != y2) && !(x = consexpr(thr, fno, y1, y2))) {
	  if (x1 != y1) qmfree(thr, qmnew(y1));
	  if (x2 != y2) qmfree(thr, qmnew(y2));
	  s = 0;
	}
      } else {
	if (x1 != y1) qmfree(thr, qmnew(y1));
	if (x2 != y2) qmfree(thr, qmnew(y2));
      }
    } else
      s = defaultstate(s, virt?0:type);
    break;
  case VECTOP: {
    int i, n = x->data.vect.n;
    EXPR **xv = x->data.vect.xv, **yv = NULL;
    if (mode==0) mode = 1;
    for (i = 0; i < n; i++)
      if ((s1 = nextstate(s, TUPLETYPE, PAIROP))) {
	EXPR *y;
	if ((s = matchx(thr, s1, mode, 0, xv[i], &y))) {
	  if (xv[i] != y) {
	    if (!yv) {
	      int j;
	      if (!(yv = malloc(n*sizeof(EXPR*)))) {
		qmfree(thr, qmnew(y));
		goto done;
	      }
	      for (j = 0; j < i; j++) yv[j] = qmnew(xv[j]);
	    }
	    yv[i] = qmnew(y);
	  } else if (yv)
	    yv[i] = qmnew(xv[i]);
	} else {
	  if (xv[i] != y) qmfree(thr, qmnew(y));
	  if (yv) {
	    int j;
	    for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	    free(yv);
	  }
	  goto done;
	}
      } else {
	s = defaultstate(s, TUPLETYPE);
	if (yv)
	  if (s) {
	    int j;
	    for (j = i; j < n; j++) yv[j] = qmnew(xv[j]);
	  } else {
	    int j;
	    for (j = 0; j < i; j++) qmfree(thr, yv[j]);
	    free(yv);
	  }
	goto done;
      }
    if ((s1 = nextstate(s, TUPLETYPE, VOIDOP)))
      s = s1;
    else
      s = defaultstate(s, TUPLETYPE);
    done:
    if (yv && !(x = vectexpr(thr, n, yv)))
      s = 0;
    break;
  }
  default:
    if ((s1 = nextstate(s, type, fno)))
      s = s1;
    else
      s = defaultstate(s, virt?0:type);
    break;
  }
  if (y) *y = x;
  return s;
}

/* interface functions */

/* match( thr, fno, x, rp, rc ) matches a term with top symbol fno and
   corresponding number of arguments taken from the beginning of the
   expression array x against the left-hand sides of rules, and sets *rp and
   *rc to the corresponding rule pointer and counter if a matching rule is
   found. Returns: nonzero if a matching rule was found, zero otherwise. If
   successful, x returns the modified arguments if any subterm was evaluated
   by the pattern matcher. Unmodified arguments are returned unchanged, with
   only their reference count incremented by 1. */

int match(THREAD *thr, int fno, EXPR **x, int **rp, int *rc)
{
  static int init = 0, nostate = 0;
  int s, type;
  EXPR *x0, *x1;
#ifdef DEBUG
  char fsym[MAXSTRLEN], tsym[MAXSTRLEN];
#endif
  if (!init) {
    init = 1;
    nostate = statetbsz == 0 || matchtbsz>0 && matchtb[0] == 0;
  }
  if (nostate || fno < RESERVED)
    return 0;
  switch (fno) {
  case CONSOP:
    type = LISTTYPE;
  match2:
    x0 = x[0]; x1 = x[1];
#ifdef DEBUG
    printf("state %d, mode %d, symbol %s, type %s\n", 0, 0,
	   fno>=RESERVED?utf8_to_sys(pname(fsym,fno)):"_",
	   type?utf8_to_sys(pname(tsym,type)):"<none>");
#endif
    if ((s = nextstate(0, type, fno)))
      (s = matchx(thr, s, 1, 0, x[0], &x[0])) &&
	(s = matchx(thr, s, 1, 0, x[1], &x[1]));
    else
      s = defaultstate(0, type);
    if (statetb[s].nrules <= 0) s = 0;
    if (!s) {
      if (x0 != x[0]) qmfree(thr, qmnew(x[0]));
      if (x1 != x[1]) qmfree(thr, qmnew(x[1]));
      x[0] = x[1] = NULL;
    } else {
      qmnew(x[0]); qmnew(x[1]);
    }
    break;
  case PAIROP:
    type = TUPLETYPE;
    goto match2;
  case APPOP:
    type = (x[0]->argc == 1)?x[0]->type:0;
    x0 = x[0]; x1 = x[1];
#ifdef DEBUG
    printf("state %d, mode %d, symbol %s, type %s\n", 0, 0,
	   fno>=RESERVED?utf8_to_sys(pname(fsym,fno)):"_",
	   type?utf8_to_sys(pname(tsym,type)):"<none>");
#endif
    if ((s = nextstate(0, type, fno))) {
      modes_t modes;
      modes = nextmodes[0][get_argv(x[0])&1];
      (s = matchx(thr, s, modes.left, 0, x[0], &x[0])) &&
	(s = matchx(thr, s, modes.right, 0, x[1], &x[1]));
    } else
      s = defaultstate(0, type);
    if (statetb[s].nrules <= 0) s = 0;
    if (!s) {
      if (x0 != x[0]) qmfree(thr, qmnew(x[0]));
      if (x1 != x[1]) qmfree(thr, qmnew(x[1]));
      x[0] = x[1] = NULL;
    } else {
      qmnew(x[0]); qmnew(x[1]);
    }
    break;
  default:
    type = !symtb[fno].argc?symtb[fno].type:0;
#ifdef DEBUG
    printf("state %d, mode %d, symbol %s, type %s\n", 0, 0,
	   fno>=RESERVED?utf8_to_sys(pname(fsym,fno)):"_",
	   type?utf8_to_sys(pname(tsym,type)):"<none>");
#endif
    if (!(s = nextstate(0, type, fno)))
      s = defaultstate(0, type);
    if (statetb[s].nrules <= 0) s = 0;
    break;
  }
#ifdef DEBUG
  printf(s?"match succeeded, final state %d\n":"match failed\n", s);
#endif
  if (s) {
    *rc = statetb[s].nrules;
    *rp = roffstb + statetb[s].roffs;
    return 1;
  } else
    return 0;
}

/* matchp( thr, s, x ) matches the expression *x, starting at the given state
   s. Returns nonzero if successful, zero otherwise. If successful, *x returns
   the (possibly modified) matched expression. */

int matchp(THREAD *thr, int s, EXPR **x)
{
  EXPR *y = *x;
  s = matchx(thr, s, 1, 0, *x, x);
#ifdef DEBUG
  printf(s?"match succeeded, final state %d\n":"match failed\n", s);
#endif
  if (!s) {
    if (*x != y) qmfree(thr, qmnew(*x));
    *x = NULL;
  } else
    qmnew(*x);
  return s;
}

/* matchtype(thr, fno, type) checks whether there is a match for an application
   of the given function to an expression of the given type. */

int matchtype(THREAD *thr, int fno, int type)
{
  static int init = 0, nostate = 0;
  int s;
#ifdef DEBUG
  char fsym[MAXSTRLEN], tsym[MAXSTRLEN];
#endif
  if (!init) {
    init = 1;
    nostate = statetbsz == 0 || matchtbsz>0 && matchtb[0] == 0;
  }
  if (nostate || fno < RESERVED)
    return 0;
  (s = nextstate(0, 0, APPOP)) &&
    (s = nextstate(s, 0, fno)) && (s = typedtrans(s, type));
#ifdef DEBUG
  if (s) printf("matched %s, type = %s\n", utf8_to_sys(pname(fsym,fno)),
		type?utf8_to_sys(pname(tsym,type)):"<none>");
#endif
  return s!=0;
}
