
%{ /* -*-C-*- */

/* qclex.l: lexical analyzer for the Q language */

/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* make sure we have flex */

#ifndef FLEX_SCANNER
#error "Sorry, this program requires flex."
#endif

#include "qcdefs.h"
#include "qc.h"

#undef yywrap
int yylineno;           /* the current line */

char *source = NULL;  	/* the source file name */

int context; /* identifier context, 1 = type identifier */

static char s1[MAXSTRLEN], s2[MAXSTRLEN];
int incmode;

static int wrapped = 0;

static int abufsz = 0, bufp = 0;
static char *buf = NULL;

static int checkid(int tok);
static comment(), string();
static void bigint();
static utf8_id(), utf8_id_or_sym(), utf8_qualid(), utf8_skip();

static char *strdupchk(char *s)
{
  if (!s || !(s = strdup(s)))
    fatal(qcmsg[MEM_OVF]);
  else
    return s;
}

/* prevent buffer overflows */

static ofchk(char *s)
{
  if (strlen(s) >= MAXSTRLEN)
    fatal(qcmsg[MEM_OVF]);
}

#ifdef HAVE_UNICODE

static inline char *
u8strind(char *s, size_t i)
{
  unsigned p = 0, q = 0;
 start:
  for (; *s && i > 0; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  break;
	case 0xe0:
	  q = 2;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0)
	    q = 3;
	  break;
	}
      }
      p = 0; if (q == 0) i--;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      if (--q == 0) {
	p = 0; i--;
      } else
	p++;
    } else {
      /* malformed char */
      i--; s -= p+1; p = q = 0;
    }
  }
  if (q > 0) {
    /* unterminated char */
    i--; s -= p; p = q = 0;
    goto start;
  }
  return s;
}

#ifdef HAVE_ICONV

#define CHUNKSZ 128

static iconv_t ic = (iconv_t)-2;

static char *toutf8(char *s)
{
  static char *mybuf = NULL;
  static size_t mybufsz = 0;

  size_t l = strlen(s);
  if (ic == (iconv_t)-2) {
    char *codeset = default_codeset?
      default_codeset:default_encoding();
    if (codeset && strcmp(codeset, "UTF-8"))
      ic = iconv_open("UTF-8", codeset);
    else
      ic = (iconv_t)-1;
  }
  if (mybufsz < l+1) {
    char *mybuf1 = realloc(mybuf, l+1);
    if (!mybuf1) return NULL;
    mybuf = mybuf1;
    mybufsz = l+1;
  }
  if (ic == (iconv_t)-1)
    return strcpy(mybuf, s);
  else {
    char *inbuf = s, *outbuf = mybuf;
    size_t inbytes = l, outbytes = mybufsz-1;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-mybuf;
	char *mybuf1 = realloc(mybuf, mybufsz+CHUNKSZ);
	if (!mybuf1) return NULL;
	mybuf = mybuf1;
	mybufsz += CHUNKSZ;
	outbuf = mybuf+k;
	outbytes += CHUNKSZ;
      } else {
	/* conversion error */
	return strcpy(mybuf, s);
      }
    /* terminate the output string */
    *outbuf = 0;
    return mybuf;
  }
}

#endif

#endif

static char *getbuf(FILE *fp)
{
  static char *mybuf = NULL;
  static size_t mybufsz = 0;

  char *t, *r;
  int l;

  if (!mybuf) {
    mybuf = malloc(MAXSTRLEN);
    if (!mybuf) return NULL;
    mybufsz = MAXSTRLEN;
  }

  *mybuf = 0; t = mybuf;
  while (fp && !feof(fp) && (r = fgets(t, MAXSTRLEN, fp)) && *t &&
	 t[(l = strlen(t))-1] != '\n') {
    /* try to enlarge the buffer: */
    int k = t-mybuf+l;
    char *mybuf1;
    if (mybuf1 = (char*)realloc(mybuf, mybufsz+MAXSTRLEN)) {
      mybuf = mybuf1;
      t = mybuf+k;
      mybufsz += MAXSTRLEN;
    } else
      return NULL;
  }
  return mybuf;
}

static char *mybuf = NULL, *mybufptr = NULL;

static inline int myinput(char *buf, int max_size)
{
 read:
  if (mybufptr && *mybufptr) {
    int l = strlen(mybufptr);
    if (l > max_size) l = max_size;
    memcpy(buf, mybufptr, l);
    mybufptr += l;
    return l;
  }
  if (mybuf) free(mybuf);
  mybuf = getbuf(yyin);
  if (!mybuf) fatal("memory overflow");
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  mybuf = toutf8(mybuf);
  if (!mybuf) fatal("memory overflow");
#endif
  mybuf = strdup(mybuf);
  if (!mybuf) fatal("memory overflow");
  mybufptr = mybuf;
  if (!*mybuf) return YY_NULL;
  goto read;
}

/* redefined input macro: */

#undef YY_INPUT
#define YY_INPUT(buf,result,max_size)\
	{\
	result = myinput(buf, max_size);\
	}
%}

%s src isrc inamesrc

L			[A-Za-z_]
O			[0-7]
D			[0-9]
X			[0-9A-Fa-f]
SF			([Ee][+-]?{D}+)
WS			[ \t\f\r]
DELIM			[\"()\[\]{},;]
NDELIM			[^0-9\1-\40\"()\[\]{},;]
SYM			({L}({L}|{D})*({NDELIM}|::)?|{NDELIM}|::)

%%

<<EOF>>			{ if (wrapped) yyterminate(); else return(EOFTOK); }
as			return(AS);
const			return(CONST);
def			return(DEF);
else			return(ELSE);
extern			return(EXTERN);
from			{ BEGIN isrc; return(FROM); }
if			return(IF);
<isrc>import		{ BEGIN inamesrc; return(IMPORT); }
<isrc>include		{ BEGIN inamesrc; return(INCLUDE); }
import			{ BEGIN isrc; return(IMPORT); }
include			{ BEGIN isrc; return(INCLUDE); }
otherwise		return(OTHERWISE);
private			return(PRIVATE);
public			return(PUBLIC);
special			return(SPECIAL);
then			return(THEN);
type			return(TYPE);
undef			return(UNDEF);
var			return(VAR);
virtual			return(VIRTUAL);
where			return(WHERE);
<isrc>\"			{
			char *buf1;
			int i, l;
			string();
			l = strlen(buf);
			buf1 = (char*)malloc((strlen(buf)+1)*sizeof(char));
			if (!buf1)
			  fatal(qcmsg[MEM_OVF]);
			if (!scanstr(buf1, buf))
			  for (i = strlen(buf); i < l-1; i += strlen(buf+i+1)+1) {
			    char msg[MAXSTRLEN];
#ifdef HAVE_UNICODE
			    char *s = buf+i+1, *t = u8strind(s, 1), c[5];
			    c[0] = '\\'; strncpy(c+1, s, t-s); c[t-s+1] = 0;
#else
			    char c[3];
			    c[0] = '\\'; c[1] = buf[i+1]; c[2] = 0;
#endif
			    sprintf(msg, qcmsg[INVALID_ESCAPE], utf8_to_sys(c));
			    yyerror(msg);
			  }
			yylval.sval = buf1;
			return(STR1);
			}
<isrc>;			{ BEGIN src; return ';'; }
<isrc>{SYM}		{ int ret;
			  yyless(0);
			  ret = utf8_id();
			  if (ret) {
			    yylval.sval = strdupchk(buf); return(ID1);
			  } else {
			    if (!*buf) utf8_skip();
			    return ERRTOK;
			  }
			}
<inamesrc>;		{ BEGIN src; return ';'; }
<inamesrc>{SYM}		{ int ret;
			  yyless(0);
			  context = 1;
			  ret = utf8_id_or_sym();
			  context = 0;
			  if (ret) {
			    yylval.sval = strdupchk(buf); return(ID1);
			  } else {
			    if (!*buf) utf8_skip();
			    return ERRTOK;
			  }
			}
0{O}+			{ bigint(yylval.zval); return(INT); }
0{O}+/\.\.		{ bigint(yylval.zval); return(INT); }
0{D}+			{ return(ERRTOK); }
0{D}+/\.\.		{ return(ERRTOK); }
{D}+			{ bigint(yylval.zval); return(INT); }
{D}+/\.\.		{ bigint(yylval.zval); return(INT); }
0[xX]{X}+		{ bigint(yylval.zval); return(INT); }
0[xX]{X}+/\.\.		{ bigint(yylval.zval); return(INT); }
{D}+{SF}		|
{D}+\.{D}*{SF}?		|
{D}*\.{D}+{SF}?		{ yylval.fval = my_strtod(yytext, NULL); return(FLOAT); }
\"			{
			char *buf1;
			int i, l;
			string();
			l = strlen(buf);
			buf1 = (char*)malloc((l+1)*sizeof(char));
			if (!buf1)
			  fatal(qcmsg[MEM_OVF]);
			if (!scanstr(buf1, buf))
			  for (i = strlen(buf); i < l-1; i += strlen(buf+i+1)+1) {
			    char msg[MAXSTRLEN];
#ifdef HAVE_UNICODE
			    char *s = buf+i+1, *t = u8strind(s, 1), c[5];
			    c[0] = '\\'; strncpy(c+1, s, t-s); c[t-s+1] = 0;
#else
			    char c[3];
			    c[0] = '\\'; c[1] = buf[i+1]; c[2] = 0;
#endif
			    sprintf(msg, qcmsg[INVALID_ESCAPE], utf8_to_sys(c));
			    yyerror(msg);
			  }
			yylval.ival = putstr(buf1);
			free(buf1);
			return(STR);
			}
^#!\ --en[^\n=]*=.*	{
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
			  char *codeset = strchr(yytext, '=');
			  if (codeset) {
			    ++codeset;
			    if (ic != (iconv_t)-1 && ic != (iconv_t)-2)
			      iconv_close(ic);
			    if (strcmp(codeset, "UTF-8")) {
			      ic = iconv_open("UTF-8", codeset);
			      if (ic == (iconv_t)-1) {
				char msg[MAXSTRLEN];
				sprintf(msg, "unknown encoding `%s'", codeset);
				yyerror(msg);
			      }
			    } else
			      ic = (iconv_t)-1;
			  }
#endif
			}
{WS}			|
"//".*			|
^#!.*			;
\n			yylineno++;
"/*"			comment();
{DELIM}			return(yytext[0]);
{SYM}			{ int ret;
			  yyless(0);
			  ret = utf8_qualid();
			  if (ret) {
			    int idtok = (ret&8)?((ret&2)?QUID:QLID):((ret&2)?UID:LID);
			    ofchk(buf); yylval.sval = strcpy(s1, buf);
			    return checkid(idtok);
			  } else {
			    if (!*buf) utf8_skip();
			    return ERRTOK;
			  }
			}
.			return(yytext[0]);

%%

static int      in_comment = 0;
static int      argc;
static char   **argv, **argv0 = NULL, **asv, **asv0 = NULL, *mainfile;

static int op_tok[] = { OP0, OP1, OP2, OP3, OP4, OP5, OP6, OP7, OP8, OP9 };

typedef struct {
  char *name;
  int tok;
} keyword;

static keyword kwtable[] = {
  {"as",	AS},
  {"const",	CONST},
  {"def",	DEF},
  {"else",	ELSE},
  {"extern",	EXTERN},
  {"from",	FROM},
  {"if",	IF},
  {"import",	IMPORT},
  {"include",	INCLUDE},
  {"otherwise",	OTHERWISE},
  {"private",	PRIVATE},
  {"public",	PUBLIC},
  {"special",	SPECIAL},
  {"then",	THEN},
  {"type",	TYPE},
  {"undef",	UNDEF},
  {"var",	VAR},
  {"virtual",	VIRTUAL},
  {"where",	WHERE},
  {"..",	DOTDOT},
  {":",		':'},
  {"|",		'|'},
  {"=",		'='},
  {"==",	EQUIV},
  {"-",		'-'},
  {"\\",	'\\'},
  {".",		'.'},
  {"@",		'@'},
  {"~",		'~'},
};

static int compkw(const void *k1, const void *k2)
{
  keyword *kw1 = (keyword*) k1;
  keyword *kw2 = (keyword*) k2;
  return strcmp(kw1->name, kw2->name);
}

static void initkws(void)
{
  qsort(kwtable, sizeof(kwtable)/sizeof(keyword), sizeof(keyword), compkw);
}

static int kwtok(char *name)
{
  keyword key, *res;
  key.name = name;
  res = bsearch(&key, kwtable, sizeof(kwtable)/sizeof(keyword),
		sizeof(keyword), compkw);
  if (res)
    return res->tok;
  else
    return NONE;
}

static int checkid(int tok)
{
  int tok1 = kwtok(yylval.sval);
  if (tok1 != NONE) {
    if (tok1 == IMPORT || tok1 == INCLUDE || tok1 == FROM)
      BEGIN isrc;
    return tok1;
  }
  if (!context) {
    char s[MAXSTRLEN];
    int fno;
    strcpy(s, yylval.sval);
    fno = getfun(s);
    if (fno != NONE && symtb[fno].prec != NONE) {
      yylval.ival = fno;
      return op_tok[symtb[fno].prec];
    }
  }
  return tok;
}

static int checksym(const char *sym)
{
  if (context)
    return 1;
  else {
    char s[MAXSTRLEN];
    strcpy(s, sym);
    return kwtok(s) != NONE || getfun(s) != NONE;
  }
}

static comment()
{
  register int    c;

  in_comment = 1;
  while ((c = input()) && c != EOF) {
    if (c == '*') {
      if ((c = input()) == '/')
	break;
      else
	unput(c);
    } else if (c == '\n')
      yylineno++;
  }
  if (!c || c == EOF) {
    yyerror(qcmsg[OPEN_COMMENT]);
    fatal("unrecoverable syntax error");
  }
  in_comment = 0;
}

static initbuf()
{
  bufp = 0;
}

static addbuf(char c)
{
  if (bufp >= abufsz)
    if (!(buf = (char*)arealloc(buf, abufsz, 100, sizeof(char))))
      fatal(qcmsg[MEM_OVF]);
    else
      abufsz += 100;
  buf[bufp++] = c;
}

static lookahead(const char *s)
{
  register long c;
  const char *t = s;
  int ret;
  while (*t && (c = input()) == *t)
    t++;
  ret = !*t;
  if (*t) {
    if (c != EOF) unput(c);
  }
  while (t > s)
    unput(*--t);
  return ret;
}

#define DELIM "\"()[]{},;_"

#ifdef HAVE_UNICODE

#define ERRC (-99)

static inline long
u8getc(void)
{
  size_t n;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  int ch;
  for (n = 0; n == 0 && (ch = input()) != EOF; ) {
    unsigned char uc = (unsigned char)ch;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return ERRC;
    }
  }
  if (n == 1)
    return c;
  else
    return EOF;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

static inline void
u8ungetc(long c)
{
  if (c && c != EOF) {
    int i;
    char s[5];
    u8encode(s, (unsigned long)c);
    for (i = strlen(s)-1; i >= 0; --i)
      unput(s[i]);
  }
}

static inline void
u8addbuf(long c)
{
  if (c != EOF) {
    char s[5], *t;
    u8encode(s, (unsigned long)c);
    for (t = s; *t; t++)
      addbuf(*t);
  }
}

static utf8_ident()
{
  register long c;
  int save_bufp = bufp, ret = 1;
  while ((c = u8getc()) && c != EOF) {
    if (c == ERRC) {
      addbuf('\0');
      return 0;
    }
    if (bufp == save_bufp)
      if (u_isalpha(c) || c == '_') {
	if (u_isupper(c))
	  ret |= 2; /* indicates a capitalized identifier */
	u8addbuf(c);
      } else
	break;
    else if (u_isalnum(c) || c == '_')
      u8addbuf(c);
    else
      break;
  }
  if (c && c != EOF)
    u8ungetc(c);
  addbuf('\0');
  if (strcmp(buf, "_") == 0) ret |= 2;
  return (*buf)?ret:0;
}

static utf8_ident_or_sym()
{
  register long c;
  int save_bufp = bufp, ret = 1, k = bufp, lastk = bufp;
  while ((c = u8getc()) && c != EOF) {
    if (c == ERRC) {
      addbuf('\0');
      return 0;
    }
    if (bufp == save_bufp)
      if (u_isalpha(c) || c == '_') {
	if (u_isupper(c))
	  ret |= 2; /* indicates a capitalized identifier */
	u8addbuf(c);
      } else if (u_ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	u8addbuf(c); addbuf('\0');
	if (checksym(buf)) k = strlen(buf);
	bufp--;
	ret |= 4; /* indicates a symbol */
      } else
	break;
    else if (ret & 4) {
      if (u_ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	int l = bufp-1;
	u8addbuf(c); addbuf('\0');
	if (strcmp(buf+l, "::") == 0 ||
	    strcmp(buf+l, "//") == 0 || strcmp(buf+l, "/*") == 0) {
	  if (k > l) k = lastk;
	  bufp--;
	  goto out;
	}
	if (checksym(buf)) {
	  lastk = k;
	  k = strlen(buf);
	}
	bufp--;
      } else
	break;
    } else if (u_isalnum(c) || c == '_')
      u8addbuf(c);
    else
      break;
  }
  if (c && c != EOF)
    u8ungetc(c);
 out:
  if (ret & 4) {
    while (bufp > k)
      unput(buf[--bufp]);
  }
  addbuf('\0');
  if (strcmp(buf, "_") == 0) ret |= 2;
#if 0
  printf("got symbol '%s' (flags: %d)\n", buf, ret);
#endif
  return (*buf)?ret:0;
}

static utf8_skip()
{
  long c;
  while ((c = input()) && c != EOF && (((unsigned char)c)&0xc0) == 0xc0) ;
  if (c == EOF)
    fatal("unrecoverable syntax error");
  else {
    yytext[0] = c;
    yytext[1] = 0;
  }
}

#else

static utf8_ident()
{
  register long c;
  int save_bufp = bufp, ret = 1;
  while ((c = input()) && c != EOF) {
    if (bufp == save_bufp)
      if (isalpha(c) || c == '_') {
	if (isupper(c))
	  ret |= 2; /* indicates a capitalized identifier */
	addbuf(c);
      } else
	break;
    else if (isalnum(c) || c == '_')
      addbuf(c);
    else
      break;
  }
  if (c && c != EOF)
    unput(c);
  addbuf('\0');
  if (strcmp(buf, "_") == 0) ret |= 2;
  return (*buf)?ret:0;
}

static utf8_ident_or_sym()
{
  register long c;
  int save_bufp = bufp, ret = 1, k = bufp, lastk = bufp;
  while ((c = input()) && c != EOF) {
    if (bufp == save_bufp)
      if (isalpha(c) || c == '_') {
	if (isupper(c))
	  ret |= 2; /* indicates a capitalized identifier */
	addbuf(c);
      } else if (ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	addbuf(c); addbuf('\0');
	if (checksym(buf)) k = strlen(buf);
	bufp--;
	ret |= 4; /* indicates a symbol */
      } else
	break;
    else if (ret & 4) {
      if (ispunct(c) && (c >= 128 || !strchr(DELIM, c))) {
	int l = bufp-1;
	addbuf(c); addbuf('\0');
	if (strcmp(buf+l, "::") == 0 ||
	    strcmp(buf+l, "//") == 0 || strcmp(buf+l, "/*") == 0) {
	  if (k > l) k = lastk;
	  bufp--;
	  goto out;
	}
	if (checksym(buf)) {
	  lastk = k;
	  k = strlen(buf);
	}
	bufp--;
      } else
	break;
    } else if (isalnum(c) || c == '_')
      addbuf(c);
    else
      break;
  }
  if (c && c != EOF)
    unput(c);
 out:
  if (ret & 4) {
    while (bufp > k)
      unput(buf[--bufp]);
  }
  addbuf('\0');
  if (strcmp(buf, "_") == 0) ret |= 2;
#if 0
  printf("got symbol '%s' (flags: %d)\n", buf, ret);
#endif
  return (*buf)?ret:0;
}

static utf8_skip()
{
  int c = input();
  if (c == EOF)
    fatal("unrecoverable syntax error");
  else {
    yytext[0] = c;
    yytext[1] = 0;
  }
}

#endif

static utf8_id()
{
  initbuf();
  return utf8_ident();
}

static utf8_id_or_sym()
{
  initbuf();
  return utf8_ident_or_sym();
}

static utf8_qualid()
{
  register long c;
  int ret, ret2;
  initbuf();
  ret = utf8_ident_or_sym();
  if (ret && (ret & 4))
    return ret;
  else if (lookahead("::")) {
    bufp--;
    addbuf(input()); addbuf(input());
    ret2 = utf8_ident_or_sym();
    if (ret2)
      return ret2 | 8; /* indicates a qualified id */
    else {
      unput(':'); unput(':');
      bufp--; bufp--;
      buf[bufp] = 0;
      return ret;
    }
  } else
    return ret;
}

static string()
{
  register int    c;

  in_comment = 1;
  initbuf();
  while ((c = input()) && c != EOF) {
    if (c == '"')
      break;
    else if (c == '\\') {
      if ((c = input()) == '\n') {
	yylineno++;
      } else {
	addbuf('\\');
	addbuf(c);
      }
    } else if (c == '\n')
      break;
    else
      addbuf(c);
  }
  addbuf('\0');
  if (c != '"') {
    yyerror(qcmsg[OPEN_STRING]);
    fatal("unrecoverable syntax error");
  }
  in_comment = 0;
}

static char *
skipz(char *s)
{
  while (*s == '0') s++;
  return s;
}

static void*
my_mpz_realloc(m, new_size)
     mpz_ptr m;
     mp_size_t new_size;
{
  mpz_t m1;
  memcpy(m1, m, sizeof(mpz_t));
  if (_mpz_realloc(m, new_size))
    return m->_mp_d;
  else {
    if (m1->_mp_d) mpz_clear(m1);
    return NULL;
  }
}

static void bigint(z)
     mpz_t z;
{
  int sz;
  if (strncmp(yytext, "0x", 2) == 0 ||
      strncmp(yytext, "0X", 2) == 0)
    sz = 4*strlen(skipz(yytext+2));
  else if (*yytext == '0')
    sz = 3*strlen(skipz(yytext+1));
  else
    sz = log(10)/log(2)*strlen(skipz(yytext))+1;
  sz = sz/(CHAR_BIT*sizeof(mp_limb_t)) + 2;
  mpz_init(z); 
  if (z->_mp_d && my_mpz_realloc(z, sz)) {
    int sz1;
    mpz_set_str(z, yytext, 0);
    sz1 = mpz_size(z);
    if (sz1 < sz && !my_mpz_realloc(z, sz1))
      fatal(qcmsg[MEM_OVF]);
  } else
    fatal(qcmsg[MEM_OVF]);
}

char *_realname[MAXFILES];
int _modtb[MAXFILES], _fnametb[MAXFILES];
int _modtbsz;

typedef struct {
  char *s, *as;
} id_alias_t;

static struct FileStack {
  YY_BUFFER_STATE state;
  FILE *yyin;
  int prio, modno, yylineno, incl;
  char *source;
  int argc;
  char **argv, **argv0;
  char **asv, **asv0;
  int namec;
  id_alias_t *namev;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  iconv_t ic;
#endif
  char *mybuf, *mybufptr;
} fst[MAXFILES], *fsp;

static
getmodno(s)
     char *s;
/* look up module name s in module table, return corresponding module
   number (NONE if not in table) */
{
  int i;
  for (i = 0; i < _modtbsz; i++)
    if (strcmp(s, strsp+_modtb[i]) == 0)
      return i;
  return NONE;
}

static
getmodno_by_fname(s)
     char *s;
/* look up module by file name */
{
  int i;
  for (i = 0; i < _modtbsz; i++)
    if (strcmp(s, strsp+_fnametb[i]) == 0)
      return i;
  return NONE;
}

static
stacked(modno)
     int modno;
/* check whether module is on include stack */
{
  struct FileStack *fsp1;
  for (fsp1 = fst; fsp1 < fsp; fsp1++)
    if (fsp1->modno == modno)
      return 1;
  return 0;
}

static
addmod(modname, realname, fname, s)
     char *modname, *realname, *fname, *s;
/* add module to module table */
{
  if (_modtbsz >= MAXFILES)
    fatal(qcmsg[TOO_MANY_FILES]);
  else {
    _modtb[_modtbsz] = putstr(modname);
    _fnametb[_modtbsz] = putstr(fname);
    _realname[_modtbsz] = strdup(realname);
    if (!_realname[_modtbsz]) fatal("memory overflow");
    modno = _modtbsz++;
    if (s == mainfile) mainno = modno;
    if (mainno > 0) putimp(0, 1);
  }
}

#define HAVE_FILE (-2)

static
opensrc(s, as, save)
     char           *s, *as;
     int save;
{
  char aname[MAXSTRLEN], fname[MAXSTRLEN], fname2[MAXSTRLEN],
    modname[MAXSTRLEN];
  int mno, have_stdin;

  if (!s || !*s) return NONE;

  /* handle stdin */

  have_stdin = mainno < 0 && strcmp(s, "-") == 0 && !as;

  if (have_stdin) {
    strcpy(fname, "<<stdin>>");
    strcpy(aname, fname);
    strcpy(modname, fname);
    as = modname;
    goto open_file;
  }

  /* parse file name and determine module id: */

  basename(modname, sys_to_utf8(s), '.');
  if (!as) as = modname;
  absname(aname, searchlib(fname, s));
  if (!chkfile(aname)) {
    strcat(strcpy(fname2, s), ".q");
    absname(aname, searchlib(fname, fname2));
  }

  /* check whether module has already been loaded: */

  if ((mno = getmodno(as)) != NONE) {
    if (s == mainfile) mainno = mno;
    /* file already loaded, check for name conflicts and cyclic inclusions */
    if (strcmp(aname, strsp+_fnametb[mno]) != 0) {
      char msg[MAXSTRLEN];
      sprintf(msg, qcmsg[AMBIG_REF], as);
      yyerror(msg);
    } else if (stacked(mno))
      yyerror(qcmsg[CYCLIC_REF]);
    return mno;
  } else if ((mno = getmodno_by_fname(aname)) != NONE) {
    /* module has already been loaded under a different alias; we handle this
       case by manufacturing aliases for all symbols of the module */
    int _modno = modno, sz = symtbsz, i;
    addmod(as, modname, aname, s);
    for (i = BUILTIN; i < sz; i++)
      if (symtb[i].modno == mno) {
	short flags = symtb[i].flags & ~EXT;
	if (flags & TSYM)
	  astype(i, 0, flags);
	else if (flags & VSYM)
	  asfvar(i, 0, flags);
	else
	  asfun(i, 0, symtb[i].argc, symtb[i].argv, flags, symtb[i].prec);
      }
    mno = modno;
    modno = _modno;
    return mno;
  }

  /* open new file: */

 open_file:
  if (save) saveimps();

  if ((yyin = have_stdin?stdin:fopen(aname, "r")) == NULL) {
    source = s;
    fatal(qcmsg[FILE_NOT_FOUND]);
  }
  if (fsp > fst)
    yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));

  /* enter into module table and initialize settings: */

  addmod(as, modname, aname, s);
  prio = 0;
  source = strdupchk(fname);
  yylineno = 1;
  BEGIN src;
  if (vflag)
    printf("%s:\n", have_stdin?fname:s);
  return HAVE_FILE;

}

static
pushfile(incl)
     int incl;
{
  if (fsp - fst >= MAXFILES)
    fatal(qcmsg[TOO_MANY_FILES]);
  else {
    fsp->state = YY_CURRENT_BUFFER;
    fsp->yyin = yyin;
    fsp->yylineno = yylineno;
    fsp->prio = prio;
    fsp->modno = modno;
    fsp->source = source;
    fsp->incl = incl;
    fsp->argc = argc;
    fsp->argv = argv;
    fsp->argv0 = argv0;
    fsp->asv = asv;
    fsp->asv0 = asv0;
    fsp->namec = 0;
    fsp->namev = NULL;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    fsp->ic = ic;
    ic = (iconv_t)-2;
#endif
    fsp->mybuf = mybuf;
    fsp->mybufptr = mybufptr;
    mybuf = mybufptr = NULL;
    fsp++;
  }
}

static popfile()
{
  fsp--;
  yy_delete_buffer(YY_CURRENT_BUFFER);
  yy_switch_to_buffer(fsp->state);
  yyin = fsp->yyin;
  yylineno = fsp->yylineno;
  prio = fsp->prio;
  modno = fsp->modno;
  source = fsp->source;
  argc = fsp->argc;
  argv = fsp->argv;
  argv0 = fsp->argv0;
  asv = fsp->asv;
  asv0 = fsp->asv0;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  if (ic != (iconv_t)-1 && ic != (iconv_t)-2)
    iconv_close(ic);
  ic = fsp->ic;
#endif
  if (mybuf) free(mybuf);
  mybuf = fsp->mybuf;
  mybufptr = fsp->mybufptr;
}

static popfile0()
{
  fsp--;
  yyin = fsp->yyin;
  yylineno = fsp->yylineno;
  prio = fsp->prio;
  modno = fsp->modno;
  source = fsp->source;
  argc = fsp->argc;
  argv = fsp->argv;
  argv0 = fsp->argv0;
  asv = fsp->asv;
  asv0 = fsp->asv0;
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  if (ic != (iconv_t)-1 && ic != (iconv_t)-2)
    iconv_close(ic);
  ic = fsp->ic;
#endif
  if (mybuf) free(mybuf);
  mybuf = fsp->mybuf;
  mybufptr = fsp->mybufptr;
}

#define MAXNAMES MAXFILES

static id_alias_t impq[MAXFILES], impname[MAXNAMES];

static int impqsz = 0, impnamesz = 0;

add_import(s, as)
     char *s, *as;
{
  if (impqsz >= MAXFILES)
    fatal(qcmsg[TOO_MANY_FILES]);
  impq[impqsz].s = utf8_to_sys_dup(s);
  if (s) free(s);
  if (!impq[impqsz].s) fatal("memory overflow");
  impq[impqsz].as = as;
  impqsz++;
}

add_import_name(s, as)
     char *s, *as;
{
  if (impnamesz >= MAXNAMES)
    fatal(qcmsg[MEM_OVF]);
  impname[impnamesz].s = s;
  impname[impnamesz].as = as;
  impnamesz++;
}

clear_imports()
{
  impqsz = impnamesz = 0;
}

static void
sym_import(int incl, int mno, int namec, id_alias_t *namev)
{
  /* make aliases for imported symbols */
  char *modname = strsp+_modtb[mno];
  int i, l = strlen(modname);
  if (namec > 0 && namev[0].s) {
    for (i = 0; i < namec; i++)
      if (namev[i].s) {
	int sym = getsym(mno, namev[i].s);
	char *as = namev[i].as?namev[i].as:namev[i].s;
	if (sym != NONE) {
	  short flags = symtb[sym].flags & ~EXT;
	  if (!incl) flags |= PRIV;
	  if (flags & TSYM) {
	    int as_sym = mkxtype(as);
	    if (as_sym != NONE) astype(sym, as_sym, flags);
	  } else if (flags & VSYM) {
	    int as_sym = mkxfvar(as);
	    if (as_sym != NONE) asfvar(sym, as_sym, flags);
	  } else {
	    int as_sym = mkxfun(as);
	    if (as_sym != NONE)
	      asfun(sym, as_sym, symtb[sym].argc, symtb[sym].argv, flags,
		    symtb[sym].prec);
	  }
	} else {
	  /* error: undeclared symbol */
	  char msg[MAXSTRLEN];
	  sprintf(msg, qcmsg[MISS_DCL], utf8_to_sys(namev[i].s),
		  utf8_to_sys(modname));
	  yyerror(msg);
	}
	free(namev[i].s);
	if (namev[i].as) free(namev[i].as);
      }
  } else {
    /* just import everything */
    int sz = symtbsz;
    for (i = BUILTIN; i < sz; i++)
      if (symtb[i].modno == mno && !(symtb[i].flags&PRIV)) {
	short flags = symtb[i].flags & ~EXT;
	if (flags & TSYM)
	  astype(i, 0, flags);
	else if (flags & VSYM)
	  asfvar(i, 0, flags);
	else
	  asfun(i, 0, symtb[i].argc, symtb[i].argv, flags, symtb[i].prec);
      }
  }
}

static
doimport(incl)
     int incl;
{
  int i, mno;
  if (impqsz == 0) return;
  pushfile(incl);
  argc = impqsz;
  argv0 = argv = aalloc(argc, sizeof(char*));
  asv0 = asv = aalloc(argc, sizeof(char*));
  if (!argv || !asv) fatal(qcmsg[MEM_OVF]);
  for (i = 0; i < argc; i++) {
    argv[i] = impq[i].s;
    asv[i] = impq[i].as;
  }
  impqsz = 0;
  if (impnamesz > 0) {
    fsp[-1].namec = impnamesz;
    fsp[-1].namev = aalloc(impnamesz, sizeof(id_alias_t));
    if (!fsp[-1].namev) fatal(qcmsg[MEM_OVF]);
    memcpy(fsp[-1].namev, impname, impnamesz*sizeof(id_alias_t));
    impnamesz = 0;
  }
  while (argc--)
    if ((mno = opensrc(*argv++, *asv++, 1)) == HAVE_FILE) {
      if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
      return;
    } else {
      if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
      fsp--;
      modno = fsp->modno;
      yylineno = fsp->yylineno;
      if (mno != NONE)
	if (fsp->namev) {
	  sym_import(incl, mno, fsp->namec, fsp->namev);
	  free(fsp->namev); fsp->namev = NULL;
	} else if (incl)
	  putinc(mno, 0);
	else
	  putimp(mno, 0);
      fsp++;
    }
  if (argv0) free(argv0);
  if (asv0) free(asv0);
  popfile0();
}

import()
{
  doimport(0);
}

include()
{
  doimport(1);
}

static int have_main = 0;

yywrap()
{
  return 1;
}

wrapover()
{
  int have_file = 0;
  if (wrapped) return 1;
  if (wflag == 1) unresolved_forwards();
  saveimps();
  if (yyin != stdin) fclose(yyin); yyin = NULL;
  if (in_comment)
    return (wrapped = 1);
  else if (fsp > fst) {
    int mno = modno;
    fsp--;
    modno = fsp->modno;
    yylineno = fsp->yylineno;
    if (source) free(source);
    source = fsp->source;
    restoreimps();
    if (fsp->namev) {
      sym_import(fsp->incl, mno, fsp->namec, fsp->namev);
      free(fsp->namev); fsp->namev = NULL;
    } else if (fsp->incl)
      putinc(mno, 0);
    else
      putimp(mno, 0);
    fsp++;
    while(argc--)
      if ((have_file = (mno = opensrc(*argv++, *asv++, 1)) == HAVE_FILE)) {
	if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
	break;
      } else {
	if (argv[-1]) free(argv[-1]); if (asv[-1]) free(asv[-1]);
	fsp--;
	if (mno != NONE)
	  if (fsp->namev) {
	    sym_import(fsp->incl, mno, fsp->namec, fsp->namev);
	    free(fsp->namev); fsp->namev = NULL;
	  } else if (fsp->incl)
	    putinc(mno, 0);
	  else
	    putimp(mno, 0);
	fsp++;
      }
    if (!have_file) {
      if (argv0) free(argv0); if (asv0) free(asv0);
      popfile();
    }
    return 1;
  } else {
    static int mno, save = 0;
    if (have_main) {
      mno = modno;
      modno = mainno;
      restoreimps();
      putimp(mno, 0);
      save = 1;
    } else if (mainno != NONE) {
      have_main = 1;
      modno = mainno;
      restoreimps();
    } else if (mainfile && !*mainfile) {
      /* create a dummy main module to hold the global imports */
      mno = modno;
      addmod("", "", "", mainfile);
      have_main = 1;
      restoreimps();
      if (mno > 0) putimp(mno, 0);
      save = 1;
    }
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
    if (ic != (iconv_t)-1 && ic != (iconv_t)-2)
      iconv_close(ic);
    ic = (iconv_t)-2;
#endif
    while(argc-- && !(have_file = (mno=opensrc(*argv++, NULL, save))
		      == HAVE_FILE)) {
      putimp(mno, 0);
      save = 1;
    }
    if (!have_file && save) saveimps();
    wrapped = !have_file;
    return 1;
  }
}

initlex(_argc, _argv)
     int             _argc;
     char          **_argv;
{
  argc = _argc;
  argv = _argv;
  mainfile = (argc < 1)?NULL:*argv;
  initkws();
  while (argc && !**argv) argc--, argv++;
  if (prelude || argc >= 1) {
    fsp = fst;
    BEGIN src;
    if (prelude)
      opensrc(prelude, NULL, 0);
    else {
      if (mainfile && !*mainfile) {
	/* create a dummy main module to hold the global imports */
	addmod("", "", "", mainfile);
	saveimps();
	have_main = 1;
      }
      argc--;
      opensrc(*argv++, NULL, 0);
    }
    return 1;
  } else
    return 0;
}

srcstate()
{
  BEGIN src;
}
