// The MixMax code below is included in the PYTHIA event generator
// by express permission from the authors.
// The coefficients tables in the skipMat17 array are not open source.
// Contact the authors for permission to use MixMax outside of PYTHIA.

// Note: the MixMaxRndm wrapper class for Pythia usage
// is found  at the bottom of this file.

//==========================================================================

/*
 * mixmax.hpp
 *
 * C++ implementation of the MIXMAX random number generator.
 *
 *  Copyright (2008-2016) by Konstantin Savvidy.
 *
 *  Free to use, academic or commercial. Do not redistribute without permission.
 *
 *	G.K.Savvidy and N.G.Ter-Arutyunian,
 *  On the Monte Carlo simulation of physical systems,
 *	J.Comput.Phys. 97, 566 (1991);
 *  Preprint EPI-865-16-86, Yerevan, Jan. 1986
 *
 *  K.Savvidy
 *  The MIXMAX random number generator
 *  Comp. Phys. Commun. 196 (2015), pp 161–165
 *  http://dx.doi.org/10.1016/j.cpc.2015.06.003
 *
 *  K.Savvidy and G.Savvidy
 *  Spectrum and Entropy of C-systems. MIXMAX random number generator
 *  Chaos, Solitons & Fractals, Volume 91, (2016) pp. 33–38
 *  http://dx.doi.org/10.1016/j.chaos.2016.05.003
 *
 */

#ifndef __MIXMAX_H
#define __MIXMAX_H

#include <array>
#include <vector>
#include <cstdint>
#include <iostream>
#include <exception>

template <typename T, T __min, T __max> class _Generator
// Boilerplate code, from Andrzej, it is required to be compatible with std::random interfaces, see example.cpp for how to use.
{
public:
    using result_type = T;
    static constexpr T min() {return __min;}
    static constexpr T max() {return __max;}
    void seed (result_type val = 1);
    T operator()();
};

typedef uint32_t myID_t;
typedef uint64_t myuint;


constexpr int Ndim = 17; // turn off for TEMPLATE use

constexpr int BITS=61;
constexpr myuint M61=2305843009213693951ULL;
constexpr myuint MERSBASE=M61;
constexpr double INV_MERSBASE=(0.43368086899420177360298E-18);


/*
 Table of parameters for MIXMAX

 Figure of merit is entropy: best generator overall is N=240

 Vector size |                                                                                    period q
 N           |    SPECIAL                |   SPECIALMUL   |           MOD_MULSPEC               | log10(q)  |   entropy  |
 ------------------------------------------------------------------------------------------------------------------------|
   8         |         0                 |     53         |                none                 |    129    |    220.4   |
  17         |         0                 |     36         |                none                 |    294    |    374.3   |
 240         | 487013230256099140        |     51         |   fmodmulM61( 0, SPECIAL , (k) )    |   4389    |   8679.2   |

*/

// Interface C++11 std::random

// template <int Ndim=240> // TEMPLATE
class mixmax_engine: public _Generator<std::uint64_t, 0, 0x1FFFFFFFFFFFFFFF> // does not work with any other values
{
static const int N = Ndim;
    static constexpr long long int SPECIAL   = ((N==17)? 0 : ((N==240)? 487013230256099140ULL:-1) ); // etc...
    static constexpr long long int SPECIALMUL= ((N==17)? 36: ((N==240)? 51                   : 0) ); // etc...
    // Note the potential for confusion...

struct rng_state_st
{
    std::array<myuint, N> V;
    myuint sumtot;
    int counter;
};

typedef struct rng_state_st rng_state_t;     // struct alias

rng_state_t S;

public:
	using T = result_type;								  // should it be double?
    static constexpr int rng_get_N() {return N;}
    static constexpr long long int rng_get_SPECIAL()    {return SPECIAL;}
    static constexpr int rng_get_SPECIALMUL() {return SPECIALMUL;}
    void seed_uniquestream( rng_state_t* Xin, myID_t clusterID, myID_t machineID, myID_t runID, myID_t  streamID );
    void print_state();
// void read_state(const char filename[] );
    myuint get_next() ;
    double get_next_float();
    //double flat() {return get_next_float();};   // Generation method for the random numbers. // for Pythia
    // mixmaxRndm();                               // constructor   // for Pythia

   // int iterate();
    mixmax_engine Branch();
    void BranchInplace();

    mixmax_engine(myID_t clusterID, myID_t machineID, myID_t runID, myID_t  streamID );	   // Constructor with four 32-bit seeds
    void seed(uint64_t seedval){seed_uniquestream( &S, 0, 0, (myID_t)(seedval>>32), (myID_t)seedval );} // seed with one 64-bit seed
    mixmax_engine(); // Constructor, no seeds

    mixmax_engine& operator=(const mixmax_engine& other );

inline T operator()()
    {
        return get_next();
    }

private:
    myuint MOD_MULSPEC(myuint k);
    void seed_vielbein(rng_state_t* X, unsigned int i); // seeds with the i-th unit vector, i = 0..N-1,  for testing only
    myuint iterate_raw_vec(myuint* Y, myuint sumtotOld);
    myuint apply_bigskip(myuint* Vout, myuint* Vin, myID_t clusterID, myID_t machineID, myID_t runID, myID_t  streamID );
    myuint modadd(myuint foo, myuint bar);
    myuint fmodmulM61(myuint cum, myuint s, myuint a);
#if defined(__x86_64__)
    inline myuint mod128(__uint128_t s);
#endif
};

#define ARRAY_INDEX_OUT_OF_BOUNDS   0xFF01
#define SEED_WAS_ZERO               0xFF02
#define ERROR_READING_STATE_FILE    0xFF03
#define ERROR_READING_STATE_COUNTER       0xFF04
#define ERROR_READING_STATE_CHECKSUM      0xFF05

#define MOD_PAYNE(k) ((((k)) & MERSBASE) + (((k)) >> BITS) )
#define MOD_MERSENNE(k) MOD_PAYNE(k)

#define PREF
#define POST
//#define PREF template <int Ndim> // TEMPLATE
//#define POST <Ndim>              // TEMPLATE

PREF myuint mixmax_engine POST::MOD_MULSPEC(myuint k){
    switch (N) {
        case 17:
            return 0;
            break;
        case 8:
            return 0;
            break;
        case 240:
            return fmodmulM61( 0, SPECIAL , (k) );
            break;
        default:
            std::cerr << "MIXMAX ERROR: " << "Disallowed value of parameter N\n";
            break;
    }
}

PREF mixmax_engine POST ::mixmax_engine()
// constructor, with no params, fast and seeds with a unit vector
{
    seed_vielbein(&S,0);
}

//PREF mixmax_engine POST ::mixmaxRndm(){seed_vielbein(&S,0);}                                         // for Pythia
//PREF mixmax_engine POST ::mixmaxRndm(uint64_t seedval){seed_uniquestream( &S,0,0,(myID_t)(seedval>>32), (myID_t)seedval);}  // for Pythia

PREF mixmax_engine POST ::mixmax_engine(myID_t clusterID, myID_t machineID, myID_t runID, myID_t  streamID)
// constructor, no need to allocate, just seed
{
    seed_uniquestream( &S, clusterID,  machineID,  runID,  streamID );
}

#define MULWU(k) (( (k)<<(SPECIALMUL) & M61) | ( (k) >> (BITS-SPECIALMUL))  )

PREF myuint mixmax_engine POST ::iterate_raw_vec(myuint* Y, myuint sumtotOld){
    // operates with a raw vector, uses known sum of elements of Y
    int i;

    myuint temp2 = Y[1];


    myuint  tempP, tempV;
    Y[0] = ( tempV = sumtotOld);
    myuint sumtot = Y[0], ovflow = 0; // will keep a running sum of all new elements
    tempP = 0;              // will keep a partial sum of all old elements
    for (i=1; i<N; i++){
        if (SPECIALMUL!=0){
            myuint tempPO = MULWU(tempP);
            tempP = modadd(tempP, Y[i]);
            tempV = MOD_MERSENNE(tempV+tempP+tempPO); // new Y[i] = old Y[i] + old partial * m
        }else{
            tempP = modadd(tempP , Y[i]);
            tempV = modadd(tempV , tempP);
        }

        Y[i] = tempV;
        sumtot += tempV; if (sumtot < tempV) {ovflow++;}
    }
    if ( SPECIAL !=0 ){
        temp2 = MOD_MULSPEC(temp2);
        Y[2] = modadd( Y[2] , temp2 );
        sumtot += temp2; if (sumtot < temp2) {ovflow++;}
    }
    return MOD_MERSENNE(MOD_MERSENNE(sumtot) + (ovflow <<3 ));
}

PREF myuint mixmax_engine POST ::get_next() {
    int i;
    i=S.counter;

    if (i<=(N-1) ){
        S.counter++;
        return S.V[i];
    }else{
        S.sumtot = iterate_raw_vec(S.V.data(), S.sumtot);
        S.counter=2;
        return S.V[1];
    }
}

PREF double mixmax_engine POST ::get_next_float()				// Returns a random double with all 53 bits random, in the range (0,1]
{    /* cast to signed int trick suggested by Andrzej Görlich     */
    int64_t Z=(int64_t)get_next();
    double F;
#if defined(__GNUC__) && (__GNUC__ < 5) && (!defined(__ICC)) && defined(__x86_64__) && defined(__SSE2_MATH__) && defined(USE_INLINE_ASM)
    //#warning Using the inline assembler
    /* using SSE inline assemly to zero the xmm register, just before int64 -> double conversion,
     not necessary in GCC-5 or better, but huge penalty on earlier compilers
     */
    __asm__  __volatile__("pxor %0, %0; "
                          :"=x"(F)
                          );
#endif
    F=Z;
    return F*INV_MERSBASE;

}

PREF void mixmax_engine POST ::seed_vielbein(rng_state_t* X, unsigned int index)
{
    //rng_state_t S=&X;
    int i;
    if (index<N){
        for (i=0; i < N; i++){
            X->V[i] = 0;
        }
        X->V[index] = 1;
    }else{
        //fprintf(stderr, "Out of bounds index, is not ( 0 <= index < N  )\n");
        std::cerr << "MIXMAX ERROR: " << ARRAY_INDEX_OUT_OF_BOUNDS << "Out of bounds index, is not ( 0 <= index < N  )\n";
        std::terminate();
    }
    X->counter = 0;  // set the counter to N if iteration should happen right away
    X->sumtot = N;   //(index ? 1:0);
}


PREF void mixmax_engine POST ::seed_uniquestream( rng_state_t* Xin, myID_t clusterID, myID_t machineID, myID_t runID, myID_t  streamID ){
    seed_vielbein(Xin,0);
    //print_state();
    Xin->sumtot = apply_bigskip(Xin->V.data(), Xin->V.data(),  clusterID,  machineID,  runID,   streamID );
    //   if (Xin->fh==NULL){Xin->fh=stdout;} // if the filehandle is not yet set, make it stdout
    // Change for PYTHIA version: replace cerr output by more extensive cout one.
    // std::cerr << "seeding with: " << clusterID << ", " << machineID << ", " <<   runID <<  ", " <<  streamID << "\n";
    std::cout << "\n The MixMax random number generator is initialized with seeds: "
              << clusterID << ", " << machineID << ", " <<   runID <<  ", " <<  streamID << "\n";
    Xin->counter = 1;
}


PREF myuint mixmax_engine POST ::apply_bigskip( myuint* Vout, myuint* Vin, myID_t clusterID, myID_t machineID, myID_t runID, myID_t  streamID ){
    /*
     makes a derived state vector, Vout, from the mother state vector Vin
     by skipping a large number of steps, determined by the given seeding ID's

     it is mathematically guaranteed that the substreams derived in this way from the SAME (!!!) Vin will not collide provided
     1) at least one bit of ID is different
     2) less than 10^100 numbers are drawn from the stream
     (this is good enough : a single CPU will not exceed this in the lifetime of the universe, 10^19 sec,
     even if it had a clock cycle of Planch time, 10^44 Hz )

     Caution: never apply this to a derived vector, just choose some mother vector Vin, for example the unit vector by seed_vielbein(X,0),
     and use it in all your runs, just change runID to get completely nonoverlapping streams of random numbers on a different day.

     clusterID and machineID are provided for the benefit of large organizations who wish to ensure that a simulation
     which is running in parallel on a large number of  clusters and machines will have non-colliding source of random numbers.

     did i repeat it enough times? the non-collision guarantee is absolute, not probabilistic

     */



    const	myuint skipMat17[128][17] =
/*
 skipping coefficients for N=17
 three-parameter generator

 *  MIXMAX
 *  A Pseudo-Random Number Generator
 *
 *
 *	G.K.Savvidy and N.G.Ter-Arutyunian,
 *  On the Monte Carlo simulation of physical systems,
 *	J.Comput.Phys. 97, 566 (1991);
 *  Preprint EPI-865-16-86, Yerevan, Jan. 1986
 *
 *  K.Savvidy
 *  The MIXMAX random number generator
 *  Comp. Phys. Commun. 196 (2015), pp 161–165
 *  http://dx.doi.org/10.1016/j.cpc.2015.06.003
 *
 */


/*
 skipping coefficients for N=17, m=2^36+1 and s=0
 */


{

{ 159701761865468764 ,  126615459087983936 ,  971935980882445987 ,  1266425933430727482 ,  330954324138038300 ,  1494390944182608462 ,  985567525599556189 ,  1444303754750856766 ,  1592836147366511740 ,  944566361294427577 ,  6336044263941214 ,  686240801639476028 ,  1446998785307329212 ,  2077656654628400897 ,  458344447554229916 ,  1731527985347537415 ,  1856587133043405783 }, // x^2^512

{ 582338566698838802 ,  782718157798125637 ,  684912925900277712 ,  1271246213246651012 ,  564133275057152122 ,  1254731637174240011 ,  899802366531028440 ,  203611955528219606 ,  2146610763813957982 ,  115972700754870533 ,  2092933746397577991 ,  1136406621159800137 ,  1067897330754616454 ,  1722611875894083236 ,  1630948851615767512 ,  989604548662398765 ,  421680943972696470 }, // x^2^513

{ 617097029144871694 ,  2275108162235384877 ,  1363862259022425560 ,  153831723325577176 ,  1962207167612439146 ,  2011895465932033980 ,  1007329645826843776 ,  1300917526580470528 ,  1171539246226411634 ,  836665263844149790 ,  1026633245683781179 ,  32389807716467901 ,  2296276234884428109 ,  409297319724863974 ,  1911430032287806834 ,  916469329336553330 ,  65677079513594107 }, // x^2^514

{ 1656997175938683235 ,  334384582636879447 ,  2289018900497662792 ,  1053313217223819648 ,  2529561867064419 ,  2085279416854675588 ,  2246944213774249775 ,  917902382070457615 ,  505594560851411438 ,  1748690703652028000 ,  1078212339021286156 ,  170401217082007747 ,  1552855376842911219 ,  2209415219208922654 ,  1174711397611724375 ,  1342069058820567972 ,  577671676013094771 }, // x^2^515

{ 200864199029196267 ,  647584785983829858 ,  449579673559383642 ,  2210795481121719067 ,  1185405708665067088 ,  310118907549337558 ,  1112237647478646030 ,  907316892605650051 ,  1241087052423417724 ,  1441515240516850614 ,  13684659557896782 ,  1932376613205111310 ,  60357690290969298 ,  403792989584565525 ,  102986206885970367 ,  1838760170077135151 ,  727651972830521938 }, // x^2^516

{ 2112835707304810889 ,  583170948234442322 ,  504636752552038260 ,  1287047182580144501 ,  2197579855548731362 ,  1121897458052508302 ,  1952770053431596080 ,  772611988382900586 ,  1985521487785539447 ,  1011576935509998521 ,  367571328103323233 ,  1395044686007468872 ,  398356602519830546 ,  1118155658788636138 ,  1242807300672421078 ,  783603574838596371 ,  2129902332243215295 }, // x^2^517

{ 476494511866938851 ,  1225043666051930516 ,  965084907571074277 ,  48495729686479132 ,  668653785847993343 ,  1110923567965684699 ,  1480594060141743070 ,  1780050366926533029 ,  1286523699078185217 ,  669624786360998719 ,  1865875391054214711 ,  1887676069452920579 ,  9982283268123603 ,  1231842656474052119 ,  1913212077924560289 ,  1062511441187205188 ,  698250642012057138 }, // x^2^518

{ 1052793057808933925 ,  1488114650898155736 ,  1809655216561165737 ,  1766730333219340872 ,  1142295800552777206 ,  1197008472054994109 ,  288311604729845194 ,  1252198368803793342 ,  1700393396484664348 ,  1561218011609178732 ,  1039496900875671159 ,  1199061431134041566 ,  217182459685479427 ,  1965769716664277780 ,  883713133563637492 ,  855967538957706448 ,  182946949964136989 }, // x^2^519

{ 521221301273005220 ,  732177595172654650 ,  1177004708676171981 ,  491026482834387582 ,  1475491185407701571 ,  338551745124704898 ,  322679017279562868 ,  1102042763149957642 ,  1781637400817890058 ,  2230976132808504329 ,  1647749103867940848 ,  363540249136304168 ,  1060964616691451642 ,  856789607085849123 ,  470641990892341750 ,  1281772529585552540 ,  603420272349730830 }, // x^2^520

{ 1803571173151157726 ,  1944596483605820771 ,  213094595441099858 ,  901693591156501692 ,  861915355835676099 ,  635260099576893571 ,  1982413183327998212 ,  1120049474254535333 ,  1505283430373136398 ,  284189413831487026 ,  1271922039532043918 ,  154314556649628344 ,  52549783574158204 ,  584612521732421173 ,  1901458131142776860 ,  1937288889567974526 ,  1212147240165209394 }, // x^2^521

{ 2034564475748864204 ,  2101470381626210311 ,  2288088283864667246 ,  1914963188887983022 ,  1095385391183929148 ,  945106352371129676 ,  1400087958043804470 ,  1843694373265941018 ,  389168898546209530 ,  624173071047154320 ,  1627380173012800657 ,  1094171310315703468 ,  1857806715885218084 ,  1326770924244540642 ,  2245083043975660260 ,  807823933843072560 ,  2287413039178247285 }, // x^2^522

{ 1438462501157028417 ,  1178282907367658045 ,  2151871713069448741 ,  2216564753928829884 ,  1135646926807917971 ,  873250056357106251 ,  1665621526869344532 ,  1732452289298951455 ,  1127204916768538887 ,  1653631650320969340 ,  1992482205762138263 ,  1836865404020205588 ,  1595463001888299444 ,  958431756408955134 ,  2110530659740056614 ,  215688460376624015 ,  1158251593711172802 }, // x^2^523

{ 1733925877085130548 ,  2093555366416251473 ,  2234101572813223254 ,  27150917859587271 ,  2120208241420965005 ,  1740681603888968422 ,  344809595448125388 ,  2297225515944255048 ,  2259042601422705288 ,  872774782224758302 ,  1492625124653803821 ,  1721307140915414448 ,  331124824482108658 ,  120601336292842643 ,  1057772943542838223 ,  904281382213867256 ,  2268149550994572836 }, // x^2^524

{ 1308619451049024014 ,  362949499662291350 ,  507935491259228365 ,  1578906684563693579 ,  1147252812024062738 ,  170426700812984671 ,  2292369007840245901 ,  2278676904939144735 ,  1661943824935578983 ,  1151791566444763239 ,  244664648985115502 ,  788605939290857339 ,  185329244981575100 ,  2234030485122694225 ,  1354886281367682733 ,  1172982108216683874 ,  1673999648423090090 }, // x^2^525

{ 12148567885654147 ,  1372231622618064454 ,  1512411127803286398 ,  91088917975722538 ,  1479647890243287328 ,  720784302152223867 ,  796410783806366571 ,  581618991323966327 ,  850700716339425563 ,  930903615199005243 ,  1411478967795549943 ,  1395258853733301256 ,  1287449265209562277 ,  2138600707136677508 ,  1952689678998032749 ,  1536157936222739421 ,  1365983300736022067 }, // x^2^526

{ 1614268848379872789 ,  1977966741412095904 ,  1868426217239032474 ,  1603841034660414664 ,  1538146046227581261 ,  686243317632590721 ,  1825936888550743483 ,  1880896341340126997 ,  368542569942961260 ,  127908944176116453 ,  2240186396348061575 ,  1299238346503882153 ,  2102660872371583004 ,  1938050823525276998 ,  1611779572790548487 ,  923339421522954197 ,  1640317376822349283 }, // x^2^527

{ 1068274751244808317 ,  478761882933885805 ,  146343578721548466 ,  2122787414678627286 ,  478616655821124071 ,  1223652533398739090 ,  1178267270852093552 ,  1585682559495334861 ,  448375108812347778 ,  28390211070681947 ,  1690010627725723003 ,  1631513459519602506 ,  344109887337730484 ,  1228367175232395654 ,  1245714930611054419 ,  2076109931139359104 ,  1028725609142702754 }, // x^2^528

{ 922695624671435383 ,  1212395461257377598 ,  2004462039540699849 ,  1640073156934071824 ,  2048256174899844882 ,  1784439921745628237 ,  1181704167380381729 ,  1002268784211250265 ,  1203700422140227637 ,  414802973276697203 ,  1506620899880171251 ,  1653959500223714911 ,  761577027275676953 ,  1394862681050805826 ,  283665158959068873 ,  1674581989169655504 ,  2170268137063855658 }, // x^2^529

{ 1531823228490667160 ,  655178204437606577 ,  1143440975987174633 ,  1469225472530244216 ,  1459815855107895954 ,  169158503015419167 ,  896992338984130145 ,  2224392167406479582 ,  334969241330017474 ,  1210934779672254914 ,  958852377915238487 ,  916143882424925025 ,  1228314367757567546 ,  576985231081272085 ,  69708611488908557 ,  419353864719896559 ,  2063012113607020176 }, // x^2^530

{ 1561213794756615274 ,  1035561623111560834 ,  236042828876112400 ,  2227997009011010087 ,  69191692217035803 ,  680016809348368644 ,  304720676536884446 ,  402216461956911069 ,  579355410310536007 ,  2067077610395244217 ,  850246508783344249 ,  1267387033405622409 ,  381166642361949681 ,  745560785470896920 ,  475789434723626541 ,  624301081669895306 ,  542440096530351799 }, // x^2^531

{ 292947975657821923 ,  1975118192564031841 ,  1096919547621493467 ,  1963319303199452777 ,  1497672437671054944 ,  587569660431161322 ,  1640139998254479090 ,  2075259177632397124 ,  1660504006066529017 ,  2098448192586763026 ,  2110557029504033173 ,  1649560687483066215 ,  1687307417474238546 ,  1953341749861443552 ,  1238894388766834138 ,  1741784409005434700 ,  961090363500770371 }, // x^2^532

{ 1383869753498066511 ,  1233178641086644718 ,  983298541932470614 ,  670427917276364832 ,  1594590175512954791 ,  1215937832423358959 ,  1903793454103261000 ,  2172773248760515308 ,  536919802769335489 ,  1192052568730958579 ,  1911240856897478532 ,  474283681972418461 ,  1641036102716815339 ,  1411044907503618945 ,  1001175747401094733 ,  301211985765609746 ,  87149012356549448 }, // x^2^533

{ 261502789562703795 ,  2039292976033461220 ,  1383214847400482168 ,  617091957131872918 ,  11536068257558752 ,  1594678644021354556 ,  75861344789992516 ,  571299992729856381 ,  1893030021771949746 ,  955316100521264931 ,  1984661955054505780 ,  2011703935837291735 ,  1088722461445557653 ,  1179450192264585869 ,  2187858293377977903 ,  925337903456050826 ,  646310246691765740 }, // x^2^534

{ 1067110047296065055 ,  1681438952663914065 ,  1912868211773513420 ,  157328871056550869 ,  1606539524193585050 ,  1826960759022391567 ,  333004198356937619 ,  1654504867491237582 ,  1675004459630487926 ,  71936368060300 ,  1918748055540842111 ,  1831638571780226898 ,  1740919230167947613 ,  1865657023192411640 ,  1938852440014266019 ,  294008211664586131 ,  1233347209473542392 }, // x^2^535

{ 1793671892574625082 ,  1130285389461734712 ,  2256576488166743893 ,  499935214652909272 ,  1154848893191704610 ,  2081455429764661190 ,  132180046418477201 ,  2017092125714317442 ,  1267352468781603191 ,  1320951561119166755 ,  1819380377385314624 ,  1968657260504504792 ,  795175392323167167 ,  138949840583543195 ,  2205578352394323161 ,  2162095450973436967 ,  144991897502595141 }, // x^2^536

{ 683024508151477474 ,  1048650848431764111 ,  222063671742581704 ,  1033879240711051787 ,  2065492192337124448 ,  1040278918101037386 ,  1671617116839141169 ,  1142933505587090141 ,  578155580871550072 ,  1397916094430897857 ,  1428297170374867004 ,  343607952975225879 ,  2212099097846398741 ,  659520961168888597 ,  1099474129656084832 ,  163872983722101162 ,  469023960460927874 }, // x^2^537

{ 1156093888444545360 ,  477156936151507247 ,  340730541722292373 ,  1631554541005175777 ,  186957487417408624 ,  926739184035859811 ,  1357420454563306968 ,  1018117004392525830 ,  363668834887049598 ,  562598157996426449 ,  1477447398167639388 ,  570432714234334068 ,  797165670329651828 ,  927307345530529305 ,  1474634744709399927 ,  2245157369991218355 ,  586948150323806327 }, // x^2^538

{ 267980979185924701 ,  1565212575331069567 ,  2036070690586106209 ,  850842928652802398 ,  1084085447966709199 ,  53832993778516816 ,  2236093215844622791 ,  361324823691972021 ,  673008249565884673 ,  1820092500335360408 ,  2155516028757275328 ,  1127096462606937572 ,  1358543969239663842 ,  2250855440900689232 ,  1750872323705159361 ,  1362418718801185396 ,  766489969361488653 }, // x^2^539

{ 259726699384885201 ,  1899046140654817590 ,  1306363085361051998 ,  2081068431945330598 ,  85108856372668378 ,  1324347710267860858 ,  637751183869623188 ,  2248344050276940170 ,  2150579538601288819 ,  1105694420952032875 ,  952722029405414348 ,  1555403505154017419 ,  2124427035047642831 ,  2043566203745579024 ,  2297745588334532887 ,  383567419144690253 ,  295720540923262008 }, // x^2^540

{ 1861487178452566675 ,  1408373324284705278 ,  1763971337054095719 ,  830694774025045496 ,  1222854782641346372 ,  1367429629279521808 ,  646871018425974561 ,  594901312199197009 ,  1347745885562466060 ,  1872975180162717691 ,  264288217725438485 ,  1587326205070728608 ,  636368619535248259 ,  1345295891447307881 ,  147325724921249282 ,  451151671184877984 ,  369152850758367779 }, // x^2^541

{ 57601454029724418 ,  8965969940832902 ,  1015753881352776595 ,  1272955433813211634 ,  1500099915165437510 ,  1505908015386217991 ,  2200135170747962774 ,  898929787588318880 ,  94174721104731682 ,  2001843482426425606 ,  1296185933469804867 ,  2067771350105959635 ,  244437767883759705 ,  115304829559788030 ,  1570793303180703663 ,  2075829501594571540 ,  1142789736568430590 }, // x^2^542

{ 1666339398200583535 ,  352498459116081565 ,  1296181205638634316 ,  943151467845475937 ,  1576590058032732201 ,  991197898685844101 ,  172929644755996237 ,  348073447626095407 ,  1119639777406826568 ,  569438225411593708 ,  2284730591697077199 ,  62008449878226744 ,  237477582274396799 ,  284434691435252140 ,  1765995421169394732 ,  1594311192946755506 ,  604156017409724002 }, // x^2^543

{ 495281382117458539 ,  205408453347821044 ,  1681558300394231421 ,  768174766423516475 ,  2178619873339022076 ,  765997180179026402 ,  102658969211254460 ,  895017413823772592 ,  415845558640901473 ,  601933504588026198 ,  969782745266307970 ,  855044719798584000 ,  2201199786666131647 ,  1734600819975899129 ,  1006570536931532428 ,  1130105133424164395 ,  1127782802378551477 }, // x^2^544

{ 2279222264094136096 ,  918140621636386578 ,  764051053229027870 ,  284578302438407812 ,  18442247946635388 ,  31558521455257101 ,  2294641003511473164 ,  1698376578572281576 ,  1937426563738038511 ,  809127821329680664 ,  1605067175404678954 ,  189108927488132646 ,  2042036507825496222 ,  311247532621290928 ,  1403615118897576721 ,  1715491077248039654 ,  1885408148480015203 }, // x^2^545

{ 1303703719061282226 ,  2273657250242045723 ,  1357599208936374367 ,  635087130862478000 ,  1079733394689912801 ,  1118552166016645994 ,  606420152917155530 ,  1461390879031614972 ,  104855646611743091 ,  1155586802792637483 ,  1332607042374593381 ,  485954066105103318 ,  755622945227398111 ,  100703703472763330 ,  2104688739715121722 ,  884188331727454949 ,  634077400952306173 }, // x^2^546

{ 951241189979241574 ,  605016052445461627 ,  2017048237067736326 ,  989577001336697190 ,  729360432477738078 ,  1891686993869218661 ,  1178287733434616934 ,  2257768800220793298 ,  386183991328298626 ,  1256576692082166247 ,  1989644154430757909 ,  677803830671194177 ,  233612802004492344 ,  1711459201958972728 ,  1493606889466009749 ,  2046914649687863407 ,  621460642415971297 }, // x^2^547

{ 1751267799283060659 ,  2256655950470869985 ,  984875549537277532 ,  2146590585318069342 ,  1539686266680012185 ,  25924698613741241 ,  633436588936544176 ,  1549785231759087109 ,  934442226797007335 ,  1754392592879566030 ,  1106511892592859350 ,  2133384541112231515 ,  483183511756818991 ,  1106565501063528265 ,  1107187069130962112 ,  2075632961004138802 ,  1789791698038574768 }, // x^2^548

{ 2168119975504999102 ,  12966540192424488 ,  681037686628849373 ,  194933206197471208 ,  1333929840850796338 ,  2213746149905142918 ,  1379421507825630221 ,  672612248824101656 ,  335325159518504203 ,  1891726571269569489 ,  838501400504296092 ,  1301803583046081493 ,  1750320150554937404 ,  67761357879073926 ,  1985524641824792115 ,  1623610528663151019 ,  1386176498075642165 }, // x^2^549

{ 1725373491573591435 ,  1047415675966614123 ,  1624486786194930396 ,  1270371958501591477 ,  915493804007886656 ,  2289688909829684654 ,  2049603525909314516 ,  148613066066958793 ,  726433066490138797 ,  1727275655898645687 ,  1975538226366019203 ,  394922847880149584 ,  462814805660060890 ,  1087564105835469854 ,  293119996590319709 ,  561334965874644379 ,  1159197305565422760 }, // x^2^550

{ 1167941940931666302 ,  186514058829683028 ,  284351732075544532 ,  1512524808600673716 ,  43881750227659649 ,  1609741131150076188 ,  1125658924019416678 ,  1011534154384997338 ,  732738003772919984 ,  277064895441615462 ,  1622677204527889015 ,  1938173807800443399 ,  2063908410477010918 ,  48488552879373011 ,  1181372505578038205 ,  135655409305671122 ,  1938059649851535706 }, // x^2^551

{ 337206759525681069 ,  1755948102219815566 ,  678765415558248151 ,  1298324546232590789 ,  484850096725079792 ,  1257105102233963714 ,  1905953290815779223 ,  1631095014301272964 ,  27718500263208979 ,  850594049252190048 ,  583977122492931140 ,  1805594056761021711 ,  1719584503850087741 ,  86179004867644487 ,  1270985715490800843 ,  644779453772114088 ,  1760476987252969926 }, // x^2^552

{ 853995884080236938 ,  369111089848434712 ,  559743835711047174 ,  1594116816433474826 ,  35842475217144553 ,  2171235383636811428 ,  765712905092749171 ,  941173958645971457 ,  608455896272321575 ,  560604720551238557 ,  1845649540169134916 ,  1718920359984151079 ,  1401755957661962268 ,  1423907959444825866 ,  1098311071256242778 ,  1528916661846841494 ,  1508764511499232241 }, // x^2^553

{ 1508791960279504205 ,  258737323991262817 ,  2078642398527055752 ,  2047350895402792396 ,  1386159783711778842 ,  2027047763369412816 ,  58440814829283402 ,  771612188948247815 ,  1023214498156083887 ,  844220082620919469 ,  868367506978956802 ,  460654633908296364 ,  2166638508094415965 ,  2152060517892743275 ,  2195460306854592144 ,  541094667496643303 ,  2293587356561486979 }, // x^2^554

{ 1214656342962583818 ,  757702747146927395 ,  2189480375257771121 ,  1816527465277679192 ,  138815664117127512 ,  1431350214961325326 ,  1187978151676236367 ,  1241806570983467832 ,  1284450519130025998 ,  1613889771871333389 ,  1061608822774622375 ,  381417471969594147 ,  2180713817066758069 ,  1573268990076834701 ,  905173682016321866 ,  1950470428221045886 ,  295485711949393017 }, // x^2^555

{ 819753035519543635 ,  538715919221404699 ,  2219429537481971437 ,  752393850271742164 ,  77092035906327214 ,  1114501521607776562 ,  1781083659997559466 ,  2154696941256077128 ,  1218985834766672241 ,  211616781192221457 ,  216411965392177616 ,  1670594910477530341 ,  714505138692032780 ,  2103941244229463221 ,  2088317230298558512 ,  1289168057502589286 ,  1125637208200814412 }, // x^2^556

{ 1962227768978994180 ,  1690314396114180321 ,  375465114688955302 ,  1798066679445937030 ,  462818969271150710 ,  1000261197673181741 ,  449380920277416447 ,  2291003141210787002 ,  1295168501047080160 ,  96762399469569918 ,  673992173479089483 ,  22635411461259319 ,  690222719954237584 ,  336145523606337534 ,  934574030654348792 ,  1843383288751365671 ,  1895192556383603959 }, // x^2^557

{ 22617970229773822 ,  1482887891197581503 ,  190614907712811083 ,  693572772014158846 ,  2013130701640461017 ,  2080394974487399438 ,  1946114308346089259 ,  990495636788197392 ,  1782544219386492119 ,  661191434765959775 ,  2044311906562246966 ,  1069940178504200830 ,  1955224049483069684 ,  2032802635089013840 ,  1312150198687724551 ,  1790320974584854699 ,  1722451636719440405 }, // x^2^558

{ 844270734204704640 ,  144569156478286025 ,  458434355726727522 ,  54107521005723860 ,  1370238437704355388 ,  635959790870056500 ,  580190401330281055 ,  1190738125435083361 ,  1508034736034112394 ,  384356207961213908 ,  2225422022608450401 ,  1307978291279169425 ,  1836303719023016910 ,  1688394455914172152 ,  1145318805385872977 ,  28214091532568837 ,  2271521727551942534 }, // x^2^559

{ 707823463534947110 ,  1322853985703915624 ,  1283965375707685336 ,  1235846354925013921 ,  2241994185339373382 ,  1492018342988994366 ,  354756557339909458 ,  2038382297798252220 ,  1998256038116472547 ,  1191608355457210551 ,  431624398742182860 ,  979989445521359703 ,  1874604718333228361 ,  863699730728041575 ,  303002850561444625 ,  351750811939826058 ,  2086150517074464097 }, // x^2^560

{ 938650030581340927 ,  1484495848437960060 ,  906987191271901948 ,  602861889366282296 ,  829217328598139608 ,  1170659179365524445 ,  136537691170306827 ,  2263710078069587303 ,  1916941180571676310 ,  601979432897153599 ,  2075793575506860756 ,  323119178978530887 ,  7962577560787577 ,  959899426480760280 ,  465731965934497666 ,  485390457639864074 ,  182578072602393068 }, // x^2^561

{ 845056639071703739 ,  2022501832369820141 ,  1551830514420859659 ,  712651911384419505 ,  1901754482233053508 ,  771255057924942569 ,  1624974410054352307 ,  1095555020243807180 ,  48292506949982043 ,  1924628272072369018 ,  354506579070195883 ,  2074354629523999876 ,  481327135146565647 ,  1980060303227111264 ,  374025770449242754 ,  2197437559867642882 ,  1347989415958383967 }, // x^2^562

{ 2284531938332082408 ,  2081446697857700129 ,  407946796335100046 ,  2284742117207953792 ,  310471079731850760 ,  1290351582175218317 ,  356273760688651989 ,  1910139122966702303 ,  380631662657835885 ,  2287374039009394440 ,  485797501634087358 ,  900166727954042323 ,  1169554169155638307 ,  1802101457472636718 ,  474217730229244850 ,  1004410789585399243 ,  588639270649921383 }, // x^2^563

{ 1502830393741044692 ,  502885788914291282 ,  1000641636223469702 ,  599101320428395707 ,  2179406366865940840 ,  358024368445093522 ,  1564063948136384394 ,  2159233836362830609 ,  1080529492271670836 ,  737489746809287059 ,  779539060254650511 ,  567680696243734902 ,  803303944361973640 ,  1778879099887965284 ,  2245121666542231208 ,  280989822576167540 ,  2256030916132704537 }, // x^2^564

{ 1678854909602034917 ,  1362544615060522442 ,  1338371892984320209 ,  1236016995662152262 ,  2266586199968442015 ,  1429545066557982811 ,  1057272578263945968 ,  1902362138019567296 ,  2027249515853796780 ,  1379458255463795129 ,  715456741316196206 ,  483096191538955481 ,  562452275970679447 ,  1899368762554121096 ,  1095218941132187764 ,  1182616069429316286 ,  2041204243971349283 }, // x^2^565

{ 1470384206682188393 ,  993056282930013636 ,  2186884113482342484 ,  1568683072711048719 ,  2141576651319216540 ,  1545634359639056795 ,  1451267712100583056 ,  583449252763690840 ,  743076343569515666 ,  880735478838481309 ,  1139615309806234587 ,  1405631565601779814 ,  1162183658026694832 ,  758880015783224342 ,  186132385266916884 ,  299496669995968613 ,  2115484609912800200 }, // x^2^566

{ 2269185097674681740 ,  368408303414772947 ,  2272053720551708375 ,  1632770988466577149 ,  2005375285078385494 ,  1280199055327212680 ,  1821048605091132853 ,  1567765420710739271 ,  1887232479893207919 ,  2230675230841783167 ,  1361156385383252011 ,  87414071150394174 ,  611216773833872434 ,  1860713875917257321 ,  1409775498386462397 ,  1690326679721511537 ,  83275546754626406 }, // x^2^567

{ 790112276274407445 ,  769228027816192054 ,  2205289911751012082 ,  696806322595942722 ,  1027622205316503460 ,  1843245508230762915 ,  181348331184190272 ,  1685443775553823790 ,  765291022909566823 ,  1159224061954534775 ,  1753833798485424233 ,  1131135065567349204 ,  1004980355827594316 ,  1518506610289278726 ,  1077654190664918382 ,  2049836184231549381 ,  186534172155365157 }, // x^2^568

{ 2214978522047075314 ,  1523958231722460737 ,  940540633818853829 ,  1502207690061290977 ,  1973166365192858632 ,  834167590580214713 ,  193198187070116104 ,  1200481096199012868 ,  1045382432042256669 ,  1043046955924667651 ,  1764037894194441387 ,  326214257353715662 ,  847610851336013740 ,  821380410810508534 ,  118611704753506829 ,  212870724809513245 ,  1049669518188822039 }, // x^2^569

{ 1186686984731968114 ,  492485282093702769 ,  2084974668148141701 ,  1954670681979820725 ,  158025799762360442 ,  1546024887967767275 ,  321250855908273132 ,  1839134851204623669 ,  2276195450093740401 ,  216550545714992720 ,  1147405869335269365 ,  1691809800568592195 ,  92303605959988217 ,  1444340652242832966 ,  1276065726666195728 ,  324774933653588943 ,  1312430778251072843 }, // x^2^570

{ 1914401401876468669 ,  1648868062260221854 ,  1464059991570664252 ,  875976384406088755 ,  859422856357056602 ,  1864309596721585546 ,  516857141291982496 ,  464095730580902783 ,  934796258134400377 ,  1817127289097008540 ,  655971001935077689 ,  680288930067826216 ,  229080197392071732 ,  921859076031383827 ,  2242366335784005876 ,  2231689867120356867 ,  102494160827714957 }, // x^2^571

{ 1047877418576672693 ,  850258032575291687 ,  545055359259387307 ,  939787324567586678 ,  588678011951547088 ,  397057856974487431 ,  226066134643513040 ,  2082177109290521202 ,  866921181848612589 ,  1377167370849823007 ,  923546628885112186 ,  1369468709683575581 ,  163347741879505736 ,  63898233712570047 ,  1837149445977275412 ,  81917523037053887 ,  1724871657333737359 }, // x^2^572

{ 2302595391708576272 ,  160985864775068781 ,  764451814383377581 ,  277491655221007957 ,  2065357052112797123 ,  2022995433719782905 ,  1992649094929659709 ,  534121146273082282 ,  2156280075941140984 ,  1374921692284132634 ,  7490180772772763 ,  724537764516397075 ,  1458937057231627989 ,  2128708058250265224 ,  1343524011236415024 ,  1632380201084339158 ,  218723970886479505 }, // x^2^573

{ 2117760628529948776 ,  2132935432038525560 ,  768781595656590901 ,  1400974401324013175 ,  1245919916200029631 ,  1432559837530198229 ,  1163713351897028899 ,  1923146399637847839 ,  1213782762495170404 ,  541538612171639467 ,  1998922401826244596 ,  1231856100509046580 ,  579861832524317272 ,  342657654818204433 ,  1849901794740458260 ,  2279916971309950571 ,  760499761682595396 }, // x^2^574

{ 2133951096918945939 ,  245427172150052114 ,  981217690063201632 ,  2240543031382098229 ,  72038350069308602 ,  597184866817343336 ,  328644676877024543 ,  183469578860761428 ,  1484808128953021465 ,  1223798115302938004 ,  622162521478669994 ,  1190522948279756621 ,  1770221081272204495 ,  300642277489993017 ,  323299607262829861 ,  1121130797903609296 ,  1341151672362966624 }, // x^2^575

{ 978460885885818525 ,  1015163128148908540 ,  1977562332100546610 ,  2245843532918136978 ,  52755975506265961 ,  105385627927894157 ,  1072178687697178180 ,  2189119972792037580 ,  1846365892857759522 ,  509344149142820345 ,  1764914861664878049 ,  1812095995141658018 ,  972953132412161570 ,  1195708733082367698 ,  49856206632166450 ,  1968416938611340445 ,  2224505860268693924 }, // x^2^576

{ 327862488024780357 ,  1457219179227842324 ,  2230322439686492524 ,  1014759634574897984 ,  268418936182947756 ,  1502956125465763704 ,  61942509184664792 ,  1760829907047336797 ,  1772759420678440112 ,  213631264477227981 ,  1260050930099550098 ,  1356809535018841668 ,  2084415297676571737 ,  392165946698433004 ,  823317575518425201 ,  1619251312183556482 ,  1475671506190286260 }, // x^2^577

{ 1056862459352327285 ,  650761739286013201 ,  711866770850654745 ,  995772669095949341 ,  638628037366816307 ,  319488467144926010 ,  146863433379310830 ,  1101916435400125848 ,  1403815239053246876 ,  1038453114803570080 ,  467259301797897322 ,  1758832454145478316 ,  359553557196445664 ,  1487375835680896666 ,  76291938912745016 ,  844090192831690228 ,  793442078170731733 }, // x^2^578

{ 1952837235717320563 ,  445430339419636400 ,  122525557445898274 ,  735595189125576324 ,  1842210194389261500 ,  1795223307042139024 ,  751864575701713084 ,  1119184159833325999 ,  700766534388037595 ,  393549636253210822 ,  1662606888175396198 ,  777088695141990512 ,  1949046976963730503 ,  659714914447882016 ,  1701619511100158747 ,  2106648518620441962 ,  2230177471130012298 }, // x^2^579

{ 2242830595691081287 ,  2153702821027923023 ,  2056432480338738486 ,  290410908425161907 ,  1981631243730724167 ,  1568909835655310667 ,  1868498486537085636 ,  443377197549410236 ,  1989171328893245017 ,  832914678880127633 ,  122754022462592655 ,  2074425172477633432 ,  450810432342594695 ,  1304742577811847983 ,  1915105538864887632 ,  398240897735172221 ,  854257484180168406 }, // x^2^580

{ 2200340348805749851 ,  459904691507133993 ,  1398407253135138579 ,  2223256024161740360 ,  605045994980733172 ,  1376873737656823533 ,  2154922394517147235 ,  1733951063198734927 ,  972687456101926553 ,  901838082782677970 ,  786315965699770073 ,  220890454000213916 ,  1082354731585990169 ,  1575625362271244736 ,  1934345059883646897 ,  610599809596399831 ,  342994941095873735 }, // x^2^581

{ 135197059374104208 ,  763606053289129943 ,  1762314800446038038 ,  1958652087897421237 ,  1759073139864458242 ,  777882068437388702 ,  1929274536720142390 ,  911476487406644691 ,  2214565556314441473 ,  52312220897893321 ,  1846728531947717379 ,  2200466005983704615 ,  262302790665678959 ,  2226903910074888712 ,  2217869614731681025 ,  56279709804267946 ,  2290324007417928442 }, // x^2^582

{ 2033888752861355042 ,  1032905065687727870 ,  1228320602981295572 ,  507759480115032778 ,  2286473996360684281 ,  2214758482416116553 ,  1809366838165194032 ,  1058372692666396235 ,  1752374720633908247 ,  783225856075389703 ,  1873729037970411918 ,  515708137390539649 ,  1758253278936485521 ,  2232321656689971091 ,  54123739319322403 ,  1645021409194515326 ,  128999663975261693 }, // x^2^583

{ 1548696340275626821 ,  1058994430734061174 ,  975903143090188182 ,  1753221425369032354 ,  642770173718849415 ,  72871801728857389 ,  1532366481913853262 ,  1685179220297983661 ,  221740024932865141 ,  572304397314497403 ,  1337436598601163606 ,  615711199196774965 ,  1856428633974911262 ,  2091160220746353073 ,  1268041182491440797 ,  1823672822413606168 ,  85882960870940069 }, // x^2^584

{ 1596386067990045239 ,  1082049082776347514 ,  964167595689777765 ,  2196584899289686937 ,  1775478739112084197 ,  668585706672723348 ,  148931594902619061 ,  1989674837193253753 ,  548537612282770315 ,  1196600058755599499 ,  40994948219245722 ,  585997221548702784 ,  1852642327512527056 ,  1962368180715517868 ,  647388617327737622 ,  1259769006990328711 ,  760195312392461746 }, // x^2^585

{ 718683317139573302 ,  320255468933257119 ,  1602065008785819027 ,  811539625884985377 ,  473380658706433597 ,  1912262500486946501 ,  470045053505719659 ,  2044632105828108925 ,  1926951883673524133 ,  2227261383404374216 ,  486603446225142281 ,  663210447975562680 ,  939660586393225605 ,  285150104962906352 ,  1736759590302217217 ,  1557356149606075095 ,  202888193082920775 }, // x^2^586

{ 1704252131283387210 ,  749862216125654467 ,  1151691390735685357 ,  1645735627001577084 ,  1704474438618968650 ,  748171802247682061 ,  2192661098187053709 ,  687285052039352483 ,  854812886608086510 ,  1254425642986060904 ,  1755448133247766409 ,  2011170119775364505 ,  2238140979402216154 ,  430682510031291698 ,  1344094596771993123 ,  1487316607310729340 ,  572027053290063200 }, // x^2^587

{ 490145643447149312 ,  139932608980010944 ,  655464257025259880 ,  394296759589777370 ,  505632773723135101 ,  1661411471151623719 ,  2151487638856430824 ,  1405022191175603197 ,  1752924822934387289 ,  540705285689857192 ,  321557534615604425 ,  357668350120260760 ,  1216461776875955281 ,  227959942379372288 ,  1843268809411056088 ,  1278964120834620296 ,  1475031348508152943 }, // x^2^588

{ 77196441284986546 ,  2116432279354126192 ,  477307664737255679 ,  2054284248076058866 ,  1902595507518183399 ,  892309383355817071 ,  632707825296532637 ,  2068144493364120654 ,  2269464712931165434 ,  1869550090717623995 ,  1230295724219128689 ,  265213902138950946 ,  2035223329405422145 ,  1942656580078323795 ,  1193709919134967859 ,  2129017786957860238 ,  1381902932849007959 }, // x^2^589

{ 1250515786673063664 ,  1170029995563455371 ,  1875333685750981198 ,  202301072852130868 ,  2043478147118479243 ,  1430654884019451218 ,  1932432698651776626 ,  906545901346634684 ,  300667563272986237 ,  1419422183404003464 ,  1991379549654933852 ,  647455449394880739 ,  694191473419776108 ,  276654368069638649 ,  327518818778853777 ,  776612031552014882 ,  754333675021774044 }, // x^2^590

{ 1494337748978015003 ,  2195753570178835409 ,  1756395833112881603 ,  1391513303657053978 ,  1469720991718755668 ,  1385546023774706695 ,  2174495928889154879 ,  1845916193012260560 ,  1508599521784450080 ,  1646177877887412686 ,  1600214236327823295 ,  481492506494972498 ,  607939032270197735 ,  1035787083329757142 ,  1867038566495963823 ,  1327899656420714797 ,  120455193991568778 }, // x^2^591

{ 1272270033586499361 ,  1355938084262029033 ,  1130547642931602057 ,  1571210037203668042 ,  109838764087692662 ,  665089929034619866 ,  1477929841071085716 ,  354404882207113353 ,  485207799804145872 ,  295271887824129191 ,  1434571796278931709 ,  1640884996563085247 ,  349639709072954139 ,  2048198598052923555 ,  524893036895320686 ,  2241615913147991183 ,  1476041495950183974 }, // x^2^592

{ 1191478803597513079 ,  179928617987606724 ,  1893767020776627979 ,  2279764014883753334 ,  2000307147025567411 ,  1317337023680470054 ,  2103432586600253524 ,  72635103884479033 ,  2120393133065940248 ,  630577942926446005 ,  184690001576938472 ,  724734720983686380 ,  2222559033240707306 ,  957823503700662442 ,  1530455664164510916 ,  1897774219578829063 ,  1565275869726572277 }, // x^2^593

{ 802169516563427070 ,  2266388954150660250 ,  1016109430849964502 ,  425155158957108184 ,  1509177586061010059 ,  735042889547641844 ,  435841014763623437 ,  552466077216086185 ,  1380304476103110982 ,  67287297735634201 ,  447647206919187346 ,  2068383112520459072 ,  1700683878491979663 ,  876911219219450197 ,  1724440678224620967 ,  2161646349907542457 ,  1646103618347380320 }, // x^2^594

{ 612487302971671307 ,  1375201488802854426 ,  381927427343438951 ,  522937542366715499 ,  344542446499511062 ,  2207818055791032517 ,  1589757712162343309 ,  800577551503529820 ,  1901158862710281177 ,  282526012842555873 ,  978266672216571827 ,  851384948740770681 ,  1847198461333883874 ,  1701120025203313330 ,  1579726260607682225 ,  425398491832428623 ,  470525020962949424 }, // x^2^595

{ 2088490725567484178 ,  485853289338907617 ,  1031877816384950332 ,  2204970857389602283 ,  1742289513696508421 ,  1887097317304814932 ,  611739697091118719 ,  620524462574164777 ,  866744533296715680 ,  113249287824674575 ,  1094771911813698402 ,  1113977650240159227 ,  597674782116109822 ,  745608676499086192 ,  1626698293359327222 ,  1428826314680818479 ,  2215603110047911506 }, // x^2^596

{ 376585604887633526 ,  1048922386135384477 ,  2142424598885808255 ,  675251024248975850 ,  1545330669815065990 ,  1857893574217764638 ,  2048982207690498307 ,  1533653988510679070 ,  335130892749828677 ,  104131490905494608 ,  2194266840041437356 ,  683616020969961024 ,  2038987316895996656 ,  917297429323765875 ,  821082630191764403 ,  1666057951671669384 ,  1130032032943836952 }, // x^2^597

{ 1041829149523127110 ,  1951902012535484719 ,  2152661804693326326 ,  1269908279296181874 ,  971953632796866049 ,  1266540697681616989 ,  2071865642155459708 ,  845692533418042777 ,  1241457687074422224 ,  161412923983560739 ,  390632925491962906 ,  353948016710720772 ,  2026700469979616031 ,  96483949026756707 ,  1773219759585896547 ,  993715984447404907 ,  1829150519300709735 }, // x^2^598

{ 877798478622581516 ,  942092329022409512 ,  1965158005364305044 ,  1441028099309753513 ,  500071448451573682 ,  1840680567469132321 ,  1468971950339879193 ,  1438547156759802726 ,  549595496973813971 ,  390734351826704403 ,  1073287705054298510 ,  48924576386470253 ,  1989283157579117827 ,  187450184005400512 ,  1949224458829795298 ,  1951496711030456157 ,  2167858042677531808 }, // x^2^599

{ 1996597802411685974 ,  114349701898949545 ,  1084220018985830117 ,  27462757590845683 ,  1081095686730339174 ,  2059987003957425282 ,  1321806358033656174 ,  1828830847664378831 ,  1455949950124722259 ,  477041639297859939 ,  2295506231622178555 ,  1152831764620264515 ,  396805345192155178 ,  2211812614000847756 ,  1962229985869550296 ,  848640427896833485 ,  548986687197667440 }, // x^2^600

{ 491594119422354073 ,  1742453895833966377 ,  1525602929914324406 ,  2076597071019376631 ,  865550929971279566 ,  997883929562138029 ,  610577039659314246 ,  1003718639233780963 ,  1951174507028809908 ,  1185710764340213666 ,  1693015712372426191 ,  317455835845832474 ,  1848795541252464763 ,  401332635199514101 ,  224456822593358236 ,  742409296289707819 ,  2118807906613325507 }, // x^2^601

{ 1293009127273644989 ,  1124274480353253667 ,  1810129434176838306 ,  259729045897854950 ,  1468101939413453436 ,  1111947300756487949 ,  501989537109533070 ,  1388491866061828170 ,  1312270603637319867 ,  1615629996017192524 ,  2134426516814571448 ,  548391958129447766 ,  1852805539182613949 ,  562903002408545356 ,  1582149854427086182 ,  1136405868045684314 ,  679089614531440081 }, // x^2^602

{ 786115001012233217 ,  572934901393961574 ,  906662352794537253 ,  1146052338661137694 ,  1717117579480047687 ,  2238979311119472866 ,  1282072454711511764 ,  913074328950615201 ,  1508062462715721091 ,  585179548016207318 ,  2263240538685112497 ,  212602680696443031 ,  2286609014980268736 ,  2160223566289961256 ,  457719410118216561 ,  892534882541613344 ,  1802928045269113919 }, // x^2^603

{ 412244890005542545 ,  882317329387396531 ,  1194216517008066508 ,  2113679728861695547 ,  754826510247388584 ,  143906396701128982 ,  1624499088946075077 ,  1517734354505524658 ,  1319665375831884375 ,  108068462219720639 ,  2061550944874121143 ,  1387192717449700004 ,  1784206536018821508 ,  1531349906234275665 ,  37851629185782003 ,  1964408593925350957 ,  675543575437141874 }, // x^2^604

{ 2224871078409190905 ,  1109593689354222314 ,  2282026642284422235 ,  1526142908234895159 ,  1185903712917801891 ,  2095637833179633450 ,  1684061887934064807 ,  1459637973270975973 ,  2080524575243813761 ,  948583036372588019 ,  1196499685761340006 ,  1838560132753716756 ,  1035066941117701308 ,  1633636158414982077 ,  2082933534230116545 ,  161677865488580416 ,  936609732430805584 }, // x^2^605

{ 409257345609329461 ,  1002655167281875678 ,  18470199587775000 ,  2184022032975947208 ,  148234944683729763 ,  2162572938835480438 ,  1216127754790813623 ,  1303762224282752882 ,  2064136757082740176 ,  518955562885803190 ,  1212819681300030394 ,  486562075491162613 ,  1283290324562294532 ,  285537074615846742 ,  321224661976450451 ,  2089560323722387545 ,  238275251292120492 }, // x^2^606

{ 2214799429964260023 ,  694747922604332887 ,  312465276764308387 ,  314561248153726635 ,  1765153767754864756 ,  1092506825381016367 ,  1475330654269567868 ,  1379961995474423138 ,  1380037933141733384 ,  767686246763038167 ,  2008926192530430712 ,  304944109134708306 ,  261183406799178779 ,  326684916374054817 ,  1565573428651338229 ,  1920871811367768904 ,  2089282340964983664 }, // x^2^607

{ 1133198628406257078 ,  977759521511265809 ,  1677025924443289061 ,  1729641585066757341 ,  2008693257322528571 ,  1421311051680525446 ,  2030362323209952329 ,  1921889928974731930 ,  1093849407549420400 ,  1836335521578497369 ,  407868381069619222 ,  10208083363410986 ,  1550647306879485613 ,  1032274398117330349 ,  889271835949247699 ,  508828554264478304 ,  313811787637759042 }, // x^2^608

{ 265345699270236054 ,  432199859212969150 ,  874558322199992458 ,  1131965915049654978 ,  298519106167257998 ,  10561017697598846 ,  1036255447357471766 ,  990617823928118898 ,  183634792806558724 ,  123066866696319933 ,  1303845119890513396 ,  1915539994626985074 ,  2074006245341867746 ,  442443726074289720 ,  1398504520177556356 ,  2177822133301482183 ,  1618315631184181685 }, // x^2^609

{ 2281731104267965972 ,  1565833165249376146 ,  2084133230658243672 ,  1466077014346623711 ,  2100113102745347532 ,  1509772133714278653 ,  1072220688498313491 ,  698970180110383152 ,  1702069119117461364 ,  786447122909266339 ,  1700725262288581680 ,  1201605067747742747 ,  1285909410275446050 ,  465944984489850383 ,  1581258227975449663 ,  1468892983675794809 ,  698369184018114847 }, // x^2^610

{ 721614432883106726 ,  1168570529527401436 ,  1558741633956623588 ,  625821206916677993 ,  1729484582639558993 ,  202818544256349027 ,  942584755749892262 ,  384853745237896372 ,  1878778990418394018 ,  2006136039057883787 ,  2000492489726132701 ,  1628427185964600209 ,  456608645446900334 ,  559232459658214239 ,  1425457030239632449 ,  287429815653732691 ,  1537646113987862352 }, // x^2^611

{ 168932144896498705 ,  1584765261048819233 ,  505170343346470483 ,  1661727371938733976 ,  58225995502531476 ,  1069662977595621678 ,  1534921440358104004 ,  1665430445118051718 ,  1421063952495178964 ,  1964407550412120431 ,  271153105959670492 ,  1303954744475330405 ,  1402417619008570585 ,  2067748336717081462 ,  635378162740815380 ,  1966388298672109925 ,  2276135507117268900 }, // x^2^612

{ 209516738545946051 ,  126364818200849051 ,  1934171883678238548 ,  964383175854976730 ,  1426910927989095855 ,  1225737711393069252 ,  545817142750996161 ,  1941470598322211846 ,  1888603351110522964 ,  1693849545801590503 ,  1145960696683862080 ,  1544154409134561092 ,  2057079562609173728 ,  333449282037994372 ,  2138527354092198252 ,  1857839034788608299 ,  96149649256385712 }, // x^2^613

{ 1680829537551411728 ,  2134567358027244766 ,  2233241443836144884 ,  882397883456946081 ,  1272913220199061746 ,  7737651019535212 ,  554065101789910033 ,  1237958145783195446 ,  1349598693130928045 ,  1212195573003581110 ,  1392362203075907328 ,  913027022784782751 ,  411026031675105306 ,  872967435734543202 ,  294712992119516029 ,  272260211295161762 ,  1636719289988127499 }, // x^2^614

{ 1664102028071512887 ,  1120090114167125698 ,  678215188753805783 ,  2138907996085142066 ,  1721907479145867302 ,  1086048121088426493 ,  282597371457515207 ,  2252562071097166874 ,  880501568810700587 ,  1513947683103494818 ,  118765667707656670 ,  133926112036721838 ,  567919878484948061 ,  79410381524546935 ,  1454226739666450321 ,  1554107108291191805 ,  1442417286842488623 }, // x^2^615

{ 1332050352400493680 ,  1321101334747797025 ,  124286076382368784 ,  2016174712781756901 ,  1731820299459346863 ,  545006347537545995 ,  1068874777126444609 ,  767902913608305873 ,  2267594054719694251 ,  1590335123866293983 ,  868156357394677648 ,  2249143243430804218 ,  347528411881171670 ,  1754162230470399965 ,  2070435897818234391 ,  1115080034382184583 ,  727133352156610422 }, // x^2^616

{ 355843900790907311 ,  363191259548386230 ,  1247995738147301243 ,  1637822561418229980 ,  2161312386590710193 ,  827479641142168242 ,  551490280055656820 ,  116433208177525158 ,  219743626260276428 ,  1031342298455165770 ,  956941458946607562 ,  1971095287608216767 ,  1197895453910374577 ,  758722071093994752 ,  777125942555276007 ,  1415023106509593626 ,  318953060159670820 }, // x^2^617

{ 2116009462654498660 ,  816261448355929278 ,  2272633719403001435 ,  573255612065573253 ,  968141095699581750 ,  1763511366639175689 ,  1173137617258455966 ,  481749672060198864 ,  750918552038818940 ,  519539565364235474 ,  1926446971667744959 ,  313336000413603815 ,  515893570456318246 ,  1497339799087374476 ,  478287049521531475 ,  1236625257683288385 ,  359335556903155074 }, // x^2^618

{ 284411256262633501 ,  1789336543617378723 ,  1511246373802383265 ,  937616018328420337 ,  138227234082520224 ,  1667286674451103850 ,  693601919163926025 ,  13505785652584434 ,  720941360686097746 ,  958299499540120680 ,  2165799682030822154 ,  569950597702024056 ,  655621404198221825 ,  1755168971312171278 ,  2043596507506252973 ,  723457099999134940 ,  542348768717201728 }, // x^2^619

{ 14465848372683520 ,  998603764202551247 ,  724938950471219595 ,  325209651844259777 ,  821482747183096187 ,  2233461975545399202 ,  1931346739821426382 ,  2293325179294131554 ,  243302969523090162 ,  1510343779638164774 ,  959965695881709576 ,  1831708054543131034 ,  883847853251423712 ,  1165812877635303160 ,  2007671568816070608 ,  44906786919162811 ,  1277515245984716365 }, // x^2^620

{ 631454168243258384 ,  1518494517880443027 ,  2287322260213126417 ,  73685056254736833 ,  942598569327822559 ,  1532992770144658039 ,  2229403027115926821 ,  842743413258963958 ,  1865437000699847820 ,  369346269649870812 ,  1254468168457467223 ,  559290945795297932 ,  1457535932113599162 ,  1926542806880420086 ,  437581909774872900 ,  1002726378631501533 ,  182139319307728305 }, // x^2^621

{ 1539785164984779758 ,  2045565025200148915 ,  120085588989386291 ,  1851688824536244542 ,  1980194980609785714 ,  1768302648237998273 ,  937420755919973678 ,  2210791368804399387 ,  328708870574712683 ,  1007467584698506438 ,  928652866696652949 ,  2228808519508067530 ,  1642940388306822455 ,  1618552641970232946 ,  108597732587521580 ,  1988412225258659433 ,  1660448156453751001 }, // x^2^622

{ 131526482323749840 ,  503434317381483751 ,  1560765838807596555 ,  1027232124412255852 ,  654779282895217397 ,  2273302731188942421 ,  1551720915692515402 ,  1826828338382462468 ,  604485879763220723 ,  1057737165588422306 ,  106531819221246981 ,  425783162197347937 ,  489984741847314768 ,  523779814301405487 ,  1865105780134618992 ,  1677103130386412459 ,  318671848522267610 }, // x^2^623

{ 1639089103549084783 ,  2260330277551218496 ,  754892153485552977 ,  2047710903557972804 ,  2229320667962732246 ,  307697180850646681 ,  423771572510991367 ,  17754501861697173 ,  1238910344976760345 ,  339140894113319789 ,  1445828176546996833 ,  2042997377381022070 ,  2207437390435628445 ,  551144982338719121 ,  384944764939825909 ,  1474198471229196249 ,  1659242296891127810 }, // x^2^624

{ 795309147716951006 ,  1452711178054134843 ,  1664782861363981358 ,  1006405310566793269 ,  971226122955741921 ,  752466486101956684 ,  1168643572163743942 ,  1369730711422397346 ,  627654132922219702 ,  2104230754718381006 ,  1588968809641801416 ,  440238626377162178 ,  732134947635506053 ,  1160506871469695759 ,  395151165185992129 ,  957703098223865589 ,  1844022625316455575 }, // x^2^625

{ 1964890520513046164 ,  829237203674997277 ,  1045128031232022678 ,  930832386865989645 ,  501678256081635892 ,  917829023495631073 ,  881831635034313098 ,  394230956869842550 ,  71895808688446751 ,  1874849501133161758 ,  645524260474403693 ,  301748792077270066 ,  1063834409487547589 ,  521966335959428443 ,  735272076777014321 ,  753999758410424987 ,  292792781171616060 }, // x^2^626

{ 1976769506118304887 ,  328773719961068411 ,  538585052711106822 ,  2240145575420715174 ,  1291008337036982313 ,  1034350880207773432 ,  1888353090737342761 ,  1978548213503129609 ,  743903982747501766 ,  2240529593386457105 ,  229308641102639039 ,  1678061221588734337 ,  1248567483083113072 ,  1768596733943584627 ,  1268676342298500384 ,  216631313378191648 ,  1830565281699080858 }, // x^2^627

{ 222335339024805767 ,  146026382943095887 ,  1745991091656491682 ,  364572344137903996 ,  491711339269051713 ,  1355802228883764600 ,  2287182613212117809 ,  1628584733201147266 ,  1730760626189599809 ,  1568968802576521126 ,  311144633575093285 ,  1257985237943883059 ,  2071814648333720000 ,  687885153740420654 ,  134464058021476074 ,  1159135545218675174 ,  1046016356313666266 }, // x^2^628

{ 2218238585802550263 ,  1796073488881922547 ,  108147160614248440 ,  2147497677742828723 ,  940377749228338777 ,  166702239352185028 ,  14837186174842935 ,  1881782378259233528 ,  442792017566957734 ,  1550966771279227259 ,  1902483003501989215 ,  660158154476503000 ,  673105302791153020 ,  749237522153030898 ,  904762939332248683 ,  1085180116177134000 ,  32410680807805190 }, // x^2^629

{ 1565357784323540059 ,  2004567400386087250 ,  563556471193393347 ,  2176504294263508416 ,  1354931238262729258 ,  1868302215996805633 ,  1039100426489789297 ,  540592058162156675 ,  2043699722662531658 ,  1444431990121166178 ,  931232743518887886 ,  2055389098129788451 ,  1235540167366576898 ,  1090861860232324736 ,  1732281293507875925 ,  1285969498415451875 ,  1757636524168521125 }, // x^2^630

{ 2235287391127506303 ,  1401636021589024058 ,  2051095569701746947 ,  644542369524563085 ,  1717409066688580033 ,  839878318363782234 ,  964338104498650467 ,  193111125860611415 ,  41226228181058700 ,  659802192172642300 ,  481644033258717967 ,  1649597320109931775 ,  1169882823869469374 ,  2274302558105332483 ,  318951044439334783 ,  1790653470898866515 ,  1273274762300650850 }, // x^2^631

{ 300315997535491308 ,  2172766343039137706 ,  1640387745692897945 ,  1527918068635484048 ,  882433389130238556 ,  519571876740982088 ,  840810181955840317 ,  967721991925117462 ,  757987265005423150 ,  692837321139327525 ,  1347675233763916865 ,  1262263002781039893 ,  366275012213686434 ,  1720518743710286974 ,  788123176573078844 ,  118677909193430055 ,  331168466283501375 }, // x^2^632

{ 2204724021283944336 ,  614667311623152683 ,  1844295995732490949 ,  810904822281747865 ,  1057803411491878257 ,  2192784617269518738 ,  525328525709235908 ,  1625675926696707314 ,  2219520795837515346 ,  1111578739349697071 ,  850264822465802254 ,  1875225445537474991 ,  447934460789202298 ,  1577693216054964532 ,  2069208581718856575 ,  1844971513814599533 ,  1100047974003437056 }, // x^2^633

{ 212063919658939349 ,  1955050766498331130 ,  1016643445611085361 ,  310923000117302398 ,  1108559594383531725 ,  1574096462279058604 ,  2058721019806527059 ,  1495555971494333315 ,  1290148625133033915 ,  1726857770258998357 ,  839176472902748525 ,  1938011636448040623 ,  1509176691584187835 ,  1684996881754410913 ,  339436873350779373 ,  1476161426144378753 ,  1443502235558451143 }, // x^2^634

{ 2154146131717687018 ,  1161290317713326050 ,  362912384144320809 ,  1110332862415808487 ,  716729690755657231 ,  1048283751964241153 ,  1336749180461386759 ,  2280348028206541371 ,  1259918326978354156 ,  1980538993489881425 ,  1333565492277683625 ,  514625871678292821 ,  1173265787958600132 ,  569893472219264300 ,  812175666077334991 ,  1255701238860433930 ,  79993120593823961 }, // x^2^635

{ 409217992666385378 ,  728224420299246128 ,  1959936179576785895 ,  1444295307550571182 ,  998063979246407739 ,  554682441648959407 ,  2073233365208088525 ,  1349861235690939694 ,  1097281043663983707 ,  1649731108960027223 ,  2214398868943408274 ,  1449553717147663103 ,  2196500661345051527 ,  2171871885380914429 ,  491427195052411299 ,  470657045051378516 ,  554494320060727622 }, // x^2^636

{ 2211653713656632985 ,  1473087100459732074 ,  345838890415354412 ,  1445623252958685565 ,  1041419654691395738 ,  2188283266242968331 ,  1417777133392604784 ,  1154814074996216406 ,  1442989098963226134 ,  901383116742510681 ,  497212216058355345 ,  999731586376214828 ,  817831951035275271 ,  159325628653133022 ,  120914548489015393 ,  1449314802372642427 ,  471104110110980884 }, // x^2^637

{ 323038291515073224 ,  265070044955668299 ,  1053676807411822176 ,  2247226908652601286 ,  2083016828474103062 ,  273234004102527575 ,  11857926472424922 ,  2055314572998661967 ,  311921811886787132 ,  1870415531374949840 ,  21265502209057804 ,  678112836970959466 ,  926490438737525589 ,  2027644346888215584 ,  495557988374717777 ,  1817993454480752541 ,  2141242677896973878 }, // x^2^638

{ 1445163042542523231 ,  1767410038768111356 ,  462427149761798838 ,  1007504177155879483 ,  1181224656949316486 ,  357022206754812124 ,  1898271363493710619 ,  1339720449183752193 ,  883758224923786172 ,  36020177946849132 ,  461791639260431974 ,  1411405712850290498 ,  701758173778161093 ,  1222887871945629218 ,  1174213062638935911 ,  326342461405188581 ,  2211032895429848005 } // x^2^639

};
    ;
//    const	myuint skipMat8[128][8] =
//#include "mixmax_skip_N8.c"
    ;

    const myuint* skipMat[128];
    switch (N) {
//        case 240:
//            for (int i=0; i<128; i++) { skipMat[i] = skipMat240[i];}
//            break;
        case 17:
            for (int i=0; i<128; i++) { skipMat[i] = skipMat17[i];}
            break;
//        case 8:
//            for (int i=0; i<128; i++) { skipMat[i] = skipMat8[i];}
//            break;

        default:
            exit(-1);
            break;
    }

    myID_t IDvec[4] = {streamID, runID, machineID, clusterID};
    int r,i,j,  IDindex;
    myID_t id;
    myuint Y[N], cum[N];
    myuint coeff;
    myuint* rowPtr;
    myuint sumtot=0;


    for (i=0; i<N; i++) { Y[i] = Vin[i]; sumtot = modadd( sumtot, Vin[i]); } ;
    for (IDindex=0; IDindex<4; IDindex++) { // go from lower order to higher order ID
        id=IDvec[IDindex];
        //printf("now doing ID at level %d, with ID = %d\n", IDindex, id);
        r = 0;
        while (id){
            if (id & 1) {
                rowPtr = (myuint*)skipMat[r + IDindex*8*sizeof(myID_t)];
                //printf("free coeff for row %d is %llu\n", r, rowPtr[0]);
                for (i=0; i<N; i++){ cum[i] = 0; }
                for (j=0; j<N; j++){              // j is lag, enumerates terms of the poly
                    // for zero lag Y is already given
                    coeff = rowPtr[j]; // same coeff for all i
                    //printf("coeff = %llu, ", coeff);
                    for (i =0; i<N; i++){
                        cum[i] =  fmodmulM61( cum[i], coeff ,  Y[i] ) ;
                    }
                    sumtot = iterate_raw_vec(Y, sumtot);
                }
                sumtot=0;
                for (i=0; i<N; i++){ Y[i] = cum[i]; sumtot = modadd( sumtot, cum[i]); } ;
            }
            id = (id >> 1); r++; // bring up the r-th bit in the ID
        }
    }
    sumtot=0;
    for (i=0; i<N; i++){ Vout[i] = Y[i]; sumtot = modadd( sumtot, Y[i]); } ;  // returns sumtot, and copy the vector over to Vout
    return (sumtot) ;
}

#if defined(__x86_64__)
PREF inline myuint mixmax_engine POST ::mod128(__uint128_t s){
    myuint s1;
    s1 = ( (  ((myuint)s)&MERSBASE )    + (  ((myuint)(s>>64)) * 8 )  + ( ((myuint)s) >>BITS) );
    return	MOD_MERSENNE(s1);
}

PREF inline myuint mixmax_engine POST ::fmodmulM61(myuint cum, myuint a, myuint b){
    __uint128_t temp;
    temp = (__uint128_t)a*(__uint128_t)b + cum;
    return mod128(temp);
}

#else // on all other platforms, including 32-bit linux, PPC and PPC64, ARM and all Windows
#define MASK32 0xFFFFFFFFULL

PREF inline myuint mixmax_engine POST ::fmodmulM61(myuint cum, myuint s, myuint a)
{
    register myuint o,ph,pl,ah,al;
    o=(s)*a;
    ph = ((s)>>32);
    pl = (s) & MASK32;
    ah = a>>32;
    al = a & MASK32;
    o = (o & M61) + ((ph*ah)<<3) + ((ah*pl+al*ph + ((al*pl)>>32))>>29) ;
    o += cum;
    o = (o & M61) + ((o>>61));
    return o;
}
#endif

PREF myuint mixmax_engine POST ::modadd(myuint foo, myuint bar){
#if (defined(__x86_64__) || defined(__i386__)) &&  defined(__GNUC__) && defined(USE_INLINE_ASM)
    //#warning Using assembler routine in modadd
    myuint out;
    /* Assembler trick suggested by Andrzej Görlich     */
    __asm__ ("addq %2, %0; "
             "btrq $61, %0; "
             "adcq $0, %0; "
             :"=r"(out)
             :"0"(foo), "r"(bar)
             );
    return out;
#else
    return MOD_MERSENNE(foo+bar);
#endif
}

PREF void mixmax_engine POST ::print_state(){ // (std::ostream& ost){
    int j;
    fprintf(stdout, "mixmax state, file version 1.0\n" );
    fprintf(stdout, "N=%u; V[N]={", rng_get_N() );
    for (j=0; (j< (rng_get_N()-1) ); j++) {
        fprintf(stdout, "%llu, ", S.V[j] );
    }
    fprintf(stdout, "%llu", S.V[rng_get_N()-1] );
    fprintf(stdout, "}; " );
    fprintf(stdout, "counter=%u; ", S.counter );
    fprintf(stdout, "sumtot=%llu;\n", S.sumtot );
}

PREF mixmax_engine POST mixmax_engine POST ::Branch(){
    S.sumtot = iterate_raw_vec(S.V.data(), S.sumtot); S.counter = N-1;
    mixmax_engine tmp=*this;
    tmp.BranchInplace();
    return tmp;
}

PREF mixmax_engine POST & mixmax_engine POST ::operator=(const mixmax_engine& other ){
    S.V = other.S.V;
    S.sumtot = other.S.sumtot;
    S.counter = other.S.counter;
    return *this;
}

PREF void mixmax_engine POST ::BranchInplace(){
    // Dont forget to iterate the mother, when branching the daughter, or else will have collisions!
    // a 64-bit LCG from Knuth line 26, is used to mangle a vector component
    constexpr myuint MULT64=6364136223846793005ULL;
    myuint tmp=S.V[1];
    S.V[1] *= MULT64; S.V[1] &= MERSBASE;
    S.sumtot = modadd( S.sumtot , S.V[1] - tmp + MERSBASE);
    S.sumtot = iterate_raw_vec(S.V.data(), S.sumtot);// printf("iterating!\n");
    S.counter = N-1;
}

//template class mixmax_engine<240>;// TEMPLATE
//template class mixmax_engine<17>;// TEMPLATE

//==========================================================================

// A derived class to generate random numbers using the MixMax algorithm.

class MixMaxRndm : public Pythia8::RndmEngine {

public:

  // Constructor.
  MixMaxRndm(uint32_t seed0 = 0, uint32_t seed1 = 0, uint32_t seed2 = 0,
             uint32_t seed3 = 0) : rndm(seed0, seed1, seed2, seed3) {;}

  // Return a flat random number.
  double flat() {return rndm.get_next_float();}

protected:

  // Internal MixMax randum number generator.
  mixmax_engine rndm;

};

//==========================================================================

#endif		// __MIXMAX_H
