#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Copyright (c) 2015, Gaurav Bansal
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

import argparse
import sys
import os
import hashlib
import zipfile
import json
import textwrap
import string
import shutil
import re
import difflib
import subprocess
import time

import requests

simple_baseurl = 'https://pypi.org/simple/'
json_baseurl = 'https://pypi.org/pypi/'
package_data = {}

def fetch_as_json(url):
    headers = {'Accept': 'application/vnd.pypi.simple.v1+json'}
    r = requests.get(url, headers=headers)
    r.raise_for_status()
    return r.json()

def populate_package_data(pkgname, pkgvers=None):
    if pkgname not in package_data:
        package_data[pkgname] = {}
    if pkgvers not in package_data[pkgname]:
        if pkgvers:
            package_data[pkgname][pkgvers] = fetch_as_json(json_baseurl+pkgname+'/'+pkgvers+'/json')
        else:
            package_data[pkgname][pkgvers] = fetch_as_json(json_baseurl+pkgname+'/json')

def get_release_urls(pkgname, pkgvers):
    populate_package_data(pkgname, pkgvers)
    return package_data[pkgname][pkgvers]['urls']

def list_all():
	""" Lists all packages available in pypi database """
	for package in fetch_as_json(simple_baseurl)['projects']:
		print(package['name'])


class Package_release_data:
	def __init__(self, attributes):
		variables = list(attributes.keys())
		for v in variables:
			setattr(self, v, attributes[v])

	def __str__(self):
		output = "Name\t\t" + self.name + "\nVersion\t\t" + self.version
		if self.maintainer and self.maintainer != 'UNKNOWN':
			output += "\nMaintainter\t" + self.maintainer
		output += "\nHome_page\t" + self.home_page
		output += "\nPackage_url\t" + self.package_url
		if self.download_url and self.download_url != 'UNKNOWN':
			output += "\nDownload_url\t" + self.download_url
		output += "\nRelease_url\t" + self.release_url
		if self.docs_url and self.docs_url != 'UNKNOWN':
			output += "\nDocs_url\t" + self.docs_url
		output += "\nDescription\t" + self.description
		return output

def release_data(pkg_name, pkg_version):
	""" Fetches the release data for a paticular package based on
	the package_name and package_version """
	populate_package_data(pkg_name, pkg_version)
	values = package_data[pkg_name][pkg_version]['info']
	if values:
		package = Package_release_data(values)
		print(package)
	else:
		print("No such package found.")
		print("Please specify the exact package name.")
	return


def fetch(pkg_name, dict):
	""" Fetches the distfile for a particular package name and release_url """
	print("Fetching distfiles...")
	checksum_sha256 = dict['digests']['sha256']
	parent_dir = './sources'
	home_dir = parent_dir + '/' + 'python'
	src_dir = home_dir + '/py-' + pkg_name
	if not os.path.exists(parent_dir):
		os.makedirs(parent_dir)
	if not os.path.exists(home_dir):
		os.makedirs(home_dir)
	if not os.path.exists(src_dir):
		os.makedirs(src_dir)

	url = dict['url']
	file_name = src_dir + '/' + dict['filename']

	if not os.path.exists(file_name):
            r = requests.get(url)
            if r.status_code == 200	:
                    with open(file_name, 'wb') as f:
                            meta = r.headers['content-length']
                            file_size = int(meta)

                            pattern = ["-","\\", "|", "/"]
                            patternIndex = 0
                            file_size_dl = 0
                            block_sz = 1024
                            # toolbar_width = int(file_size/block_sz)+1
                            toolbar_width = 30
                            # sys.stdout.write("["+"-"*int(file_size_dl/block_sz)+pattern[patternIndex]+" "*int((file_size-file_size_dl)/block_sz-1)+"] "+" "+"(%5d Kb of %5d Kb)"% (file_size_dl, file_size))
                            print(file_size)
                            incr = int(file_size/50)
                            print(incr)
                            count = 0
                            left = 49
                            sys.stdout.write("["+"-"*int(count)+pattern[patternIndex]+" "*int(left)+"]"+"(%5d Kb of %5d Kb)"% (file_size_dl, file_size))
                            sys.stdout.flush()
                            buff = 0
                            for chunk in r.iter_content(block_sz):
                                    f.write(chunk)
                                    if file_size_dl+block_sz > file_size:
                                            file_size_dl = file_size
                                            count += 1
                                            left -= 1						
                                            sys.stdout.write("\r")
                                            sys.stdout.write("["+"-"*int(count+1)+"]"+"(%5d Kb of %5d Kb)"% (file_size_dl, file_size))
                                            time.sleep(0.1)
                                            sys.stdout.flush()
                                            buff = 0
                                            patternIndex = (patternIndex + 1)%4
                                    else:
                                            file_size_dl += block_sz
                                    buff += block_sz
                                    if(buff >= incr):
                                            count += 1
                                            left -= 1						
                                            sys.stdout.write("\r")
                                            time.sleep(0.1)
                                            sys.stdout.flush()
                                            buff = 0
                                            patternIndex = (patternIndex + 1)%4
                                    patternIndex = (patternIndex + 1)%4
                                    sys.stdout.write("\r")
                                    if(file_size_dl+block_sz >= file_size):
                                                    sys.stdout.write("["+"-"*int(count+1)+"]"+"(%5d Kb of %5d Kb)"% (file_size_dl, file_size))
                                    else:
                                            sys.stdout.write("["+"-"*int(count)+pattern[patternIndex]+" "*int(left)+"]"+"(%5d Kb of %5d Kb)"% (file_size_dl, file_size))
                    sys.stdout.write(" OK\n")
                    sys.stdout.flush()

	checksum_sha256_calc = hashlib.sha256(open(file_name,'rb').read()).hexdigest()

	if str(checksum_sha256) == str(checksum_sha256_calc):
		print('Successfully fetched')
		ext = file_name.split('.')[-1]
		if ext == 'egg':
			zip = zipfile.ZipFile(file_name)
			for name in zip.namelist():
				if name.split("/")[0] == "EGG-INFO":
					zip.extract(name, src_dir)
		return file_name
	else:
		print('Aborting due to inconsistency on checksums (expected {0} != downloaded {1})\n'.format(checksum_sha256, checksum_sha256_calc))
		try:
			os.remove(file_name)
		except OSError as e:
			print(("Error: {0} - {1}.".format(e.filename, e.strerror)))
		return False


def fetch_url(pkg_name, pkg_version, checksum=False, deps=False):
	""" Checks for the checksums and dependecies for a particular python package
	on the basis of package_name and package_version """
	urls = get_release_urls(pkg_name, pkg_version)
	if checksum:
		for value in urls:
			if value['packagetype'] == 'sdist':
				return fetch(pkg_name, value)
	else:
		for value in urls:
			return fetch(pkg_name, value)


def dependencies(pkg_name, pkg_version, deps=False):
	""" Finds dependencies for a particular package on the basis of
	package_name and package_version """
	flag = False
	if not deps:
		return
	urls = get_release_urls(pkg_name, pkg_version)
	for value in urls:
		if value['packagetype'] == 'sdist':
			fetch(pkg_name, value)
	try:
		with open('./sources/python/py-'
				  + pkg_name + '/EGG-INFO/requires.txt') as f:
			list = f.readlines()
			list = [x.strip('\n') for x in list]
		f.close()
		try:
			if flag:
				shutil.rmtree('./sources/python/py-' + pkg_name + '/EGG-INFO',
							  ignore_errors=True)
				items = os.listdir('./sources/python/py-' + pkg_name)
				for item in items[:]:
					if item.split('.')[-1] not in ('gz', 'zip'):
						os.remove('./sources/python/py-'
								  + pkg_name + '/' + item)
						items.remove(item)
				if not items:
					os.rmdir('./sources/python/py-' + pkg_name)
		except:
			pass
		return list
	except:
		try:
			if flag:
				shutil.rmtree('./sources/python/py-'+pkg_name+'/EGG-INFO',
							  ignore_errors=True)
				items = os.listdir('./sources/python/py-'+pkg_name)
				for item in items[:]:
					if item.split('.')[-1] not in ('gz', 'zip'):
						os.remove('./sources/python/py-'+pkg_name+'/'+item)
						items.remove(item)
				if not items:
					os.rmdir('./sources/python/py-'+pkg_name)
		except:
			pass
		return False


def create_diff(old_file, new_file, diff_file):
	""" Creates a diff file for an existent port """
	with open(old_file) as f:
		a = f.readlines()

	with open(new_file) as f:
		b = f.readlines()

	diff_string = difflib.unified_diff(a, b, "Portfile.orig", "Portfile")
	with open(diff_file, 'w') as d:
		d.writelines(diff_string)


def search_port(name):
	""" Searches for an existent port by its name """
	try:
		command = "port file name:^py-" + name + "$"
		command = command.split()
		existing_portfile = \
			subprocess.check_output(command, stderr=subprocess.STDOUT).strip()
		return existing_portfile
	except Exception:
		return False


def checksums(pkg_name, pkg_version):
	""" Generates checksums for a package on the basis of the distfile fetched by
	its package_name and package_version """
	flag = False
	print("Attempting to fetch distfiles...")
	file_name = fetch_url(pkg_name, pkg_version, True)
	if file_name:
		checksums = {}
		try:
			print("Generating checksums...")

			for chk in ['rmd160', 'sha256']:
				command = "openssl " + chk + " " + file_name
				command = command.split()
				val = str(subprocess.check_output(command, stderr=subprocess.STDOUT))
				val = val.split('=')[1][1:-1]
				checksums[chk] = val
			checksums['size'] = os.path.getsize(file_name)

			dir = '/'.join(file_name.split('/')[0:-1])
			if flag:
				os.remove(file_name)
			try:
				if flag:
					os.rmdir(dir)
			except OSError:
				pass
			return checksums
		except:
			print("Error\n")
			return


def search_distfile(name, version):
	""" Searches if the distfile listed is present or not """
	try:
		url = get_release_urls(name, version)[0]['url']
		r = requests.get(url)
		if not r.status_code == 200:
			raise Exception('No distfile')
	except:
		print("No distfile found")
		print("Please set a DISTFILE env var before generating the portfile")
		sys.exit(0)


def search_license(license):
	""" Maps the license passed to the already present list of
	licences available in Macports """
	license = license.lower()
	patterns = ['.*mit.*', '.*apache.*2', '.*apache.*', '.*bsd.*', '.*agpl.*3',
				'.*agpl.*2', '.*agpl.*', '.*affero.*3', '.*affero.*2',
				'.*affero.*', '.*lgpl.*3', '.*lgpl.*2', '.*lgpl.*', '.*gpl.*3',
				'.*gpl.*2', '.*gpl.*', '.*general.*public.*license.*3',
				'.*general.*public.*license.*2',
				'.*general.*public.*license.*', '.*mpl.*3', '.*mpl.*2',
				'.*mpl.*', '.*python.*license.*', '^python$', '.*']
	licenses = ['MIT', 'Apache-2', 'Apache', 'BSD', 'AGPL-3', 'AGPL-2', 'AGPL',
				'AGPL-3', 'AGPL-2', 'AGPL', 'LGPL-3', 'LGPL-2', 'LGPL',
				'GPL-3', 'GPL-2', 'GPL', 'GPL-3', 'GPL-2', 'GPL', 'MPL-3',
				'MPL-2', 'MPL', 'Python', 'Python', 'NULL']
	for i in range(len(patterns)):
		match = re.search(patterns[i], license)
		if match:
			return licenses[i]

def is_purepython(pkg_name, pkg_version):
    """
    Return True if the specified package is pure Python, or False if it
    contains native code.
    """
    urls = get_release_urls(pkg_name, pkg_version)
    for url in urls:
        if url['packagetype'] == 'bdist_wheel':
            if url['filename'].endswith('-none-any.whl'):
                return True
            else:
                return False
    return False


def port_testing(name, portv='311'):
	""" Port Testing function for various phase implementations """
	euid = os.geteuid()
	if euid:
		args = ['sudo', sys.executable] + sys.argv + [os.environ]
		os.execlpe('sudo', *args)

	for phase in [port_fetch, port_checksum, port_extract, port_configure,
				  port_build, port_destroot, port_clean]:
		print((phase.__name__))
		phase_output = phase(name, portv)
		if phase_output:
			print((phase.__name__ + " - SUCCESS"))
		else:
			print((phase.__name__ + " FAILED"))
			port_clean(name, portv)
			print("Exiting")
			sys.exit(1)

		euid = os.geteuid()
		if euid:
			args = ['sudo', sys.executable] + sys.argv + [os.environ]
			os.execlpe('sudo', *args)


def port_fetch(name, portv='311'):
	""" Fetch phase implementation """
	try:
		command = "sudo port -t fetch dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_checksum(name, portv='311'):
	""" Checksum phase implementation """
	try:
		command = "sudo port -t checksum dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_extract(name, portv='311'):
	""" Checksum phase implementation """
	try:
		command = "sudo port -t extract dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_patch(name, portv='311'):
	""" Patch phase implementation """
	try:
		command = "sudo port -t patch dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_configure(name, portv='311'):
	""" Configure phase implementation """
	try:
		command = "sudo port -t configure dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_build(name, portv='311'):
	""" Build phase implementation """
	try:
		command = "sudo port -t build dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_destroot(name, portv='311'):
	""" Destroot phase implementation """
	try:
		command = "sudo port -t destroot dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False


def port_clean(name, portv='311'):
	""" Clean phase implementation """
	try:
		command = "sudo port -t clean dports/python/py-" + \
				  name + " subport=py" + portv + "-" + name
		command = command.split()
		subprocess.check_call(command, stderr=subprocess.STDOUT)
		return True
	except:
		return False

def create_portfile(dict, file_name, dict2):
	""" Creates a portfile on the basis of the release_data and release_url fetched
	on the basis of package_name and package_version """
	search_distfile(dict['name'], dict['version'])
	print(("Creating Portfile for pypi package " + dict['name'] + "..."))
	with open(file_name, 'w') as file:
		file.write('# -*- coding: utf-8; mode: tcl; tab-width: 4; ')
		file.write('indent-tabs-mode: nil; c-basic-offset: 4 ')
		file.write('-*- vim:fenc=utf-8:ft=tcl:et:sw=4:ts=4:sts=4\n')
		file.write('\n')
		file.write('PortSystem          1.0\n')
		file.write('PortGroup           python 1.0\n\n')

		file.write('name                py-{0}\n'.format(dict['name']))
		file.write('version             {0}\n'.format(dict['version']))

		if is_purepython(dict['name'], dict['version']):
			file.write('platforms           {darwin any}\n')
			file.write('supported_archs     noarch\n')
		license = dict['license']
		license = search_license(license)
		file.write('license             {0}\n'.format(license))

		if dict['maintainer']:
			maintainers = ' '.join(dict['maintainer'])
			if not maintainers == "UNKNOWN":
				file.write('maintainers         {0}\n\n'.format(maintainers))
			else:
				file.write('maintainers         {0}\n\n'.format(
						   os.getenv('maintainer', 'nomaintainer')))
		else:
			print("No maintainers found...")
			print("Looking for maintainers in environment variables...")
			file.write('maintainers         {0}\n\n'.format(
					   os.getenv('maintainer', 'nomaintainer')))

		summary = dict['summary']
		if summary:
			summary = re.sub(r'[\[\]\{\}\;\:\$\t\"\'\`\=(--)]+',
							 ' ', summary)
			summary = re.sub(r'\s(\s)+', ' ', summary)
			# print(summary)
			# print(type(summary))
			summary = ''.join([x for x in summary if x in string.printable])
			# print(type(summary))
			sum_lines = textwrap.wrap(summary)
			file.write('description         ')
			for sum_line in sum_lines:
				if sum_line:
					if not sum_lines.index(sum_line) == 0:
						file.write('                    ')
					if sum_line == sum_lines[-1]:
						file.write("{0}\n".format(sum_line))
					else:
						file.write("{0} \\\n".format(sum_line))
		else:
			file.write('description         None\n\n')

		file.write('long_description    {*}${description}\n\n')
		home_page = dict['home_page']

		if home_page and not home_page == 'UNKNOWN':
			file.write('homepage            {0}\n'.format(home_page))
		else:
			print("No homepage found...")
			print("Looking for homepage in environment variables...")
			file.write('homepage            {0}\n'.format(
					   os.getenv('home_page', '')))

		try:
			for item in dict2:
				if item['python_version'] == 'source':
					master_var = item['url']
					break

			if master_var:
				master_site = '/'.join(master_var.split('/')[0:-1])
				ext = master_var.split('/')[-1].split('.')[-1]
				if ext == 'zip':
					zip_set = True
				else:
					zip_set = False
		except:
			if dict['release_url']:
				master_site = dict['release_url']
				zip_set = False
			else:
				print("No master site found...")
				print("Looking for master site in environment variables...")
				master_site = os.getenv('master_site', '')
				zip_set = False

		if master_site:
                        if not re.match(r'^https?://files\.pythonhosted\.org/packages/', master_site):
                            file.write('master_sites        {0}\n'.format(master_site))
                        master_site_exists = True
		else:
			master_site_exists = False

		if zip_set:
			file.write('use_zip             yes\n')
			file.write('extract.mkdir       yes\n')

		print(("Attempting to generate checksums for " + dict['name'] + "..."))
		checksums_values = checksums(dict['name'], dict['version'])
		if checksums_values:
			first=True
			for chk in sorted(checksums_values.keys()):
				if first:
					file.write('checksums           ')
					first=False
				else:
					file.write(' \\\n')
					file.write('                    ')
				file.write('{0: <6}  {1}'.format(chk, checksums_values[chk]))
			file.write('\n\n')
		else:
			file.write('checksums           rmd160  XXX \\\n')
			file.write('                    sha256  XXX \\\n')
			file.write('                    size    XXX\n\n')


		python_vers = dict['requires_python']
		if python_vers:
			file.write('python.versions     311 {0}\n'.format(
					   dict['requires_python']))
		else:
			file.write('python.versions     311\n')
		file.write('python.pep517       yes\n\n')

		print("Finding dependencies...")
		file.write('if {${name} ne ${subport}} {')
		deps = dependencies(dict['name'], dict['version'], True)
		if deps:
			for i, dep in enumerate(deps):
				dep = dep.split('>')[0].split('=')[0]
				dep = dep.replace('[', '').replace(']', '')
				deps[i] = dep
			for dep in deps:
				if dep in ['setuptools', '', '\n']:
					while deps.count(dep) > 0:
						deps.remove(dep)

			if len(deps) > 0:
				file.write('    depends_run-append \\\n')

				for dep in deps[:-1]:
					file.write('                        ' +
							   'port:py${python.version}-' +
							   dep + ' \\\n')
				else:
					file.write('                        ' +
							   'port:py${python.version}-' +
							   deps[-1] + '\n')
			else:
				file.write("\n")
		file.write('\n')
		file.write('    livecheck.type      none\n')
		file.write('}\n')
	print("Searching for existent port...")
	port_exists = search_port(dict['name'])
	if port_exists:
		print("Creating diff...")
		old_file = port_exists
		new_file = './dports/python/py-'+dict['name']+'/Portfile'
		diff_file = './dports/python/py-'+dict['name']+'/patch.Portfile.diff'
		create_diff(old_file, new_file, diff_file)
		print((str(os.path.abspath(diff_file))+"\n"))
		print("\nIf you want to open a new ticket. Please visit")
		print("https://trac.macports.org/newticket")
		print("to open a new ticket after logging in with your GitHub account.")
	else:
		print("No port found.")


def print_portfile(pkg_name, pkg_version=None):
	""" Creates the directories and other commands necessary
	for a development environment """
	root_dir = os.path.abspath("./dports")
	port_dir = os.path.join(root_dir, 'python')
	home_dir = os.path.join(port_dir, 'py-'+pkg_name)
	if not os.path.exists(root_dir):
		os.makedirs(root_dir)
		try:
			command = 'portindex dports/'
			command = command.split()
			subprocess.call(command, stderr=subprocess.STDOUT)
		except:
			pass
	if not os.path.exists(port_dir):
		os.makedirs(port_dir)
	if not os.path.exists(home_dir):
		os.makedirs(home_dir)

	print("Attempting to fetch data from pypi...")

	populate_package_data(pkg_name, pkg_version)
	dict = package_data[pkg_name][pkg_version]['info']
	dict2 = get_release_urls(pkg_name, pkg_version)
	if dict and dict2:
		print("Data fetched successfully.")
	elif dict:
		print("Release Data fetched successfully.")
	elif dict2:
		print("Release url fetched successfully.")
	else:
		print("No data found.")

	file_name = os.path.join(home_dir, "Portfile")
	create_portfile(dict, file_name, dict2)
	print("SUCCESS\n")


def main(argv):
	""" Main function - Argument Parser """
	parser = argparse.ArgumentParser(description="Pypi2Port Tester")
# Calls list_all() which lists al available python packages
	parser.add_argument('-l', '--list', action='store_true', dest='list',
						default=False, required=False,
						help='List all packages')
# Calls release_data with package_name and package_version
	parser.add_argument('-d', '--data', action='store',
						dest='packages_data', nargs='*', type=str,
						help='Releases data for a package')
# Calls fetch_url with the various package_releases
	parser.add_argument('-f', '--fetch', action='store', type=str,
						dest='package_fetch', nargs='*', required=False,
						help='Fetches distfiles for a package')
# Calls print_portfile with the release data available
	parser.add_argument('-p', '--portfile', action='store', type=str,
						dest='package_portfile', nargs='*', required=False,
						help='Prints the portfile for a package')
# Calls port_testing
	parser.add_argument('-t', '--test', action='store', type=str,
						dest='package_test', nargs='*', required=False,
						help='Tests the portfile for various phase tests')
	options = parser.parse_args()

	if options.list:
		list_all()
		return

	if options.packages_data:
		pkg_name = options.packages_data[0]
		if len(options.packages_data) > 1:
			pkg_version = options.packages_data[1]
			release_data(pkg_name, pkg_version)
		else:
			populate_package_data(pkg_name)
			if package_data[pkg_name][None]['releases']:
				pkg_version = package_data[pkg_name][None]['info']['version']
				release_data(pkg_name, pkg_version)
			else:
				print("No release found\n")
		return

	if options.package_fetch:
		pkg_name = options.package_fetch[0]
		if len(options.package_fetch) > 1:
			pkg_version = options.package_fetch[1]
			fetch_url(pkg_name, pkg_version)
		else:
			populate_package_data(pkg_name)
			releases = package_data[pkg_name][None]['releases']
			if releases:
				pkg_version = package_data[pkg_name][None]['info']['version']
				fetch_url(pkg_name, pkg_version)
			else:
				print("No release found\n")
		return

	if options.package_portfile:
		pkg_name = options.package_portfile[0]
		if len(options.package_portfile) > 1:
			pkg_version = options.package_portfile[1]
			print_portfile(pkg_name, pkg_version)
		else:
			populate_package_data(pkg_name)
			vers = package_data[pkg_name][None]['releases']
			if vers:
				pkg_version = package_data[pkg_name][None]['info']['version']
				print_portfile(pkg_name, pkg_version)
			else:
				print("No release found\n")
		return

	if options.package_test:
		if len(options.package_test) > 0:
			pkg_name = options.package_test[0]
			port_testing(pkg_name)
		else:
			print("No package name specified\n")
		return

	parser.print_help()
	parser.error("No input specified")

if __name__ == "__main__":
	main(sys.argv[1:])
