#
# dihedral.py -> dihedral class of PyNMR
#
# Copyright (C) 2005  Dr. Stephane Gagne
# the full copyright notice is found in the LICENSE file in this directory
# 
# This program is free software; 
# you can redistribute it and/or modify it under the terms of the 
# GNU General Public License as published by the Free Software Foundation; 
# either version 2 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, 
# but WITHOUT ANY WARRANTY; without even the implied warranty of 
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
# See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License 
# along with this program; if not, write to the Free Software Foundation, Inc., # 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Contact:  nmr@rsvs.ulaval.ca
#

import os, string, math, os.path
import ltorsions, pbar, common
from verify import *


#..........................................................................
# generic restraint object.
#
class DihConstraints:

    def __init__(self):
        self.numCsts = 0
        self.constraints = []

#   ............................
    class CstItem:
        def __init__(self):
            self.norm = 0
            self.tolerance = 0
            self.atoms = []
            self.res = ""
            self.serial = 0
            self.type = ""
            
    def assign_line(self, rstline, mycst):
        if rstline == "":
            return
        rstline = string.split(string.strip(rstline))
        try:
            mycst.atoms = [rstline[5],rstline[10],rstline[15],rstline[20]]
        except:
            return
        if mycst.atoms == ['C)', 'N)', 'CA)', 'C)']:
            mycst.type = "phi"
        elif mycst.atoms == ['N)', 'CA)', 'C)', 'N)']:
            mycst.type = "psi"
#        elif mycst.atoms == ['N)', 'CA)', 'CB)', 'CG)']:
#            mycst.type = "chi"
        else:
            return
        mycst.resnum = string.atoi(rstline[7])
        mycst.norm = string.atof(rstline[22])
        mycst.tolerance = string.atof(rstline[23])
        self.numCsts = self.numCsts + 1
        mycst.serial = self.numCsts
        self.constraints.append(mycst)

#   ............................
    def readCnsCnsts(self, maingui):
        import re
                                                                                
#       open the file, then read the whole thing into memory
        fullfname = os.path.join(common.workingdir, maingui.fname.get())
        try: rstfile = open(fullfname, "r")
        except:
            print "** ERROR: cannot open restraints file"
            return
        rstlines = rstfile.readlines()
        rstfile.close()

        rstline = ""
        for line in rstlines:
            if line[0] == "!":
                continue
            mycst = self.CstItem()
            splitline = string.split(string.strip(line))
            # at the start of a restraint line, which can be split across lines
            if splitline[0] == "assign":
                # try reading the data out of the previous line(s)
                self.assign_line(rstline, mycst)
                # re-set rstlines
                rstline = line
            # append to the line
            else: rstline += line 
        # and do the last line in the file...
        self.assign_line(rstline, mycst)


    def readCyanaCnsts(self, maingui):
        return

#   ............................
    def readCnsts(self, maingui):
        if maingui.ftype.get() == "CNS":
            self.readCnsCnsts(maingui)
        else:
            self.readCyanaCnsts(maingui)
        

#..........................................................................
# class of dihedral angles, calculated from the pdb files.
# calls the ltorsions routines
#
class DihAngles:


    def __init__(self):
        self.myphipsi = dict()
        self.angtypes = ["phi", "psi", "chi", "omega"]


    # function to calculate the dihedral angles from the pdb files
    def calc_dihedrals(self, obj):

        phipsi = ltorsions.get_torsions(obj+".pdb")

        # parse input line.  store results.
        iterate = phipsi.iteritems()
        while True:
            try: ((resnum,type), angle) = iterate.next() 
            except StopIteration: break
            self.myphipsi[obj, int(resnum), self.angtypes[type]]  = angle


    def getAngle(self, obj, resnum, angtype):

        try: ang = self.myphipsi[obj, resnum, angtype]
        except KeyError:
            print ">> Sorry.  cannot compute violations for ", obj
            print """>>    either this is a group structure, """
            print """>>    or there is an error in pynmr."""
            return 0

        return ang




#---------------------------------------------------------------------------
# this is the routine that puts everything together.  It compares
# the constraints against the actual angles, and when it finds
# a violation it stores it in the violations object.
def checkViolations(obj, rsts, angles, violations, maingui):

    cutoff = maingui.cutoff.get()

    # foreachline in the constraint file - DihedralCOnstraint

    for cst in rsts.constraints:

#        uplim = cst.upperlimit
#        lowlim = cst.lowerlimit
        norm = cst.norm
        tol = cst.tolerance
        resnum = cst.resnum
        type = cst.type


        realAngle = angles.getAngle(obj,resnum, type)
        diff = math.fabs(realAngle - norm)
        if diff > 180: diff = 360-diff
        if diff > (tol + cutoff) :
            violations.vappend(obj, norm, tol, realAngle,
                cst.serial, resnum, type, diff-tol , '', '')

# debug.  added next code at Oli's suggestion.  what we are doing here is
# calling EVERYTHING a violation, no matter what.  So each restraint will
# generate a lien in violations, but the value of the violation will be "0".
# later we will weed these out.

        else:
            violations.vappend(obj, norm, tol, realAngle,
                cst.serial, resnum, type, 0, '', '')

        # angles can be negative or positive.  convert them to positive.
#        realAngle = math.fmod(360 + realAngle, 360)
#        uplim = math.fmod(360 + uplim, 360)
#        lowlim = math.fmod(360 + lowlim, 360)
#        print "actual = ", realAngle, " uplim = ", uplim, "lowlim = ", lowlim

        # check to see if there is a violation.
#        lowchk = lowlim - realAngle
#        if (lowchk > 180): lowchk = -1 * ( 360 - lowchk)
#        if (lowchk < -180): lowchk = 360 + lowchk
#        if (lowchk > cutoff):
#            violations.vappend(obj, lowlim, uplim, realAngle,
#                cst.serial, resnum, type, lowchk, '', '')
#
#        upchk = realAngle - uplim
#        if (upchk > 180): upchk = -1 * ( 360 - upchk)
#        if (upchk < -180): upchk = 360 + upchk
#        if (upchk > cutoff):
#            violations.vappend(obj, lowlim, uplim, realAngle,
#                cst.serial, resnum, type, upchk, '', '')



#        lowchk = (lowlim - realAngle)
#        if (lowchk > 180):
#            lowchk = (lowlim - 360 - realAngle)
#        upchk = (realAngle - uplim)
#        if (upchk > 180):
#            upchk = (360 - realAngle - uplim)



class DihViolation(Violations):

    # function to do the checking of dihedral restraints.
    # this is called from verify.py
    def doVerifyDih(maingui, myverify):
        from pymol import cmd 

        # for each on-screen object (pdb file)
        objs = cmd.get_names('objects',1)

        # put up progress bar because this takes a while
        max = len(objs)
        mybar = pbar.ProgressBar(max=max)
        counter = 1

        for obj in objs:

            # calulcate the dihedral angles
            myAngles = DihAngles()
            myAngles.calc_dihedrals(obj)

            checkViolations(obj, myverify.constraints, myAngles, myverify.violations, myverify.maingui)

            mybar.update(counter)
            counter += 1

        mybar.done()

