# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['yaspin']

package_data = \
{'': ['*'], 'yaspin': ['data/*']}

install_requires = \
['termcolor>=1.1.0,<2.0.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['dataclasses>=0.8,<0.9']}

setup_kwargs = {
    'name': 'yaspin',
    'version': '2.1.0',
    'description': 'Yet Another Terminal Spinner',
    'long_description': '|Logo|\n\n=====================================================================\n``yaspin``: **Y**\\ et **A**\\ nother Terminal **Spin**\\ ner for Python\n=====================================================================\n\n|Build Status| |Coverage| |pyup| |black-fmt|\n\n|pypi| |Versions| |Wheel| |Examples|\n\n|DownloadsTot| |DownloadsW|\n\n\n``Yaspin`` provides a full-featured terminal spinner to show the progress during long-hanging operations.\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/demo.gif\n\nIt is easy to integrate into existing codebase by using it as a `context manager`_\nor as a function `decorator`_:\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n\n    # Context manager:\n    with yaspin():\n        time.sleep(3)  # time consuming code\n\n    # Function decorator:\n    @yaspin(text="Loading...")\n    def some_operations():\n        time.sleep(3)  # time consuming code\n\n    some_operations()\n\n\n**Yaspin** also provides an intuitive and powerful API. For example, you can easily summon a shark:\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n\n    with yaspin().white.bold.shark.on_blue as sp:\n        sp.text = "White bold shark in a blue sea"\n        time.sleep(5)\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/shark.gif\n\n\nFeatures\n--------\n\n- Runs at all major **CPython** versions (*3.6*, *3.7*, *3.8*, *3.9*), **PyPy**\n- Supports all (70+) spinners from `cli-spinners`_\n- Supports all *colors*, *highlights*, *attributes* and their mixes from `termcolor`_ library\n- Easy to combine with other command-line libraries, e.g. `prompt-toolkit`_\n- Flexible API, easy to integrate with existing code\n- User-friendly API for handling POSIX `signals`_\n- Safe **pipes** and **redirects**:\n\n.. code-block:: bash\n\n    $ python script_that_uses_yaspin.py > script.log\n    $ python script_that_uses_yaspin.py | grep ERROR\n\n\nInstallation\n------------\n\nFrom `PyPI`_ using ``pip`` package manager:\n\n.. code-block:: bash\n\n    pip install --upgrade yaspin\n\n\nOr install the latest sources from GitHub:\n\n.. code-block:: bash\n\n    pip install https://github.com/pavdmyt/yaspin/archive/master.zip\n\n\nUsage\n-----\n\nBasic Example\n/////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/basic_example.gif\n\n.. code:: python\n\n    import time\n    from random import randint\n    from yaspin import yaspin\n\n    with yaspin(text="Loading", color="yellow") as spinner:\n        time.sleep(2)  # time consuming code\n\n        success = randint(0, 1)\n        if success:\n            spinner.ok("✅ ")\n        else:\n            spinner.fail("💥 ")\n\n\nIt is also possible to control spinner manually:\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n\n    spinner = yaspin()\n    spinner.start()\n\n    time.sleep(3)  # time consuming tasks\n\n    spinner.stop()\n\n\nRun any spinner from `cli-spinners`_\n////////////////////////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/cli_spinners.gif\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n    from yaspin.spinners import Spinners\n\n    with yaspin(Spinners.earth, text="Earth") as sp:\n        time.sleep(2)                # time consuming code\n\n        # change spinner\n        sp.spinner = Spinners.moon\n        sp.text = "Moon"\n\n        time.sleep(2)                # time consuming code\n\n\nAny Colour You Like `🌈`_\n/////////////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/basic_colors.gif\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n\n    with yaspin(text="Colors!") as sp:\n        # Support all basic termcolor text colors\n        colors = ("red", "green", "yellow", "blue", "magenta", "cyan", "white")\n\n        for color in colors:\n            sp.color, sp.text = color, color\n            time.sleep(1)\n\n\nAdvanced colors usage\n/////////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/advanced_colors.gif\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n    from yaspin.spinners import Spinners\n\n    text = "Bold blink magenta spinner on cyan color"\n    with yaspin().bold.blink.magenta.bouncingBall.on_cyan as sp:\n        sp.text = text\n        time.sleep(3)\n\n    # The same result can be achieved by passing arguments directly\n    with yaspin(\n        Spinners.bouncingBall,\n        color="magenta",\n        on_color="on_cyan",\n        attrs=["bold", "blink"],\n    ) as sp:\n        sp.text = text\n        time.sleep(3)\n\n\nRun any spinner you want\n////////////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/custom_spinners.gif\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin, Spinner\n\n    # Compose new spinners with custom frame sequence and interval value\n    sp = Spinner(["😸", "😹", "😺", "😻", "😼", "😽", "😾", "😿", "🙀"], 200)\n\n    with yaspin(sp, text="Cat!"):\n        time.sleep(3)  # cat consuming code :)\n\n\nChange spinner properties on the fly\n////////////////////////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/sp_properties.gif\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n    from yaspin.spinners import Spinners\n\n    with yaspin(Spinners.noise, text="Noise spinner") as sp:\n        time.sleep(2)\n\n        sp.spinner = Spinners.arc  # spinner type\n        sp.text = "Arc spinner"    # text along with spinner\n        sp.color = "green"         # spinner color\n        sp.side = "right"          # put spinner to the right\n        sp.reversal = True         # reverse spin direction\n\n        time.sleep(2)\n\n\nSpinner with timer\n//////////////////\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n\n    with yaspin(text="elapsed time", timer=True) as sp:\n        time.sleep(3.1415)\n        sp.ok()\n\n\nDynamic text\n////////////\n\n.. code:: python\n\n    import time\n    from datetime import datetime\n    from yaspin import yaspin\n\n    class TimedText:\n        def __init__(self, text):\n            self.text = text\n            self._start = datetime.now()\n\n        def __str__(self):\n            now = datetime.now()\n            delta = now - self._start\n            return f"{self.text} ({round(delta.total_seconds(), 1)}s)"\n\n    with yaspin(text=TimedText("time passed:")):\n        time.sleep(3)\n\n\nWriting messages\n////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/write_text.gif\n\nYou should not write any message in the terminal using ``print`` while spinner is open.\nTo write messages in the terminal without any collision with ``yaspin`` spinner, a ``.write()`` method is provided:\n\n.. code:: python\n\n    import time\n    from yaspin import yaspin\n\n    with yaspin(text="Downloading images", color="cyan") as sp:\n        # task 1\n        time.sleep(1)\n        sp.write("> image 1 download complete")\n\n        # task 2\n        time.sleep(2)\n        sp.write("> image 2 download complete")\n\n        # finalize\n        sp.ok("✔")\n\n\nIntegration with other libraries\n////////////////////////////////\n\n.. image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/gifs/hide_show.gif\n\nUtilizing ``hidden`` context manager it is possible to toggle the display of\nthe spinner in order to call custom methods that write to the terminal. This is\nhelpful for allowing easy usage in other frameworks like `prompt-toolkit`_.\nUsing the powerful ``print_formatted_text`` function allows you even to apply\nHTML formats and CSS styles to the output:\n\n.. code:: python\n\n    import sys\n    import time\n\n    from yaspin import yaspin\n    from prompt_toolkit import HTML, print_formatted_text\n    from prompt_toolkit.styles import Style\n\n    # override print with feature-rich ``print_formatted_text`` from prompt_toolkit\n    print = print_formatted_text\n\n    # build a basic prompt_toolkit style for styling the HTML wrapped text\n    style = Style.from_dict({\n        \'msg\': \'#4caf50 bold\',\n        \'sub-msg\': \'#616161 italic\'\n    })\n\n\n    with yaspin(text=\'Downloading images\') as sp:\n        # task 1\n        time.sleep(1)\n        with sp.hidden():\n            print(HTML(\n                u\'<b>></b> <msg>image 1</msg> <sub-msg>download complete</sub-msg>\'\n            ), style=style)\n\n        # task 2\n        time.sleep(2)\n        with sp.hidden():\n            print(HTML(\n                u\'<b>></b> <msg>image 2</msg> <sub-msg>download complete</sub-msg>\'\n            ), style=style)\n\n        # finalize\n        sp.ok()\n\n\nHandling POSIX `signals`_\n/////////////////////////\n\nHandling keyboard interrupts (pressing Control-C):\n\n.. code:: python\n\n    import time\n\n    from yaspin import kbi_safe_yaspin\n\n\n    with kbi_safe_yaspin(text="Press Control+C to send SIGINT (Keyboard Interrupt) signal"):\n        time.sleep(5)  # time consuming code\n\n\nHandling other types of signals:\n\n.. code:: python\n\n    import os\n    import time\n    from signal import SIGTERM, SIGUSR1\n\n    from yaspin import yaspin\n    from yaspin.signal_handlers import default_handler, fancy_handler\n\n\n    sigmap = {SIGUSR1: default_handler, SIGTERM: fancy_handler}\n    with yaspin(sigmap=sigmap, text="Handling SIGUSR1 and SIGTERM signals") as sp:\n        sp.write("Send signals using `kill` command")\n        sp.write("E.g. $ kill -USR1 {0}".format(os.getpid()))\n        time.sleep(20)  # time consuming code\n\n\nMore `examples`_.\n\n\nDevelopment\n-----------\n\nClone the repository:\n\n.. code-block:: bash\n\n    git clone https://github.com/pavdmyt/yaspin.git\n\n\nInstall dev dependencies:\n\n.. code-block:: bash\n\n    poetry install\n\n    # if you don\'t have poetry installed:\n    pip install -r requirements.txt\n\n\nLint code:\n\n.. code-block:: bash\n\n    make lint\n\n\nFormat code:\n\n.. code-block:: bash\n\n    make black-fmt\n\n\nRun tests:\n\n.. code-block:: bash\n\n    make test\n\n\nContributing\n------------\n\n1. Fork it!\n2. Create your feature branch: ``git checkout -b my-new-feature``\n3. Commit your changes: ``git commit -m \'Add some feature\'``\n4. Push to the branch: ``git push origin my-new-feature``\n5. Submit a pull request\n6. Make sure tests are passing\n\n\nLicense\n-------\n\n* MIT - Pavlo Dmytrenko; https://twitter.com/pavdmyt\n* Contains data from `cli-spinners`_: MIT License, Copyright (c) Sindre Sorhus sindresorhus@gmail.com (sindresorhus.com)\n\n\n.. |Logo| image:: https://raw.githubusercontent.com/pavdmyt/yaspin/master/static/logo_80.png\n   :alt: yaspin Logo\n.. |Build Status| image:: https://travis-ci.org/pavdmyt/yaspin.svg?branch=master\n   :target: https://travis-ci.org/pavdmyt/yaspin\n.. |Coverage| image:: https://codecov.io/gh/pavdmyt/yaspin/branch/master/graph/badge.svg\n   :target: https://codecov.io/gh/pavdmyt/yaspin\n.. |pypi| image:: https://img.shields.io/pypi/v/yaspin.svg\n   :target: https://pypi.org/project/yaspin/\n.. |Versions| image:: https://img.shields.io/pypi/pyversions/yaspin.svg\n   :target: https://pypi.org/project/yaspin/\n.. |Wheel| image:: https://img.shields.io/pypi/wheel/yaspin.svg\n   :target: https://pypi.org/project/yaspin/\n.. |Examples| image:: https://img.shields.io/badge/learn%20by-examples-0077b3.svg\n   :target: https://github.com/pavdmyt/yaspin/tree/master/examples\n.. |pyup| image:: https://pyup.io/repos/github/pavdmyt/yaspin/shield.svg\n   :target: https://pyup.io/repos/github/pavdmyt/yaspin/\n.. |black-fmt| image:: https://img.shields.io/badge/code%20style-black-000000.svg\n   :target: https://github.com/ambv/black\n.. |DownloadsTot| image:: https://pepy.tech/badge/yaspin\n   :target: https://pepy.tech/project/yaspin\n.. |DownloadsW| image:: https://pepy.tech/badge/yaspin/week\n   :target: https://pepy.tech/project/yaspin\n\n\n.. _context manager: https://docs.python.org/3/reference/datamodel.html#context-managers\n.. _decorator: https://www.thecodeship.com/patterns/guide-to-python-function-decorators/\n.. _cli-spinners: https://github.com/sindresorhus/cli-spinners\n.. _termcolor: https://pypi.org/project/termcolor/\n.. _PyPI: https://pypi.org/\n.. _🌈: https://en.wikipedia.org/wiki/Any_Colour_You_Like\n.. _examples: https://github.com/pavdmyt/yaspin/tree/master/examples\n.. _prompt-toolkit: https://github.com/jonathanslenders/python-prompt-toolkit/\n.. _signals: https://www.computerhope.com/unix/signals.htm\n',
    'author': 'Pavlo Dmytrenko',
    'author_email': 'mail@pavdmyt.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pavdmyt/yaspin',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
