from inspect import isclass
import logging, re
from itertools import chain

from dispatch import strategy

import formencode
from formencode.foreach import ForEach
from formencode import Invalid

import toscawidgets
from toscawidgets.api import Widget, adapt_value, lazystring
from toscawidgets.widgets.forms import (InputWidget, InputWidgetRepeater,
                                        validators)

log = logging.getLogger(__name__)

__all__ = [
    "FormField", "FormFieldRepeater", "ContainerMixin", "Form", "FieldSet",
    "TableMixin", "ListMixin", "ListForm", "ListFieldSet", "TextArea",
    "InputField", "TextField", "PasswordField", "HiddenField", "FileField",
    "Button", "SubmitButton", "ResetButton", "ImageButton", "SelectionField",
    "SingleSelectField", "SelectionList", "SingleSelectionMixin",
    "MultipleSelectionMixin", "MultipleSelectField", "RadioButtonList",
    "CheckBoxList", "TableForm", "CheckBox", "RadioButton",
    "SecureTicketField", "SecureFormMixin", "BooleanRadioButtonList",
    "TableFieldSet",
    ]

_ = lazystring

def name2label(name):
    """
    Convert a column name to a Human Readable name.

    Yanked from TGFastData
    """
    # Create label from the name:
    #   1) Convert _ to spaces
    #   2) Convert CamelCase to Camel Case
    #   3) Upcase first character of Each Word
    # Note: I *think* it would be thread-safe to
    #       memoize this thing.
    return ' '.join([s.capitalize() for s in
               re.findall(r'([A-Z][a-z0-9]+|[a-z0-9]+|[A-Z0-9]+)', name)])


class FormField(InputWidget):
    """
    Base class for all Widgets that can be attached to a Form or FieldSet.

    Form and FieldSets are in turn FormFields so they can be arbitrarily nested.
    These widgets can provide a validator that should validate and coerce the
    input they generate when submitted.
    """
    params = [
        "is_required", "label_text", "help_text", "attrs", "show_error",
        "disabled", "style",
        ]
    show_error = False
    show_error__doc = ("Should the field display it's own errors? Defaults to "
                       "False because normally they're displayed by the "
                       "container widget")
    disabled = None
    disabled__doc = ("Should the field be disbaled on render and it's input "
                     "ignored by the validator? UNIMPLEMENTED")
    attrs = {}
    attrs__doc = ("Extra attributes for the outermost DOM node")
    help_text = None
    help_text__doc = ("Description of the field to aid the user")
    label_text = None
    label_text__doc = "The text that should label this field"
    style = None
    style__doc = ("Style properties for the field. It's recommended to use "
                  "css classes and stylesheets instead of this parameter")


    @property
    def is_required(self):
        try:
            self.validate('', use_request_local=False)
            return False
        except formencode.Invalid:
            return True
    is_required__doc = ("Computed flag indicating if input is required from "
                        "this field")

    file_upload = False
    engine_name = "genshi"

    def __init__(self, id=None, parent=None, children=[], **kw):
        super(FormField, self).__init__(id,parent,children, **kw)
        if self.label_text is None and self.name is not None:
            pos = self.name.rfind('.')
            name = self.name[pos+1:]
            self.label_text = name2label(name)


    def update_params(self,d):
        super(FormField,self).update_params(d)
        if self.is_required:
            d.css_classes.append('required')
        self.update_attrs(d, 'disabled')

    def update_attrs(self, d, *args):
        """
        Fetches values from the dict and inserts the in the attrs dict.

        This is useful when you want to avoid boiler-place at the template:

        Instead of:

        .. code-block:: xhtml

            <foo bar='$bar' zoo='$zoo' />

        Do:

        .. code-block:: xhtml

            <foo py:attrs="attrs" />

        And inside update_params:

        .. code-block:: python

            self.update_attrs(d, 'bar', 'zoo')

        ('bar' and 'zoo' need to be listed at ``params``)

        """
        for name in args:
            d.setdefault('attrs',{}).setdefault(name, d[name])



class ContainerMixin(Widget):
    """
    A mix-in class for FormFields that contain other FormFields

    The following parameters are available:

    `show_children_errors`
      A flag indicating if the container should display it's children's errors

    It provides the template with two iterators:

    `ifields`
      Iterates over all the container's visible FormFields present in `children`

    `ihidden_fields`
      Iterates over all the container's hidden FormFields
    """
    params = ["ifields", "ihidden_fields", "show_children_errors"]
    show_error = False
    show_children_errors = True

    def __new__(cls, id=None, parent=None, children=[], **kw):
        fields = kw.pop('fields', None)
        if fields is not None:
            children = fields
        else:
            children = getattr(cls, 'fields', children)
        return super(ContainerMixin, cls).__new__(cls, id,parent,children,**kw)

    @property
    def ifields(self):
        return self.ifilter_children(
            lambda x: isinstance(x,FormField) and not isinstance(x,HiddenField)
            )

    @property
    def ihidden_fields(self):
        return self.ifilter_children(lambda x: isinstance(x,HiddenField))

    def _has_file_upload(self):
        for field in self.ifilter_children(
            lambda x: getattr(x, 'file_upload', False)
        ):
            return True
        return False

    def post_init(self, *args, **kw):
        log.debug("Setting 'file_upload' for %r", self)
        self.file_upload = self._has_file_upload()



class FormFieldRepeater(InputWidgetRepeater, ContainerMixin, FormField):
    show_error = True   # Trick containers not to display its errors
    engine_name = 'toscawidgets'





class Form(ContainerMixin, FormField):
    """
    A base class for all forms.

    Use this class as a base for your custom form. You should override it's
    template because it's a dummy one which does not display errors, help text
    or anything besides it's fields.

    The form will take care of setting its ``enctype`` if it contains any
    FileField
    """
    template = "genshi:toscawidgets.widgets.forms.templates.form"
    params = ["action", "method", "submit_text"]
    strip_name_if_root_container = True
    action = ''
    action__doc = "The url where the form's contents should be submitted"
    method = 'post'
    method__doc = "The HTTP request method to be used"
    submit_text = "Submit"
    submit_text__doc = ("Text that should appear in the auto-generated Submit "
                        "button. If None then no submit button will be "
                        "autogenerated.")

    def __init__(self, id=None, parent=None, children=[], **kw):
        super(Form, self).__init__(id, parent, children, **kw)
        if not hasattr(self.c, 'submit') and self.submit_text is not None:
            SubmitButton('submit', self, default=self.submit_text, 
                         label_text='')

    def post_init(self, *args, **kw):
        log.debug("Setting 'enctype' for %r", self)
        if self._has_file_upload():
            self.attrs.setdefault('enctype', 'multipart/form-data')
     
    def update_params(self, d):
        super(Form, self).update_params(d)
        d['value']['submit'] = d['submit_text'] 
        d.method = d.method.lower()
        # Fails W3C validation if present
        d.attrs.pop('disabled', None)
    

class FieldSet(ContainerMixin, FormField):
    """
    Base class for a fieldset.

    Use this class for your custom fieldset. You should override it's template
    because it's a dummy one which does not display errors, help text or
    anything besides it's fields.
    """
    template = "genshi:toscawidgets.widgets.forms.templates.fieldset"
    params = ["legend"]
    legend__doc = ("The legend for the fieldset. If none is provided it will "
                   "use its name")

    def __init__(self, id=None, parent=None, children=[], **kw):
        super(FieldSet, self).__init__(id,parent,children, **kw)
        if self.legend is None:
            self.legend = self.label_text


class TableMixin:
    """
    Mix-in class for containers that use a table to render their fields
    """
    params = ["table_attrs"]
    table_attrs = {}

class ListMixin:
    """
    Mix-in class for containers that use a list to render their fields
    """
    params = ["list_attrs"]
    list_attrs = {}


class ListForm(Form, ListMixin):
    """
    A form that renders it's fields as an unordered list
    """
    template = "genshi:toscawidgets.widgets.forms.templates.list_form"

class TableForm(Form, TableMixin):
    """
    A form that renders it's fields in a table
    """
    template = "genshi:toscawidgets.widgets.forms.templates.table_form"




class ListFieldSet(FieldSet, ListMixin):
    """
    A fieldset that renders it's fields as an unordered list
    """
    template = "genshi:toscawidgets.widgets.forms.templates.list_fieldset"


class TableFieldSet(FieldSet, TableMixin):
    """
    A fieldset that renders it's fields in a table
    """
    template = "genshi:toscawidgets.widgets.forms.templates.table_fieldset"

class TextArea(FormField):
    """
    Displays a textarea.
    """
    params = ["rows", "cols"]
    rows__doc = "Number of rows to render"
    cols__doc = "Number of columns to render"
    template = "genshi:toscawidgets.widgets.forms.templates.textarea"
    rows = 7
    cols = 50

    def update_params(self,d):
        super(TextArea, self).update_params(d)
        self.update_attrs(d, "rows", "cols")







class InputField(FormField):
    """Base class for <input> fields"""
    params = ["type"]
    template = "genshi:toscawidgets.widgets.forms.templates.input_field"




class TextField(InputField):
    """A text field"""
    params = ["size", "max_size"]
    size__doc = "The size of the text field."
    size__doc = "The maximum size of the field"
    type = "text"
    def update_params(self,d):
        super(TextField, self).update_params(d)
        self.update_attrs(d, "size", "max_size")


class PasswordField(InputField):
    """A password field."""
    type = "password"


class HiddenField(InputField):
    """A hidden field """
    type = "hidden"

class FileField(InputField):
    """A file upload field"""
    type = "file"
    file_upload = True

    [adapt_value.when(strategy.default, priority=1)]
    def __set_field_to_None(self, value):
        # This is needed because genshi doesn't seem to like displaying
        # cgi.FieldStorage instances
        return None

class Button(InputField):
    """A button field"""
    type = "button"
    params = ["named_button"]
    named_button = False

    def __init__(self, id=None, parent=None, children=[], **kw):
        super(Button, self).__init__(id,parent,children, **kw)
        if not self.named_button:
            self.validator = None

    @property
    def name(self):
        if not self.named_button:
            return None
        return '.'.join(reversed(
            [w.name_path_elem for w in self.path if w.name_path_elem]#[:-1]
            )) or None

class SubmitButton(Button):
    type = "submit"

    def update_params(self,d):
        super(SubmitButton, self).update_params(d)
        # A submit button with and id of 'submit' will make
        # form.submit == 'object' in JS code. See ticket #1295
        if d.id == 'submit':
            d.id = None

class ResetButton(Button):
    type = "reset"

class ImageButton(Button):
    params = ["src", "width", "height"]
    type = "image"
    def update_params(self,d):
        super(ImageButton, self).update_params(d)
        self.update_attrs(d, "src", "width", "height")


class CheckBox(InputField):
    type = "checkbox"
    validator = validators.Bool
    def update_params(self, d):
        super(CheckBox, self).update_params(d)
        d.attrs['checked'] = self.validator.to_python(d.value) or None

class RadioButton(InputField):
    type = "radio"


class SelectionField(FormField):
    selected_verb = None
    params = ["options"]
    options = []

    def update_params(self, d):
        super(SelectionField, self).update_params(d)
        grouped_options = []
        options = []
        d['options'] = self._iterate_options(d['options'])
        value = d['value']
        for optgroup in d["options"]:
            if isinstance(optgroup[1], (list,tuple)):
                group = True
                optlist = optgroup[1][:]
            else:
                group = False
                optlist = [optgroup]
            for i, option in enumerate(self._iterate_options(optlist)):
                if len(option) is 2:
                    option_attrs = {}
                elif len(option) is 3:
                    option_attrs = dict(option[2])
                if self._is_option_selected(option[0], value):
                    option_attrs[self.selected_verb] = self.selected_verb
                optlist[i] = (self.adjust_value(option[0]), option[1],
                              option_attrs)
            options.extend(optlist)
            if group:
                grouped_options.append((optgroup[0], optlist))
        # options provides a list of *flat* options leaving out any eventual
        # group, useful for backward compatibility and simpler widgets
        d["options"] = options
        if grouped_options:
            d["grouped_options"] = grouped_options
        else:
            d["grouped_options"] = [(None, options)]

        
    def _iterate_options(self, options):
        for option in options:
            if not isinstance(option, (tuple,list)):
                yield (option, option)
            else:
                yield option



class SelectionList(ListMixin, SelectionField):
    params = ["field_type"]
    selected_verb = "checked"
    template = "genshi:toscawidgets.widgets.forms.templates.selection_list"




class SingleSelectionMixin:
    def _is_option_selected(self, option_value, value):
        return option_value == value


class MultipleSelectionMixin:
    def _is_option_selected(self, option_value, value):
        return value is not None and option_value in value

    def post_init(self, *args, **kw):
        # Only override the user-provided validator if it's not a ForEach one,
        # which usually means the user needs to perform validation on the list
        # as a whole.
        if not (isinstance(self.validator, ForEach) or 
                (isclass(self.validator) and 
                 issubclass(self.validator, ForEach))):
            self.validator = ForEach(self.validator)





class SingleSelectField(SingleSelectionMixin, SelectionField):
    selected_verb = 'selected'
    template = "genshi:toscawidgets.widgets.forms.templates.select_field"

class MultipleSelectField(MultipleSelectionMixin, SelectionField):
    params = ["size"]
    size = 5
    selected_verb = 'selected'
    template = "genshi:toscawidgets.widgets.forms.templates.select_field"

    def update_params(self,d):
        super(MultipleSelectField, self).update_params(d)
        self.update_attrs(d, "size")
        d['attrs']['multiple'] = True
        




class RadioButtonList(SingleSelectionMixin, SelectionList):
    field_type = "radio"

class BooleanRadioButtonList(RadioButtonList):
    options = [(False, _("No")), (True, _("Yes"))]
    validator = validators.StringBoolean


class CheckBoxList(MultipleSelectionMixin, SelectionList):
    field_type = "checkbox"




class SecureTicketField(HiddenField):
    """
    Hidden form field that offers some protection against Cross-Site
    Request Forgery:
    
        http://en.wikipedia.org/wiki/Cross-site_request_forgery

    This protection is not complete against XSS or web browser bugs, see
    
        http://www.cgisecurity.com/articles/csrf-faq.shtml
    
    A per-session per-form authentication key is generated, and
    injected to this hidden field. They are compared on form
    validation. On mismatch, validation error is displayed.
    """

    def __init__(self, id=None, parent=None, children=[],
                 session_secret_cb=None, **kw):
        """Initialize the CSRF form token field.

        session_secret_cb() should return (session_secret, token)
        where session_secret is a random per-session secret string,
        and token some string associated with the current user.
        """
        super(SecureTicketField, self).__init__(id, parent, children, **kw)
        self.validator = validators.SecureTicketValidator(self,
                                                          session_secret_cb)
    
    def update_params(self, d):
        super(SecureTicketField, self).update_params(d)
        d['value'] = self.validator.get_hash()
        return d

class SecureFormMixin(FormField):
    """
    Protect against Cross-site request forgery, by adding
    SecureTicketField to the form.
    
    This can be use like this:
    
        class MyForm(ListForm, SecureFormMixin):
            ...

        def session_secret_cb():
            "Return session-specific secret data string and some data.
             Both of them should stay constant per form per user."
            secret = session['secret']
            user_id = c.user.user_name
            return str(secret), str(user_id)
    
        form = MyForm(session_secret_cb=session_secret_cb)

    or

        class MyForm(ListForm, SecureFormMixin):

            ...
            
            def session_secret_cb():
                ...

        form = MyForm()
    """
    
    def post_init(self, *args, **kw):
        session_secret_cb = kw.pop('session_secret_cb', None)
        #super(SecureFormMixin, self).__init__(id, parent, children, **kw)
        if hasattr(self, 'session_secret_cb') and session_secret_cb is None:
            session_secret_cb = self.session_secret_cb
        SecureTicketField("form_token__", self,
                          session_secret_cb=session_secret_cb)
