#!/usr/bin/python

#    pytiff -- a library for using TIFF files and advanced imaging in Python
#    Copyright (C) 2004 by Oliver M. Haynold
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    $Id: test_pytiff.py,v 1.5 2005/02/03 23:45:52 oliverh Exp oliverh $


import unittest
import pytiff
import os

tiffreader_imagefiles = [
    ("cramps.tif",
     1,
     {"bits_per_sample" : 8,
      "samples_per_pixel" : 1,
      "photometric_interpretation" : "grey",
      "sample_format" : "unsigned integer",
      "size" : (800,607)},
     {"compression": "packbits"}
     ),
    ("fax2d.tif",
     1,
     {"bits_per_sample" : 1,
      "samples_per_pixel" : 1,
      "photometric_interpretation" : "grey",
      "size" : (1728,1082)},
     {"compression": "T4"}
     ),
    ("horse.tif",
     1,
     {"bits_per_sample" : 8,
      "samples_per_pixel" : 3,
      "photometric_interpretation" : "RGB",
      "size" : (294,390)},
     {"compression": "deflate",
      "resolution" : (300, 300),
      "resolution unit" : "inch"}
     ),
    ("horse-tiled.tif",
     1,
     {"bits_per_sample" : 8,
      "samples_per_pixel" : 3,
      "photometric_interpretation" : "RGB",
      "size" : (294,390)},
     {"compression": "deflate",
      "resolution" : (300, 300),
      "resolution unit" : "inch"}
     ),
    ("horse-separate-planes.tif",
     1,
     {"bits_per_sample" : 8,
      "samples_per_pixel" : 3,
      "photometric_interpretation" : "RGB",
      "size" : (294,390)},
     {"compression": "deflate",
      "resolution" : (300, 300),
      "resolution unit" : "inch"}
     ),
    ("g3test.tif",
     1,
     {"bits_per_sample" : 1,
      "samples_per_pixel" : 1,
      "photometric_interpretation" : "grey",
      "size" : (1728,1103)},
     {"compression": "T4"}
     ),
    ("caspian.tif",
     1,
     {"bits_per_sample" : 64,
      "samples_per_pixel" : 3,
      "photometric_interpretation" : "RGB",
      "sample_format" : "IEEE floating point",
      "size" : (279,220)},
     {"compression": "deflate"}
     )
    ]

imaging_imagefiles = ["g3test.tif", "cramps.tif", "horse.tif"]

equal_series = [["horse.tif", "horse-tiled.tif", "horse-separate-planes.tif"]]

imagefiledir = "testimages"

more_bits_series = [ [("horse.tif", 0), ("horse-16bit-col-bigendian.tif", 8),
                      ("horse-16bit-col-littleendian.tif", 8)] ]

testimages = []
for fn in imaging_imagefiles:
    r = pytiff.TiffReader(imagefiledir + os.sep + fn)
    p = r[0]
    cp = pytiff.CropFilter(p,(0,0),(104,104))
    d = cp.data
    m = pytiff.MemoryImage(cp)    
    testimages.append((fn,m))

class TestTiffImageReader(unittest.TestCase):
    def test_load(self):
        for i in tiffreader_imagefiles:
            fn, length, attrs, metadata = i
            r = pytiff.TiffReader(imagefiledir + os.sep + fn)
            self.assertEqual(len(r), length)
            p = r[0]
            for a in attrs:
                self.assertEqual(getattr(p, a), attrs[a])
            for a in metadata:
                self.assert_(p.metadata.has_key(a))
                self.assertEqual(p.metadata[a], metadata[a])
            x = p.data
            self.assert_(len(p.data)>=(p.size[0] * p.size[1] / 8))
    def test_data_equality(self):
        for series in equal_series:
            first_img = pytiff.TiffReader(imagefiledir + os.sep +series[0])[0]
            for fn in series[1:]:
                other_img = pytiff.TiffReader(imagefiledir + os.sep +fn)[0]
                self.assert_(first_img.data == other_img.data)
    def test_more_bits_series(self):
        for series in more_bits_series:
            first_img = pytiff.MemoryImage(pytiff.TiffReader(imagefiledir + os.sep +series[0][0])[0])
            for img in series[1:]:
                other_img = pytiff.MemoryImage(pytiff.TiffReader(imagefiledir + os.sep + img[0])[0])
                for x in range(20):
                    for y in range(20):
                        first_pix = first_img.get_pixel((x,y))
                        other_pix = other_img.get_pixel((x,y))
                        for channel in range(len(first_pix)):
                            first_val = first_pix[channel]
                            other_val = other_pix[channel] >> img[1]
                            diff = abs(first_val - other_val)
                            self.assert_(diff <= 1)
                        
def pumpup_filter_test(testcase):
    methodnames = dir(testcase)
    for methodname in methodnames:
        if methodname[:4] == "test":
            for (fn, image) in testimages:
                def proxy_func(self):
                    getattr(self, "old_" + methodname)(image)
                setattr(testcase, methodname + "_" + fn.replace(".", "_"),
                        proxy_func)
            setattr(testcase, "old_" + methodname, getattr(testcase, methodname))
            delattr(testcase, methodname)
                

class TestIdentityFilter(unittest.TestCase):
    def test_identity(self, i):
        m = pytiff.IdentityFilter(i)
        self.assertEqual(i.bits_per_sample, m.bits_per_sample)
        self.assertEqual(i.samples_per_pixel, m.samples_per_pixel)
        self.assertEqual(i.photometric_interpretation, m.photometric_interpretation)
        self.assert_(i.data == m.data)
        self.assertEqual(i.size, m.size)
        self.assertEqual(i.metadata, m.metadata)
pumpup_filter_test(TestIdentityFilter)

class TestInvertFilter(unittest.TestCase):
    def test_it(self, i):
        i1 = pytiff.InvertFilter(i)
        i2 = pytiff.InvertFilter(i1)
        self.assert_(i.data != i1.data)
        self.assert_(i.data == i2.data)
        m1 = pytiff.MemoryImage(i1)
        size_x, size_y = i.size
        for p in [(0,0), (1,0), (0,1), (0,size_y-1), (size_x-1,0), (size_x-1, size_y-1)]:
            odata = i.get_pixel(p)
            invdata = m1.get_pixel(p)
            for s in range(len(odata)):
                oval = odata[s]
                invval = invdata[s]
                if(i.sample_format=="unsigned integer"):
                    expval = 2 ** i.bits_per_sample - 1 - oval
                else:
                    expval = -oval;
                self.assertEqual(invval, expval)
pumpup_filter_test(TestInvertFilter)

class TestFlipFilter(unittest.TestCase):
    def do_test_once(self, image, direction):
        f = pytiff.FlipFilter(image, direction)
        self.assertEqual(image.bits_per_sample, f.bits_per_sample)
        self.assertEqual(image.samples_per_pixel, f.samples_per_pixel)
        self.assertEqual(image.photometric_interpretation, f.photometric_interpretation)
        self.assertEqual(image.size, f.size)
        self.assertEqual(image.metadata, f.metadata)
        # self.assert_(image.data <> f.data) This doesn't need to be true
    def do_test_twice(self, image, direction):
        f = pytiff.FlipFilter(pytiff.FlipFilter(image, direction),direction)
        self.assert_(image.data == f.data)
    def test_horizontal_once(self, i):
        self.do_test_once(i, "horizontal")
    def test_horizontal_twice(self, i):
        self.do_test_twice(i, "horizontal")
    def test_vertical_once(self, i):
        self.do_test_once(i, "vertical")
    def test_vertical_twice(self, i):
        self.do_test_twice(i, "vertical")
pumpup_filter_test(TestFlipFilter)

class TestRotateFilter(unittest.TestCase):
    def test_90_with_reverse(self, image):
        f = pytiff.RotateFilter(image, 90)
        self.assertEqual(image.bits_per_sample, f.bits_per_sample)
        self.assertEqual(image.samples_per_pixel, f.samples_per_pixel)
        self.assertEqual(image.photometric_interpretation, f.photometric_interpretation)
        self.assertEqual(image.size, (f.size[1], f.size[0]))
        self.assertEqual(image.metadata, f.metadata)
        # self.assert_(image.data <> f.data) This doesn't need to be true
        f = pytiff.RotateFilter(f, -90)
        self.assertEqual(image.size, f.size)
        self.assert_(image.data == f.data)
    def test_180_with_flip(self, image):
        f = pytiff.RotateFilter(image, 180)
        f1 = pytiff.FlipFilter(pytiff.FlipFilter(image, "vertical"),"horizontal")
        self.assertEqual(f.size, f1.size)
        self.assert_(f.data == f1.data)
pumpup_filter_test(TestRotateFilter)

class TestCropFilter(unittest.TestCase):
    def test_crop(self, i):
        f1 = pytiff.CropFilter(i, (10,10), (60,60))
        self.assertEqual(i.bits_per_sample, f1.bits_per_sample)
        self.assertEqual(i.samples_per_pixel, f1.samples_per_pixel)
        self.assertEqual(i.photometric_interpretation, f1.photometric_interpretation)
        self.assertEqual(f1.size, (60,60))
        f2 = pytiff.CropFilter(i, (30,30), (24,20))
        f3 = pytiff.CropFilter(f1, (20,20), (24,20))
        self.assertEqual(f2.size, f3.size)
        self.assert_(f2.data == f3.data)
pumpup_filter_test(TestCropFilter)

class TestChangeBitsPerSampleFilter(unittest.TestCase):
    def test_changebps(self, i):
        print repr(self), repr(i)
        f16 = pytiff.ChangeBitsPerSampleFilter(i, 16)
        f3 = pytiff.ChangeBitsPerSampleFilter(i, 3)
        f1 = pytiff.ChangeBitsPerSampleFilter(i, 1)
        f16_back = pytiff.ChangeBitsPerSampleFilter(f16, i.bits_per_sample)
        f3_1 = pytiff.ChangeBitsPerSampleFilter(f3, 1)
        self.assertEqual(i.bits_per_sample, f16_back.bits_per_sample)
        self.assertEqual(i.samples_per_pixel, f16_back.samples_per_pixel)
        self.assertEqual(i.photometric_interpretation, f16_back.photometric_interpretation)
        self.assertEqual(i.size, f16_back.size)
        self.assert_(f1.data == f3_1.data)
        self.assert_(i.data == f16_back.data)
pumpup_filter_test(TestChangeBitsPerSampleFilter)

class TestScaleDownToRGBAFilter(unittest.TestCase):
    def test_scaledown(self, image):
        f = pytiff.ScaleDownToRGBAFilter(image, 10)
        self.assertEqual(8, f.bits_per_sample)
        self.assertEqual(4, f.samples_per_pixel)
        self.assertEqual("RGBA", f.photometric_interpretation)
        new_size_x = image.size[0] / 10
        if image.size[0] % 10:
            new_size_x += 1
        new_size_y = image.size[1] / 10
        if image.size[1] % 10:
            new_size_y += 1
        self.assertEqual((new_size_x, new_size_y), f.size)
        self.assertEqual(image.metadata, f.metadata)
        self.assertEqual(len(f.data), f.size[0] * f.size[1] * 4);
    def test_scale_identity(self, image):
        f = pytiff.ScaleDownToRGBAFilter(image, 1)
        self.assertEqual(8, f.bits_per_sample)
        self.assertEqual(4, f.samples_per_pixel)
        self.assertEqual("RGBA", f.photometric_interpretation)
        self.assertEqual(image.size, f.size)
        self.assertEqual(image.metadata, f.metadata)
        self.assertEqual(len(f.data), f.size[0] * f.size[1] * 4);
pumpup_filter_test(TestScaleDownToRGBAFilter)

class TestTiffWriter(unittest.TestCase):
    def test_write_from_testimage(self, image):
        w = pytiff.TiffWriter("testoutput.tiff")
        w.append(image)
        del w
        r = pytiff.TiffReader("testoutput.tiff")
        p = r[0]
        self.assertEqual(image.bits_per_sample, p.bits_per_sample)
        self.assertEqual(image.samples_per_pixel, p.samples_per_pixel)
        self.assertEqual(image.photometric_interpretation, p.photometric_interpretation)
        self.assertEqual(p.size, image.size)
        self.assert_(p.data == image.data)
        self.assertEqual(len(p.metadata), len(image.metadata))
        for key in p.metadata:
            self.assertEqual(p.metadata[key], image.metadata[key])
pumpup_filter_test(TestTiffWriter)

class TestMemoryImage(unittest.TestCase):
    def test_convert_to_64bit_and_back(self, i):
        i = pytiff.MemoryImage(pytiff.CropFilter(i,(0,0),(20,20)))
        i64 = pytiff.MemoryImage(i.size, None, i.metadata, 64, i.samples_per_pixel,
                                 i.photometric_interpretation, "IEEE floating point")
        for x in range(i.size[0]):
            for y in range(i.size[0]):
                i64.set_pixel((x,y),i.get_pixel((x,y)))
        for x in range(i.size[0]):
            for y in range(i.size[0]):
                self.assertEqual(i.get_pixel((x,y)),i64.get_pixel((x,y)))
pumpup_filter_test(TestMemoryImage)


    
if __name__ == '__main__':
    unittest.main()

