/*
    pytiff.c -- a library for using TIFF files and advanced imaging in Python
    Copyright (C) 2005 by Oliver M. Haynold

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    $Id: pytiff.c,v 1.48 2005/02/23 06:38:34 oliverh Exp oliverh $
    
*/

/*

TODO:
- Reorganize data format as continuous bitstream
   - Change code in MemoryImage
   - Allow for 3D images
   - Provide C-level cropped area/row/pixel/sample routines (separate for byte-aligned and not)
   - Provide C-level MemoryImage constructor (original_image, new_data)
   - MemoryImage should make a reference to an existing data block if it can be reused
   - Change all other routines accordingly
   - Add 3D support for reading and writing
- Accept file descriptor objects in TiffReader/TiffWriter
- handle datatypes other than unsigned int gracefully (Check all existing code.)
- fromstring / tostring (bitorder, byteorder, padding)
- Handle exotic photometric interpretations
- do something sensible with tifflib errors and warnings
- use new libtiff tag access methods for metadata
- speed up MemoryImage.getpixel/setpixel by caching data access methods
- read and write scaled-down subdirectories; for writing them, automatically
  generate them from a policy given to TiffWriter. e.g:
     def the_policy(image):
        res = []
	scale_fac = image.size[0]*image.size[1] / 100000
	if scale_fac > 1:
	   res.append(ScaleDownFilter(image, scale_fac))
	return res
- see how libtiff handles byte-swapping for words other than 16 and 32 bit.
  Right now, the we assume that if we work on a big-endian machine we have
  to swap 16, 32, and 64 bits. Also what should happen if a file claims to
  be big-endian with another number of bits per sample, possible one that is
  not even divisible by 8? In these situations only little-endianness seems to
  make sense. The same goes for writing files.
- add separate data reading routines for 1, 8 and 16 bit uint 
- add separate data writing routine for 1 bit uint

Wishlist (may or may not happen):
- file locking
*/

#include <tiffio.h>
#include "Python.h"
#include "structmember.h"
#include "stddef.h"
#include "pythread.h"

#include "pytiff-constants.h"

#define ATTRIBUTE_NAME_BITS_PER_SAMPLE "bits_per_sample"
#define ATTRIBUTE_NAME_SAMPLES_PER_PIXEL "samples_per_pixel"
#define ATTRIBUTE_NAME_DATA "data"
#define ATTRIBUTE_NAME_METADATA "metadata"
#define ATTRIBUTE_NAME_SAMPLE_FORMAT "sample_format"
#define ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION "photometric_interpretation"
#define ATTRIBUTE_NAME_SIZE "size"


int is_big_endian;		/* initialized by init routine */

/* Type definitions used throughout the module */

typedef PyObject *(*bits_to_value_routine) (const char *, int);
typedef int (*value_to_bits_routine) (PyObject *, char *, int);

typedef struct
{
PyObject_HEAD} AbstractImage;

typedef struct
{
	PyObject_HEAD PyTupleObject *size;
	PyDictObject *metadata;
	PyObject *data;
	PyObject *real_metadata;
	int bits_per_sample;
	int samples_per_pixel;
	PyObject *photometric_interpretation;
	PyObject *sample_format;
} MemoryImage;

typedef struct
{
	PyObject_HEAD PyObject *base_image;
} IdentityFilter;

typedef struct
{
	IdentityFilter base_data;
} InvertFilter;

typedef struct
{
	IdentityFilter base_data;
	int direction;		/* 0=vertical, 1=horizontal */
} FlipFilter;

typedef struct
{
	IdentityFilter base_data;
	int angle;		/* in degrees counterclockwise */
} RotateFilter;

typedef struct
{
	IdentityFilter base_data;
	int pos_x;
	int pos_y;
	int size_x;
	int size_y;
} CropFilter;

typedef struct
{
	IdentityFilter base_data;
	int bits_per_sample;	/* in degrees counterclockwise */
} ChangeBitsPerSampleFilter;

typedef struct
{
	IdentityFilter base_data;
	int scale_factor;
} ScaleDownToRGBAFilter;

typedef struct
{
	PyObject_HEAD TIFF *tif;
	PyListObject *pages;
	PyThread_type_lock *tiflock;
} TiffReader;

typedef struct
{
	PyObject_HEAD PyTupleObject *size;
	PyDictObject *metadata;
	TiffReader *reader;
	int seq_no;
	PyDictObject *real_metadata;
	int bits_per_sample;
	int samples_per_pixel;
	PyObject *photometric_interpretation;
	PyObject *sample_format;
} TiffReaderPage;

typedef struct
{
	PyObject_HEAD TIFF *tif;
	PyThread_type_lock *tiflock;
} TiffWriter;

/* Constants describing the TIFF format */

typedef struct
{
	uint16 tag;
	char *name;
} TagNames;

static TagNames AsciiTags[] = {
	{TIFFTAG_DOCUMENTNAME, PYTIFF_METADATA_DOCUMENTNAME},
	{TIFFTAG_IMAGEDESCRIPTION, PYTIFF_METADATA_IMAGEDESCRIPTION},
	{TIFFTAG_MAKE, PYTIFF_METADATA_SCANNER_MAKE},
	{TIFFTAG_MODEL, PYTIFF_METADATA_SCANNER_MODEL},
	{TIFFTAG_SOFTWARE, PYTIFF_METADATA_SOFTWARE},
	{TIFFTAG_ARTIST, PYTIFF_METADATA_ARTIST},
	{TIFFTAG_INKNAMES, PYTIFF_METADATA_INKNAMES},
	{TIFFTAG_DOCUMENTNAME, PYTIFF_METADATA_DOCUMENTNAME},
	{TIFFTAG_TARGETPRINTER, PYTIFF_METADATA_TARGETPRINTER},
	{0, NULL}
};

static TagNames CompressionValues[] = {
	{COMPRESSION_NONE, PYTIFF_COMPRESSION_NONE},
	{COMPRESSION_CCITTRLE, PYTIFF_COMPRESSION_CCITT_RLE}, 
	{COMPRESSION_CCITT_T4, PYTIFF_COMPRESSION_CCITT_T4},
	{COMPRESSION_CCITT_T6, PYTIFF_COMPRESSION_CCITT_T6},
	{COMPRESSION_LZW, PYTIFF_COMPRESSION_LZW},
	{COMPRESSION_JPEG, PYTIFF_COMPRESSION_JPEG},
	{COMPRESSION_OJPEG, PYTIFF_COMPRESSION_JPEG},
	{COMPRESSION_NEXT, PYTIFF_COMPRESSION_NEXT},
	{COMPRESSION_PACKBITS, PYTIFF_COMPRESSION_PACKBITS},
	{COMPRESSION_DEFLATE, PYTIFF_COMPRESSION_DEFLATE},
	{COMPRESSION_ADOBE_DEFLATE, PYTIFF_COMPRESSION_ADOBE_DEFLATE},
	{0, NULL}
};

static TagNames ResolutionUnitValues[] = {
	{RESUNIT_NONE, PYTIFF_RESUNIT_NONE},
	{RESUNIT_INCH, PYTIFF_RESUNIT_INCH},
	{RESUNIT_CENTIMETER, PYTIFF_RESUNIT_CENTIMETER},
	{0, NULL}
};

typedef struct
{
	uint16 tag;
	char *name;
	TagNames *values;
} CodedValuesTagEntry;

static CodedValuesTagEntry CodedValuesTags[] = {
	{TIFFTAG_COMPRESSION, PYTIFF_METADATA_COMPRESSION, CompressionValues},
	{TIFFTAG_RESOLUTIONUNIT, PYTIFF_METADATA_RESOLUTIONUNIT, ResolutionUnitValues},
	{0, NULL, NULL}
};

static TagNames SampleFormatValues[] = {
	{SAMPLEFORMAT_UINT, PYTIFF_SAMPLEFORMAT_UINT},
	{SAMPLEFORMAT_INT, PYTIFF_SAMPLEFORMAT_INT},
	{SAMPLEFORMAT_IEEEFP, PYTIFF_SAMPLEFORMAT_IEEEFP},
	{SAMPLEFORMAT_VOID, PYTIFF_SAMPLEFORMAT_VOID},
	{SAMPLEFORMAT_COMPLEXINT, PYTIFF_SAMPLEFORMAT_COMPLEXINT},
	{SAMPLEFORMAT_COMPLEXIEEEFP, PYTIFF_SAMPLEFORMAT_COMPLEXIEEEFP},
	{0, NULL}
};


void
swap_byte_order_n_bytes (char *data, int n, int len)
{
	int i, j;
	for (i = 0; i < len; i += n)
	{
		for (j = 0; j < n / 2; j++)
		{
			char temp;
			temp = data[i + j];
			data[i + j] = data[i + n - 1 - j];
			data[i + n - 1 - j] = temp;
		}
	}
}



/* Image utility functions for use in C; these offer generic access and don't depend
 on particular data structures */

int
PyObject_attr_as_int (PyObject * object, char *attr_name, int *attr_value)
{
	PyObject *attr;
	PyObject *pyint;
	attr = PyObject_GetAttrString (object, attr_name);
	if (!attr)
		return 0;	/* Exception already set */
	if (!PyNumber_Check (attr))
	{
		Py_XDECREF (attr);
		return 0;
	}
	pyint = PyNumber_Int (attr);
	Py_XDECREF (attr);
	if (!pyint)
		return 0;
	*attr_value = PyInt_AsLong (pyint);
	Py_XDECREF (pyint);
	return 1;
}

char *
PyObject_attr_as_string (PyObject * object, char *attr_name)
{
	PyObject *attr;
	char *res;
	attr = PyObject_GetAttrString (object, attr_name);
	if (!attr)
		return NULL;	/* Exception already set */
	if (!PyString_Check (attr))
	{
		Py_XDECREF (attr);
		return NULL;
	}
	res = PyString_AsString (attr);
	Py_XDECREF (attr);
	return res;
}

static int
Image_size_as_ints (PyObject * image, int *size_x, int *size_y)
{
	/* returns 0 on failure, 1 on success */
	PyObject *attr;
	attr = NULL;
	attr = PyObject_GetAttrString (image, ATTRIBUTE_NAME_SIZE);
	if (attr == NULL)
		return 0;
	if (!PyArg_ParseTuple (attr, "ii", size_x, size_y))
	{
		Py_XDECREF (attr);
		return 0;	/* Exception set automatically */
	}
	Py_XDECREF (attr);
	if ((*size_x < 1) || (*size_y < 1))
	{
		PyErr_SetString (PyExc_ValueError,
				 "Both dimensions of an image must be greater than zero.");
		return 0;
	}
	return 1;
}

int
Image_bits_per_sample_as_int (PyObject * image)
{
	/* returns -1 on failure, result on success */
	int result;
	if (PyObject_attr_as_int (image, ATTRIBUTE_NAME_BITS_PER_SAMPLE, &result))
		return result;
	else
		return -1;
}

int
Image_samples_per_pixel_as_int (PyObject * image)
{
	/* returns -1 on failure, result on success */
	int result;
	if (PyObject_attr_as_int (image, ATTRIBUTE_NAME_SAMPLES_PER_PIXEL, &result))
		return result;
	else
		return -1;
}

int
Image_bits_per_pixel_as_int (PyObject * image)
{
	int bps, spp;
	bps = Image_bits_per_sample_as_int (image);
	if (bps < 0)
		return -1;	/* Exception already set. */
	spp = Image_samples_per_pixel_as_int (image);
	if (spp < 0)
		return -1;	/* Exception already set. */
	return spp * bps;
}

int
Image_rowstride_as_int (PyObject * image)
{
	/* returns -1 on failure, rowstride in bytes on success */
	int size_x, size_y;
	int bits_per_pixel;
	if (!Image_size_as_ints (image, &size_x, &size_y))
		return -1;
	if ((bits_per_pixel = Image_bits_per_pixel_as_int (image)) < 1)
		return -1;
	return (size_x * bits_per_pixel) / 8 +
		(((size_x * bits_per_pixel) % 8) ? 1 : 0);
}


int
Image_data_size_as_int (PyObject * image)
{
	/* returns -1 on failure, data size in bytes on success */
	int rowstride;
	int size_x, size_y;
	rowstride = Image_rowstride_as_int (image);
	if (rowstride == -1)
		return -1;	/* Exception already set */
	if (!Image_size_as_ints (image, &size_x, &size_y))
		return -1;	/* Exception already set */
	return rowstride * size_y;
}

char *
Image_photometric_interpretation_as_string (PyObject * image)
{
	return PyObject_attr_as_string (image, ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION);
}

int
Image_data_as_buf (PyObject * image, const void **data, PyObject ** attr_ref)
{
	/* returns 0 on failure, 1 on success; if successful call Py_XDECREF when finished */
	int data_len;
	*attr_ref = NULL;
	*data = NULL;
	*attr_ref = PyObject_GetAttrString (image, ATTRIBUTE_NAME_DATA);
	if (*attr_ref == NULL)
		return 0;	/* Exception already set */
	PyObject_AsReadBuffer (*attr_ref, data, &data_len);
	if (*data == NULL)
	{
		Py_XDECREF (*attr_ref);
		*attr_ref = NULL;
		return 0;	/* Exception already set */
	};
	if (data_len != Image_data_size_as_int (image))
	{
		Py_XDECREF (*attr_ref);
		PyErr_SetString (PyExc_IndexError,
				 "Wrong image data length for this mode and size.");
		*attr_ref = NULL;
		*data = NULL;
		return 0;
	}
	return 1;
}

char *
Image_sample_format_defaulted (PyObject * image, PyObject ** attr_ref)
{
	*attr_ref = PyObject_GetAttrString (image, ATTRIBUTE_NAME_SAMPLE_FORMAT);
	if (*attr_ref == NULL)
		return PYTIFF_SAMPLEFORMAT_UINT;
	else
		return PyString_AsString (*attr_ref);
}

/* Data conversion functions from bits to Python value*/

PyObject *
bits_to_unsigned_integer (const char *data, int bits)
{
	long long res;
	int i;
	res = 0;
	for (i = 0; i < bits; i++)
	{
		if (data[i / 8] & (128 >> (i % 8)))
			res |= (1 << (bits - i - 1));
	}
	return PyLong_FromLongLong (res);
}


PyObject *
bits_to_unsigned_integer_char (const char *data, int bits)
{
	return PyInt_FromLong ((long) (*((unsigned char *) data)));
}

PyObject *
bits_to_signed_integer (const char *data, int bits)
{
	long long res;
	int i;
	res = 0;
	for (i = 1; i < bits; i++)
	{
		if (data[i / 8] & (128 >> (i % 8)))
			res |= (1 << (bits - i - 1));
	}
	if (*data & 128)
		res = !res;
	return PyLong_FromLongLong (res);
}

PyObject *
bits_to_float  (const char *data, int bits)
{
	double res;
	res = *((float *) data);
	switch (bits)
	{
	case sizeof (float) * 8:
	  res = *((float *) data);
	  break;
	case sizeof (double) * 8:
	  res = *((double *) data);
	  break;
	case sizeof (long double) * 8:
	  res = *((long double *) data);
	  break;
	default:
	  PyErr_SetString (PyExc_NotImplementedError,
			   "Bits per sample must be supported by the machine.");
	  return NULL;
	}
	if (is_big_endian)
	  swap_byte_order_n_bytes ((char *) &res, sizeof (res),
				   sizeof (res));
	return PyFloat_FromDouble (res);
}

PyObject *
bits_to_complex_from_float (const char *data, int bits)
{
	PyObject* real;
	PyObject* imag;
	if((bits/2)%8!=0){
	  PyErr_SetString (PyExc_NotImplementedError,
			   "Complex floats must have samples size in bits divisible by 8.");
	}

	real = bits_to_float(data, bits/2);
	imag = bits_to_float(data + (bits/8), bits/2);
	return PyComplex_FromDoubles (PyFloat_AsDouble(real), PyFloat_AsDouble(imag));
}

PyObject *
bits_to_complex_from_integer (const char *data, int bits)
{
	long long real, imag;
	int i;
	real = 0;
	for (i = 1; i < bits / 2; i++)
	{
		if (data[i / 8] & (128 >> (i % 8)))
			real |= (1 << (bits - i - 1));
	}
	if (*data & 128)
		real = !real;
	imag = 0;
	for (i = bits / 2 + 1; i < bits; i++)
	{
		if (data[i / 8] & (128 >> (i % 8)))
			imag |= (1 << (bits - i - 1));
	}
	if (data[bits / 16] & 128)
		imag = !imag;
	return PyComplex_FromDoubles ((double) real, (double) imag);
}

bits_to_value_routine
pick_bits_to_value_routine (char *sample_format, int bits_per_sample)
{
	if (bits_per_sample < 1)
	{
		PyErr_SetString (PyExc_ValueError,
				 "I could not find out the bits per sample of this image.");
		return NULL;
	}
	if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_UINT) == 0)
	{
		switch (bits_per_sample)
		{
		case sizeof (char) * 8:
			return bits_to_unsigned_integer_char;
		default:
			if (bits_per_sample < sizeof (long long) * 8 - 1)
				return bits_to_unsigned_integer;
			else
			{
				PyErr_SetString (PyExc_ValueError,
						 "Bits per sample must be less than size of long long.");
				return NULL;
			}
		}
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_INT) == 0)
	{
		if (bits_per_sample < sizeof (long long) * 8)
			return bits_to_signed_integer;
		else
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample must be less than size of long long.");
			return NULL;
		}
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_IEEEFP) == 0)
	{
	  return bits_to_float;
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_COMPLEXINT) == 0)
	{
		if (bits_per_sample % 2 != 0)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample for complex numbers must be divisible by 2.");
			return NULL;
		}
		if (bits_per_sample < sizeof (long long) * 8 * 2)
			return bits_to_complex_from_integer;
		else
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample must be less than the size of long long.");
			return NULL;
		}
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_COMPLEXIEEEFP) == 0)
	{
	  return bits_to_complex_from_float;
	}
	else
	{
		PyErr_SetString (PyExc_NotImplementedError,
				 "I don't understand this sample format.");
		return NULL;
	}
	return NULL;
}


/* Data conversion functions from Python value to bits;
   these functions return 0 on success, -1 and Python Exception
   set on failure */

int
make_bits_unsigned_integer (PyObject * indata, char *outbuf, int bits)
{
	unsigned long long value;
	PyObject *coerced_value;
	int result;
	int i;
	result = -1;
	coerced_value = PyNumber_Long (indata);
	if (coerced_value == NULL)
		goto clean_up;
	value = PyLong_AsUnsignedLongLong (coerced_value);
	if (PyErr_Occurred ())
		goto clean_up;
	if (value >= (1 << bits))
	{
		PyErr_SetString (PyExc_OverflowError,
				 "Maximum value for this data type exceeded.");
		goto clean_up;
	}
	if (value >= (1 << bits))
	{
		PyErr_SetString (PyExc_OverflowError,
				 "Minimum value for this data type exceeded.");
		goto clean_up;
	}
	for (i = 0; i < bits; i++)
	{
		if (i % 8 == 0)
			outbuf[i / 8] = 0;
		if (value & (1 << (bits - 1 - i)))
			outbuf[i / 8] |= 128 >> (i % 8);
	}
	result = 0;
      clean_up:
	if (coerced_value)
	{
		Py_XDECREF (coerced_value);
	}
	return result;
}

int
make_bits_signed_integer (PyObject * indata, char *outbuf, int bits)
{
	signed long long value;
	PyObject *coerced_value;
	int result;
	int i;
	result = -1;
	coerced_value = PyNumber_Long (indata);
	if (coerced_value == NULL)
		goto clean_up;
	value = PyLong_AsLongLong (coerced_value);
	if (PyErr_Occurred ())
		goto clean_up;
	if (value >= ((1 << (bits - 1))))
	{
		PyErr_SetString (PyExc_OverflowError,
				 "Maximum value for this data type exceeded.");
		goto clean_up;
	}
	if (value < (-(1 << (bits - 1))))
	{
		PyErr_SetString (PyExc_OverflowError,
				 "Minimum value for this data type exceeded.");
		goto clean_up;
	}
	if (value < 0)
	{
		outbuf[0] = 128;
	}
	else
	{
		outbuf[0] = 0;
	}
	for (i = 1; i < bits; i++)
	{
		if (i % 8 == 0)
			outbuf[i / 8] = 0;
		if (value & (1 << (bits - 1 - i)))
			outbuf[i / 8] |= 128 >> (i % 8);
	}
	result = 0;
      clean_up:
	if (coerced_value)
	{
		Py_XDECREF (coerced_value);
	}
	return result;
}

int
make_bits_float (PyObject * indata, char *outbuf, int bits)
{
	double value;
	PyObject *coerced_value;
	int result;
	result = -1;
	coerced_value = PyNumber_Float (indata);
	if (coerced_value == NULL)
		goto clean_up;
	value = PyFloat_AsDouble (coerced_value);
	if (PyErr_Occurred ())
		goto clean_up;
	switch (bits)
	{
	case sizeof (float) * 8:
		*(float *) outbuf = (float) value;
		break;
	case sizeof (double) * 8:
		*(double *) outbuf = (double) value;
		break;
	case sizeof (long double) * 8:
		*(long double *) outbuf = (long double) value;
		break;
	default:
		PyErr_SetString (PyExc_NotImplementedError,
				 "Bits per sample must be supported by the machine.");
		goto clean_up;
	}
	if (is_big_endian)
		swap_byte_order_n_bytes (outbuf, bits / 8, bits / 8);
	result = 0;
      clean_up:
	if (coerced_value)
	{
		Py_XDECREF (coerced_value);
	}
	return result;
}

int
make_bits_integer_complex (PyObject * indata, char *outbuf, int bits)
{
	PyObject *real_val, *imag_val;
	int result;
	result = 1;
	real_val = PyObject_GetAttrString (indata, "real");
	imag_val = PyObject_GetAttrString (indata, "imag");
	if ((!real_val) || (!imag_val))
		goto clean_up;
	result = make_bits_signed_integer (real_val, outbuf, bits / 2);
	if (result != 0)
		goto clean_up;
	result = make_bits_signed_integer (real_val, outbuf + bits / 16,
					   bits / 2);
      clean_up:
	Py_XDECREF (real_val);
	Py_XDECREF (imag_val);
	return result;
}

int
make_bits_float_complex (PyObject * indata, char *outbuf, int bits)
{
	PyObject *real_val, *imag_val;
	int result;
	result = 1;
	real_val = PyObject_GetAttrString (indata, "real");
	imag_val = PyObject_GetAttrString (indata, "imag");
	if ((!real_val) || (!imag_val))
		goto clean_up;
	result = make_bits_float (real_val, outbuf, bits / 2);
	if (result != 0)
		goto clean_up;
	result = make_bits_float (real_val, outbuf + bits / 16, bits / 2);
      clean_up:
	Py_XDECREF (real_val);
	Py_XDECREF (imag_val);
	return result;
}

value_to_bits_routine
pick_value_to_bits_routine (char *sample_format, int bits_per_sample)
{
	if (bits_per_sample < 1)
	{
		PyErr_SetString (PyExc_ValueError,
				 "I could not find out the bits per sample of this image.");
		return NULL;
	}
	if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_UINT) == 0)
	{
		switch (bits_per_sample)
		{
			/*    case sizeof (char) * 8:
			 * return bits_to_unsigned_integer_char; */
		default:
			if (bits_per_sample < sizeof (long long) * 8 - 1)
				return make_bits_unsigned_integer;
			else
			{
				PyErr_SetString (PyExc_ValueError,
						 "Bits per sample must be less than size of long long.");
				return NULL;
			}
		}
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_INT) == 0)
	{
		if (bits_per_sample < sizeof (long long) * 8)
			return make_bits_signed_integer;
		else
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample must be less than size of long long.");
			return NULL;
		}
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_IEEEFP) == 0)
	{
		return make_bits_float;
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_COMPLEXINT) == 0)
	{
		if (bits_per_sample % 2 != 0)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample for complex numbers must be divisible by 2.");
			return NULL;
		}
		if (bits_per_sample < sizeof (long long) * 8 * 2)
			return make_bits_integer_complex;
		else
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample must be less than the size of long long.");
			return NULL;
		}
	}
	else if (strcmp (sample_format, PYTIFF_SAMPLEFORMAT_COMPLEXIEEEFP) == 0)
	{
		if (bits_per_sample % 2 != 0)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample for complex numbers must be divisible by 2.");
			return NULL;
		}
		switch (bits_per_sample)
		{
		case sizeof (float) * 2 * 8:
			break;
		case sizeof (double) * 2 * 8:
			break;
		case sizeof (long double) * 2 * 8:
			break;
		default:
			PyErr_SetString (PyExc_ValueError,
					 "Bits per sample must match a machine-supported float type.");
			return NULL;
		}
		return make_bits_float_complex;
	}
	else
	{
		PyErr_SetString (PyExc_NotImplementedError,
				 "I don't understand this sample format.");
		return NULL;
	}
	return NULL;
}

/* MemoryImage definitions */

static PyObject *
MemoryImage_new_from_Cdata (PyTypeObject * cls, int size_x, int size_y,
			    PyObject * data, PyObject * metadata,
			    int bits_per_sample, int samples_per_pixel,
			    char *photometric_interpretation,
			    char *sample_format)
{
	MemoryImage *self;
	void *buf, *databuf;
	int buflen;
	int rowstride;
	self = (MemoryImage *) cls->tp_alloc (cls, 0);
	self->size =
		(PyTupleObject *) Py_BuildValue ("(l,l)", size_x, size_y);
	if (bits_per_sample < 1)
	{
		PyErr_SetString (PyExc_ValueError,
				 "Bits per sample must be at least 1");
		return NULL;
	}
	self->bits_per_sample = bits_per_sample;
	if (samples_per_pixel < 1)
	{
		PyErr_SetString (PyExc_ValueError,
				 "Samples per pixel must be at least 1");
		return NULL;
	}
	self->samples_per_pixel = samples_per_pixel;
	self->photometric_interpretation =
		PyString_FromString (photometric_interpretation);
	self->sample_format = PyString_FromString (sample_format);
	self->real_metadata = PyDict_New ();
	PyDict_Update (self->real_metadata, metadata);
	self->metadata =
		(PyDictObject *) PyDictProxy_New ((PyObject *) self->
						  real_metadata);
	if ((data != Py_None)
	    && (PyObject_AsReadBuffer (data, (const void **) &buf, &buflen) !=
		0))
		return NULL;	/* Exception already set */
	rowstride = Image_rowstride_as_int ((PyObject *) self);
	if (rowstride < 0)
	{
		return NULL;	/* Exception already set */
	}
	if (data == Py_None)
	{
		buflen = rowstride * size_y;
	}
	if ((data != Py_None) && (rowstride * size_y != buflen))
	{
		char msg[100];
		snprintf (msg, sizeof (msg),
			  "Data has a different length (%d) than what I expected (%d).",
			  buflen, rowstride * size_y);
		PyErr_SetString (PyExc_IndexError, msg);
		return NULL;
	}
	self->data = PyBuffer_New (buflen);
	PyObject_AsWriteBuffer (self->data, &databuf, &buflen);
	if (data != Py_None)
	{
		memcpy (databuf, buf, buflen);
	}
	else
	{
		memset (databuf, 0, buflen);
	}
	/* Py_INCREF (data); I don't know how that got here. */
	return (PyObject *) self;
}

static PyObject *
MemoryImage_new (PyTypeObject * cls, PyObject * args, PyObject * kwd)
{
	int size_x, size_y;
	PyObject *data;
	PyObject *metadata;
	PyObject *original_image;
	PyObject *photometric_interpretation;
	PyObject *sample_format;
	PyObject *sample_format_ref;
	int bps, spp;
	char *pi_str;
	char *sample_format_str;
	PyObject *res;

/*  static char *kwlist[] =
    { ATTRIBUTE_NAME_SIZE, ATTRIBUTE_NAME_DATA, ATTRIBUTE_NAME_METADATA, ATTRIBUTE_NAME_BITS_PER_SAMPLE, ATTRIBUTE_NAME_SAMPLES_PER_PIXEL,
    ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION, ATTRIBUTE_NAME_SAMPLE_FORMAT, NULL
  }; */

	data = metadata = original_image = photometric_interpretation =
		sample_format = sample_format_ref = NULL;
	pi_str = NULL;
	if (PyArg_ParseTuple (args, "O", &original_image))
	{
		if (!Image_size_as_ints (original_image, &size_x, &size_y))
			return NULL;
		bps = Image_bits_per_sample_as_int (original_image);
		if (bps < 1)
			return NULL;
		spp = Image_samples_per_pixel_as_int (original_image);
		if (bps < 1)
			return NULL;
		photometric_interpretation =
			PyObject_GetAttrString (original_image,
						ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION);
		if (!photometric_interpretation)
			return NULL;
		sample_format_str =
			Image_sample_format_defaulted (original_image,
						       &sample_format_ref);
		if (!sample_format_str)
		{
			Py_XDECREF (sample_format_ref);
			return NULL;
		}
		pi_str = PyString_AsString (photometric_interpretation);
		if (!pi_str)
		{
			/* Py_XDECREF (photometric_interpretation); */
			Py_XDECREF (sample_format_ref);
			return NULL;
		}
		data = PyObject_GetAttrString (original_image, ATTRIBUTE_NAME_DATA);
		if (!data)
		{
			Py_XDECREF (sample_format_ref);
			return NULL;
		}
		metadata =
			PyObject_GetAttrString (original_image, ATTRIBUTE_NAME_METADATA);
		if (!metadata)
		{
			Py_XDECREF (sample_format_ref);
			Py_XDECREF (data);
			return NULL;
		}
	}
	else
	{
		PyErr_Clear ();
		if (!PyArg_ParseTuple
		    (args, "(ii)OOiiss", &size_x, &size_y, &data,
		     &metadata, &bps, &spp, &pi_str, &sample_format_str))
			return NULL;	/* Exception already set */
		if (!sample_format_str)
			sample_format_str = PYTIFF_SAMPLEFORMAT_UINT;
		Py_INCREF (data);
		Py_INCREF (metadata);
	}
	res = MemoryImage_new_from_Cdata (cls, size_x, size_y, data, metadata,
					  bps, spp, pi_str,
					  sample_format_str);
	Py_XDECREF (data);
	Py_XDECREF (sample_format_ref);
	Py_XDECREF (metadata);
	return res;
}

static void
MemoryImage_dealloc (MemoryImage * self)
{
	Py_XDECREF (self->photometric_interpretation);
	Py_XDECREF (self->size);
	Py_XDECREF (self->metadata);
	Py_XDECREF (self->data);
	Py_XDECREF (self->real_metadata);
	((MemoryImage *) self)->ob_type->tp_free ((PyObject *) self);
};


char *
sample_buffer_alloc (int bits_per_sample)
{
	return (char *) calloc (bits_per_sample / 8 +
				((bits_per_sample % 8) ? 1 : 0), 1);
}

void
get_sample_data (char *result, char *data, int bits_per_sample,
		 int samples_per_pixel, int size_x, int x, int y, int sample)
{
	int rowstride;
	rowstride = (size_x * bits_per_sample * samples_per_pixel) / 8 +
		(((size_x * bits_per_sample * samples_per_pixel) %
		  8) ? 1 : 0);
	if (bits_per_sample == 8)
	{
		result[0] =
			data[y * rowstride + x * samples_per_pixel + sample];
	}
	else
	{
		int b;
		long current_bit;
		current_bit =
			y * rowstride * 8 +
			x * bits_per_sample * samples_per_pixel +
			sample * bits_per_sample;
		for (b = 0;
		     b <
		     (bits_per_sample / 8 + ((bits_per_sample % 8) ? 1 : 0));
		     b++)
			result[b] = 0;
		for (b = 0; b < bits_per_sample; b++)
		{
			if (data[current_bit / 8] &
			    (128 >> (current_bit % 8)))
				result[b / 8] |= (1 << (7 - (b % 8)));
			current_bit++;
		}
	}
}

void
set_sample_data (char *sample_data, unsigned char *data, int bits_per_sample,
		 int samples_per_pixel, int size_x, int x, int y, int sample)
{
	int rowstride;
	rowstride = (size_x * bits_per_sample * samples_per_pixel) / 8 +
		(((size_x * bits_per_sample * samples_per_pixel) %
		  8) ? 1 : 0);
	if (bits_per_sample == 8)
	{
		data[y * rowstride + x * samples_per_pixel + sample] =
			sample_data[0];
	}
	else
	{
		int b;
		long current_bit;
		current_bit =
			y * rowstride * 8 +
			x * bits_per_sample * samples_per_pixel +
			sample * bits_per_sample;
		for (b = 0; b < bits_per_sample; b++)
		{
			if (sample_data[b / 8] & (128 >> (b % 8)))
				data[current_bit / 8] |=
					(unsigned char) (128 >>
							 (current_bit % 8));
			else
			{
				data[current_bit / 8] &=
					(255 - (128 >> (current_bit % 8)));
			}
			current_bit++;
		}
	}
}

static PyObject *
MemoryImage_get_pixel (MemoryImage * self, PyObject * args, PyObject * kwd)
{
	PyObject *res;
	int x, y, bps, spp, data_len;
	char *data;
	int s;
	bits_to_value_routine btvr;
	int size_x, size_y;
	char *bitbuffer;
	if (!PyArg_ParseTuple (args, "(ii)", &x, &y))
		return NULL;	/* Exception set */
	if (!Image_size_as_ints ((PyObject *) self, &size_x, &size_y))
		return NULL;
	if (x<0 || y<0 || x>=size_x || y>=size_y){
	  PyErr_SetString (PyExc_IndexError,
			   "Illegal pixel coordinates.");
	}
	PyObject_AsReadBuffer (self->data, (const void **) &data, &data_len);
	bps = self->bits_per_sample;
	spp = self->samples_per_pixel;
	res = PyTuple_New (spp);
	btvr = pick_bits_to_value_routine (PyString_AsString
					   (self->sample_format), bps);
	if (btvr == NULL)
		return NULL;
	bitbuffer = sample_buffer_alloc (bps);
	if (!bitbuffer)
		return NULL;
	for (s = 0; s < spp; s++)
	{
		PyObject *pyval;
		get_sample_data (bitbuffer, data, bps, spp, size_x, x, y, s);
		pyval = btvr (bitbuffer, bps);
		PyTuple_SetItem (res, s, pyval);
	}
	free (bitbuffer);
	return res;
}

static PyObject *
MemoryImage_set_pixel (MemoryImage * self, PyObject * args, PyObject * kwd)
{
	PyObject *value, *sample_value;
	int x, y, bps, spp, data_len;
	char *data;
	int s;
	value_to_bits_routine vtbr;
	int size_x, size_y;
	char *bitbuffer;
	if (!PyArg_ParseTuple (args, "(ii)O", &x, &y, &value))
		return NULL;	/* Exception set */
	if (!Image_size_as_ints ((PyObject *) self, &size_x, &size_y))
		return NULL;
	if (x<0 || y<0 || x>=size_x || y>=size_y){
	  PyErr_SetString (PyExc_IndexError,
			   "Illegal pixel coordinates.");
	}
	bps = self->bits_per_sample;
	spp = self->samples_per_pixel;
	if (PySequence_Size (value) != spp)
	{
		PyErr_SetString (PyExc_IndexError,
				 "The value must be a sequence with as many members as the image has samples per pixel.");
		return NULL;
	}
	if (PyObject_AsWriteBuffer (self->data, (void **) &data, &data_len) ==
	    -1)
		return NULL;
	vtbr = pick_value_to_bits_routine (PyString_AsString
					   (self->sample_format), bps);
	if (vtbr == NULL)
		return NULL;
	bitbuffer = sample_buffer_alloc (bps);
	if (!bitbuffer)
		return NULL;
	for (s = 0; s < spp; s++)
	{
		sample_value = PySequence_GetItem (value, s);
		vtbr (sample_value, bitbuffer, bps);
		set_sample_data (bitbuffer, data, bps, spp, size_x, x, y, s);
	}
	free (bitbuffer);
	Py_INCREF (Py_None);
	return Py_None;
}

static PyObject *
MemoryImage_get_data (MemoryImage * self, void *closure)
{
	char *buf;
	int buflen;
	PyObject_AsReadBuffer (self->data, (const void **) &buf, &buflen);
	return PyString_FromStringAndSize (buf, buflen);
}

static PyMethodDef MemoryImage_methods[] = {
	{"get_pixel", (PyCFunction) MemoryImage_get_pixel, METH_VARARGS,
	 "Get the value of a pixel as a tuple."},
	{"set_pixel", (PyCFunction) MemoryImage_set_pixel, METH_VARARGS,
	 "Set the value of a pixel from a tuple."},
	{NULL}			/* Sentinel */
};



static PyMemberDef MemoryImage_members[] = {
	{ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION, T_OBJECT,
	 offsetof (MemoryImage, photometric_interpretation),
	 READONLY, "Image photometric interpretation"},
	{ATTRIBUTE_NAME_SAMPLE_FORMAT, T_OBJECT, offsetof (MemoryImage, sample_format),
	 READONLY, "Image sample format"},
	{ATTRIBUTE_NAME_BITS_PER_SAMPLE, T_INT, offsetof (MemoryImage, bits_per_sample),
	 READONLY, "Image bits per sample"},
	{ATTRIBUTE_NAME_SAMPLES_PER_PIXEL, T_INT,
	 offsetof (MemoryImage, samples_per_pixel),
	 READONLY, "Image samples per pixel"},
	{ATTRIBUTE_NAME_SIZE, T_OBJECT, offsetof (MemoryImage, size), READONLY,
	 "Image size"},
	{ATTRIBUTE_NAME_METADATA, T_OBJECT, offsetof (MemoryImage, metadata), READONLY,
	 "Image metadata"},
/*  {ATTRIBUTE_NAME_DATA, T_OBJECT, offsetof (MemoryImage, data), READONLY, "Image data"}, */
	{NULL}			/* Sentinel */
};

static PyGetSetDef MemoryImage_getset[] = {
	{ATTRIBUTE_NAME_DATA, (getter) MemoryImage_get_data, NULL,
	 "Image data", NULL},
	{NULL}
};

static PyTypeObject MemoryImageType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.MemoryImage",	/*tp_name */
	sizeof (MemoryImage),	/*tp_basicsize */
	0,			/*tp_itemsize */
	(destructor) MemoryImage_dealloc,	/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"An image held in memory. Instantiate with pytiff.MemoryImage(image) or \n"
		"pytiff.MemoryImage(size, data, metadata, bits_per_sample, \n"
		"samples_per_pixel, photometric_interpretation) .\n"
		"The following conventions apply:\n"
		"- Where applicable, small values signify low intensity."
		"- Data must be in MSB to LSB byte order and MSB to LSB bit order. Note \n"
		"  that this is _not_ the Intel byte order.\n"
		"- Within a row, bits follow each other irrespective of byte alignment.\n"
		"- Rows are byte-aligned.\n",
	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	MemoryImage_methods,	/* tp_methods */
	MemoryImage_members,	/* tp_members */
	MemoryImage_getset,	/* tp_getset */
	0,			/* tp_base */
	0, /* tp_dict */ 0,	/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	MemoryImage_new,	/* tp_new */
};

/* IdentityFilter definition */

static PyObject *
IdentityFilter_new_from_baseimage (PyTypeObject * cls, PyObject * base_image)
{
	IdentityFilter *self;
	self = (IdentityFilter *) cls->tp_alloc (cls, 0);
	if (!base_image)
	{
		PyErr_SetString (PyExc_ValueError,
				 "I need to have a base image to filter.");
		return NULL;
	}
	self->base_image = base_image;
	Py_INCREF (self->base_image);
	return (PyObject *) self;
}

static PyObject *
IdentityFilter_new (PyTypeObject * cls, PyObject * args, PyObject * kwd)
{
	PyObject *self;
	PyObject *base_image;
	if (PyArg_ParseTuple (args, "O", &base_image) == 0)
		return NULL;	/* Exception already set */
	self = IdentityFilter_new_from_baseimage (cls, base_image);
	return (PyObject *) self;
}

static void
IdentityFilter_dealloc (IdentityFilter * self)
{
	Py_XDECREF (self->base_image);
	self->ob_type->tp_free ((PyObject *) self);
};

static PyObject *
IdentityFilter_get_bits_per_sample (IdentityFilter * self)
{
	return PyObject_GetAttrString (self->base_image, ATTRIBUTE_NAME_BITS_PER_SAMPLE);
}

static PyObject *
IdentityFilter_get_samples_per_pixel (IdentityFilter * self)
{
	return PyObject_GetAttrString (self->base_image, ATTRIBUTE_NAME_SAMPLES_PER_PIXEL);
}

static PyObject *
IdentityFilter_get_photometric_interpretation (IdentityFilter * self)
{
	return PyObject_GetAttrString (self->base_image,
				       ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION);
}

static PyObject *
IdentityFilter_get_sample_format (IdentityFilter * self)
{
	PyObject *attr_ref;
	char *sample_format;
	PyObject *res;
	sample_format =
		Image_sample_format_defaulted (self->base_image, &attr_ref);
	if (sample_format)
		res = PyString_FromString (sample_format);
	else
		res = PyObject_GetAttrString (self->base_image,
					      ATTRIBUTE_NAME_SAMPLE_FORMAT);
	Py_XDECREF (attr_ref);
	return res;
}

static PyObject *
IdentityFilter_get_size (IdentityFilter * self)
{
	return PyObject_GetAttrString (self->base_image, ATTRIBUTE_NAME_SIZE);
}

static PyObject *
IdentityFilter_get_data (IdentityFilter * self)
{
	return PyObject_GetAttrString (self->base_image, ATTRIBUTE_NAME_DATA);
}

static PyObject *
IdentityFilter_get_metadata (IdentityFilter * self)
{
	return PyObject_GetAttrString (self->base_image, ATTRIBUTE_NAME_METADATA);
}

static PyGetSetDef IdentityFilter_getseters[] = {
	{ATTRIBUTE_NAME_BITS_PER_SAMPLE,
	 (getter) IdentityFilter_get_bits_per_sample,
	 NULL,
	 "Image bits per sample",
	 NULL},
	{ATTRIBUTE_NAME_SAMPLES_PER_PIXEL,
	 (getter) IdentityFilter_get_samples_per_pixel,
	 NULL,
	 "Image samples per pixel",
	 NULL},
	{ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION,
	 (getter) IdentityFilter_get_photometric_interpretation,
	 NULL,
	 "Image photometric interpretation",
	 NULL},
	{ATTRIBUTE_NAME_SAMPLE_FORMAT,
	 (getter) IdentityFilter_get_sample_format,
	 NULL,
	 "Image sample format",
	 NULL},
	{ATTRIBUTE_NAME_SIZE,
	 (getter) IdentityFilter_get_size,
	 NULL,
	 "Image size",
	 NULL},
	{ATTRIBUTE_NAME_DATA,
	 (getter) IdentityFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{ATTRIBUTE_NAME_METADATA,
	 (getter) IdentityFilter_get_metadata,
	 NULL,
	 "Image metadata",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyTypeObject IdentityFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.IdentityFilter",	/*tp_name */
	sizeof (IdentityFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	(destructor) IdentityFilter_dealloc,	/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The IdentityFilter does nothing. Instantiate with pytiff.IdentityFilter(baseimage) .",
	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	0,			/* tp_members */
	IdentityFilter_getseters,	/* tp_getset */
	0,			/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	IdentityFilter_new,	/* tp_new */
};

/* InvertFilter */

static void invert_image_data(char* data, int size_x, int size_y, int samples_per_pixel, int bits_per_sample, 
                            char* sample_format){
	int valuebits;
	int i,y,rowstride;
	int bits_per_pixel;
	valuebits = bits_per_sample;
	bits_per_pixel = samples_per_pixel * bits_per_sample;
	rowstride = (size_x * bits_per_pixel) / 8 + (((size_x * bits_per_pixel) % 8) ? 1 : 0);
        if(strcmp(sample_format, PYTIFF_SAMPLEFORMAT_COMPLEXINT)==0 ||
            strcmp(sample_format, PYTIFF_SAMPLEFORMAT_COMPLEXIEEEFP)==0)
            valuebits /= 2;
		Py_BEGIN_ALLOW_THREADS
            if(strcmp(sample_format, PYTIFF_SAMPLEFORMAT_UINT)==0){
                for (i = 0; i < size_y * rowstride; i++)
                    data[i] = ~(data[i]);
            }
            else {
                /* we have already checked that the sample format
                   isn't void. All sample formats except for unsigned int
                   have a sign at the beginning, so inverting the first
                   bit of each number should do the trick. */
                if(valuebits % 8 == 0){
                    for (i = 0; i < size_y * rowstride; i+=(valuebits/8))
                        data[i] ^= 128;
                }
                else {
		  for(y=0; y < size_y; y++)
		    for(i=0; i < rowstride * 8; i+= valuebits)
		      data[y*rowstride + i/8] ^= 128 >> (i % 8);
                }
            }
        Py_END_ALLOW_THREADS
}

static PyObject *
InvertFilter_get_data (InvertFilter * self)
{
  int size_x, size_y, rowstride, bps, spp;
	char *data;
	char *newdata;
        char* sample_format;
	PyObject *data_attr;
	PyObject *res;
	PyObject *sample_format_attr;


	/* boilerplate initialization */
	newdata = NULL;
	data_attr = NULL;
        sample_format_attr = NULL;
	res = NULL;
	rowstride = 0;
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		goto finish;	/* Exception already set */
	rowstride =
		Image_rowstride_as_int (((IdentityFilter *) self)->
					base_image);
	if (rowstride < 0)
		goto finish;	/* Exception already set */
	bps = Image_bits_per_sample_as_int (((IdentityFilter *) self)->
					   base_image);
	if (bps < 0)
		goto finish;	/* Exception already set */
	spp = Image_samples_per_pixel_as_int (((IdentityFilter *) self)->
					   base_image);
	if (spp < 0)
		goto finish;	/* Exception already set */
    sample_format = Image_sample_format_defaulted(((IdentityFilter *) self)->
					   base_image, &sample_format_attr);
    if(!sample_format)
        goto finish;
    if(strcmp(sample_format, PYTIFF_SAMPLEFORMAT_VOID)==0){
        PyErr_SetString (PyExc_ValueError,
				 "I cannot very well invert an image if I don't know its\n"
                 "sample format. (Sample format 'void' is illegal for this filter.)\n");
		goto finish;   
    }
	if (!Image_data_as_buf
	    (((IdentityFilter *) self)->base_image, (const void **) &data,
	     &data_attr))
		goto finish;	/* Exception already set */
	/* do stuff */
    newdata = malloc (rowstride * size_y);
    memcpy(newdata, data, rowstride * size_y);
    invert_image_data(newdata, size_x, size_y, spp, bps, sample_format);
		/* boilerplate cleanup */
      finish:
	if (newdata)
	{
		res = PyString_FromStringAndSize (newdata,
						  size_y * rowstride);
		free (newdata);
	}
	Py_XDECREF (data_attr);
	Py_XDECREF (sample_format_attr);
	return res;
}


static PyGetSetDef InvertFilter_getseters[] = {
	{ATTRIBUTE_NAME_DATA,
	 (getter) InvertFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyTypeObject InvertFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.InvertFilter",	/*tp_name */
	sizeof (InvertFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	0,			/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The InvertFilter inverts an image. For unsigned integer it flips all\n"
    "data bits, for other type excpet for void (which raises and error) it\n"
    "flips the first bit of each number.",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	0,			/* tp_members */
	InvertFilter_getseters,	/* tp_getset */
	&IdentityFilterType,	/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	0,		/* tp_new */
};

/* FlipFilter */

static PyObject *
FlipFilter_new_from_baseimage_and_direction (PyTypeObject * cls,
					     PyObject * base_image,
					     char *direction)
{
	FlipFilter *self;
	self = (FlipFilter *) cls->tp_alloc (cls, 0);
	((IdentityFilter *) self)->base_image = base_image;
	if (strcmp (direction, "horizontal") == 0)
	{
		self->direction = 1;
	}
	else
	{
		if (strcmp (direction, "vertical") == 0)
		{
			self->direction = 0;
		}
		else
		{
			PyErr_SetString (PyExc_ValueError,
					 "The flip direction must be either 'horizontal' or 'vertical'.");
			return NULL;
		}
	}
	Py_INCREF (((IdentityFilter *) self)->base_image);
	return (PyObject *) self;
}

static PyObject *
FlipFilter_new (PyTypeObject * cls, PyObject * args, PyObject * kwd)
{
	PyObject *self;
	PyObject *base_image;
	char *direction;
	if (PyArg_ParseTuple (args, "Os", &base_image, &direction) == 0)
		return NULL;	/* Exception already set */
	self = FlipFilter_new_from_baseimage_and_direction (cls, base_image,
							    direction);
	return (PyObject *) self;
}

static PyObject *
FlipFilter_get_data (FlipFilter * self)
{
	int size_x, size_y, rowstride, bpp;
	char *data;
	char *newdata;
	PyObject *data_attr;
	PyObject *res;

	int x, y, b;

	/* boilerplate initialization */
	newdata = NULL;
	data_attr = NULL;
	res = NULL;
	rowstride = 0;
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		goto finish;	/* Exception already set */
	rowstride =
		Image_rowstride_as_int (((IdentityFilter *) self)->
					base_image);
	if (rowstride < 0)
		goto finish;	/* Exception already set */
	bpp = Image_bits_per_pixel_as_int (((IdentityFilter *) self)->
					   base_image);
	if (bpp < 0)
		goto finish;	/* Exception already set */
	if (!Image_data_as_buf
	    (((IdentityFilter *) self)->base_image, (const void **) &data,
	     &data_attr))
		goto finish;	/* Exception already set */
	/* do stuff */
	Py_BEGIN_ALLOW_THREADS if (self->direction == 0)
	{
		/* vertical */
		newdata = calloc (rowstride * size_y, 1);
		for (y = 0; y < size_y; y++)
			memcpy ((char *) (newdata +
					  ((size_y - y - 1) * rowstride)),
				(char *) (data + (y * rowstride)), rowstride);
	}
	else
	{
		/* horizontal */
		if (bpp % 8 == 0)
		{
			int bytes_pp;
			newdata = calloc (rowstride * size_y, 1);
			bytes_pp = bpp / 8;
			for (y = 0; y < size_y; y++)
				for (x = 0; x < size_x; x++)
					memcpy ((void *) (newdata +
							  y * rowstride +
							  (size_x - x -
							   1) * bytes_pp),
						(void *) (data +
							  y * rowstride +
							  x * bytes_pp),
						bytes_pp);
		}
		else
		{
			newdata = calloc (rowstride * size_y, 1);
			for (y = 0; y < size_y; y++)
				for (x = 0; x < size_x; x++)
					for (b = 0; b < bpp; b++)
						if ((data
						     [y * rowstride +
						      (x * bpp +
						       b) / 8] & (128 >> ((x *
									   bpp
									   +
									   b)
									  %
									  8)))
						    != 0)
							newdata[y *
								rowstride +
								((size_x - x -
								  1) * bpp +
								 b) / 8] |=
								128 >>
								(((size_x -
								   x -
								   1) * bpp +
								  b) % 8);
		}
	}
	Py_END_ALLOW_THREADS
		/* boilerplate cleanup */
      finish:
	if (newdata)
	{
		res = PyString_FromStringAndSize (newdata,
						  size_y * rowstride);
		free (newdata);
	}
	Py_XDECREF (data_attr);
	return res;
}

static PyGetSetDef FlipFilter_getseters[] = {
	{ATTRIBUTE_NAME_DATA,
	 (getter) FlipFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyTypeObject FlipFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.FlipFilter",	/*tp_name */
	sizeof (FlipFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	0,			/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The FlipFilter flips an image vertically or horizontally. Instantiate with \n" "pytiff.FlipFilter(baseimage, \"vertical\")\n" "or pytiff.FlipFilter(baseimage, \"horizontal\") .",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	0,			/* tp_members */
	FlipFilter_getseters,	/* tp_getset */
	&IdentityFilterType,	/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	FlipFilter_new,		/* tp_new */
};


/* RotateFilter */

static PyObject *
RotateFilter_new_from_baseimage_and_angle (PyTypeObject * cls,
					   PyObject * base_image, int angle)
{
	RotateFilter *self;
	self = (RotateFilter *) cls->tp_alloc (cls, 0);
	((IdentityFilter *) self)->base_image = base_image;
	self->angle = angle % 360;
	if (self->angle < 0)
		self->angle += 360;
	if (self->angle % 90 != 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "I can only rotate in multiples of 90 degrees.");
		return NULL;
	}
	Py_INCREF (((IdentityFilter *) self)->base_image);
	return (PyObject *) self;
}

static PyObject *
RotateFilter_new (PyTypeObject * cls, PyObject * args, PyObject * kwd)
{
	PyObject *self;
	int angle;
	PyObject *base_image;
	if (PyArg_ParseTuple (args, "Oi", &base_image, &angle) == 0)
		return NULL;	/* Exception already set */
	self = RotateFilter_new_from_baseimage_and_angle (cls, base_image,
							  angle);
	return (PyObject *) self;
}

static PyObject *
RotateFilter_get_data (RotateFilter * self)
{
	int size_x, size_y, rowstride, bpp, newrowstride;
	char *data;
	char *newdata;
	PyObject *data_attr;
	PyObject *res;
	int newdata_size;
	int x, y, b;

	/* Real quick if we have identity */
	if (self->angle == 0)
	{
		return PyObject_GetAttrString (((IdentityFilter *) self)->
					       base_image, ATTRIBUTE_NAME_DATA);
	}
	/* boilerplate initialization */
	newdata = NULL;
	data_attr = NULL;
	res = NULL;
	rowstride = 0;
	newdata_size = 0;
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		goto finish;	/* Exception already set */
	rowstride =
		Image_rowstride_as_int (((IdentityFilter *) self)->
					base_image);
	if (rowstride < 0)
		goto finish;	/* Exception already set */
	bpp = Image_bits_per_pixel_as_int (((IdentityFilter *) self)->
					   base_image);
	if (bpp < 0)
		goto finish;	/* Exception already set */
	if (!Image_data_as_buf
	    (((IdentityFilter *) self)->base_image, (const void **) &data,
	     &data_attr))
		goto finish;	/* Exception already set */
	/* do stuff */
	Py_BEGIN_ALLOW_THREADS if (bpp % 8 == 0)
	{			/* pixels are aligned with bytes */
		int bytes_pp;
		newdata_size = rowstride * size_y;
		newdata = calloc (newdata_size, 1);
		bytes_pp = bpp / 8;
		if (self->angle == 180)
		{
			for (y = 0; y < size_y; y++)
				for (x = 0; x < size_x; x++)
					memcpy ((void *) (newdata +
							  (size_y - y -
							   1) * rowstride +
							  (size_x - x -
							   1) * bytes_pp),
						(void *) (data +
							  y * rowstride +
							  x * bytes_pp),
						bytes_pp);
		}
		else
		{
			newrowstride = size_y * bytes_pp;
			newdata_size = newrowstride * size_x;
			newdata = calloc (newdata_size, 1);
			if (self->angle == 90)
			{
				for (y = 0; y < size_y; y++)
					for (x = 0; x < size_x; x++)
						memcpy ((void *) (newdata +
								  (size_x -
								   x -
								   1) *
								  newrowstride
								  +
								  y *
								  bytes_pp),
							(void *) (data +
								  y *
								  rowstride +
								  x *
								  bytes_pp),
							bytes_pp);
			}
			else
			{	/* angle = 270 */
				for (y = 0; y < size_y; y++)
					for (x = 0; x < size_x; x++)
						memcpy ((void *) (newdata +
								  x *
								  newrowstride
								  + (size_y -
								     y -
								     1) *
								  bytes_pp),
							(void *) (data +
								  y *
								  rowstride +
								  x *
								  bytes_pp),
							bytes_pp);
			}
		}
	}
	else /* pixels are not aligned with bytes */ if (self->angle == 180)
	{
		newdata_size = rowstride * size_y, 1;
		newdata = calloc (newdata_size, 1);
		for (y = 0; y < size_y; y++)
			for (x = 0; x < size_x; x++)
				for (b = 0; b < bpp; b++)
					if ((data
					     [y * rowstride +
					      (x * bpp +
					       b) / 8] & (128 >> ((x * bpp +
								   b) %
								  8))) != 0)
						newdata[(size_y - y -
							 1) * rowstride +
							((size_x - x -
							  1) * bpp +
							 b) / 8] |=
							128 >>
							(((size_x - x -
							   1) * bpp + b) % 8);
	}
	else
	{
		newrowstride = size_y / 8 + (size_y % 8 ? 1 : 0);
		newdata_size = newrowstride * size_x;
		newdata = calloc (newdata_size, 1);
		if (self->angle == 90)
		{
			for (y = 0; y < size_y; y++)
				for (x = 0; x < size_x; x++)
					for (b = 0; b < bpp; b++)
						if ((data
						     [y * rowstride +
						      (x * bpp +
						       b) / 8] & (128 >> ((x *
									   bpp
									   +
									   b)
									  %
									  8)))
						    != 0)
							newdata[(size_x - x -
								 1) *
								newrowstride +
								(y * bpp +
								 b) / 8] |=
								128 >>
								((y * bpp +
								  b) % 8);
		}
		else
		{		/* angle = 270 */
			for (y = 0; y < size_y; y++)
				for (x = 0; x < size_x; x++)
					for (b = 0; b < bpp; b++)
						if ((data
						     [y * rowstride +
						      (x * bpp +
						       b) / 8] & (128 >> ((x *
									   bpp
									   +
									   b)
									  %
									  8)))
						    != 0)
							newdata[x *
								newrowstride +
								((size_y - y -
								  1) * bpp +
								 b) / 8] |=
								128 >>
								(((size_y -
								   y -
								   1) * bpp +
								  b) % 8);
		}
	}
	/* boilerplate cleanup */
	Py_END_ALLOW_THREADS finish:if (newdata)
	{
		res = PyString_FromStringAndSize (newdata, newdata_size);
		free (newdata);
	}
	Py_XDECREF (data_attr);
	return res;
}

static PyObject *
RotateFilter_get_size (RotateFilter * self)
{
	int size_x, size_y;
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		return NULL;	/* Exception already set */
	if (self->angle == 180)
		return Py_BuildValue ("(ii)", size_x, size_y);
	else
		return Py_BuildValue ("(ii)", size_y, size_x);
}

static PyGetSetDef RotateFilter_getseters[] = {
	{ATTRIBUTE_NAME_DATA,
	 (getter) RotateFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{ATTRIBUTE_NAME_SIZE,
	 (getter) RotateFilter_get_size,
	 NULL,
	 "Image size",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyMemberDef RotateFilter_members[] = {
	{"angle", T_INT, offsetof (RotateFilter, angle),
	 READONLY, "rotation angle in degrees counterclockwise"},
	{NULL}			/* Sentinel */
};

static PyTypeObject RotateFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.RotateFilter",	/*tp_name */
	sizeof (RotateFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	0,			/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The RotateFilter rotates images. Instantiate with pytiff.RotateFilter(baseimage, angle) .\n"
		"The angle is interpreted as degrees counterclockwise and right now must be a multiple of 90.",
	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	RotateFilter_members,	/* tp_members */
	RotateFilter_getseters,	/* tp_getset */
	&IdentityFilterType,	/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	RotateFilter_new,	/* tp_new */
};


/* CropFilter */

static PyObject *
CropFilter_new_from_baseimage_and_size (PyTypeObject * cls,
					PyObject * base_image,
					int pos_x,
					int pos_y, int size_x, int size_y)
{
	CropFilter *self;
	self = (CropFilter *) cls->tp_alloc (cls, 0);
	((IdentityFilter *) self)->base_image = base_image;
	self->pos_x = pos_x;
	self->pos_y = pos_y;
	self->size_x = size_x;
	self->size_y = size_y;
	Py_INCREF (((IdentityFilter *) self)->base_image);
	return (PyObject *) self;
}

static PyObject *
CropFilter_new (PyTypeObject * cls, PyObject * args, PyObject * kwd)
{
	PyObject *self;
	PyObject *base_image;
	int pos_x, pos_y, size_x, size_y;
	if (PyArg_ParseTuple
	    (args, "O(ii)(ii)", &base_image, &pos_x, &pos_y, &size_x,
	     &size_y) == 0)
		return NULL;	/* Exception already set */
	self = CropFilter_new_from_baseimage_and_size (cls, base_image, pos_x,
						       pos_y, size_x, size_y);
	return (PyObject *) self;
}

int
CropFilter_compute_corners (CropFilter * self, int *x1, int *y1, int *x2,
			    int *y2)
{
	int size_x, size_y, temp;
	PyObject *size_attr;
	size_attr =
		PyObject_GetAttrString (((IdentityFilter *) self)->base_image,
					ATTRIBUTE_NAME_SIZE);
	if (!size_attr)
		return -0;
	if (!PyArg_ParseTuple (size_attr, "ii", &size_x, &size_y))
	{
		*x1 = *y1 = *x2 = *y2 = 0;
		Py_XDECREF (size_attr);
		return -0;
	}
	Py_XDECREF (size_attr);
	*x1 = self->pos_x;
	*x2 = self->pos_x + self->size_x;
	*y1 = self->pos_y;
	*y2 = self->pos_y + self->size_y;
	if (*x1 > *x2)
	{
		temp = *x1;
		*x1 = *x2;
		*x2 = temp;
	}
	if (*y1 > *y2)
	{
		temp = *y1;
		*y1 = *y2;
		*y2 = temp;
	}
	if (*x1 < 0)
		*x1 = 0;
	if (*x2 < 0)
		*x2 = 0;
	if (*y1 < 0)
		*y1 = 0;
	if (*y2 < 0)
		*y2 = 0;
	if (*x1 > (size_x - 1))
		*x1 = size_x - 1;
	if (*y1 > (size_y - 1))
		*y1 = size_y - 1;
	if (*x2 > (size_x - 1))
		*x2 = size_x - 1;
	if (*y2 > (size_y - 1))
		*y2 = size_y - 1;
	return 1;
}

static PyObject *
CropFilter_get_size (CropFilter * self)
{
	int x1, y1, x2, y2;
	if (!CropFilter_compute_corners (self, &x1, &y1, &x2, &y2))
		return NULL;
	return Py_BuildValue ("(ii)", x2 - x1, y2 - y1);
}


static PyObject *
CropFilter_get_data (CropFilter * self)
{
	int size_x, size_y, new_size_x, new_size_y, rowstride, new_rowstride,
		x1, x2, y1, y2, bpp;
	char *data;
	char *newdata;
	PyObject *data_attr;
	PyObject *res;

	int x, y, b;

	/* boilerplate initialization */
	newdata = NULL;
	data_attr = NULL;
	res = NULL;
	rowstride = 0;
	new_size_y = new_rowstride = 0;
	if (!CropFilter_compute_corners (self, &x1, &y1, &x2, &y2))
		return NULL;	/* Exception already set */
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		goto finish;	/* Exception already set */
	rowstride =
		Image_rowstride_as_int (((IdentityFilter *) self)->
					base_image);
	if (rowstride < 0)
		goto finish;	/* Exception already set */
	bpp = Image_bits_per_pixel_as_int (((IdentityFilter *) self)->
					   base_image);
	if (bpp < 0)
		goto finish;	/* Exception already set */
	if (!Image_data_as_buf
	    (((IdentityFilter *) self)->base_image, (const void **) &data,
	     &data_attr))
		goto finish;	/* Exception already set */
	new_size_x = x2 - x1;
	new_size_y = y2 - y1;
	new_rowstride =
		(bpp * new_size_x) / 8 +
		(((bpp * new_size_x) % 8 > 0) ? 1 : 0);
	newdata = calloc (new_rowstride * new_size_y, 1);
	if (!newdata)
	{
		PyErr_SetString (PyExc_MemoryError,
				 "Could not allocate enough memory.");
		goto finish;
	}
	/* do stuff */
	Py_BEGIN_ALLOW_THREADS if ((x1 * bpp) % 8 == 0)	/* if the start of each row is byte aligned, it's easy */
		for (y = 0; y < new_size_y; y++)
			memcpy ((void *) (newdata + y * new_rowstride),
				(void *) (data + (y1 + y) * rowstride +
					  x1 * bpp / 8), new_rowstride);
	else			/* we have to copy bit by bit */
		for (y = 0; y < new_size_y; y++)
			for (x = 0; x < new_size_x; x++)
				for (b = 0; b < bpp; b++)
					if (data
					    [(y1 + y) * rowstride +
					     ((x1 + x) * bpp +
					      b) / 8] & (128 >> (((x1 + x) *
								  bpp +
								  b) % 8)))
						newdata[y * new_rowstride +
							(x * bpp + b) / 8] |=
							(128 >>
							 ((x * bpp + b) % 8));
	Py_END_ALLOW_THREADS
		/* boilerplate cleanup */
      finish:
	if (newdata)
	{
		res = PyString_FromStringAndSize (newdata,
						  new_size_y * new_rowstride);
		free (newdata);
	}
	Py_XDECREF (data_attr);
	return res;
}

static PyGetSetDef CropFilter_getseters[] = {
	{ATTRIBUTE_NAME_DATA,
	 (getter) CropFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{ATTRIBUTE_NAME_SIZE,
	 (getter) CropFilter_get_size,
	 NULL,
	 "Image size",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyTypeObject CropFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.CropFilter",	/*tp_name */
	sizeof (CropFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	0,			/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The CropFilter crops an image. Instantiate with pytiff.CropFilter(baseimage, (pos_x, pos_y), \n" "(size_x, size_y)) .",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	0,			/* tp_members */
	CropFilter_getseters,	/* tp_getset */
	&IdentityFilterType,	/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	CropFilter_new,		/* tp_new */
};

/* ChangeBitsPerSampleFilter */

static PyObject *
ChangeBitsPerSampleFilter_new (PyTypeObject * cls, PyObject * args,
			       PyObject * kwd)
{
	ChangeBitsPerSampleFilter *self;
	PyObject *base_image;
	self = (ChangeBitsPerSampleFilter *) cls->tp_alloc (cls, 0);
	if (PyArg_ParseTuple
	    (args, "Oi", &base_image, &(self->bits_per_sample)) == 0)
		return NULL;	/* Exception already set */
	((IdentityFilter *) self)->base_image = base_image;
	if (self->bits_per_sample < 1)
	{
		PyErr_SetString (PyExc_ValueError,
				 "Bits per sample must be at least 1.");
		return NULL;
	}
	Py_INCREF (((IdentityFilter *) self)->base_image);
	return (PyObject *) self;
}

static PyObject *
ChangeBitsPerSampleFilter_get_data (ChangeBitsPerSampleFilter * self)
{
	int size_x, size_y, rowstride, newrowstride, bps, newbps, spp;
	char *data;
	char *newdata;
	PyObject *data_attr;
	PyObject *res;

	/* boilerplate initialization */
	newdata = NULL;
	data_attr = NULL;
	res = NULL;
	newrowstride = rowstride = 0;
	newbps = self->bits_per_sample;
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		goto finish;	/* Exception already set */
	rowstride =
		Image_rowstride_as_int (((IdentityFilter *) self)->
					base_image);
	if (rowstride < 0)
		goto finish;	/* Exception already set */
	spp = Image_samples_per_pixel_as_int (((IdentityFilter *) self)->
					      base_image);
	if (spp < 0)
		goto finish;	/* Exception already set */
	bps = Image_bits_per_sample_as_int (((IdentityFilter *) self)->
					    base_image);
	if (bps < 0)
		goto finish;	/* Exception already set */
	if (!Image_data_as_buf
	    (((IdentityFilter *) self)->base_image, (const void **) &data,
	     &data_attr))
		goto finish;	/* Exception already set */
	newrowstride =
		(size_x * newbps * spp) / 8 +
		(((size_x * newbps * spp) % 8) ? 1 : 0);
	newdata = calloc (newrowstride * size_y, 1);
	if (!newdata)
	{
		PyErr_SetFromErrno (PyExc_MemoryError);
		goto finish;
	}
	if (bps == newbps)
	{
		memcpy ((void *) newdata, (void *) data, size_y * rowstride);
		goto finish;
	}
	/* do stuff */
	Py_BEGIN_ALLOW_THREADS
	{
		int x, y, sample, bit_no, yoffset, ynewoffset, xoffsetbits,
			xnewoffsetbits, sampleoffsetbits, samplenewoffsetbits;
		for (y = 0; y < size_y; y++)
		{
			yoffset = y * rowstride;
			ynewoffset = y * newrowstride;
			for (x = 0; x < size_x; x++)
			{
				xoffsetbits = x * spp * bps;
				xnewoffsetbits = x * spp * newbps;
				for (sample = 0; sample < spp; sample++)
				{
					sampleoffsetbits = sample * bps;
					samplenewoffsetbits = sample * newbps;
					for (bit_no = 0; bit_no < newbps;
					     bit_no++)
					{
						int old_bit_no;
						old_bit_no = bit_no % bps;
						if ((data
						     [yoffset +
						      (xoffsetbits +
						       sampleoffsetbits +
						       old_bit_no) /
						      8] & (128 >>
							    ((xoffsetbits +
							      sampleoffsetbits
							      +
							      old_bit_no) %
							     8))) != 0)
							newdata[ynewoffset +
								(xnewoffsetbits
								 +
								 samplenewoffsetbits
								 +
								 bit_no) /
								8] |=
								128 >>
								((xnewoffsetbits + samplenewoffsetbits + bit_no) % 8);
					}
				}
			}
		}
	}
	Py_END_ALLOW_THREADS
		/* boilerplate cleanup */
      finish:
	if (newdata)
	{
		res = PyString_FromStringAndSize (newdata,
						  size_y * newrowstride);
		free (newdata);
	}
	Py_XDECREF (data_attr);
	return res;
}

static PyGetSetDef ChangeBitsPerSampleFilter_getseters[] = {
	{ATTRIBUTE_NAME_DATA,
	 (getter) ChangeBitsPerSampleFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyMemberDef ChangeBitsPerSampleFilter_members[] = {
	{ATTRIBUTE_NAME_BITS_PER_SAMPLE, T_INT,
	 offsetof (ChangeBitsPerSampleFilter, bits_per_sample),
	 READONLY, "Image bits per sample"},
	{NULL}			/* Sentinel */
};

static PyTypeObject ChangeBitsPerSampleFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.ChangeBitsPerSampleFilter",	/*tp_name */
	sizeof (ChangeBitsPerSampleFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	0,			/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The ChangeBitsPerSampleFilter changes the bits_per_sample of an image. Instantiate with \n" "pytiff.ChangeBitsPerSampleFilter(baseimage, bits_per_sample) ." "* Currently this only works for unsigned int. *\n",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	ChangeBitsPerSampleFilter_members,	/* tp_members */
	ChangeBitsPerSampleFilter_getseters,	/* tp_getset */
	&IdentityFilterType,	/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	ChangeBitsPerSampleFilter_new,	/* tp_new */
};



/* ScaleDownToRGBAFilter */

static PyObject *
ScaleDownToRGBAFilter_new (PyTypeObject * cls, PyObject * args,
			   PyObject * kwd)
{
	ScaleDownToRGBAFilter *self;
	self = (ScaleDownToRGBAFilter *) cls->tp_alloc (cls, 0);
	if (PyArg_ParseTuple
	    (args, "Oi", &((IdentityFilter *) self)->base_image,
	     &self->scale_factor) == 0)
		return NULL;
	Py_INCREF (((IdentityFilter *) self)->base_image);
	if (self->scale_factor < 1)
		return NULL;
	return (PyObject *) self;
}

int
ScaleDownToRGBAFilter_compute_size (ScaleDownToRGBAFilter * self,
				    long *new_size_x, long *new_size_y)
{
	long size_x, size_y;
	PyObject *size_attr;
	size_attr =
		PyObject_GetAttrString (((IdentityFilter *) self)->base_image,
					ATTRIBUTE_NAME_SIZE);
	if (!size_attr)
		return -0;
	if (!PyArg_ParseTuple (size_attr, "ll", &size_x, &size_y))
	{
		*new_size_x = *new_size_y = 0;
		Py_XDECREF (size_attr);
		return -0;
	}
	Py_XDECREF (size_attr);
	(*new_size_x) =
		(size_x / self->scale_factor) +
		((size_x % self->scale_factor > 0) ? 1 : 0);
	(*new_size_y) =
		(size_y / self->scale_factor) +
		((size_y % self->scale_factor > 0) ? 1 : 0);
	return 1;
}

static PyObject *
ScaleDownToRGBAFilter_get_size (ScaleDownToRGBAFilter * self)
{
	long new_size_x, new_size_y;
	if (!ScaleDownToRGBAFilter_compute_size
	    (self, &new_size_x, &new_size_y))
		return NULL;
	return Py_BuildValue ("(ll)", new_size_x, new_size_y);
}

static PyObject *
ScaleDownToRGBAFilter_get_data (ScaleDownToRGBAFilter * self)
{
	int size_x, size_y, rowstride, bpp, bps, spp;
	long new_size_x, new_size_y;
	char *data;
	char *newdata;
	PyObject *data_attr;
	PyObject *res;
	char *photint;

	int modecode;
	int x, y, x1, y1, x2, y2;
	long new_r, new_g, new_b, new_a;
	int pix_visited;

	/* boilerplate initialization */
	newdata = NULL;
	res = NULL;
	rowstride = 0;
	if (!Image_size_as_ints
	    (((IdentityFilter *) self)->base_image, &size_x, &size_y))
		goto finish;	/* Exception already set */
	rowstride =
		Image_rowstride_as_int (((IdentityFilter *) self)->
					base_image);
	if (rowstride < 0)
		goto finish;	/* Exception already set */
	bpp = Image_bits_per_pixel_as_int (((IdentityFilter *) self)->
					   base_image);
	if (bpp < 0)
		goto finish;	/* Exception already set */
	bps = Image_bits_per_sample_as_int (((IdentityFilter *) self)->
					    base_image);
	if (bpp < 0)
		goto finish;	/* Exception already set */
	spp = Image_samples_per_pixel_as_int (((IdentityFilter *) self)->
					      base_image);
	if (spp < 0)
		goto finish;	/* Exception already set */
	if (!Image_data_as_buf
	    (((IdentityFilter *) self)->base_image, (const void **) &data,
	     &data_attr))
		goto finish;	/* Exception already set */
	if (!ScaleDownToRGBAFilter_compute_size
	    (self, &new_size_x, &new_size_y))
		goto finish;	/* Exception already set */
	photint =
		Image_photometric_interpretation_as_string (((IdentityFilter
							      *) self)->
							    base_image);
	/* do stuff */
	if ((strcmp (photint, PYTIFF_PHOTOMETRIC_INTERPRETATION_GREY) == 0) && (bps == 1) && (spp == 1))
	{
		modecode = 1;
		goto mode_set;
	}
	if ((strcmp (photint, PYTIFF_PHOTOMETRIC_INTERPRETATION_GREY) == 0) && (bps == 8) && (spp == 1))
	{
		modecode = 2;
		goto mode_set;
	}
	if ((strcmp (photint, PYTIFF_PHOTOMETRIC_INTERPRETATION_RGB) == 0) && (bps == 8) && (spp == 3))
	{
		modecode = 3;
		goto mode_set;
	}
	if ((strcmp (photint, PYTIFF_PHOTOMETRIC_INTERPRETATION_RGBA) == 0) && (bps == 8) && (spp == 4))
	{
		modecode = 4;
		goto mode_set;
	}
	PyErr_SetString (PyExc_ValueError,
			 "I can't scale images of this mode.");
	goto finish;
      mode_set:
	Py_BEGIN_ALLOW_THREADS newdata =
		calloc (4 * new_size_x * new_size_y, 1);
	if (!newdata)
	{
		PyErr_SetString (PyExc_MemoryError,
				 "Could not allocate memory.");
		goto finish;
	}
	for (x = 0; x < new_size_x; x++)
		for (y = 0; y < new_size_y; y++)
		{
			new_r = new_g = new_b = new_a = pix_visited = 0;
			for (x1 = 0; x1 < self->scale_factor; x1++)
				for (y1 = 0; y1 < self->scale_factor; y1++)
				{
					x2 = x * self->scale_factor + x1;
					y2 = y * self->scale_factor + y1;
					if (x2 < size_x && y2 < size_y)
					{
						switch (modecode)
						{
						case 1:
							if (data
							    [y2 * rowstride +
							     x2 /
							     8] & (128 >> (x2
									   %
									   8)))
								new_r++;
							break;
						case 2:
							new_r += data[y2 *
								      rowstride
								      + x2];
							break;
						case 3:
							new_r += data[y2 *
								      rowstride
								      +
								      x2 * 3];
							new_g += data[y2 *
								      rowstride
								      +
								      x2 * 3 +
								      1];
							new_b += data[y2 *
								      rowstride
								      +
								      x2 * 3 +
								      2];
							break;
						case 4:
							new_r += data[y2 *
								      rowstride
								      +
								      x2 * 4];
							new_g += data[y2 *
								      rowstride
								      +
								      x2 * 4 +
								      1];
							new_b += data[y2 *
								      rowstride
								      +
								      x2 * 4 +
								      2];
							new_a += data[y2 *
								      rowstride
								      +
								      x2 * 4 +
								      3];
							break;
						}
						pix_visited++;
					}
				}
			switch (modecode)
			{
			case 1:
				new_r = new_g = new_b =
					new_r * 255 / pix_visited;
				new_a = 255;
				break;
			case 2:
				new_r = new_g = new_b =
					new_r / pix_visited +
					((new_r % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_a = 255;
				break;
			case 3:
				new_r = new_r / pix_visited +
					((new_r % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_g = new_g / pix_visited +
					((new_g % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_b = new_b / pix_visited +
					((new_b % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_a = 255;
				break;
			case 4:
				new_r = new_r / pix_visited +
					((new_r % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_g = new_g / pix_visited +
					((new_g % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_b = new_b / pix_visited +
					((new_b % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				new_a = new_a / pix_visited +
					((new_a % pix_visited >=
					  pix_visited / 2) ? 1 : 0);
				break;
			}
			newdata[(x + y * new_size_x) * 4] = new_r;
			newdata[(x + y * new_size_x) * 4 + 1] = new_g;
			newdata[(x + y * new_size_x) * 4 + 2] = new_b;
			newdata[(x + y * new_size_x) * 4 + 3] = new_a;
		}
	Py_END_ALLOW_THREADS
		/* boilerplate cleanup */
      finish:
	if (newdata)
	{
		res = PyString_FromStringAndSize (newdata,
						  4 * new_size_x *
						  new_size_y);
		free (newdata);
	}
	Py_XDECREF (data_attr);
	return res;
}

static PyObject *
ScaleDownToRGBAFilter_get_bits_per_sample (ScaleDownToRGBAFilter * self)
{
	return PyInt_FromLong (8);
}

static PyObject *
ScaleDownToRGBAFilter_get_samples_per_pixel (ScaleDownToRGBAFilter * self)
{
	return PyInt_FromLong (4);
}

static PyObject *
ScaleDownToRGBAFilter_get_photometric_interpretation (ScaleDownToRGBAFilter *
						      self)
{
	return PyString_FromString (PYTIFF_PHOTOMETRIC_INTERPRETATION_RGBA);
}

static PyGetSetDef ScaleDownToRGBAFilter_getseters[] = {
	{ATTRIBUTE_NAME_BITS_PER_SAMPLE,
	 (getter) ScaleDownToRGBAFilter_get_bits_per_sample,
	 NULL,
	 "Image bits per sample",
	 NULL},
	{ATTRIBUTE_NAME_SAMPLES_PER_PIXEL,
	 (getter) ScaleDownToRGBAFilter_get_samples_per_pixel,
	 NULL,
	 "Image samples per pixel",
	 NULL},
	{ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION,
	 (getter) ScaleDownToRGBAFilter_get_photometric_interpretation,
	 NULL,
	 "Image photometric interpretation",
	 NULL},
	{ATTRIBUTE_NAME_SIZE,
	 (getter) ScaleDownToRGBAFilter_get_size,
	 NULL,
	 "Image size",
	 NULL},
	{ATTRIBUTE_NAME_DATA,
	 (getter) ScaleDownToRGBAFilter_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{NULL}			/* Sentinel */
};

static PyMemberDef ScaleDownToRGBAFilter_members[] = {
	{"scaledown_factor", T_INT,
	 offsetof (ScaleDownToRGBAFilter, scale_factor),
	 READONLY,
	 "factor by which the image is scaled down; 2 means the result is half the original size"},
	{NULL}			/* Sentinel */
};

static PyTypeObject ScaleDownToRGBAFilterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.ScaleDownToRGBAFilter",	/*tp_name */
	sizeof (ScaleDownToRGBAFilter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	0,			/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"The ScaleDownToRGBAFilter simultaneously converts an image to RGBA mode and scales it down\n" "by a given integral factor. This is useful to save memory for the screen presentation of bitmap\n" "images. Instantiate with pytiff.ScaleDownToRGBAFilter(baseimage, factor) . A factor of 2 gives\n" "an image half the size of the original.\n" "* Currently this only works for unsigned int images. *",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	ScaleDownToRGBAFilter_members,	/* tp_members */
	ScaleDownToRGBAFilter_getseters,	/* tp_getset */
	&IdentityFilterType,	/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	ScaleDownToRGBAFilter_new,	/* tp_new */
};


/* Tiff objects utility routines */

PyObject *
tiffcode_from_number (int number)
{
	char buf[100];
	snprintf (buf, 100, "TIFF code %d", number);
	return PyString_FromString (buf);
}

int
tiffcode_string_as_number (char *buf)
{
	/* return -1 in case of error */
	char *err;
	int res;
	if (!buf)
		return -1;
	if (strlen (buf) < 11)
	{
		PyErr_SetString (PyExc_ValueError,
				 "I can't interpret this string as a TIFF coded value.");
		return -1;
	}
	buf += 10;
	res = strtol (buf, &err, 10);
	if (err != (buf + strlen (buf)))
	{
		PyErr_SetString (PyExc_ValueError,
				 "I can't interpret this string as a TIFF coded value.");
		return -1;
	}
	return res;
}

int
tiffcode_as_number (PyObject * s)
{
	/* return -1 in case of error */
	char *buf;
	buf = PyString_AsString (s);
	return tiffcode_string_as_number (buf);
}

PyObject *
tiffcode_or_const_from_number (int number, TagNames * values)
{
	int i;
	for (i = 0; values[i].name != NULL; i++)
		if (values[i].tag == number)
			return PyString_FromString (values[i].name);
	return tiffcode_from_number (number);
}

int
tiffcode_or_const_as_number (PyObject * s, TagNames * values)
{
	/* return -1 in case of error */
	char *buf;
	int i;
	if (!s)
		return -1;
	if (!PyString_Check (s))
		return -1;
	buf = PyString_AsString (s);
	for (i = 0; values[i].name != NULL; i++)
		if (strcmp (values[i].name, buf) == 0)
			return values[i].tag;
	return tiffcode_string_as_number (buf);
}

/* TiffReaderPage */

static TiffReaderPage *
TiffReaderPage_new_from_reader (PyTypeObject * cls, TiffReader * reader)
{
	TiffReaderPage *self;
	uint32 w, l, temp1;
	uint16 bitspersample, samplesperpixel, orientation, photometric,
		sampleformat;
	int i, j;
	char *ascii_value;
	TIFF *tif;
	float x_res, y_res;
	uint16 short_value;
	self = (TiffReaderPage *) cls->tp_alloc (cls, 0);
	self->reader = reader;
	Py_INCREF (reader);
	tif = reader->tif;
	self->seq_no = TIFFCurrentDirectory (tif);
	if (TIFFGetField (tif, TIFFTAG_IMAGEWIDTH, &w) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "TIFF image doesn't have a width tag.");
		return NULL;
	}
	if (TIFFGetField (tif, TIFFTAG_IMAGELENGTH, &l) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "TIFF image doesn't have a height tag.");
		return NULL;
	}
	if (TIFFGetFieldDefaulted (tif, TIFFTAG_BITSPERSAMPLE, &bitspersample) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "TIFF image doesn't have a bits per sample tag.");
		return NULL;
	}
	if (TIFFGetFieldDefaulted (tif, TIFFTAG_SAMPLESPERPIXEL, &samplesperpixel) ==
	    0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "TIFF image doesn't have a samples per pixel tag.");
		return NULL;
	}
	if (TIFFGetFieldDefaulted (tif, TIFFTAG_ORIENTATION, &orientation) ==
	    0)
	{
		return NULL;
	}
	if (TIFFGetField (tif, TIFFTAG_PHOTOMETRIC, &photometric) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "This image doesn't have a photometric field.");
		return NULL;
	}
	if (TIFFGetFieldDefaulted (tif, TIFFTAG_SAMPLEFORMAT, &sampleformat)
	    == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "This image doesn't have a sampleformat field.");
		return NULL;
	}
	if (TIFFGetField (tif, TIFFTAG_IMAGEDEPTH, &short_value))
		if (short_value != 1)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Image depths other than 1 are not currently supported.");
			return NULL;
		}
	if (TIFFGetField (tif, TIFFTAG_TILEDEPTH, &short_value))
		if (short_value != 1)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Tile depths other than 1 are not currently supported.");
			return NULL;
		}
	if (orientation >= ORIENTATION_LEFTTOP
	    && orientation <= ORIENTATION_LEFTBOT)
	{
		temp1 = l;
		l = w;
		w = temp1;
	}
	self->size =
		(PyTupleObject *) Py_BuildValue ("(i,i)", (int) w, (int) l);
	self->bits_per_sample = bitspersample;
	self->samples_per_pixel = samplesperpixel;
	switch (photometric)
	{
	case PHOTOMETRIC_MINISWHITE:
	case PHOTOMETRIC_MINISBLACK:
		self->photometric_interpretation =
			PyString_FromString (PYTIFF_PHOTOMETRIC_INTERPRETATION_GREY);
		break;
	case PHOTOMETRIC_RGB:
		if (self->samples_per_pixel == 3)
			self->photometric_interpretation =
				PyString_FromString (PYTIFF_PHOTOMETRIC_INTERPRETATION_RGB);
		else
			self->photometric_interpretation =
				PyString_FromString (PYTIFF_PHOTOMETRIC_INTERPRETATION_RGBA);
		break;
	default:
		self->photometric_interpretation =
			tiffcode_from_number (photometric);
	}
	self->sample_format =
		tiffcode_or_const_from_number (sampleformat,
					       SampleFormatValues);
	/* set metadata */
	self->real_metadata = (PyDictObject *) PyDict_New ();
	self->metadata =
		(PyDictObject *) PyDictProxy_New ((PyObject *) self->
						  real_metadata);
	for (i = 0; AsciiTags[i].tag > 0; i++)
	{
		if (TIFFGetField (tif, AsciiTags[i].tag, &ascii_value) == 1)
		{
			PyDict_SetItemString ((PyObject *) self->
					      real_metadata,
					      AsciiTags[i].name,
					      PyString_FromString
					      (ascii_value));
		}
	}
	/* set resolution */
	if (TIFFGetField (tif, TIFFTAG_XRESOLUTION, &x_res) == 1)
	{
		if (TIFFGetField (tif, TIFFTAG_YRESOLUTION, &y_res) == 1)
		{
			PyDict_SetItemString ((PyObject *) self->
					      real_metadata, "resolution",
					      Py_BuildValue ("(d, d)",
							     (double) x_res,
							     (double) y_res));
		}
	}
	/* set resolution */
	if (TIFFGetField (tif, TIFFTAG_DATETIME, &ascii_value) == 1)
	{
		if (strlen (ascii_value) == 20)
		{
			int year, month, day, hour, minute, second;
			char buf[5];
			char *errchr;
			strncpy (buf, ascii_value + 0, 4);
			buf[4] = 0;
			year = strtol (buf, &errchr, 10);
			if (errchr != buf + 4)
				goto nothing;
			strncpy (buf, ascii_value + 5, 2);
			buf[2] = 0;
			month = strtol (buf, &errchr, 10);
			if (errchr != buf + 2)
				goto nothing;
			strncpy (buf, ascii_value + 8, 2);
			day = strtol (buf, &errchr, 10);
			if (errchr != buf + 2)
				goto nothing;
			strncpy (buf, ascii_value + 11, 2);
			hour = strtol (buf, &errchr, 10);
			if (errchr != buf + 2)
				goto nothing;
			strncpy (buf, ascii_value + 14, 2);
			minute = strtol (buf, &errchr, 10);
			if (errchr != buf + 2)
				goto nothing;
			strncpy (buf, ascii_value + 17, 2);
			second = strtol (buf, &errchr, 10);
			if (errchr != buf + 2)
				goto nothing;
			PyDict_SetItemString ((PyObject *) self->
					      real_metadata, "date/time",
					      Py_BuildValue ("(i,i,i,i,i,i)",
							     year, month, day,
							     hour, minute,
							     second));
		      nothing:
			{
			}
		}
	}
	/* coded values */
	for (i = 0; CodedValuesTags[i].tag > 0; i++)
	{
		if (TIFFGetFieldDefaulted
		    (tif, CodedValuesTags[i].tag, &short_value) == 1)
		{
			ascii_value = NULL;
			for (j = 0; CodedValuesTags[i].values[j].tag > 0; j++)
				if (CodedValuesTags[i].values[j].tag ==
				    short_value)
				{
					ascii_value =
						CodedValuesTags[i].values[j].
						name;
					break;
				}
			if (ascii_value != NULL)
				PyDict_SetItemString ((PyObject *) self->
						      real_metadata,
						      CodedValuesTags[i].name,
						      PyString_FromString
						      (ascii_value));
			else
				PyDict_SetItemString ((PyObject *) self->
						      real_metadata,
						      CodedValuesTags[i].name,
						      tiffcode_from_number
						      (short_value));
		}
	}
	return self;
}

PyObject *
TiffReaderPage_get_data (TiffReaderPage * self)
{
	TIFF *tif;
	PyObject *result;
	uint16 orientation;
	char *data;
	int rowstride, planerowstride;
	int width, length, planes;
	uint16 planconf, photometric;
	uint32 temp32, tilewidth, tilelength;
	int x, y, s;
	char *planebuf;
	int samples_per_pixel, bits_per_sample;

	tif = self->reader->tif;
	Py_BEGIN_ALLOW_THREADS
		PyThread_acquire_lock (self->reader->tiflock, WAIT_LOCK);
	Py_END_ALLOW_THREADS if (TIFFSetDirectory (tif, self->seq_no) == 0)
	{
		PyErr_SetString (PyExc_IndexError,
				 "Could not seek the image on disc.");
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}
	if (TIFFGetFieldDefaulted (tif, TIFFTAG_ORIENTATION, &orientation) ==
	    0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "This image doesn't have an orientation field.");
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}
	if (TIFFGetField (tif, TIFFTAG_IMAGEWIDTH, &width) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "This image doesn't have a width field.");
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}
	if (TIFFGetField (tif, TIFFTAG_IMAGELENGTH, &length) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "This image doesn't have a length field.");
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}
	if (TIFFGetField (tif, TIFFTAG_PHOTOMETRIC, &photometric) == 0)
	{
		PyErr_SetString (PyExc_ValueError,
				 "This image doesn't have a photometric field.");
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}
	if (self->samples_per_pixel == 1)
		planconf = PLANARCONFIG_CONTIG;
	else
	{
		if (TIFFGetFieldDefaulted
		    (tif, TIFFTAG_PLANARCONFIG, &planconf) == 0)
		{
			PyErr_SetString (PyExc_ValueError,
					 "This image doesn't have a planar configuration field.");
			PyThread_release_lock (self->reader->tiflock);
			return NULL;
		}
	}
	samples_per_pixel = self->samples_per_pixel;
	bits_per_sample = self->bits_per_sample;
	rowstride = self->bits_per_sample * self->samples_per_pixel * width;
	rowstride = (rowstride / 8) + ((rowstride % 8) ? 1 : 0);
	if (photometric == PHOTOMETRIC_MINISWHITE &&
	    strcmp(PyString_AsString(self->sample_format), PYTIFF_SAMPLEFORMAT_VOID)==0)
	{
	  PyErr_SetString (PyExc_ValueError,
			   "I can't invert this image, but the photometric interpretation demands it.");
	  PyThread_release_lock (self->reader->tiflock);
	  return NULL;
	}
	data = malloc (rowstride * length);
	if (!data)
	{
		PyErr_SetFromErrno (PyExc_MemoryError);
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}

	switch (planconf)
	{
	case PLANARCONFIG_CONTIG:
		planerowstride = rowstride;
		planes = 1;
		break;
	case PLANARCONFIG_SEPARATE:
		planerowstride = self->bits_per_sample * width;
		planerowstride =
			(planerowstride / 8) + ((planerowstride % 8) ? 1 : 0);
		planes = samples_per_pixel;
		break;
	default:
		PyErr_SetString (PyExc_ValueError,
				 "Unknown planar configuration.");
		PyThread_release_lock (self->reader->tiflock);
		return NULL;
	}

	if (planes == 1)
		planebuf = data;
	else
	{
		planebuf = calloc (planerowstride * length, 1);
		if (!planebuf)
		{
			PyErr_SetFromErrno (PyExc_MemoryError);
			free (data);
			PyThread_release_lock (self->reader->tiflock);
			return NULL;
		}
	}

	for (s = 0; s < planes; s++)
	{
		if (TIFFGetField (tif, TIFFTAG_ROWSPERSTRIP, &temp32))
		{
			for (y = 0; y < length; y++)
			{
				int res;
				Py_BEGIN_ALLOW_THREADS
					res =
					TIFFReadScanline (tif,
							  planebuf +
							  y * planerowstride,
							  y, s);
				Py_END_ALLOW_THREADS if (res == -1)
				{
					PyErr_SetString (PyExc_ValueError,
							 "Tifflib could not read the strip.");
					if (data != planebuf)
						free (planebuf);
					free (data);
					PyThread_release_lock (self->reader->
							       tiflock);
					return NULL;
				}
			}
		}
		else if (TIFFGetField (tif, TIFFTAG_TILEWIDTH, &tilewidth) &&
			 TIFFGetField (tif, TIFFTAG_TILELENGTH, &tilelength))
		{
			char *tile_buf;
			int tilerowstride;
			tilerowstride = TIFFTileSize (tif) / tilelength;
			tile_buf = malloc (TIFFTileSize (tif));
			if (!tile_buf)
			{
				PyErr_SetFromErrno (PyExc_MemoryError);
				if (data != planebuf)
					free (planebuf);
				free (data);
				PyThread_release_lock (self->reader->tiflock);
				return NULL;
			}
			for (x = 0; x < width; x += tilewidth)
			{
				for (y = 0; y < length; y += tilelength)
				{
					int y1;
					int res;
					Py_BEGIN_ALLOW_THREADS
						res =
						TIFFReadTile (tif, tile_buf,
							      x, y, 0, s);
					Py_END_ALLOW_THREADS if (res == -1)
					{
						PyErr_SetString
							(PyExc_ValueError,
							 "Tifflib could not read the tile.");
						if (data != planebuf)
							free (planebuf);
						free (data);
						free (tile_buf);
						PyThread_release_lock (self->
								       reader->
								       tiflock);
						return NULL;
					}
					Py_BEGIN_ALLOW_THREADS
						for (y1 = 0;
						     (y1 < tilelength)
						     && (y + y1 < length);
						     y1++)
					{
						int pix_in_line;
						int buflen;
						char *dest;
						pix_in_line =
							(x + tilewidth <
							 width) ? tilewidth :
							width - x;
						if (planes == 1)
						{
							buflen = (bits_per_sample * samples_per_pixel * pix_in_line) / 8;
							dest = planebuf + (y +
									   y1)
								*
								planerowstride
								+
								x *
								bits_per_sample
								*
								samples_per_pixel
								/ 8;
						}
						else
						{
							buflen = (bits_per_sample * pix_in_line) / 8;
							dest = planebuf + (y +
									   y1)
								*
								planerowstride
								+
								x *
								bits_per_sample
								/ 8;
						}
						memcpy ((void *) dest,
							(void *) (tile_buf +
								  y1 *
								  tilerowstride),
							buflen);
					}
				Py_END_ALLOW_THREADS}
			}
			free (tile_buf);
		}
		else
		{
			PyErr_SetString (PyExc_ValueError,
					 "Image seems to have neither strips nor tiles.");
			if (data != planebuf)
				free (planebuf);
			free (data);
			PyThread_release_lock (self->reader->tiflock);
			return NULL;
		}

		if (planes > 1)
		{
			Py_BEGIN_ALLOW_THREADS for (y = 0; y < length; y++)
				for (x = 0; x < width; x++)
					if (bits_per_sample % 8 == 0)
						memcpy ((void *) (data +
								  y *
								  rowstride +
								  x *
								  (bits_per_sample
								   / 8) *
								  samples_per_pixel
								  +
								  s *
								  (bits_per_sample
								   / 8)),
							(void *) (planebuf +
								  y *
								  planerowstride
								  +
								  x *
								  (bits_per_sample
								   / 8)),
							bits_per_sample / 8);
					else
					{
						int b;
						for (b = 0;
						     b < bits_per_sample; b++)
							if ((planebuf
							     [y *
							      planerowstride +
							      (x *
							       bits_per_sample
							       +
							       b) /
							      8] & (128 >>
								    ((x *
								      bits_per_sample
								      +
								      b) %
								     8))) !=
							    0)
								data[y *
								     rowstride
								     +
								     (x *
								      bits_per_sample
								      *
								      samples_per_pixel
								      +
								      s *
								      bits_per_sample
								      +
								      b) /
								     8] |=
								    128 >>
								    ((x *
								      bits_per_sample
								      *
								      samples_per_pixel
								      +
								      s *
								      bits_per_sample
								      +
								      b) % 8);

					}
		Py_END_ALLOW_THREADS}
	}

	/* NO MORE TIF ACCESS AFTER THIS */
	PyThread_release_lock (self->reader->tiflock);

	if (planes > 1)
		free (planebuf);

	if (is_big_endian)
	{
		int bits_per_sample_number;
		bits_per_sample_number = bits_per_sample;
		if (strcmp(PyString_AsString(self->sample_format), PYTIFF_SAMPLEFORMAT_COMPLEXINT) == 0
		    || strcmp(PyString_AsString(self->sample_format), PYTIFF_SAMPLEFORMAT_COMPLEXIEEEFP)==0)
			bits_per_sample_number /= 2;
		if (bits_per_sample_number == 16)
			swap_byte_order_n_bytes (data, 2, length * rowstride);
		else if (bits_per_sample_number == 32)
			swap_byte_order_n_bytes (data, 4, length * rowstride);
		else if (bits_per_sample_number == 64)
			swap_byte_order_n_bytes (data, 8, length * rowstride);
	}

	if (photometric == PHOTOMETRIC_MINISWHITE)
	  invert_image_data(data, width, length, samples_per_pixel,
			    bits_per_sample, PyString_AsString(self->sample_format));

	result = PyString_FromStringAndSize (data, length * rowstride);
	free (data);

	if (orientation != ORIENTATION_TOPLEFT
	    && orientation <= ORIENTATION_LEFTBOT)
	{
		MemoryImage *uncorrected_image;
		FlipFilter *flipf;
		RotateFilter *rotatef;
		PyObject *new_result;
		flipf = NULL;
		rotatef = NULL;
		new_result = NULL;
		uncorrected_image =
			(MemoryImage *)
			MemoryImage_new_from_Cdata (&MemoryImageType, width,
						    length, result,
						    (PyObject *) self->
						    metadata,
						    self->bits_per_sample,
						    self->samples_per_pixel,
						    PyString_AsString (self->
								       photometric_interpretation),
						    PyString_AsString (self->
								       sample_format));
		if (!uncorrected_image)
			return NULL;	/* Exception already set */
		switch (orientation)
		{
		case ORIENTATION_TOPRIGHT:
			flipf = (FlipFilter *)
				FlipFilter_new_from_baseimage_and_direction
				(&FlipFilterType,
				 (PyObject *) uncorrected_image,
				 "horizontal");
			new_result = FlipFilter_get_data (flipf);
			break;
		case ORIENTATION_BOTRIGHT:
			rotatef =
				(RotateFilter *)
				RotateFilter_new_from_baseimage_and_angle
				(&RotateFilterType,
				 (PyObject *) uncorrected_image, 180);
			new_result = RotateFilter_get_data (rotatef);
			break;
		case ORIENTATION_BOTLEFT:
			flipf = (FlipFilter *)
				FlipFilter_new_from_baseimage_and_direction
				(&FlipFilterType,
				 (PyObject *) uncorrected_image, "vertical");
			new_result = FlipFilter_get_data (flipf);
			break;
		case ORIENTATION_LEFTTOP:
			flipf = (FlipFilter *)
				FlipFilter_new_from_baseimage_and_direction
				(&FlipFilterType,
				 (PyObject *) uncorrected_image, "vertical");
			rotatef =
				(RotateFilter *)
				RotateFilter_new_from_baseimage_and_angle
				(&RotateFilterType, (PyObject *) flipf, -90);
			new_result = RotateFilter_get_data (rotatef);
			break;
		case ORIENTATION_RIGHTTOP:
			rotatef =
				(RotateFilter *)
				RotateFilter_new_from_baseimage_and_angle
				(&RotateFilterType,
				 (PyObject *) uncorrected_image, -90);
			new_result = RotateFilter_get_data (rotatef);
			break;
		case ORIENTATION_RIGHTBOT:
			flipf = (FlipFilter *)
				FlipFilter_new_from_baseimage_and_direction
				(&FlipFilterType,
				 (PyObject *) uncorrected_image, "vertical");
			rotatef =
				(RotateFilter *)
				RotateFilter_new_from_baseimage_and_angle
				(&RotateFilterType, (PyObject *) flipf, 90);
			new_result = RotateFilter_get_data (rotatef);
			break;
		case ORIENTATION_LEFTBOT:
			rotatef =
				(RotateFilter *)
				RotateFilter_new_from_baseimage_and_angle
				(&RotateFilterType,
				 (PyObject *) uncorrected_image, 90);
			new_result = RotateFilter_get_data (rotatef);
			break;
		}
		Py_XDECREF (result);
		result = new_result;
		Py_XDECREF ((PyObject *) flipf);
		Py_XDECREF ((PyObject *) rotatef);
	}


	/* TODO C routines for orientation filters */

	return result;
}

static void
TiffReaderPage_dealloc (TiffReaderPage * self)
{
	Py_XDECREF (self->size);
	Py_XDECREF (self->metadata);
	Py_XDECREF (self->photometric_interpretation);
	Py_XDECREF (self->reader);
	Py_XDECREF (self->real_metadata);
	((TiffReaderPage *) self)->ob_type->tp_free ((PyObject *) self);
};


static PyGetSetDef TiffReaderPage_getseters[] = {
	{ATTRIBUTE_NAME_DATA,
	 (getter) TiffReaderPage_get_data,
	 NULL,
	 "Image data",
	 NULL},
	{NULL}			/* Sentinel */
};


static PyMemberDef TiffReaderPage_members[] = {
	{ATTRIBUTE_NAME_PYTIFF_PHOTOMETRIC_INTERPRETATION, T_OBJECT,
	 offsetof (TiffReaderPage, photometric_interpretation),
	 READONLY, "Image photometric interpretation"},
	{ATTRIBUTE_NAME_SAMPLE_FORMAT, T_OBJECT, offsetof (TiffReaderPage, sample_format),
	 READONLY, "Image sample format"},
	{ATTRIBUTE_NAME_BITS_PER_SAMPLE, T_INT, offsetof (TiffReaderPage, bits_per_sample),
	 READONLY, "Image bits per sample"},
	{ATTRIBUTE_NAME_SAMPLES_PER_PIXEL, T_INT,
	 offsetof (TiffReaderPage, samples_per_pixel),
	 READONLY, "Image samples per pixel"},
	{ATTRIBUTE_NAME_SIZE, T_OBJECT, offsetof (TiffReaderPage, size), READONLY,
	 "Image size"},
	{ATTRIBUTE_NAME_METADATA, T_OBJECT, offsetof (TiffReaderPage, metadata), READONLY,
	 "Image metadata"},
	{NULL}			/* Sentinel */
};


static PyTypeObject TiffReaderPageType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.TiffReaderPage",	/*tp_name */
	sizeof (TiffReaderPage),	/*tp_basicsize */
	0,			/*tp_itemsize */
	(destructor) TiffReaderPage_dealloc,	/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0, /*tp_str */ 0,	/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"This is one page of a TIFF file. It is automatically instantiated by the TiffReader",
	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	TiffReaderPage_members,	/* tp_members */
	TiffReaderPage_getseters,	/* tp_getset */
	0,			/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	0,			/* tp_new */
};

/* TiffReader */

static PyObject *
TiffReader_new (PyTypeObject * cls, PyObject * args, PyObject * kwds)
{
	TiffReader *self;
	char *filename;
	TiffReaderPage *page;
	if (!PyArg_ParseTuple (args, "s", &filename))
		return NULL;	/* Exception already set */
	self = (TiffReader *) cls->tp_alloc (cls, 0);
	self->tif = TIFFOpen (filename, "r");
	if (!self->tif)
	{
		PyErr_SetString (PyExc_IOError, "Could not open TIFF file.");
		return NULL;
	}
	self->tiflock = PyThread_allocate_lock ();
	self->pages = (PyListObject *) PyList_New (0);
	do
	{
		page = TiffReaderPage_new_from_reader (&TiffReaderPageType,
						       self);
		if (!page)
			return NULL;	/* Exception already set */
		PyList_Append ((PyObject *) self->pages, (PyObject *) page);
		Py_DECREF ((PyObject *) page);
	}
	while (TIFFReadDirectory (self->tif));
	return (PyObject *) self;
}


static void
TiffReader_dealloc (TiffReader * self)
{
	if (self->tiflock)
	{
		PyThread_free_lock (self->tiflock);
	}
	Py_XDECREF (self->pages);
	if (self->tif)
		TIFFClose (self->tif);
	self->ob_type->tp_free ((PyObject *) self);
}

static int
TiffReader_sq_length (TiffReader * self)
{
	return PyList_Size ((PyObject *) (self->pages));
}

static PyObject *
TiffReader_sq_item (TiffReader * self, int item)
{
	PyObject *the_item;
	the_item = PyList_GetItem ((PyObject *) (self->pages), item);
	if (the_item)
		Py_INCREF (the_item);
	return the_item;
}


static PySequenceMethods TiffReaderSequenceMethods = {
	(inquiry) TiffReader_sq_length,	/* sq_length */
	0,			/* sq_concat */
	0,			/* sq_repeat */
	(intargfunc) TiffReader_sq_item,	/* sq_item */
	0,			/* sq_slice */
	0,			/* sq_ass_item */
	0,			/* sq_ass_slice */
	0,			/* sq_contains */
	0,			/* sq_inplace_concat */
	0,			/* sq_inplace_repeat */
};


static PyTypeObject TiffReaderType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.TiffReader",	/*tp_name */
	sizeof (TiffReader),	/*tp_basicsize */
	0,			/*tp_itemsize */
	(destructor) TiffReader_dealloc,	/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	&TiffReaderSequenceMethods,	/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0,			/*tp_str */
	0,			/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"TiffReader reads TIFF files. Instantiate it with pytiff.TiffReader(filename) and then\n" "access the data via the sequence interface (i.e. with reader[] and len(reader) ).",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	0,			/* tp_methods */
	0,			/* tp_members */
	0,			/* tp_getset */
	0,			/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	TiffReader_new,		/* tp_new */
};

/* TiffWriter */

static PyObject *
TiffWriter_new (PyTypeObject * cls, PyObject * args, PyObject * kwds)
{
	TiffWriter *self;
	char *filename;
	if (!PyArg_ParseTuple (args, "s", &filename))
		return NULL;
	self = (TiffWriter *) cls->tp_alloc (cls, 0);
	self->tif = TIFFOpen (filename, "w");
	if (!self->tif)
		return NULL;
	self->tiflock = PyThread_allocate_lock ();
	return (PyObject *) self;
}


static void
TiffWriter_dealloc (TiffWriter * self)
{
	if (self->tiflock)
		PyThread_free_lock (self->tiflock);
	if (self->tif)
		TIFFClose (self->tif);
	self->ob_type->tp_free ((PyObject *) self);
}

static PyObject *
TiffWriter_append (TiffWriter * self, PyObject * args, PyObject * kwd)
{
	PyObject *res;
	PyObject *image;
	if (!PyArg_ParseTuple (args, "O", &image))
		return NULL;	/* Exception set */
	/* TODO: Copy verbatim if this is a ReaderPage */
	res = NULL;
	if (1)
	{
		int size_x, size_y;
		char *data;
		int data_size;
		PyObject *metadata;
		PyObject *data_attr;
		int16 bitspersample, samplesperpixel;
		int16 photometric;
		int i;
		int rowstride;
		char *newdata;
		char *photometric_interpretation;
		int tifres;
		int sampleformat;
		Py_BEGIN_ALLOW_THREADS PyThread_acquire_lock (self->tiflock,
							      WAIT_LOCK);
		Py_END_ALLOW_THREADS metadata = NULL;
		newdata = NULL;
		if (!Image_size_as_ints (image, &size_x, &size_y))
			goto finish;	/* Exception already set */
		if (!Image_data_as_buf
		    (image, (const void **) &data, &data_attr))
			goto finish;	/* Exception already set */
		data_size = Image_data_size_as_int (image);
		if (data_size < 0)
			goto finish;	/* Exception already set. */
		metadata = PyObject_GetAttrString (image, ATTRIBUTE_NAME_METADATA);
		samplesperpixel = Image_samples_per_pixel_as_int (image);
		if (samplesperpixel < 0)
			goto finish;	/* Exception already set. */
		rowstride = Image_rowstride_as_int (image);
		if (rowstride < 0)
			goto finish;	/* Exception already set. */
		bitspersample = Image_bits_per_sample_as_int (image);
		if (bitspersample < 0)
			goto finish;	/* Exception already set. */
		sampleformat =
			tiffcode_or_const_as_number (PyObject_GetAttrString
						     (image, ATTRIBUTE_NAME_SAMPLE_FORMAT),
						     SampleFormatValues);
		if (sampleformat == -1)
		{
			sampleformat = SAMPLEFORMAT_UINT;
		}
		if (!PyMapping_Check (metadata))
		{
			PyErr_SetString (PyExc_TypeError,
					 "Metadata must be a mapping object.");
			goto finish;
		}
		photometric_interpretation =
			Image_photometric_interpretation_as_string (image);
		if (strcmp (PYTIFF_PHOTOMETRIC_INTERPRETATION_GREY, photometric_interpretation) == 0)
		{
			photometric = PHOTOMETRIC_MINISBLACK;
			goto photo_set;
		}
		if (strcmp (PYTIFF_PHOTOMETRIC_INTERPRETATION_RGB, photometric_interpretation) == 0)
		{
			photometric = PHOTOMETRIC_RGB;
			goto photo_set;
		}
		if (strcmp (PYTIFF_PHOTOMETRIC_INTERPRETATION_RGBA, photometric_interpretation) == 0)
		{
			photometric = PHOTOMETRIC_RGB;
			goto photo_set;
		}
		photometric =
			tiffcode_string_as_number
			(photometric_interpretation);
		if (photometric != -1)
			goto photo_set;
		PyErr_SetString (PyExc_ValueError,
				 "I don't know this photometric interpretation.");
		goto finish;
	      photo_set:
		if ((photometric == PHOTOMETRIC_MINISBLACK)
		    && (bitspersample == 1))
		{
			photometric = PHOTOMETRIC_MINISWHITE;
			newdata = (char *) malloc (data_size);
			if (!newdata)
			{
				PyErr_SetString (PyExc_MemoryError,
						 "Could not allocate memory to save data.");
				goto finish;
			}
			for (i = 0; i < data_size; i++)
				newdata[i] = ~(data[i]);
			data = newdata;
		}

		/* Reverse byte order if need be */
		if (is_big_endian)
		{
			int bits_per_sample_number;
			if (!newdata)
			{
				newdata = (char *) malloc (data_size);
				if (!newdata)
				{
					PyErr_SetString (PyExc_MemoryError,
							 "Could not allocate memory to save data.");
					goto finish;
				}
				memcpy (newdata, data, data_size);
				data = newdata;
			}
			bits_per_sample_number = bitspersample;
			if ((sampleformat == SAMPLEFORMAT_COMPLEXINT)
			    || (sampleformat == SAMPLEFORMAT_COMPLEXIEEEFP))
				bits_per_sample_number /= 2;
			if (bits_per_sample_number == 16)
				swap_byte_order_n_bytes (data, 2, data_size);
			else if (bits_per_sample_number == 32)
				swap_byte_order_n_bytes (data, 4, data_size);
			else if (bits_per_sample_number == 64)
				swap_byte_order_n_bytes (data, 8, data_size);

		}

		/* The basics */
		TIFFSetField (self->tif, TIFFTAG_IMAGEWIDTH, size_x);
		TIFFSetField (self->tif, TIFFTAG_IMAGELENGTH, size_y);
		TIFFSetField (self->tif, TIFFTAG_BITSPERSAMPLE,
			      bitspersample);
		TIFFSetField (self->tif, TIFFTAG_SAMPLESPERPIXEL,
			      samplesperpixel);
		TIFFSetField (self->tif, TIFFTAG_PHOTOMETRIC, photometric);
		TIFFSetField (self->tif, TIFFTAG_SAMPLEFORMAT, sampleformat);
		TIFFSetField (self->tif, TIFFTAG_ORIENTATION,
			      ORIENTATION_TOPLEFT);
		TIFFSetField (self->tif, TIFFTAG_PLANARCONFIG,
			      PLANARCONFIG_CONTIG);
		TIFFSetField (self->tif, TIFFTAG_ROWSPERSTRIP, size_y);
		/* Stuff from metadata */
		for (i = 0; AsciiTags[i].tag > 0; i++)
			if (PyMapping_HasKeyString
			    (metadata, AsciiTags[i].name))
			{
				PyObject *item;
				item = PyMapping_GetItemString (metadata,
								AsciiTags[i].
								name);
				if (item)
				{
					char *ascii_value;
					ascii_value =
						PyString_AsString (item);
					if (ascii_value)
						TIFFSetField (self->tif,
							      AsciiTags[i].
							      tag,
							      ascii_value);
					Py_DECREF (item);
				}
			}
		if (PyMapping_HasKeyString (metadata, "resolution"))
		{
			double x_res, y_res;
			PyObject *item;
			item = PyMapping_GetItemString (metadata,
							"resolution");
			if (item)
			{
				if (PyArg_ParseTuple
				    (item, "dd", &x_res, &y_res))
				{
					TIFFSetField (self->tif,
						      TIFFTAG_XRESOLUTION,
						      (float) x_res);
					TIFFSetField (self->tif,
						      TIFFTAG_YRESOLUTION,
						      (float) y_res);
				}
				Py_DECREF (item);
			}
		}
		if (PyMapping_HasKeyString (metadata, "date/time"))
		{
			PyObject *item;
			item = PyMapping_GetItemString (metadata,
							"resolution");
			if (item)
			{
				unsigned int year, month, day, hour, minute,
					second;
				if (PyArg_ParseTuple
				    (item, "(iiiiii)", &year, &month, &day,
				     &hour, &minute, &second))
				{
					char ascii_value[20];
					snprintf (ascii_value, 20,
						  "%04d:%02d:%02d %02d:%02d:%02d",
						  year, month, day, hour,
						  minute, second);
					TIFFSetField (self->tif,
						      TIFFTAG_DATETIME,
						      ascii_value);
				}
			}
			Py_DECREF (item);
		}
		for (i = 0; CodedValuesTags[i].tag > 0; i++)
		{
			if (PyMapping_HasKeyString
			    (metadata, CodedValuesTags[i].name))
			{
				PyObject *item;
				item = PyMapping_GetItemString (metadata,
								CodedValuesTags
								[i].name);
				if (item)
				{
					if (PyString_Check (item))
					{
						char *ascii_value;
						int j;
						uint16 num_value;
						ascii_value =
							PyString_AsString
							(item);
						num_value = 0;
						for (j = 0;
						     CodedValuesTags[i].
						     values[j].tag > 0; j++)
						{
							if (strcmp
							    (CodedValuesTags
							     [i].values[j].
							     name,
							     ascii_value) ==
							    0)
							{
								num_value =
									CodedValuesTags
									[i].
									values
									[j].
									tag;
								break;
							}
						}
						if (num_value)
							TIFFSetField (self->
								      tif,
								      CodedValuesTags
								      [i].tag,
								      num_value);
					}
					else if (tiffcode_as_number (item) !=
						 -1)
					{
						TIFFSetField (self->tif,
							      CodedValuesTags
							      [i].tag,
							      tiffcode_as_number
							      (item));
					}
					Py_DECREF (item);
				}
			}
		}
		/* write data */
		Py_BEGIN_ALLOW_THREADS
			tifres =
			TIFFWriteEncodedStrip (self->tif, 0, data, data_size);
		Py_END_ALLOW_THREADS if (tifres != data_size)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Could not write the TIFF strip to disk.");
			goto finish;
		}
		Py_BEGIN_ALLOW_THREADS tifres =
			TIFFWriteDirectory (self->tif);
		Py_END_ALLOW_THREADS if (!tifres)
		{
			PyErr_SetString (PyExc_ValueError,
					 "Could not write the TIFF directory to disk.");
		}
		res = Py_None;
		Py_INCREF (res);
	      finish:
		if (newdata)
			free (newdata);
		Py_XDECREF (metadata);
		Py_XDECREF (data_attr);
		PyThread_release_lock (self->tiflock);
	}
	return res;
}

static int
TiffWriter_sq_length (TiffWriter * self)
{
	return TIFFCurrentDirectory (self->tif);
}

static PySequenceMethods TiffWriterSequenceMethods = {
	(inquiry) TiffWriter_sq_length,	/* sq_length */
	0,			/* sq_concat */
	0,			/* sq_repeat */
	0,			/* sq_item */
	0,			/* sq_slice */
	0,			/* sq_ass_item */
	0,			/* sq_ass_slice */
	0,			/* sq_contains */
	0,			/* sq_inplace_concat */
	0,			/* sq_inplace_repeat */
};

static PyMethodDef TiffWriter_methods[] = {
	{"append", (PyCFunction) TiffWriter_append, METH_VARARGS,
	 "Append an image to this file."},
	{NULL}			/* Sentinel */
};

static PyTypeObject TiffWriterType = {
	PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
	"pytiff.TiffWriter",	/*tp_name */
	sizeof (TiffWriter),	/*tp_basicsize */
	0,			/*tp_itemsize */
	(destructor) TiffWriter_dealloc,	/*tp_dealloc */
	0,			/*tp_print */
	0,			/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	0,			/*tp_repr */
	0,			/*tp_as_number */
	&TiffWriterSequenceMethods,	/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	0,			/*tp_str */
	0,			/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
	"TiffWriter writes images to a TIFF file. Instantiate with pytiff.TiffWriter(filename) and then\n" "append pages with the append() method.",	/* tp_doc */
	0,			/* tp_traverse */
	0,			/* tp_clear */
	0,			/* tp_richcompare */
	0,			/* tp_weaklistoffset */
	0,			/* tp_iter */
	0,			/* tp_iternext */
	TiffWriter_methods,	/* tp_methods */
	0,			/* tp_members */
	0,			/* tp_getset */
	0,			/* tp_base */
	0,			/* tp_dict */
	0,			/* tp_descr_get */
	0,			/* tp_descr_set */
	0,			/* tp_dictoffset */
	0,			/* tp_init */
	0,			/* tp_alloc */
	TiffWriter_new,		/* tp_new */
};

/* initialization */

static PyMethodDef pytiffmethods[] = {
	{NULL, NULL, 0, NULL}
};


typedef struct
{
	PyTypeObject *type_obj;
	char *type_name;
} register_type_t;

register_type_t register_types[] = {
	{&MemoryImageType, "MemoryImage"},
	{&IdentityFilterType, "IdentityFilter"},
	{&InvertFilterType, "InvertFilter"},
	{&FlipFilterType, "FlipFilter"},
	{&CropFilterType, "CropFilter"},
	{&RotateFilterType, "RotateFilter"},
	{&ChangeBitsPerSampleFilterType, "ChangeBitsPerSampleFilter"},
	{&ScaleDownToRGBAFilterType, "ScaleDownToRGBAFilter"},
	{&TiffReaderType, "TiffReader"},
	{&TiffWriterType, "TiffWriter"},
	{&TiffReaderPageType, "TiffReaderPage"},
	{NULL, NULL}
};

char *module_doc =
	"Tiff handling and advanced imaging for Python.\n"
	"Copyright (c) 2005 by Oliver M. Haynold\n\n"
	"A little code example:\n"
	"To crop all pages of a TIFF file to 200 * 200 pixels say:\n\n"
	"r=pytiff.TiffReader('source.tif')\n"
	"w=pytiff.TiffWriter('dest.tiff')\n"
	"for p in r:\n"
	"   w.append(pytiff.CropFilter(p, (0,0), (200, 200)))\n\n"
	"That's all!\n\n"
	"All image objects have these attributes:\n"
	"- bits_per_sample \n"
	"- samples_per_pixel\n"
	"- photometric_interpretation: a string;\n"
	"    for most operations arbitrary values are acceptable, but the library currently understands:\n"
	"    -'grey' for grey or black/white data with 0 being black\n"
	"    -'RGB' and and 'RGBA'\n"
	"- sample_format\n"
	"   - 'unsigned integer' (default if this attribute is missing)\n"
	"   - 'signed integer'\n"
	"   - 'IEEE floating point'\n"
	"   - 'integer complex'\n"
	"   - 'IEEE floating point complex'\n"
	"   - 'void' (this really implies the data is meaningless, so avoid it!)\n"
	"- size: a tuple of two integers, width first\n"
	"- data: the image data\n"
	"- metadata: a dictionary with the image metadata\n\n"
	"Filters generate their results on the fly, and all objects in this modules, with the exception \n"
	"of TiffWriter, are immutable.";

void
determine_endianness (void)
{
	short one = 1;
	char *cp = (char *) &one;

	if (*cp == 0)
		is_big_endian = 1;
	else
		is_big_endian = 0;
	return;
}

void
initpytiff (void)
{
	PyObject *m;
	int i;

	determine_endianness ();

	for (i = 0; register_types[i].type_obj != NULL; i++)
		if (PyType_Ready (register_types[i].type_obj) < 0)
			return;

	m = Py_InitModule3 ("pytiff", pytiffmethods, module_doc);
	if (m == NULL)
		return;

	for (i = 0; register_types[i].type_obj != NULL; i++)
		if (register_types[i].type_name)
		{
			Py_INCREF (register_types[i].type_obj);
			PyModule_AddObject (m, register_types[i].type_name,
					    (PyObject *) register_types[i].
					    type_obj);
		}
}
