"""\
Test cases for Template
Copyright (C) 2005 Remy Blank

This file is part of SvnReporter.

SvnReporter is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, version 2.

SvnReporter is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with SvnReporter; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
"""

import unittest

import SvnReporter


class TestTemplate(SvnReporter.Template):
    def foreach_main(self, locals):
        return self.apply(locals)

    def foreach_entry(self, locals):
        return self.apply(locals) + self.apply(locals)


class TemplateTest(unittest.TestCase):
    def testSimpleLoop(self):
        """Template with a simple loop"""
        t = SvnReporter.Template("""Item 1 <?foreach entry?>Item2<?end entry?> Item 3""")
        self.assertEqual(("main", "Item 1 %(foreach(0))s Item 3"), t.loops[1])
        self.assertEqual(("entry", "Item2"), t.loops[0])

    def testMultiple(self):
        """Multiple loops in series"""
        t = SvnReporter.Template("""\
Item 1 <?foreach first?>Item 2<?end first?> Item 3
Item 4 <?foreach second?>Item 5<?end second?> Item 6
""")
        self.assertEqual(("main", """\
Item 1 %(foreach(0))s Item 3
Item 4 %(foreach(1))s Item 6
"""), t.loops[2])
        self.assertEqual(("first", "Item 2"), t.loops[0])
        self.assertEqual(("second", "Item 5"), t.loops[1])

    def testNesting(self):
        """Template with nested loops"""
        t = SvnReporter.Template("""\
Item 1 <?foreach frame?>Item 2 <?foreach field?>Item 3<?end field?> Item 4<?end frame?> Item 5
""")
        self.assertEqual(("main", """\
Item 1 %(foreach(1))s Item 5
"""), t.loops[2])
        self.assertEqual(("frame", "Item 2 %(foreach(0))s Item 4"), t.loops[1])
        self.assertEqual(("field", "Item 3"), t.loops[0])

    def testApply(self):
        """Applying a template"""
        t = TestTemplate("""Item 1 <?foreach entry?>Item 2<?end entry?> Item 3""")
        self.assertEqual("""Item 1 Item 2Item 2 Item 3""", t.generate({}))


class Evaluator(unittest.TestCase):
    def testSimple(self):
        """Simple expressions in an Evaluator"""
        def f(n, b):
            if b:
                return str(-n)
            else:
                return str(n)
        
        e = SvnReporter.Evaluator({"a": 2, "b": 3, "f": f})
        self.assertEqual("3", e.substitute("%(1 + 2)d"))
        self.assertEqual("5", e.substitute("%(a + b)d"))
        self.assertEqual("(1, 2)", e.substitute("%((1, 2))s"))
        self.assertEqual("-3", e.substitute("%(f(3, True))s"))
        
    def testRecursion(self):
        """Maximum recursion depth of an Evaluator"""
        def f(n):
            if n:
                return "%%(f(%d))s" % (n - 1)
            else:
                return "Done"

        e = SvnReporter.Evaluator({"f": f}, 10)
        self.assertEqual("Done", e.substitute("%(f(9))s"))
        self.assertEqual("Done", e.substitute(f(10)))
        self.assertRaises(RuntimeError, e.substitute, "%(f(10))s")
        self.assertRaises(RuntimeError, e.substitute, f(11))
        
        e = SvnReporter.Evaluator({}, 0)
        self.assertEqual("Done", e.substitute("Done"))
        self.assertRaises(RuntimeError, e.substitute, "%(1 + 2)d")

