#!/bin/env python
"""\
Subversion commit reporter
Copyright (C) 2005 Remy Blank

This file is part of SvnReporter.

SvnReporter is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, version 2.

SvnReporter is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with SvnReporter; if not, write to the Free Software Foundation,
Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
"""

# TODO: Base classes for single-event and event-list generators
# TODO: Allow customising the section regexps
# TODO: Add a News generator
# TODO: Add prefix smtp: or sendmail: to specify delivery of mail

import ConfigParser
import cPickle as pickle
import imp
from optparse import OptionParser
import os
import re
import smtplib
from svn import core, delta, fs, repos
import sys
import time
import traceback
from xml.sax.saxutils import escape, quoteattr

__metaclass__ = type

# Project metadata
class _metadata_:
    project   = "SvnReporter"
    version   = "0.4"
    date      = "2005.12.03"
    author    = "Remy Blank"
    email     = "software@calins.ch"
    copyright = "Copyright (C) %s %s <%s>" % (date[0:4], author, email)
    license   = "GPL-2"
    url       = "http://www.calins.ch/software/SvnReporter.html"
    download  = "http://www.calins.ch/download/SvnReporter/%s-%s.tar.gz" % (project, version)
    description = "Subversion commit reporter"
    longDescription = """\
SvnReporter generates various reports in response to commits happening
in a Subversion repository. It is intended to be called from the 
post-commit hook.

Two types of reports are supported: single-event and event list reports.
The former generate reports relative to the current commit only, and are 
typically used to generate post-commit mails. The latter generate reports
relative to a list of commits, e.g. an RSS feed or a web page showing the 
latest commits.

Reports can be restricted to commits whose objects match certain criteria,
specified by a list of regular expressions.

The format of the reports can be defined freely and flexibly using a
very simple template format.
"""
    keywords = ["Subversion", "SVN", "commit", "post-commit", "hook", 
                "e-mail", "feed", "atom", "rss"]
    platforms = ["POSIX", "Windows"]
    classifiers = [
        "Development Status :: 4 - Beta",
        "Environment :: Console",
        "Intended Audience :: Developers",
        "License :: OSI Approved :: GNU General Public License (GPL)",
        "Operating System :: OS Independent",
        "Programming Language :: Python",
        "Topic :: Software Development :: Version Control",
    ]

__author__ = _metadata_.author
__version__ = _metadata_.version
__date__ = _metadata_.date

# Test hooks
_open = open

# Module error exception
class Error(Exception): pass

# Helper functions
def inetTime(timestamp=None, local=False):
    """Format a timestamp as internet time (RFC3339)."""
    if timestamp is None:
        timestamp = time.time()
    if local:
        localTime = time.localtime(timestamp)
        if localTime.tm_isdst:
            zoneSecs = time.altzone
        else:
            zoneSecs = time.timezone
        if zoneSecs >= 0:
            zone = "-" + time.strftime("%H:%M", time.gmtime(zoneSecs))
        else:
            zone = "+" + time.strftime("%H:%M", time.gmtime(-zoneSecs))
        return time.strftime("%Y-%m-%dT%H:%M:%S", localTime) + zone
    else:
        return time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime(timestamp))


def viewCvsObjectLink(urlBase, revision, change):
    """Create a ViewCVS object link from a change."""
    if change.action == "deleted":
        return escape(change.path)
    href = "%s/%s?rev=%d" % (urlBase, change.path, revision)
    if not change.path.endswith("/"):
        href += "&view=markup"
    return "<a href=%s>%s</a>" % (quoteattr(href), escape(change.path))


def viewCvsDiffLink(urlBase, revision, change, text):
    """Create a viewcvs diff link from a change."""
    if not change.textChanged:
        return ""
    if change.action == "added":
        return ""
    if change.basePath:
        basePath = change.basePath
        baseRevision = change.baseRevision
    else:
        basePath = change.path
        baseRevision = revision - 1 
    href = "%s/%s?rev=%d&view=diff&r1=%d&r2=%d&p1=%s&p2=%s" % (urlBase, change.path, revision, 
        revision, baseRevision, change.path, basePath)
    return "<a href=%s>%s</a>" % (quoteattr(href), text)

    
def readFile(fileName):
    """Read a file into a string."""
    f = _open(fileName, "r")
    try:
        return f.read()
    finally:
        f.close()


class ChainDict(dict):
    """Chained dictionary"""
    def __init__(*args, **kwargs):
        self = args[0]
        if len(args) > 1:
            self.chain = args[1]
            args = args[2:]
        else:
            self.chain = {}
            args = args[1:]
        super(ChainDict, self).__init__(*args, **kwargs)
    
    def __contains__(self, key):
        return super(ChainDict, self).__contains__(key) or key in self.chain

    def __getitem__(self, key):
        try:
            return super(ChainDict, self).__getitem__(key)
        except KeyError:
            return self.chain[key]


class Struct:
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)


class Evaluator:
    """Dict proxy that returns eval(key) as a value"""
    def __init__(self, locals, maxDepth=None):
        self.locals = locals
        self.maxDepth = maxDepth
    
    def substitute(self, s):
        return s % self

    def __getitem__(self, key):
        if self.maxDepth is not None:
            if self.maxDepth <= 0:
                raise RuntimeError, "maximum substitution depth exceeded"
            self.maxDepth -= 1
        result = eval(key, {}, self.locals)
        if isinstance(result, basestring):
            result = result % self
        if self.maxDepth is not None:
            self.maxDepth += 1
        return result        


class Template:
    """Template processor"""
    begin = re.compile(r"<\?foreach\s+(?P<name>[a-zA-Z][a-zA-Z0-9_]*)\s*\?>")
    end = r"<\?end\s+%s\s*\?>"

    def __init__(self, content):
        self.loops = []
        self.parse("main", content)
        self.stack = []
        
    def generate(self, locals):
        self.locals = ChainDict(locals)
        return self.foreach(len(self.loops) - 1)    # main is always last

    def apply(self, locals):
        return self.applyLoop(locals, self.stack[-1])

    def applyLoop(self, locals, index):
        self.locals = ChainDict(locals, foreach=self.foreach)
        evaluator = Evaluator(self.locals, 100)
        return evaluator.substitute(self.loops[index][1])

    def foreach(self, index):
        self.stack.append(index)
        try:
            return getattr(self, "foreach_" + self.loops[index][0])(self.locals)
        finally:
            self.stack.pop()

    def parse(self, name, content):
        pos = 0
        while True:
            begin = self.begin.search(content, pos)
            if not begin:
                break
            loopName = begin.group("name")
            
            endPattern = re.compile(self.end % loopName)
            end = endPattern.search(content, begin.end())
            if not end:
                raise Error, "End of foreach '%s' not found" % loopName
                
            index = self.parse(loopName, content[begin.end():end.start()])
            newContent = content[:begin.start()] + "%(foreach(" + str(index) + "))s"
            pos = len(newContent)
            content = newContent + content[end.end():]

        self.loops.append((name, content))
        return len(self.loops) - 1


def taskPoolUser(f):
    """Decorates a function to call self.clearTaskPool() after a method."""
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        finally:
            args[0].clearTaskPool()
    wrapper.__name__ = f.__name__
    wrapper.__doc__ = f.__doc__
    return wrapper


class SvnDataSource:
    """Data source for SVN repository"""
    @taskPoolUser
    def __init__(self, pool, path, revision=None, cachePath=None):
        self.pool = pool
        self.path = path
        self.cachePath = cachePath
        self.taskPool = core.svn_pool_create(self.pool)
        self.repos = repos.open(path, self.pool)
        self.fsPtr = repos.fs(self.repos)

        self.uuid = fs.get_uuid(self.fsPtr, self.taskPool)
        self.youngest = fs.youngest_rev(self.fsPtr, self.taskPool)
        if not revision:
            self.revision = self.youngest
        else:
            self.revision = revision

        self.cache = {}
        if cachePath is not None:
            try:
                f = _open(cachePath, "r")
                try:
                    (uuid, version, cache) = pickle.load(f)
                finally:
                    f.close()
                if (uuid == self.uuid) and (version == _metadata_.version):
                    self.cache = cache
            except:
                pass

    def storeCache(self):
        if self.cachePath is None:
            return
        f = _open(self.cachePath, "w")
        try:
            try:
                pickle.dump((self.uuid, _metadata_.version, self.cache), f, pickle.HIGHEST_PROTOCOL)
            finally:
                f.close()
        except:
            os.remove(self.cachePath)
            raise

    def getRepositoryData(self):
        return Struct(
            youngest=self.youngest,
            uuid=self.uuid
        )
        
    @taskPoolUser
    def getRevisionData(self, revision):
        try:
            data = self.cache[revision]
        except KeyError:
            root = fs.revision_root(self.fsPtr, revision, self.taskPool)
            editor = repos.ChangeCollector(self.fsPtr, root, self.taskPool)
            (ePtr, eBaton) = delta.make_editor(editor, self.taskPool)
            repos.replay(root, ePtr, eBaton, self.taskPool)

            author = self.getRevisionProp(revision, core.SVN_PROP_REVISION_AUTHOR) or ""
            dateStamp = self.getRevisionProp(revision, core.SVN_PROP_REVISION_DATE)
            date = int(core.secs_from_timestr(dateStamp, self.taskPool))
            log = self.getRevisionProp(revision, core.SVN_PROP_REVISION_LOG) or ""

            # TODO: Use this when repos.ChangeCollector is fixed
#            rootProps = editor.get_root_props()
#            author = str(rootProps["svn:author"])
#            date = int(core.secs_from_timestr(str(rootProps["svn:date"]), self.taskPool))
#            log = str(rootProps["svn:log"])

            changes = {}
            for (path, change) in editor.get_changes().iteritems():
                basePath = change.base_path
                if change.item_kind == core.svn_node_dir:
                    path += "/"
                    if basePath:
                        basePath += "/"
                changedStr = [" ", " ", " "]
                if not change.path:
                    action = 'deleted'
                    changedStr[0] = "D"
                elif change.added:
                    changedStr[0] = "A"
                    if change.base_path and change.base_rev != core.SWIG_SVN_INVALID_REVNUM:
                        action = "copied"
                        changedStr[2] = "+"
                    else:
                        action = "added"
                else:
                    action = "modified"
                    changedStr[0] = "M"
                if change.prop_changes:
                    changedStr[1] = "M"
                changedStr = "".join(changedStr)
                changes[path] = Struct(
                    path=path,
                    action=action,
                    propChanged=change.prop_changes,
                    textChanged=change.text_changed,
                    changedStr=changedStr,
                    basePath=basePath,
                    baseRevision=change.base_rev
                )
            
            data = Struct(
                number=revision,
                author=author,
                date=date,
                log=log,
                changes=changes
            )
            self.cache[revision] = data
        return data
        
    def getRevisionProp(self, revision, property):
        return fs.revision_prop(self.fsPtr, revision, property, self.taskPool)

    def clearTaskPool(self):
        core.svn_pool_clear(self.taskPool)


class SvnFeedTemplate(Template):
    """SVN feed template"""
    def __init__(self, template, source, match, entries, maxDays, maxDepth):
        content = readFile(template)
        super(SvnFeedTemplate, self).__init__(content)
        self.source = source
        self.match = match
        self.entries = entries
        self.maxDays = maxDays
        self.maxDepth = maxDepth

    def foreach_main(self, locals):
        """Generate feed."""
        locals["repository"] = self.source.getRepositoryData()
        return self.apply(locals)

    def foreach_revision(self, locals):
        """Generate feed entries."""
        if self.maxDays is not None:
            minDate = time.time() - self.maxDays * 24 * 3600
        else:
            minDate = 0
        if self.maxDepth is not None:
            minRevision = max(0, self.source.revision - self.maxDepth)
        else:
            minRevision = 0
        entries = []
        count = 0
        for rev in range(self.source.revision, minRevision, -1):
            if (self.entries is not None) and (count >= self.entries):
                break
            data = self.source.getRevisionData(rev)
            if data.date < minDate:
                break
            for each in data.changes.iterkeys():
                if self.match.matches(each):
                    break
            else:
                continue
            self.revision = data
            locals["revision"] = data
            locals["revisionCount"] = count
            entries.append(self.apply(locals))
            del self.revision
            count += 1
        return "".join(entries)

    def foreach_change(self, locals):
        """Generate changes."""
        changes = []
        count = 0
        for each in sorted(self.revision.changes.iterkeys()):
            data = self.revision.changes[each]
            locals["change"] = data
            locals["changeCount"] = count
            changes.append(self.apply(locals))
            count += 1
        return "".join(changes)
        

class Feed:
    """Feed generator"""
    def __init__(self, template, destination, entries=None, maxDays=None, maxDepth=None):
        self.template = template
        self.destination = destination
        self.entries = entries
        self.maxDays = maxDays
        self.maxDepth = maxDepth

    def toString(self, source, section):
        template = SvnFeedTemplate(self.template, source, section.match,
            self.entries, self.maxDays, self.maxDepth)
        return template.generate(section)
    
    def generate(self, source, section):
        out = open(self.destination, "w")
        try:
            out.write(self.toString(source, section))
        finally:
            out.close()

    def flush(self, source, section):
        pass


class SvnRevisionTemplate(Template):
    """SVN revision template"""
    def __init__(self, template, source):
        content = readFile(template)
        super(SvnRevisionTemplate, self).__init__(content)
        self.source = source

    def foreach_main(self, locals):
        """Generate revision."""
        locals["repository"] = self.source.getRepositoryData()
        self.revision = self.source.getRevisionData(self.source.revision)
        locals["revision"] = self.revision
        return self.apply(locals)

    def foreach_change(self, locals):
        """Generate changes."""
        changes = []
        count = 0
        for each in sorted(self.revision.changes.iterkeys()):
            data = self.revision.changes[each]
            locals["change"] = data
            locals["changeCount"] = count
            changes.append(self.apply(locals))
            count += 1
        return "".join(changes)
        

class Mail:
    """Mail generator"""
    store = {}
    
    def __init__(self, template, mailer, mailFrom, recipients):
        self.template = template
        self.mailer = mailer
        self.mailFrom = mailFrom
        self.recipients = set(each for each in re.split("\s*(?:,|\s)\s*", recipients) if each)

    def storeKey(self):
        return (self.template, self.mailer, self.mailFrom)

    def toString(self, source, locals):
        template = SvnRevisionTemplate(self.template, source)
        return template.generate(locals)
    
    def generate(self, source, section):
        # Just collect mail addresses for matching sections
        data = source.getRevisionData(source.revision)
        for each in data.changes.iterkeys():
            if section.match.matches(each):
                storeData = self.store.setdefault(self.storeKey(), ([], set()))
                storeData[0].append(section.name)
                storeData[1].update(self.recipients)
                return
        
    def flush(self, source, section):
        # Generate message and send once to all
        if self.storeKey() not in self.store:
            return
        (sections, recipients) = self.store.pop(self.storeKey())
        sections.sort()
        recipients = sorted(recipients)
        locals = ChainDict(section, __names__=sections)
        message = self.toString(source, locals)

        server = smtplib.SMTP(self.mailer)
        try:
            server.sendmail(self.mailFrom, recipients, message)
        finally:
            server.quit()
        

class Option:
    """Configuration file option descriptor"""
    def __init__(self, name, type):
        self.name = name
        self.type = type

    def __get__(self, instance, owner):
        if instance is None:
            return self
        try:
            return getattr(instance, "_" + self.name)
        except AttributeError:
            evaluator = Evaluator(instance, 100)
            value = self.type(evaluator.substitute(instance[self.name]), instance)
            setattr(instance, "_" + self.name, value)
            return value


class RegexList:
    """List of regular expressions"""
    def __init__(self, lines, section):
        self.regexes = [re.compile(each) for each in lines.splitlines() if each]
        
    def matches(self, s):
        for each in self.regexes:
            if each.match(s):
                return True
        return False


def generatorList(lines, section):
    """Evaluate a generator description list to a list of generators."""
    locals = ChainDict(section,
        Feed=Feed,
        Mail=Mail,
    )
    evaluator = Evaluator(locals, 100)
    return evaluator["[" + lines + "]"]


def loadModule(path, section):
    """Load a module by name and path and return its __dict__."""
    if not path:
        return {}
    name = os.path.basename(path).split(".")[0]
    dirName = os.path.dirname(path)
    searchPath = [dirName] + sys.path
    (f, pathName, description) = imp.find_module(name, searchPath)
    module = imp.load_module(name, f, pathName, description)
    return module.__dict__
    

class ConfigSection:
    """Proxy for one section of the configuration file"""
    generators = Option("generators", generatorList)
    match      = Option("match",      RegexList)
    module     = Option("module",     loadModule)
    
    def __init__(self, config, name):
        self.config = config
        self.name = name

    def generate(self, source):
        for each in self.generators:
            each.generate(source, self)
    
    def flush(self, source):
        for each in self.generators:
            each.flush(source, self)

    def __getitem__(self, key):
        try:
            return self.config.parser.get(self.name, key)
        except ConfigParser.NoOptionError:
            return self.module[key]

    def toString(self, source, genIndex):
        return self.generators[genIndex].toString(source, self)


class Parser(ConfigParser.RawConfigParser):
    """Case-sensitive configuration file parser"""
    def optionxform(self, s):
        return s


class Config:
    """Program configuration"""
    def __init__(self, configFile):
        self.sections = {}
        self.parser = Parser()
        self.parser.readfp(configFile)
        for each in self.parser.sections():
            self.sections[each] = ConfigSection(self, each)
            
    def cache(self):
        """Return DEFAULT option 'cache'."""
        try:
            return self.parser.get("DEFAULT", "cache")
        except ConfigParser.NoOptionError:
            return None


def indent(indent, s):
    result = indent + s.replace("\n", "\n" + indent)
    if s.endswith("\n"):
        return result[:-len(indent)]
    else:
        return result


def svnApplication(pool, path, revision, configPath):
    """Execute application."""
    status = 0
    
    # Read configuration file
    configFile = _open(configPath, "r")
    try:
        config = Config(configFile)
    finally:
        configFile.close()

    # Create data source
    source = SvnDataSource(pool, path, revision, config.cache())
    
    # Generate all sections
    for each in config.sections.itervalues():
        try:
            each.generate(source)
        except:
            status = 1
            sys.stderr.write("Error generating section '%s'\n" % each.name)
            sys.stderr.write(indent("  ", traceback.format_exc()))
    
    # Flush all sections
    for each in config.sections.itervalues():
        try:
            each.flush(source)
        except:
            status = 1
            sys.stderr.write("Error flushing section '%s'\n" % each.name)
            sys.stderr.write(indent("  ", traceback.format_exc()))
            
    # Store data source cache
    source.storeCache()

    return status

    
def main(argv):
    """Parse command line and execute application."""
    parser = OptionParser("%prog [options] config repos_path [revision]",
        version="%prog " + _metadata_.version,
        description=_metadata_.longDescription[:_metadata_.longDescription.index("\n\n")])
    (options, args) = parser.parse_args()

    if not (2 <= len(args) <= 3):
        parser.error("incorrect number of arguments")
        
    config = args[0]
    repository = args[1]
    if len(args) == 3:
        revision = int(args[2])
    else:
        revision = None

    try:
        return core.run_app(svnApplication, repository, revision, config)
    except:
        sys.stderr.write("Error while running application\n")
        sys.stderr.write(indent("  ", traceback.format_exc()))
        return 1
    

if __name__ == "__main__":
    sys.exit(main(sys.argv))
