#! /usr/bin/env python

## Copyright 2000-2003 by LivingLogic AG, Bayreuth, Germany.
## Copyright 2000-2003 by Walter Drwald
##
## All Rights Reserved
##
## Permission to use, copy, modify, and distribute this software and its documentation
## for any purpose and without fee is hereby granted, provided that the above copyright
## notice appears in all copies and that both that copyright notice and this permission
## notice appear in supporting documentation, and that the name of LivingLogic AG or
## the author not be used in advertising or publicity pertaining to distribution of the
## software without specific, written prior permission.
##
## LIVINGLOGIC AG AND THE AUTHOR DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
## INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
## LIVINGLOGIC AG OR THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
## DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
## IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR
## IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

"""
<par><module>sql</module> is a module that contains three classes
<class>Insert</class>, <class>Update</class> and <class>Delete</class>,
that simplify generating &sql; statements.</par>

<par>Creating an Insert statement simply works like this:</par>
<prog>
query = sql.Insert(
	"TABLE",
	{"FIELD1": "'Value1'", "FIELD2": 42, "FIELD3": None}
)
</prog>
<par><lit>print str(query)</lit> will give you (but of course without the line feeds):</par>
<prog>
INSERT INTO TABLE
	(FIELD1, FIELD2, FIELD3)
	VALUES('''Value1''', 42, NULL)
</prog>

<par>Executing the query can be done via:</par>
<prog>
<rep>cursor</rep>.execute(str(query))
</prog>
<par>or via:</par>
<prog>
query.execute(<rep>cursor</rep>)
</prog>

<par>Creating an Update statement works like this:</par>
<prog>
query = sql.Update(
	"TABLE",
	{"FIELD1": "'Value1'"},
	{"FIELD2": 42, "FIELD3": None}
)
</prog>

<par>The query generated from this object is:</par>
<prog>
UPDATE TABLE
	SET FIELD1='''Value1'''
	WHERE FIELD2=42 AND FIELD3 IS NULL
</prog>

<par><lit>WHERE</lit> clauses that are more involved than that are currently not possible,
and probably never will.</par>
"""

__version__ = tuple(map(int, "$Revision: 1.28 $"[11:-2].split(".")))
# $Source: /data/cvsroot/LivingLogic/Python/sql/sql.py,v $

try:
	from mx import DateTime
	mxdttype = type(DateTime.now())
except ImportError:
	mxdttype = None

import datetime

class Command:
	"""
	<par>encapsulates an &sql; command and provides a
	bunch of services for derived classes.</par>
	"""

	def _formatValue(self, value):
		if value is None:
			return "NULL"
		elif isinstance(value, str):
			return "'%s'" % value.replace("'", "''")
		elif isinstance(value, unicode):
			return ("'%s'" % value.replace("'", "''")).encode("latin1")
		elif isinstance(value, (int, long, float)):
			return str(value)
		elif mxdttype is not None and isinstance(value, mxdttype):
			return "'%s'" % value.Format("%Y-%m-%d %H:%M:%S")
		elif isinstance(value, datetime.datetime):
			return "'%s'" % value.strftime("%Y-%m-%d %H:%M:%S")
		else:
			raise ValueError("unrecognised type %s for database field" % value)

	def _formatField(self, name, value, format=0):
		"""
		format == 0: setting
		format == 1: testing
		format == 2: inserting
		"""
		if value is None:
			if format==0:
				return name + "=NULL"
			elif format==1:
				return name + " IS NULL"
			else:
				return "NULL"
		else:
			if format==0 or format==1:
				return name + "=" + self._formatValue(value)
			else:
				return self._formatValue(value)

	def _formatFields(self, fields, format=0):
		v = []
		for (fieldname, fieldvalue) in fields.iteritems():
			v.append(self._formatField(fieldname, fieldvalue, format))
		if format==0:
			return ",".join(v)
		elif format==1:
			return " AND ".join(v)
		else:
			return ",".join(v)

	def execute(self, cursor):
		return cursor.execute(str(self))

class Insert(Command):
	"""
	<par>An &sql; insert command. Convert the <class>Insert</class>
	instance to the query string with <function>str</function>.</par>
	"""

	def __init__(self, table, set):
		"""
		<par>Create a new <class>Insert</class> instance.
		Arguments are <arg>table</arg>, the table name,
		and <arg>set</arg>, the dictionary with the field names
		as keys and the field values as values. <lit>None</lit> will
		be treated as <lit>NULL</lit>.</par>
		"""
		self.table = table
		self.set = set

	def __str__(self):
		v = []
		v.append("INSERT INTO ")
		v.append(self.table)
		v.append(" (")
		vv = []
		for field in self.set:
			vv.append(field)
		v.append(",".join(vv))
		v.append(") VALUES (")
		v.append(self._formatFields(self.set, 2))
		v.append(")")
		return "".join(v)

class Update(Command):
	"""
	<par>An SQL update command. Convert the <class>Update</class>
	object to the query string with <function>str</function>.</par>
	"""
	def __init__(self, table, set, where):
		"""
		<par>Create a new <class>Insert</class> instance.
		Arguments are <arg>table</arg>, the table name,
		<arg>set</arg>, the dictionary with the field names
		as keys and the field values as values and
		<arg>where</arg>, a dictionary that maps field names
		to required field values. <lit>None</lit> will be treated as
		<lit>NULL</lit> for both <arg>set</arg> and <arg>where</arg>.</par>
		"""
		self.table = table
		self.set = set
		self.where = where

	def __str__(self):
		v = []
		v.append("UPDATE " + self.table + " SET ")
		v.append(self._formatFields(self.set, 0))
		if len(self.where):
			v.append(" WHERE ")
			v.append(self._formatFields(self.where, 1))
		return "".join(v)

class Delete(Command):
	"""
	<par>An &sql; delete command. Convert the <class>Delete</class>
	object to the query string with <function>str</function>.</par>
	"""
	def __init__(self, table, where):
		"""
		<par>Create a new <class>Delete</class> instance.
		Arguments are <arg>table</arg>, the table name and
		<arg>where</arg>, a dictionary that maps field names
		to required field values. <lit>None</lit> will be treated as
		<lit>NULL</lit> for both <arg>set</arg> and <arg>where</arg>.</par>
		"""
		self.table = table
		self.where = where

	def __str__(self):
		v = []
		v.append("DELETE FROM " + self.table)
		if len(self.where):
			v.append(" WHERE ")
			v.append(self._formatFields(self.where, 1))
		return "".join(v)
