"""
    sphinx.ext.autodoc.typehints
    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Generating content for autodoc using typehints

    :copyright: Copyright 2007-2020 by the Sphinx team, see AUTHORS.
    :license: BSD, see LICENSE for details.
"""

import re
from collections import OrderedDict
from typing import Any, Dict, Iterable
from typing import cast

from docutils import nodes
from docutils.nodes import Element

from sphinx import addnodes
from sphinx.application import Sphinx
from sphinx.config import ENUM
from sphinx.util import inspect, typing


def config_inited(app, config):
    if config.autodoc_typehints == 'description':
        # HACK: override this to make autodoc suppressing typehints in signatures
        config.autodoc_typehints = 'none'

        # preserve user settings
        app._autodoc_typehints_description = True
    else:
        app._autodoc_typehints_description = False


def record_typehints(app: Sphinx, objtype: str, name: str, obj: Any,
                     options: Dict, args: str, retann: str) -> None:
    """Record type hints to env object."""
    try:
        if callable(obj):
            annotations = app.env.temp_data.setdefault('annotations', {})
            annotation = annotations.setdefault(name, OrderedDict())
            sig = inspect.signature(obj)
            for param in sig.parameters.values():
                if param.annotation is not param.empty:
                    annotation[param.name] = typing.stringify(param.annotation)
            if sig.return_annotation is not sig.empty:
                annotation['return'] = typing.stringify(sig.return_annotation)
    except TypeError:
        pass


def merge_typehints(app: Sphinx, domain: str, objtype: str, contentnode: Element) -> None:
    if domain != 'py':
        return
    if app._autodoc_typehints_description is False:  # type: ignore
        return

    signature = cast(addnodes.desc_signature, contentnode.parent[0])
    if signature['module']:
        fullname = '.'.join([signature['module'], signature['fullname']])
    else:
        fullname = signature['fullname']
    annotations = app.env.temp_data.get('annotations', {})
    if annotations.get(fullname, {}):
        field_lists = [n for n in contentnode if isinstance(n, nodes.field_list)]
        if field_lists == []:
            field_list = insert_field_list(contentnode)
            field_lists.append(field_list)

        for field_list in field_lists:
            modify_field_list(field_list, annotations[fullname])


def insert_field_list(node: Element) -> nodes.field_list:
    field_list = nodes.field_list()
    desc = [n for n in node if isinstance(n, addnodes.desc)]
    if desc:
        # insert just before sub object descriptions (ex. methods, nested classes, etc.)
        index = node.index(desc[0])
        node.insert(index - 1, [field_list])
    else:
        node += field_list

    return field_list


def modify_field_list(node: nodes.field_list, annotations: Dict[str, str]) -> None:
    arguments = {}  # type: Dict[str, Dict[str, bool]]
    fields = cast(Iterable[nodes.field], node)
    for field in fields:
        field_name = field[0].astext()
        parts = re.split(' +', field_name)
        if parts[0] == 'param':
            if len(parts) == 2:
                # :param xxx:
                arg = arguments.setdefault(parts[1], {})
                arg['param'] = True
            elif len(parts) > 2:
                # :param xxx yyy:
                name = ' '.join(parts[2:])
                arg = arguments.setdefault(name, {})
                arg['param'] = True
                arg['type'] = True
        elif parts[0] == 'type':
            name = ' '.join(parts[1:])
            arg = arguments.setdefault(name, {})
            arg['type'] = True
        elif parts[0] == 'rtype':
            arguments['return'] = {'type': True}

    for name, annotation in annotations.items():
        if name == 'return':
            continue

        arg = arguments.get(name, {})
        field = nodes.field()
        if arg.get('param') and arg.get('type'):
            # both param and type are already filled manually
            continue
        elif arg.get('param'):
            # only param: fill type field
            field += nodes.field_name('', 'type ' + name)
            field += nodes.field_body('', nodes.paragraph('', annotation))
        elif arg.get('type'):
            # only type: It's odd...
            field += nodes.field_name('', 'param ' + name)
            field += nodes.field_body('', nodes.paragraph('', ''))
        else:
            # both param and type are not found
            field += nodes.field_name('', 'param ' + annotation + ' ' + name)
            field += nodes.field_body('', nodes.paragraph('', ''))

        node += field

    if 'return' in annotations and 'return' not in arguments:
        field = nodes.field()
        field += nodes.field_name('', 'rtype')
        field += nodes.field_body('', nodes.paragraph('', annotation))
        node += field


def setup(app):
    app.setup_extension('sphinx.ext.autodoc')
    app.config.values['autodoc_typehints'] = ('signature', True,
                                              ENUM("signature", "description", "none"))
    app.connect('config-inited', config_inited)
    app.connect('autodoc-process-signature', record_typehints)
    app.connect('object-description-transform', merge_typehints)
