# BSD 3-Clause License; see https://github.com/scikit-hep/uproot4/blob/master/LICENSE

"""
Defines versionless models of ``TRef`` and ``TRefArray``.
"""

from __future__ import absolute_import

import struct

try:
    from collections.abc import Sequence
except ImportError:
    from collections import Sequence

import numpy

import uproot4.model


_tref_format1 = struct.Struct(">xxIxxxxxx")


class Model_TRef(uproot4.model.Model):
    """
    A versionless :py:class:`~uproot4.model.Model` for ``TRef``.

    This model does not deserialize all fields, only the reference number.
    """

    @property
    def ref(self):
        """
        The reference number as an integer.
        """
        return self._ref

    def read_numbytes_version(self, chunk, cursor, context):
        pass

    def read_members(self, chunk, cursor, context, file):
        if self.is_memberwise:
            raise NotImplementedError(
                """memberwise serialization of {0}
in file {1}""".format(
                    type(self).__name__, self.file.file_path
                )
            )
        self._ref = cursor.field(chunk, _tref_format1, context)

    def __repr__(self):
        return "<TRef {0}>".format(self._ref)

    @classmethod
    def strided_interpretation(
        cls, file, header=False, tobject_header=True, original=None
    ):
        members = []
        members.append(("@pidf", numpy.dtype(">u2")))
        members.append(("ref", numpy.dtype(">u4")))
        members.append(("@other1", numpy.dtype(">u2")))
        members.append(("@other2", numpy.dtype(">u4")))

        return uproot4.interpretation.objects.AsStridedObjects(
            cls, members, original=original
        )

    @classmethod
    def awkward_form(cls, file, index_format="i64", header=False, tobject_header=True):
        import awkward1

        contents = {}
        if tobject_header:
            contents["@pidf"] = uproot4._util.awkward_form(
                numpy.dtype("u2"), file, index_format, header, tobject_header
            )
            contents["ref"] = uproot4._util.awkward_form(
                numpy.dtype("u4"), file, index_format, header, tobject_header
            )
            contents["@other1"] = uproot4._util.awkward_form(
                numpy.dtype("u2"), file, index_format, header, tobject_header
            )
            contents["@other2"] = uproot4._util.awkward_form(
                numpy.dtype("u4"), file, index_format, header, tobject_header
            )
        return awkward1.forms.RecordForm(contents, parameters={"__record__": "TRef"})


_trefarray_format1 = struct.Struct(">i")
_trefarray_dtype = numpy.dtype(">i4")


class Model_TRefArray(uproot4.model.Model, Sequence):
    """
    A versionless :py:class:`~uproot4.model.Model` for ``TRefArray``.

    This also satisfies Python's abstract ``Sequence`` protocol.
    """

    @property
    def refs(self):
        """
        The reference number as a ``numpy.ndarray`` of ``dtype(">i4")``.
        """
        return self._data

    @property
    def nbytes(self):
        """
        The number of bytes in :py:attr:`~uproot4.models.TRef.TRefArray.refs`.
        """
        return self._data.nbytes

    @property
    def name(self):
        """
        The name of this TRefArray.
        """
        return self._members["fName"]

    def read_members(self, chunk, cursor, context, file):
        if self.is_memberwise:
            raise NotImplementedError(
                """memberwise serialization of {0}
in file {1}""".format(
                    type(self).__name__, self.file.file_path
                )
            )
        cursor.skip(10)
        self._members["fName"] = cursor.string(chunk, context)
        self._members["fSize"] = cursor.field(chunk, _trefarray_format1, context)
        cursor.skip(6)
        self._data = cursor.array(
            chunk, self._members["fSize"], _trefarray_dtype, context
        )

    def __getitem__(self, where):
        return self._data[where]

    def __len__(self):
        return len(self._data)

    def __repr__(self):
        if self.class_version is None:
            version = ""
        else:
            version = " (version {0})".format(self.class_version)
        return "<{0}{1} {2} at 0x{3:012x}>".format(
            self.classname,
            version,
            numpy.array2string(
                self._data,
                max_line_width=numpy.inf,
                separator=", ",
                formatter={"float": lambda x: "%g" % x},
                threshold=6,
            ),
            id(self),
        )

    @classmethod
    def awkward_form(cls, file, index_format="i64", header=False, tobject_header=True):
        import awkward1

        contents = {}

        contents["fName"] = uproot4.containers.AsString(
            False, typename="TString"
        ).awkward_form(file, index_format, header, tobject_header)
        contents["fSize"] = uproot4._util.awkward_form(
            numpy.dtype("i4"), file, index_format, header, tobject_header
        )
        contents["refs"] = uproot4._util.awkward_form(
            numpy.dtype("i4"), file, index_format, header, tobject_header
        )
        return awkward1.forms.RecordForm(
            contents, parameters={"__record__": "TRefArray"}
        )


uproot4.classes["TRef"] = Model_TRef
uproot4.classes["TRefArray"] = Model_TRefArray
