import os
import io
import warnings
try:
    from StringIO import StringIO
except ImportError:
    from io import StringIO
from math import e

import numpy as np
from numpy.testing import assert_array_equal, assert_allclose, \
    assert_array_almost_equal

from pyne.utils import VnVWarning
warnings.simplefilter("ignore", VnVWarning)

from pyne.endf import Library
from pyne.utils import endftod
from pyne.rxdata import DoubleSpinDict
from pyne.xs.data_source import ENDFDataSource
from pyne import nucname

import nose
from nose.tools import assert_equal

str_library = StringIO(
""" $Rev:: 532      $  $Date:: 2011-12-05#$                             1 0  0
 1.002000+3 1.996800+0          1          0          0          0 128 1451    1
 0.000000+0 0.000000+0          0          0          0          6 128 1451    2
 1.000000+0 1.500000+8          1          0         10          7 128 1451    3
 0.000000+0 0.000000+0          0          0        211         17 128 1451    4
  1-H -  2 LANL       EVAL-FEB97 WEBDubois,Q.W.Alle,H.N.Casnowck   128 1451    5
 CH97,CH99            DIST-DEC06                       20111222    128 1451    6
----ENDF/B-VII.1      MATERIAL  128                                128 1451    7
-----INCIDENT NEUTRON DATA                                         128 1451    8
------ENDF-6 FORMAT                                                128 1451    9
                                                                   128 1451   10
 ****************************************************************  128 1451   11
                                                                   128 1451   12
 Lorem ipsum dolor sit amet, consectetur adipiscing elit.          128 1451   13
 Integer nec odio. Praesent libero. Sed cursus ante dapibus        128 1451   14
 diam. Sed nisi. Nulla quis sem at nibh elementum imperdiet.       128 1451   15
 Duis sagittis ipsum. Praesent mauris. Fusce nec tellus sed        128 1451   16
 augue semper porta.                                               128 1451   17
                                                                   128 1451   18
 ************************ C O N T E N T S ************************ 128 1451   19
                                1        451         28          0 128 1451   20
                                1        452          1          0 128 1451   21
                                2        151        107          0 128 1451   22
                                3          1          1          0 128 1451   23
                                3          2          1          0 128 1451   24
                                3          3          1          0 128 1451   25
                                3         16          1          0 128 1451   26
                                3        102          1          0 128 1451   27
                                4          2          1          0 128 1451   28
 0.000000+0 0.000000+0          0          0          0          0 128 1  099999
 1.111111+1                                                        128 1452    1
                                                                   128 1  099999
 0.000000+0 0.000000+0          0          0          0          0 128 0  0    0
 1.876876+3 1.776576+0          0          0          1          0 128 2151    1
 1.002000+3 1.000000+0          0          0          5          0 128 2151    2
 2.700000+3 2.000000+5          1          1          1          2 128 2151    3
 0.000000+0 0.000000+0          0          0          4         10 128 2151    4
          2          3          5          2          7          1 128 2151    5
         10          6                                             128 2151    6
 7.364350+7-2.862931+9-1.654883-4 5.067101+5 1.848039+3-1.523885+0 128 2151    7
-4.444773-3 1.292684+4 7.128604-3-4.362816+7 9.971208+5 6.697571-7 128 2151    8
-6.877209-5-8.102489+8 2.740343+8 1.809683+1-2.561444-7 1.803708+3 128 2151    9
-4.434634-2-4.496024+2                                             128 2151   10
 0.500000+0 1.000000-1          0          0          2          0 128 2151   11
 1.000000+0 1.100000-1 2.000000+0          1         12          2 128 2151   12
 2.000000+5 1.000000+0 1.000000+1 2.000000+0 1.100000+0 3.100000+0 128 2151   13
 3.000000+6 2.000000+0 2.000000+1 3.000000+0 1.200000+0 3.200000+0 128 2151   14
 1.100000+0 1.110000-1 2.500000+0          1         12          2 128 2151   15
 3.100000+5 3.000000+0 1.100000+1 2.100000+0 2.100000+0 4.100000+0 128 2151   16
 4.100000+6 4.000000+0 2.100000+1 3.100000+0 2.200000+0 4.200000+0 128 2151   17
 2.300000+3 3.000000+5          1          1          0          1 128 2151   18
 5.000000-1 1.000000-1          0          0          1          0 128 2151   19
 1.000000+0 1.100000-1 1.000000+0          1         12          2 128 2151   20
 3.500000+5 1.000000+0 1.000000+1 2.000000+0 1.100000+0 3.100000+0 128 2151   21
 4.500000+6 2.000000+0 2.000000+1 3.000000+0 1.200000+0 3.200000+0 128 2151   22
 2.197178+3 1.781069+6          1          3          1          1 128 2151   23
 0.000000+0 0.000000+0          0          0          2          7 128 2151   24
          3          4          7          1                       128 2151   25
-3.274193-7 1.413926+0-1.852205-1 6.591655+5-2.190334-1 4.766215-5 128 2151   26
 2.555015-4 1.596638+5-4.687759-2 4.664150+3 4.857577-1 3.441778+6 128 2151   27
-1.891328-3 5.048539-8                                             128 2151   28
 2.500000+0 3.699209+3          1          0          1          0 128 2151   29
 2.399684-9 2.626370+5 2.000000+0          0          6          1 128 2151   30
 4.127773+3-3.956950-7 3.739684-5-3.872199+7 2.259559+5-3.096948-8 128 2151   31
 1.787886-1 1.634787+6          1          4          1          1 128 2151   32
 0.000000+0 0.000000+0          0          0          3         10 128 2151   33
          3          2          6          5         10          1 128 2151   34
 1.420823-7 3.476614+5 3.719237+9-1.281719-9 3.055872+7 2.448907+3 128 2151   35
 2.257804+4 3.481293-2-3.238532-6-1.090767+9-3.328793-1 4.956508-9 128 2151   36
 1.874769+9-3.980820+5-9.256614+9 5.585164-5-4.219474+1-3.512732+6 128 2151   37
 2.095667+4 3.870564+1                                             128 2151   38
 3.500000+0 2.145387-2          0          0          2          0 128 2151   39
 1.523605+6 0.000000+0          7          0         18          3 128 2151   40
 9.143204-3 1.601509+1-3.312654-7-3.460776+8-3.947879-5-1.646877-5 128 2151   41
 1.009363-5-1.861342-7-1.613360+7-7.549728+5-3.064120+9-2.961641+0 128 2151   42
-4.390193-5 2.303605+0-4.630212+5-3.237353+1-4.037885+4-3.231304+0 128 2151   43
 0.000000+0 0.000000+0 4.000000+0          0          2          0 128 2151   44
 1.000000+0 0.000000+0          0          0         24          2 128 2151   45
 1.029199+3-4.611259+5-4.348653+0 5.072331-8-2.722664+8 4.238546+4 128 2151   46
 2.881576+6-4.663196+5-4.754578-7-4.578510+1 1.040734+3 2.171429-7 128 2151   47
 4.485210+0-1.075272+3 5.228296-9 4.636330+4 9.996406-7-2.860366+4 128 2151   48
-2.751527+7 3.763842-8-3.333810-5-1.250613+0-4.636583-6 8.651338+1 128 2151   49
 2.000000+0 0.000000+0          0          0         12          1 128 2151   50
-1.703457-4 2.722273+9 2.950158-1 4.767819-1 4.506927-3 2.960170+3 128 2151   51
-4.006882+8-2.188708-4 3.262820+4 2.372937+6 4.251123-6 4.998481-6 128 2151   52
 0.000000+0 0.000000+0 5.000000+0          0          1          0 128 2151   53
 3.000000+0 0.000000+0          0          0         12          1 128 2151   54
-3.211014+1-2.011165+3 4.178337+5 1.640997+2 1.122313-5 1.537114-2 128 2151   55
 4.634918-8-3.884155-4 2.384144-9-3.745465-7-1.646941-2-2.894650-8 128 2151   56
 1.685738+5 1.659888-5          1          7          0          0 128 2151   57
 0.000000+0 0.000000+0          1          3          2          1 128 2151   58
 0.000000+0 0.000000+0          2          0         24          4 128 2151   59
 7.916271+6-3.532347-6 4.469905+7-2.134022+4-3.500000+0-3.500000+0 128 2151   60
 5.307428-7          0          1          7          1         -1 128 2151   61
 2.807643-8-4.478596+0 3.274758+3-2.760395+9 1.356440+3 3.447654+4 128 2151   62
 4.790839-8          1         -1        800          1         -1 128 2151   63
-4.000000+0          0          1          0         12          2 128 2151   64
 1.981937+3 9.740279-7-2.450194+5-1.304844+4 1.856158-9-1.218463-9 128 2151   65
-4.097837+1-2.765873-9-0.913351+1 1.591290+5-2.379063+0 2.066455-6 128 2151   66
 0.000000+0 0.000000+0          0          3         18          3 128 2151   67
 2.949030-1 1.156625+7 7.255199-6                                  128 2151   68
 4.453964+1 5.062864-5-1.110875-3                                  128 2151   69
 2.208407-7 9.942677-6-3.134503-8                                  128 2151   70
 0.000000+0 0.000000+0          0          0          1          1 128 2151   71
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 128 2151   72
 0.000000+0 0.000000+0          0          0          4          5 128 2151   73
          1          2          2          1          3          3 128 2151   74
          5          4                                             128 2151   75
 2.733413-7 4.281371-6-4.024400-6 3.630494+8-2.453233-4-1.087679+1 128 2151   76
-4.395957-3-3.776962+9 3.274197+6 3.799711-7                       128 2151   77
 0.000000+0 0.000000+0          0          0          2          6 128 2151   78
          4          1          6          2                       128 2151   79
-3.184906-9 1.563214-7-3.079936+9 4.383830+7 4.624237-9 1.878097+2 128 2151   80
-2.545276-9 4.802278-2-4.134103-3 5.039026+5-6.153657-9-1.927630-3 128 2151   81
 1.960831+3-1.053619+4          0          0          2          1 128 2151   82
 1.298772+5-2.834965+3 8.381641-6 3.338353+5-1.012675-7 0.000000+0 128 2151   83
 3.000000+0          0          1          1         12          2 128 2151   84
 0.951092+5 7.932944-5-3.716218-8 4.007761-3 1.277498-6 2.041832+6 128 2151   85
-7.375896+8-4.822942+4 4.491725+9 3.018430+8 2.238307-5-3.591395+9 128 2151   86
 0.000000+0 0.000000+0          0          3         18          3 128 2151   87
 5.088175-6-2.282938+0-4.236786-6                                  128 2151   88
 8.930267-9-3.115607+8-2.521300-4                                  128 2151   89
 3.978418+5 4.821547-6 3.110373-3                                  128 2151   90
 1.960831+3-1.053619+4          0          0          3          1 128 2151   91
 3.941056-6-0.524089+0-2.023965-8 0.000000+0 0.000000+0 0.000000+0 128 2151   92
 0.000000+0 0.000000+0          0          0          0          1 128 2151   93
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 128 2151   94
 0.000000+0 0.000000+0          0          0          0          1 128 2151   95
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 128 2151   96
 0.000000+0 0.000000+0          0          0          1          1 128 2151   97
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 128 2151   98
 0.000000+0 0.000000+0          0          0          1          3 128 2151   99
          3          2                                             128 2151  100
 4.016335+2 2.076736-3 4.668090-5 9.776415+2-3.940740+2-2.296483+8 128 2151  101
 0.000000+0 0.000000+0          0          0          3         10 128 2151  102
          3          1          6          2         10          3 128 2151  103
-4.803282+6-1.114539-5 9.465304+2-1.436769-9 7.889727+2 4.824983+9 128 2151  104
 4.020763+6 2.308443-6-4.188441-2 1.778263+8-3.408683+7 2.845463+7 128 2151  105
 3.371147+1 2.054714+3-2.746606-3-9.635977-6-1.387257-2 7.042637+0 128 2151  106
 6.917628+9-2.912896-7                                             128 2151  107
 0.000000+0 0.000000+0          0          0          0          0 128 2  099999
 0.000000+0 0.000000+0          0          0          0          0 128 0  0    0
 1.989875+3 1.998644+0          0          0          0          0 128 3  1    1
 0.000000+0 0.000000+0          0          0          0          0 128 3  099999
 1.012000+3 1.231241+0          0          0          0          0 128 3  2    1
 0.000000+0 0.000000+0          0          0          0          0 128 3  099999
 1.123123+3 1.123113+0          0          0          0          0 128 3  3    1
 0.000000+0 0.000000+0          0          0          0          0 128 3  099999
 7.984533+3 1.125535+0          1          0          0          0 128 3 16    1
 0.000000+0 0.000000+0          0          0          0          0 128 3  099999
 1.948720+3 1.145243+0          0          0          0          0 128 3102    1
 0.000000+0 0.000000+0          0          0          0          0 128 3  099999
 0.000000+0 0.000000+0          0          0          0          0 128 0  0    0
 1.983232+3 1.934732+0          0          2          0          0 128 4  2    1
 0.000000+0 0.000000+0          0          0          0          0 128 4  099999
 0.000000+0 0.000000+0          0          0          0          0 128 0  0    0
 0.000000+0 0.000000+0          0          0          0          0   0 0  0    0
 0.000000+0 0.000000+0          0          0          0          0  -1 0  0    0
 $Rev:: 513      $  $Date:: 2006-12-05#$                             1 0  0    0
 1.003100+3 2.098312+0          1          0          0          1 131 1451    1
 0.564324+0 1.123121+0          0          0          0          6 131 1451    2
 1.905018+0 2.401998+7          1          0         10          7 131 1451    3
 0.109590+0 0.123112+0          0          0         90          8 131 1451    4
  1-H -  3 LANL       EVAL-NOV01 W.P.Pike                          131 1451    5
PRC  42, 438 (1990)   DIST-DEC06                       20111222    131 1451    6
----ENDF/B-VII.1      MATERIAL  131                                131 1451    7
-----INCIDENT NEUTRON DATA                                         131 1451    8
------ENDF-6 FORMAT                                                131 1451    9
                                                                   131 1451   10
*** Mauris massa. Vestibulum lacinia arcu eget nulla. Class *****  131 1451   11
aptent taciti sociosqu ad litora torquent per conubia nostra, per  131 1451   12
inceptos himenaeos. Curabitur sodales ligula in libero. Sed        131 1451   13
*****************************************************************  131 1451   14
                                                                   131 1451   15
                                1        451         21          1 131 1451   16
                                2        151         22          1 131 1451   17
                                3          1          1          1 131 1451   18
                                3          2          1          1 131 1451   19
                                3         16          1          1 131 1451   20
                                4          2          1          1 131 1451   21
 0.000000+0 0.000000+0          0          0          0          0 131 1  099999
 0.000000+0 0.000000+0          0          0          0          0 131 0  0    0
 1.003100+3 2.898897+0          0          0          1          0 131 2151    1
 1.003100+3 1.000000+0          0          0          2          0 131 2151    2
 1.700000+3 1.000000+5          2          1          0          0 131 2151    3
 3.500000+0 5.101200-1          0          0          2          0 131 2151    4
 1.357310+2 0.000000+0          0          0         18          3 131 2151    5
 1.800000+3          6 1.000000+0 2.078520-1 1.000000-2 0.000000+0 131 2151    6
 2.100000+3          7 2.000000+0 6.088000-1 2.000000-2 0.000000+0 131 2151    7
 3.100000+3          8 3.000000+0 3.120000-1 3.000000-2 0.000000+0 131 2151    8
 1.357310+2 0.000000+0          1          0         18          3 131 2151    9
 1.810000+3          9 4.000000+0 4.489400-1 8.000000-2 0.000000+0 131 2151   10
 2.110000+3         10 5.000000+0 8.497500-1 7.000000-2 0.000000+0 131 2151   11
 3.110000+3         11 6.000000+0 9.524900-1 6.000000-2 0.000000+0 131 2151   12
 1.100000+5 2.000000+7          2          1          0          0 131 2151   13
 2.000000+0 5.101200-1          0          0          2          0 131 2151   14
 1.357310+2 0.000000+0          2          0         18          3 131 2151   15
 1.801000+3          0 1.100000+0 3.078520-1 1.000000-2 0.000000+0 131 2151   16
 2.101000+3          1 2.100000+0 7.088000-1 2.000000-2 0.000000+0 131 2151   17
 3.101000+3          2 3.100000+0 2.120000-1 3.000000-2 0.000000+0 131 2151   18
 1.357310+2 0.000000+0          1          0         18          3 131 2151   19
 1.812000+3          3 4.100000+0 5.489400-1 8.000000-2 0.000000+0 131 2151   20
 2.112000+3          4 5.100000+0 9.497500-1 7.000000-2 0.000000+0 131 2151   21
 3.112000+3          5 6.100000+0 0.524900-1 6.000000-2 0.000000+0 131 2151   22
-2.000000+0 0.000000+0          0          0          0          0 131 2  099999
-1.000000+0 0.000000+0          0          0          0          0 131 0  0    0
 1.304918+3 2.582082+0          0          0          0          0 131 3  1    1
 1.000000+0 0.000000+0          0          0          0          0 131 3  099999
 2.000000+0 2.912396+0          0          0          0          0 131 3  2    1
 0.000000+0 0.000000+0          0          0          0          0 131 3  099999
 1.001900+3 2.988596+0          0          0          0          0 131 3 16    1
 0.000000+0 0.000000+0          0          0          0          0 131 3  099999
 0.000000+0 0.000000+0          0          0          0          0 131 0  0    0
 1.001900+3 2.989116+0          0          1          0          0 131 4  2    1
 0.000000+0 0.000000+0          0          0          0          0 131 4  099999
 0.000000+0 0.000000+0          0          0          0          0 131 0  0    0
 0.000000+0 0.000000+0          0          0          0          0   0 0  0    0
 0.000000+0 0.000000+0          0          0          0          0  -1 0  0    0
 $Rev:: 514      $  $Date:: 2006-12-05#$                             1 0  0    0
 4.000000+3 6.192500+0          1          0          0          1 419 1451    1
 0.123000+0 0.063200+0          0          0          0          6 419 1451    2
 1.000230+0 8.100130+6          1          0         10          7 419 1451    3
 0.001200+0 0.001600+0          0          0         47         10 419 1451    4
  4-Be-  7 LANL       EVAL-JUN04 E.N.Trix                          419 1451    5
                      DIST-DEC06                       20111222    419 1451    6
----ENDF/B-VII.1      MATERIAL  419                                419 1451    7
-----INCIDENT NEUTRON DATA                                         419 1451    8
------ENDF-6 FORMAT                                                419 1451    9
                                                                   419 1451   10
 ***************************************************************** 419 1451   11
dignissim lacinia nunc. Curabitur tortor. Pellentesque nibh.       419 1451   12
Aenean quam. In scelerisque sem at dolor. Maecenas mattis. Sed     419 1451   13
convallis tristique sem.                                           419 1451   14
 ***************************************************************** 419 1451   15
                                1        451         22          1 419 1451   16
                                2        151         57          1 419 1451   17
                                3          2         12          1 419 1451   18
                                3        600         11          1 419 1451   19
                                3        650          1          1 419 1451   20
                                3        800          1          1 419 1451   21
                                4          2          2          1 419 1451   22
 0.000000+0 0.000000+0          0          0          0          0 419 1  099999
 0.000000+0 0.000000+0          0          0          0          0 419 0  0    0
 4.000000+3 4.400259+9          0          0          2          0 419 2151    1
 4.019200+3 2.776505-3          0          1          1          0 419 2151    2
 3.724629+1 3.538532+6          2          1          1          1 419 2151    3
 0.000000+0 0.000000+0          0          0          2         10 419 2151    4
          2          3          10         6                       419 2151    5
 3.871526-8-9.663371-2 2.386010+0-4.265862-4-4.243279-3-4.277051-7 419 2151    6
-1.264228+8-4.377257+3 4.850497-7-2.027212+0-4.647138+6-2.023107-1 419 2151    7
 4.505022+0-8.430411+0 2.510353-7-0.305028-1 3.299939+3 2.493202+0 419 2151    8
 3.763615-9-4.211922+8                                             419 2151    9
 1.500000+0 1.865419+9          0          0          5          1 419 2151   10
-2.625837-2-1.457011+4-3.267593-4 3.959905+3-3.440459-2            419 2151   11
-1.263108-1 0.000000+0 4.000000+0          0          1          0 419 2151   12
 0.000000+0 0.000000+0 4.000000+0          3         11          0 419 2151   13
-3.276558-9 3.000000+0 1.499031-3-4.455678-9 2.910100-5 0.000000+0 419 2151   14
-4.284304+2-3.856415-8 9.065534-7-3.355711+6-1.847975+6            419 2151   15
 4.004000+3-4.242156+9          0          1          3          0 419 2151   16
-4.315613-5 2.891921+6          2          1          0          0 419 2151   17
 4.500000+0 3.751025-7          1          0          6          2 419 2151   18
-4.130223+6-3.793486+5 3.513877+2-6.072241-9-1.787864+3-1.162246+3 419 2151   19
-7.832899-5 0.000000+0 2.000000+0          0          2          0 419 2151   20
 0.000000+0 0.000000+0 2.000000+0          2         12          0 419 2151   21
 3.636791-1 3.000000+0 4.476397+9 8.789695+2 1.833746-9 0.000000+0 419 2151   22
-2.210171-1-2.561209+6 3.534863-9 4.970383+3-4.524614+0-3.986952-3 419 2151   23
 0.000000+0 0.000000+0 2.000000+0          1         12          0 419 2151   24
 1.979410-6-4.000000+0-4.264745-1 3.042416+0-4.427212+7 0.000000+0 419 2151   25
 2.163723-2 2.421080+5-9.370260+7-2.706321-7-3.037705+0 1.889285+1 419 2151   26
 4.648092-4 0.000000+0 3.000000+0          0          1          0 419 2151   27
 0.000000+0 0.000000+0 3.000000+0          3         12          0 419 2151   28
 2.804009-5 4.000000+0 3.181707+3 3.885315-9-3.382438+3 0.000000+0 419 2151   29
 2.376630+2 7.198625-2-5.887887-8-4.380016-5 1.747888-6-4.104291-9 419 2151   30
-3.639532-6-1.965467+3          2          1          0          0 419 2151   31
 3.500000+0 4.391826-7          1          0          7          1 419 2151   32
-2.723837-2-8.755303-2 2.245337-2-9.034520+2 2.252098+5 2.666587+2 419 2151   33
 3.747872-3                                                        419 2151   34
-2.368259-8 0.000000+0 4.000000+0          0          1          0 419 2151   35
 0.000000+0 0.000000+0 4.000000+0          4         13          0 419 2151   36
-9.824193-5 5.000000+0 4.676826-4-4.336597+0-9.045122+2 0.000000+0 419 2151   37
 3.699655-9-3.919000+5 8.467144-3-3.737007+9-5.750577+7-9.588021+8 419 2151   38
-3.280571+7                                                        419 2151   39
 1.754489+2 1.593056+1          2          2          0          0 419 2151   40
 5.000000-1 4.145900-5          1          0          2          0 419 2151   41
-1.187125-7 0.000000+0 5.000000+0          0          2          0 419 2151   42
 6.000000+0 0.000000+0          3          0         12          1 419 2151   43
 0.000000+0 0.000000+0 4.003466-3-2.709252-8 0.000000+0 5.075078+7 419 2151   44
 6.469007-9-2.062519+7-7.116815-5 1.562553-4 2.341246-9 4.397092+0 419 2151   45
 7.000000+0 0.000000+0          3          0         12          1 419 2151   46
 0.000000+0 0.000000+0 3.215655-1 2.913648-8 0.000000+0-4.040338+3 419 2151   47
-6.162365+2 4.264049+5 2.088009+8 2.756941+1 6.978406-1-1.334121-9 419 2151   48
-5.702860+9 0.000000+0 6.000000+0          0          2          0 419 2151   49
 8.000000+0 0.000000+0          2          0         18          2 419 2151   50
 0.000000+0 0.000000+0 6.019205-4 4.315267+8 0.000000+0 4.241206+5 419 2151   51
 4.199534+1 1.769801-1 1.378667-8 9.070620+2-4.490878-9 2.721648+1 419 2151   52
 4.859555+8-2.330988-6-1.872580+5-2.816019-1-2.982221+6-1.048786+1 419 2151   53
 9.000000+0 0.000000+0          2          0         18          2 419 2151   54
 0.000000+0 0.000000+0-4.253833-1-2.269388+0 0.000000+0 4.732644-4 419 2151   55
-5.873521-3-4.808214+9 5.089619+5 4.836683+0 2.772702-3-4.865151-8 419 2151   56
-2.659480-9 1.044275+8-1.393749+2-4.189996-6-9.596467-4 3.942829+9 419 2151   57
 0.000000+0 0.000000+0          0          0          0          0 419 2  099999
 0.000000+0 0.000000+0          0          0          0          0 419 0  0    0
 4.284918+3 6.292347+0          0          0          0          0 419 3  2    1
 4.019200+3-4.245658-8          0-4.651348+3          7         20 419 3  2    2
          6          4          9          2         12          1 419 3  2    3
         13          5         15          3         17          4 419 3  2    4
         20          1                                             419 3  2    5
-3.802784+7-1.460835-4-3.862664+2 5.712822-1 1.482715+9 5.922770+8 419 3  2    6
-3.993688-3-1.907553-7 2.609470-5-1.634402-8 4.313437+0 4.055371+6 419 3  2    7
-1.975543+7 1.511556+5 3.234085-6-3.281996-1 2.573704-2 7.695823+4 419 3  2    8
-2.984883+0 3.567168-6 8.964894-3-2.114121+2 3.497215-6-4.671332-5 419 3  2    9
-3.631599+9 3.855194-7-4.361452-7-1.209872+2-3.014999-2 1.579273+8 419 3  2   10
 3.374580-6 6.257083-8 4.598797+7-2.047819-3 4.406774+3-3.498203+2 419 3  2   11
 2.140116+7 3.120507-6-1.741408+8 1.783210-6                       419 3  2   12
 0.000000+0 0.000000+0          0          0          0          0 419 3  099999
 4.004000+3 6.287192+0          0          0          0          0 419 3600    1
 3.863437-5-7.373532-7          0 8.675483-1          5         20 419 3600    2
          4          1          8          2         12          3 419 3600    3
         16          4         20          5                       419 3600    4
          1         15          4         12         10         -7 419 3600    5
         20         10          1         15          4         12 419 3600    6
         10         -7         20         10          1          1 419 3600    7
 2.718282+0          3 2.718282+0          3 7.389056+0          0 419 3600    8
          1          1          2 2.718282+0          2 7.389056+0 419 3600    9
          3 2.718281+0          1 3.678794-1          2 1.471517+0 419 3600   10
          2 1.847264+0          3 8.210062-1                       419 3600   11
 0.000000+0 0.000000+0          0          0          0          0 419 3  099999
 4.192847+3 6.874398+0          0          0          0          0 419 3650    1
 0.000000+0 0.000000+0          0          0          0          0 419 3  099999
 4.897498+3 6.287322+0          0          0          0          0 419 3800    1
 0.000000+0 0.000000+0          0          0          0          0 419 3  099999
 0.000000+0 0.000000+0          0          0          0          0 419 0  0    0
 4.898421+3 6.768123+0          0          1          0          0 419 4  2    1
 2.123124+6 8.123142-6 2.123212+6 8.231231-6-2.231211+6 8.123421-6 419 4  2    2
 0.000000+0 0.000000+0          0          0          0          0 419 4  099999
 0.000000+0 0.000000+0          0          0          0          0 419 0  0    0
 0.000000+0 0.000000+0          0          0          0          0   0 0  0    0
 0.000000+0 0.000000+0          0          0          0          0  -1 0  0    0
""")


library = Library(str_library)
nuc1002, nuc10031, nuc40000 = nucname.id(1002), nucname.id(10031), nucname.id(40000)
library._read_res(nuc1002)
library._read_res(nuc10031)
library._read_res(nuc40000)
nuc40040 = nucname.id(40040)

def array_from_ENDF(fh):
    "Convert a chunk of ENDF, stripped of metadata, into a numpy array."
    return np.genfromtxt(fh,
                         dtype='float',
                         delimiter=11,
                         converters={0: endftod, 1: endftod, 2: endftod,
                                     3: endftod, 4: endftod, 5: endftod})


def test_endftod():
    from pyne._utils import endftod
    obs = [endftod(" 3.28559+12"),
           endftod(" 2.328559+4"),
           endftod(" 3.28559-12"),
           endftod(" 2.328559-2"),
           endftod("-3.28559+12"),
           endftod("-2.328559+2"),
           endftod("-3.28559-12"),
           endftod("-2.328559-2"),
           endftod("        121"),
           endftod("       -121")]
    exp = [ 3.28559e+12,
            2.328559e+4,
            3.28559e-12,
            2.328559e-2,
           -3.28559e+12,
           -2.328559e+2,
           -3.28559e-12,
           -2.328559e-2,
                  121.0,
                 -121.0]
    obs = np.array(obs)
    exp = np.array(exp)
    assert_allclose(obs, exp, rtol = 1e-8)


def test_get():
    obs = library.get_rx(nuc40000, 4, 2)
    exp = [4.898421e+3,6.768123e+0,0,1,0,0,2.123124e+6,8.123142e-6,2.123212e+6,
           8.231231e-6,-2.231211e+6,8.123421e-6]
    try:
        badkey = library.get_rx(111, 1, 1)
        assert(False)
    except ValueError:
        assert(True)
    assert_array_equal(exp, obs)


def test_unresolved_resonances_a():
    # Case A (ENDF Manual p.70)
    obs = library.structure[nuc10031]['data'][nuc10031]['unresolved']
    obs_LIST = obs[1][2][2,2]

    exp = array_from_ENDF(io.BytesIO(
        b""" 1.801000+3          0 1.100000+0 3.078520-1 1.000000-2 0.000000+0
 2.101000+3          1 2.100000+0 7.088000-1 2.000000-2 0.000000+0
 3.101000+3          2 3.100000+0 2.120000-1 3.000000-2 0.000000+0"""))
    exp_LIST = dict(zip(('D','AJ','AMUN','GN0','GG'), exp.transpose()))

    for key in exp_LIST:
        assert_array_equal(exp_LIST[key], obs_LIST[key])

def test_unresolved_resonances_b():
    # Case B (ENDF Manual p. 70)
    obs = library.structure[nuc40000]['data'][nuc40040]['unresolved']
    # For the spin=4.5, L=3, J=4 section in the first isotope
    obs_1 = obs[0][2][4.5,3,4]
    exp_1_a = array_from_ENDF(io.BytesIO(
        b""" 0.000000+0 0.000000+0 3.000000+0          3         12          0
 2.804009-5 4.000000+0 3.181707+3 3.885315-9-3.382438+3 0.000000+0
 2.376630+2 7.198625-2-5.887887-8-4.380016-5 1.747888-6-4.104291-9"""))
    exp_1 = dict(zip((0,0,'L','MUF','NE+6',0,'D','AJ','AMUN','GN0','GG'),
                     exp_1_a[:2].flat))
    exp_1['GF'] = exp_1_a[2]
    del exp_1[0]

    for key in exp_1:
        assert_array_equal(obs_1[key], exp_1[key])
    # For the spin=3.5, L=4, J=5 section in the second isotope
    obs_2 = obs[1][2][3.5,4,5]
    exp_2_a = array_from_ENDF(io.BytesIO(
        b""" 0.000000+0 0.000000+0 4.000000+0          4         13          0
-9.824193-5 5.000000+0 4.676826-4-4.336597+0-9.045122+2 0.000000+0
 3.699655-9-3.919000+5 8.467144-3-3.737007+9-5.750577+7-9.588021+8
-3.280571+7                                                       """))
    exp_2 = dict(zip((0,0,'L','MUF','NE+6',0,'D','AJ','AMUN','GN0','GG'),
                     exp_2_a[:2].flat))
    num_e = exp_2['NE+6']-6
    exp_2['GF'] = exp_2_a[2:].flat[:num_e]
    del exp_2[0]
    for key in exp_2:
        assert_array_equal(obs_2[key], exp_2[key])

    # Check the energy values.
    obs_ES = obs[1][2]['ES']
    exp_ES_a = array_from_ENDF(io.BytesIO(
        b"""-2.723837-2-8.755303-2 2.245337-2-9.034520+2 2.252098+5 2.666587+2
 3.747872-3                                                       """))
    exp_ES = exp_ES_a.flat[:num_e]
    assert_array_equal(obs_ES, exp_ES)


def test_unresolved_resonances_c():
    # Case C (ENDF Manual p. 70)
    obs = library.structure[nuc40000]['data'][nuc40040]['unresolved'][2][2][0.5,6,9]

    exp_a = array_from_ENDF(io.BytesIO(
        b""" 9.000000+0 0.000000+0          2          0         18          2
 0.000000+0 0.000000+0-4.253833-1-2.269388+0 0.000000+0 4.732644-4
-5.873521-3-4.808214+9 5.089619+5 4.836683+0 2.772702-3-4.865151-8
-2.659480-9 1.044275+8-1.393749+2-4.189996-6-9.596467-4 3.942829+9"""))
    exp = dict(zip(('ES','D','GX','GN0','GG','GF'),
                   exp_a[2:].transpose()))
    exp.update(dict(zip(('AJ',0,'INT',0,'6*NE+6','NE',0,0,'AMUX','AMUN','AMUG',
                         'AMUF'),
                        exp_a[:2].flat)))
    exp['AWRI'] = -5.702860e+9
    for key in obs:
        assert_array_equal(obs[key], exp[key])


def test_DoubleSpinDict():
    subject = DoubleSpinDict({(3.499999999998, 2, 1):{'A':'a', 'B':'b'},
                              (2.000000000012, 3, 4):{'C':'c', 'D':'d'}})
    subject.update({(3.500000000011,8,9):{'E':'e', 'F':'f'}})

    obs = subject[(3.48, 8, 9)]
    exp = {'E':'e', 'F':'f'}
    assert_equal(exp, obs)

def test_resolved_breitwigner():
    """ The section looks like this:
         EL         EH        LRU        LRF        NRO       NAPS 419 2151    3
 0.000000+0 0.000000+0          0          0         NR         NP 419 2151    4
        SPI         AP          0          0        NLS          0 419 2151    5
       AWRI         QX          L        LRX      6*NRS        NRS 419 2151    6
         ER         AJ         GT         GN         GG         GF 419 2151    7
         ER         AJ         GT         GN         GG         GF 419 2151    8
       AWRI         QX          L        LRX      6*NRS        NRS 419 2151    6
         ER         AJ         GT         GN         GG         GF 419 2151    7
         ER         AJ         GT         GN         GG         GF 419 2151    8
"""

    data = library.structure[nuc1002]['data'][nuc1002]['resolved']
    # Check to see if NRO is reading from the right place.
    # NRO = 0 case
    range_nro_0 = data[2]
    assert_equal(range_nro_0[-1]['NRO'], 0)
    # NRO = 1 case
    # Check to see if NAPS is reading from the right place
    assert_equal(range_nro_0[-1]['NAPS'], 1)
    # Check to see if SPI, NLS are reading from the right place
    assert_equal(range_nro_0[-1]['SPI'], 0.5)
    assert_equal(range_nro_0[-1]['NLS'], 1)
    # Check to see if the data is alright...
    expected = {'ER': [350000.0, 4500000.0], 'AJ': [1.0, 2.0], 'GT': [10., 20.],
                'GN': [2., 3.], 'GG': [1.1, 1.2], 'GF': [3.1,3.2]}

    for key in range_nro_0[2][0.5,1]:
        assert_allclose(range_nro_0[2][0.5,1][key],expected[key], rtol = 1e-8)

def test_resolved_reichmoore():
    """The section looks like this:
        EL         EH        LRU        LRF        NRO       NAPS 419 2151    3
0.000000+0 0.000000+0          0          0         NR         NP 419 2151    4
       SPI         AP        LAD          0        NLS       NLSC 419 2151    5
      AWRI        APL          L          0      6*NRS        NRS 419 2151    6
        ER         AJ         GN         GG        GFA        GFB 419 2151    7
"""

    subsection = library.structure[nuc1002]['data'][nuc1002]['resolved'][1]
    assert_array_equal(subsection[2]['int']['intpoints'], [3,7])
    assert_array_equal(subsection[2]['int']['intschemes'], [4,1])

    obs_data = subsection[2][2.5,2]
    exp_data = {'ER': 4.127773e+3, 'AJ': -3.956950e-7, 'GN': 3.739684e-5,
                'GG': -3.872199e+7, 'GFA': 2.259559e+5, 'GFB': -3.096948e-8}
    for key in subsection[2][2.5,2]:
        assert_allclose(obs_data[key], exp_data[key], rtol = 1e-8)

def test_resolved_adleradler():
    """The section looks like this:
[MAT, 2,151/  0.0,  0.0,    0,    0,    NR,   NP/ points / AP(E) ] TAB1
[MAT, 2,151/  SPI,   AP,    0,    0,   NLS,    0 ] CONT
[MAT, 2,151/ AWRI,  0.0,   LI,    0,  6*NX,   NX
              AT1,  AT2,  AT3,  AT4,   BT1,  BT2,
              AF1, -----------------------,  BF2,
              AC1, -----------------------,  BC2 ] LIST
[MAT, 2,151/  0.0,  0.0,    L,    0,   NJS,    0] CONT(l)
[MAT, 2,151/   AJ,  0.0,    0,    0,12*NLJ,  NLJ/
             DET1, DWT1, GRT1, GIT1,  DEF1, DWF1,
             GRF1, GIF1, DEC1, DWC1,  GRC1, GIC1,
             DET2, DWT2, GIC2,---- --------------
             DET3,-------------------------------
             ------------------------------------
             ------------------------, GICN LJ ] LIST
"""
    subsection = library.structure[nuc1002]['data'][nuc1002]['resolved'][0]

    obs_LIST = subsection[2][3.5,5,3]
    obs_bg = subsection[2]['bg']

    # Test to see if the LIST records read in right
    exp_LIST = {'DET': -3.211014e+1, 'DWT': -2.011165e+3,'GRT': 4.178337e+5,
                'GIT': 1.640997e+2, 'DEF': 1.122313e-5, 'DWF': 1.537114e-2,
                'GRF': 4.634918e-8, 'GIF': -3.884155e-4, 'DEC': 2.384144e-9,
                'DWC': -3.745465e-7, 'GRC': -1.646941e-2, 'GIC': -2.894650e-8}


    for key in exp_LIST:
        assert_allclose(exp_LIST[key],obs_LIST[key], rtol = 1e-8)

    exp_bg_string = io.BytesIO(
        b""" 9.143204-3 1.601509+1-3.312654-7-3.460776+8-3.947879-5-1.646877-5
 1.009363-5-1.861342-7-1.613360+7-7.549728+5-3.064120+9-2.961641+0
-4.390193-5 2.303605+0-4.630212+5-3.237353+1-4.037885+4-3.231304+0""")
    exp_bg = dict(zip(('A1','A2','A3','A4','B1','B2'),
                      array_from_ENDF(exp_bg_string).transpose()))

    for key in exp_bg:
        assert_array_equal(exp_bg[key],obs_bg[key])

def test_resolved_r_matrix_kbk_kps():

    obs_3 = library.structure[nuc1002]['data'][nuc1002]['resolved'][-1][2][3.0]
    obs_4 = library.structure[nuc1002]['data'][nuc1002]['resolved'][-1][2][-4.0]

    exp_3 = array_from_ENDF(io.BytesIO(
        b""" 1.960831+3-1.053619+4          0          0          3          1
 3.941056-6-0.524089+0-2.023965-8 0.000000+0 0.000000+0 0.000000+0
 0.000000+0 0.000000+0          0          0          0          1
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0
 0.000000+0 0.000000+0          0          0          0          1
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0
 0.000000+0 0.000000+0          0          0          1          1
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0
 0.000000+0 0.000000+0          0          0          1          3
          3          2                                            
 4.016335+2 2.076736-3 4.668090-5 9.776415+2-3.940740+2-2.296483+8
 0.000000+0 0.000000+0          0          0          3         10
          3          1          6          2         10          3
-4.803282+6-1.114539-5 9.465304+2-1.436769-9 7.889727+2 4.824983+9
 4.020763+6 2.308443-6-4.188441-2 1.778263+8-3.408683+7 2.845463+7
 3.371147+1 2.054714+3-2.746606-3-9.635977-6-1.387257-2 7.042637+0
 6.917628+9-2.912896-7                                            
"""))


    ch0_obs = obs_3['ch0']

    # lbk = 3: [MAT,2,151/ ED, EU, 0, 0, LBK, 1/ R0, SO, GA, 0.0, 0.0, 0.0]LIST
    ch0_exp = dict(zip(('ED','EU',0,0,'LBK',0,'R0','SO','GA'),
                       (exp_3[:2].flat)))
    ch0_exp['LPS'] = 0.0
    del ch0_exp[0]

    ch1_obs = obs_3['ch1']

    # lbk = 0: no data
    # lps = 1: [MAT,2,151/ 0.0, 0.0,   0,   0, LPS,   1/
    #                      0.0, 0.0, 0.0, 0.0, 0.0, 0.0]LIST
    #          [MAT,2,151/ 0.0,0.0,0,0, NR, NP/points/PSR(E)]TAB1
    #                      0.0,0.0,0,0, NR, NP/points/PSI(E)]TAB1

    ch1_exp = {'PSI':{'intpoints': [3.,6.,10.],
                      'intschemes': [1.,2.,3.],
                      'e_int': exp_3[13:17].flat[:-4:2],
                      'PSI': exp_3[13:17].flat[1:-4:2]},
               'PSR':{'intpoints': 3.,
                      'intschemes': 2.,
                      'e_int': exp_3[10].flat[::2],
                      'PSR': exp_3[10].flat[1::2]},
               'LBK': 0.,
               'LPS': 1.}
    for key in ch1_exp:
        if isinstance(ch1_exp[key], dict):
            for intkey in ch1_exp[key]:
                assert_array_equal(ch1_exp[key][intkey], ch1_obs[key][intkey])
        else:
            assert_array_equal(ch1_obs[key], ch1_exp[key])
    assert_equal(ch0_obs, ch0_exp)

    lbk1_obs = obs_4['ch0']
    lbk2_obs = obs_4['ch1']
    lbk_exp = array_from_ENDF(io.BytesIO(
        b""" 0.000000+0 0.000000+0          0          0          1          1
 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0 0.000000+0
 0.000000+0 0.000000+0          0          0          4          5
          1          2          2          1          3          3
          5          4                                            
 0.000000+0 0.000000+0          0          0          2          6
          4          1          6          2                      
 1.960831+3-1.053619+4          0          0          2          1
 1.298772+5-2.834965+3 8.381641-6 3.338353+5-1.012675-7 0.000000+0
"""))

    # lbk = 1: [MAT,2,151/ 0.0, 0.0,   0,   0, LBK,   1/
    #                      0.0, 0.0, 0.0, 0.0, 0.0, 0.0]LIST
    #          [MAT,2,151/ 0.0,0.0,0,0, NR, NP/points/RBR(E)]TAB1
    #                      0.0,0.0,0,0, NR, NP/points/RBI(E)]TAB1
    lbk1_exp = {'LBK': 1,
                'RBR':{'intpoints':[1,2,3,5], 'intschemes':[2,1,3,4]},
                'RBI':{'intpoints':[4,6], 'intschemes':[1,2]}}

    for key in lbk1_exp:
        if isinstance(lbk1_exp[key], dict):
            for intkey in lbk1_exp[key]:
                assert_array_equal(lbk1_exp[key][intkey], lbk1_obs[key][intkey])
        else:
            assert_array_equal(lbk1_obs[key], lbk1_exp[key])

    # lbk = 2: [MAT,2,151/ ED, EU,  0,  0, LBK,   1/
    #                      R0, R1, R2, S0,  S1, 0.0]LIST

    lbk2_exp = dict(zip(('ED','EU',0,0,'LBK',0,'R0','R1','R2','S0','S1'),
                        lbk_exp[7:9].flat))
    lbk2_exp['LBK'] = 2
    del lbk2_exp[0]

    assert_equal(lbk2_exp, lbk2_obs)


def test_resolved_r_matrix():
    pp_exp_a = array_from_ENDF(io.BytesIO(
        b"""1.685738+5 1.659888-5          1          7          0          0
 0.000000+0 0.000000+0          1          3          2          1
 0.000000+0 0.000000+0          2          0         24          4
 7.916271+6-3.532347-6 4.469905+7-2.134022+4-3.500000+0-3.500000+0
 5.307428-7          0          1          7          1         -1
 2.807643-8-4.478596+0 3.274758+3-2.760395+9 1.356440+3 3.447654+4
 4.790839-8          1         -1        800          1         -1"""))
    pp_obs = library.structure[nuc1002]['data'][nuc1002]['resolved'][-1][3]
    pp_exp = dict(zip(('MA','MB','ZA','ZB','IA','IB','Q','PNT','SHF','MT',
                       'PA','PB'),
                       pp_exp_a[3:7].reshape(2,12).transpose()))
    pp_exp.update(dict(zip((0,0,'IFG','KRM','NJS','KRL'),
                           pp_exp_a[1])))
    del pp_exp[0]

    ch_exp_a = array_from_ENDF(io.BytesIO(
        b"""-4.000000+0          0 1.914541-3-4.683290+5         12          2
 1.981937+3 9.740279-7-2.450194+5-1.304844+4 1.856158-9-1.218463-9
-4.097837+1-2.765873-9-0.913351+1 1.591290+5-2.379063+0 2.066455-6
 3.000000+0          0-2.924403-5-4.840218-1         12          2
 0.951092+5 7.932944-5-3.716218-8 4.007761-3 1.277498-6 2.041832+6
-7.375896+8-4.822942+4 4.491725+9 3.018430+8 2.238307-5-3.591395+9"""))
    ch_exp = {}
    ch_exp[3.0] = dict(zip(('IPP','L','SCH','BND','APE','APT'),
                         ch_exp_a[4:6].transpose()))
    ch_exp[-4.0] = dict(zip(('IPP','L','SCH','BND','APE','APT'),
                          ch_exp_a[1:3].transpose()))
    ch_obs = library.structure[nuc1002]['data'][nuc1002]['resolved'][-1][2]

    gam_4_a = array_from_ENDF(io.BytesIO(
        b""" 2.949030-1 1.156625+7 7.255199-6          0          0          0
 4.453964+1 5.062864-5-1.110875-3          0          0          0
 2.208407-7 9.942677-6-3.134503-8          0          0          0"""))
    gam_4_a = gam_4_a.transpose()
    gam_4_exp = {'ER': gam_4_a[0],
                 'GAM': gam_4_a[1:3].transpose()}
    ch_exp[-4.0].update(gam_4_exp)

    gam_3_a = array_from_ENDF(io.BytesIO(
        b""" 5.088175-6-2.282938+0-4.236786-6          0          0          0
 8.930267-9-3.115607+8-2.521300-4          0          0          0
 3.978418+5 4.821547-6 3.110373-3          0          0          0"""))
    gam_3_a = gam_3_a
    gam_3_a = gam_3_a.transpose()
    gam_3_exp = {'ER': gam_3_a[0],
                 'GAM': gam_3_a[1:3].transpose()}

    ch_exp[3.0].update(gam_3_exp)

    for key in pp_exp:
        assert_array_equal(pp_obs[key], pp_exp[key])
    for spin_group in ch_exp:
        spin_group_obs = ch_obs[spin_group]
        spin_group_exp = ch_exp[spin_group]
        for key in spin_group_exp:
            assert_array_equal(spin_group_obs[key], spin_group_exp[key])

def test_xs():
    # Read in the data
    nuc_i = nucname.id(40192)
    library._read_xs(nuc40000, 2, nuc_i)
    library._read_xs(nuc40000, 600, nuc40040)

    # Manually find where the data should be reading from and check if it is
    # consistent with what the program is doing.
    exp_2_str = io.BytesIO(
        b""" 4.284918+3 6.292347+0          0          0          0          0
 4.047593+5-4.245658-8          0-4.651348+3          7         20
          6          4          9          2         12          1
         13          5         15          3         17          4
         20          1                                            """)
    exp_2_a = array_from_ENDF(exp_2_str)
    exp_2 = dict(zip(('intpoints', 'intschemes'),
                     (exp_2_a[2:].flat[:14:2], exp_2_a[2:].flat[1:14:2])))
    obs_2 = library.structure[nuc40000]['data'][nuc_i]['xs'][2][0]

    exp_600_a = array_from_ENDF(io.BytesIO(
        b""" 4.193742+3 6.287192+0          0          0          0          0
 3.863437-5-7.373532-7          0 8.675483-1          5         20
          4          1          8          2         12          3
         16          4         20          5                     """))

    exp_600 = dict(zip(('intpoints', 'intschemes'),
                       (exp_600_a[2:].flat[:-2:2], exp_600_a[2:].flat[1:-1:2])))
    obs_600 = library.structure[nuc40000]['data'][nuc40040]['xs'][600][0]

    for key in exp_2:
        assert_array_equal(obs_2[key], exp_2[key])
        assert_array_equal(obs_600[key], exp_600[key])

    # Heck, why not check the flags too?
    obs_600_flags = library.structure[nuc40000]['data'][nuc40040]['xs'][600][1]
    exp_600_flags = dict(zip(('QM','QI',0,'LM','NR','NP'),
        exp_600_a[1]))#
    exp_600_flags.update({'ZA': 4.004e+3, 'AWR': 6.287192})
    del exp_600_flags[0]
    assert_equal(obs_600_flags, exp_600_flags)


def test_u235():
    try:
        assert(os.path.isfile('U235.txt'))
    except AssertionError:
        try:
            import urllib.request as urllib
        except ImportError:
            import urllib
        urllib.urlretrieve("http://t2.lanl.gov/nis/data/data/ENDFB-VII.1-neutron/U/235",
                    "U235.txt")
    from hashlib import md5
    with open("U235.txt", "rb") as f:
        obs_hash = md5(f.read()).hexdigest()
    exp_hash = "1b71da3769d8b1e675c3c579ba5cb2d3"
    try:
        assert_equal(obs_hash, exp_hash)
    except AssertionError:
        raise AssertionError("U235.txt hash check failed; please try redownloading the U235 data file.")

    u235 = Library('U235.txt')
    nuc = 922350000
    u235._read_res(nuc)
    u235._read_xs(nuc, 37)
    exp_a = array_from_ENDF(io.BytesIO
         (b""" 9.223500+4 2.330248+2          0          0          0          0
-1.788560+7-1.788560+7          0          0          1          6
          6          2                                            
 1.796240+7 5.05980-10 1.800000+7 3.810030-7 1.850000+7 8.441785-5
 1.900000+7 2.387410-4 1.950000+7 1.348763-3 2.000000+7 4.785594-3
"""))
    obs =  u235.structure[nuc]['data'][nuc]['xs'][37][0]
    exp = {'intpoints': 6, 'intschemes': 2,
           'e_int': exp_a[3:5].flat[::2],
           'xs': exp_a[3:5].flat[1::2]}

    for key in obs:
        assert_array_equal(obs[key], exp[key])


# Test ENDF Data Source

def test_int_hist():
    exp_Eint = np.array([1,4,10, 20])
    exp_xs = np.array([15, 12, -7, 10])
    obs = library.integrate_tab_range(1, exp_Eint, exp_xs)
    exp = (3*15 + 6*12+10*-7)/19.
    assert_allclose(exp, obs, rtol = 1e-12)

def test_int_hist_interpolation():
    exp_Eint = np.array([1,4,10, 20])
    exp_xs = np.array([15, 12, -7, 10])
    obs = library.integrate_tab_range(1, exp_Eint, exp_xs, low = 2, high = 15)
    exp = (2*15 + 6*12 + 5*-7)/13.
    assert_allclose(exp, obs, rtol = 1e-12)

def test_int_hist_only_interpolate_one_endpoint():
    endfds = ENDFDataSource(str_library)
    obs = endfds.integrate_dst_group((1, 5), np.array([2, 5, 8]),
                                     {2: 1, 5: 1, 8:1},
                                     np.array([0., 2, 4, 6, 8]),
                                     np.array([0., 1, 0, 0, 0]))
    exp = 0.5
    assert_equal(exp, obs)

def test_int_linlin():
    exp_Eint = np.array([1,4,10, 20])
    exp_xs = np.array([15, 12, -7, 10])
    obs = library.integrate_tab_range(2, exp_Eint, exp_xs)
    exp = (3*13.5 + 6*2.5 + 10 * 1.5)/19.
    assert_allclose(exp, obs, rtol = 1e-12)
    return exp

def test_int_linlin_interpolation():
    exp_Eint = np.array([1, 4, 10, 20.])
    exp_xs = np.array([15, 12, -7, 10.])
    obs = library.integrate_tab_range(2, exp_Eint, exp_xs, low = 2, high = 15)
    exp = (2*13 + 6*2.5 + 5*(-7+1.5)/2)/13.
    assert_allclose(exp, obs, rtol = 1e-12)

def test_int_linlin_interpolation_2():
    endfds = ENDFDataSource(str_library)
    obs = endfds.integrate_dst_group((1, 5), np.array([3, 5, 8]),
                                     {3: 2, 5: 2, 8:1},
                                     np.array([0., 2, 4, 6, 8]),
                                     np.array([0., 1, 0, 0, 0]))
    exp = (0.75 + 1) / 4
    assert_equal(exp, obs)

def test_int_linlin_only_interpolate_one_endpoint():
    exp_Eint = np.array([1, 4, 10, 20.])
    exp_xs = np.array([15, 12, -7, 10.])
    obs = library.integrate_tab_range(2, exp_Eint, exp_xs, low = 1, high = 15)
    exp = (3*13.5 + 6*2.5 + 5*(-7+1.5)/2)/14.
    assert_allclose(exp, obs, rtol = 1e-12)

def test_int_linlog():
    exp_Eint = np.array([1, e, e, e**2])
    exp_xs = np.array([1, 3, 3, 0])
    obs = library.integrate_tab_range(3, exp_Eint, exp_xs)
    exp = (e+1+3*e**2-6*e)/(e**2-1)
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_linlog_interpolation():
    exp_Eint = np.array([1, e**2, e**4, e**6])
    exp_xs = np.array([0, 2, 4, 6.])
    obs = library.integrate_tab_range(3, exp_Eint, exp_xs, low=e, high=e**5)
    exp = 4*e**5/(e**5-e)
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_linlog_only_interpolate_one_endpoint():
    exp_Eint = np.array([1, e**2, e**4, e**6])
    exp_xs = np.array([0, 2, 4, 6.])
    obs = library.integrate_tab_range(3, exp_Eint, exp_xs, low=1, high=e**5)
    exp = (1+4*e**5)/(e**5-1)
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_loglin():
    exp_Eint = np.array([1., 2., 2., 3.])
    exp_xs = np.array([1, e, e**2, e])
    obs = library.integrate_tab_range(4, exp_Eint, exp_xs)
    exp = (e**2-1)/2
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_loglin_interpolation():
    exp_Eint = np.array([0, 2, 4, 6])
    exp_xs = np.array([1, e**2, e**4, e**6])
    obs = library.integrate_tab_range(4, exp_Eint, exp_xs, low=1, high=5)
    exp = (e**5 - e)/(5-1)
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_loglin_only_interpolate_one_endpoint():
    exp_Eint = np.array([0, 2, 4, 6], dtype="float64")
    exp_xs = np.array([1, e**2, e**4, e**6])
    obs = library.integrate_tab_range(4, exp_Eint, exp_xs, low=2, high=5)
    exp = (e**5 - e**2)/(5-2)
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_loglog():
    exp_Eint = np.array([1., 2., 2., 3.])
    exp_xs = np.array([1/e, 4/e, (e**2)/4, (e**2)/9])
    obs = library.integrate_tab_range(5, exp_Eint, exp_xs)
    exp = (7/(3*e) + (e**2)/6)/2.
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_loglog_interpolation():
    # ln y = 2 ln x + 1
    # y = e * x ** 2
    # integral of y = e/3 * x**3
    exp_Eint = np.array([1, 3, 5, 7], dtype="float64")
    exp_xs = np.array([e, 9*e, 25*e, 49*e])
    obs = library.integrate_tab_range(5, exp_Eint, exp_xs, low = 2, high = 6)
    exp = e/3 * (6**3 - 2 **3) / (6-2)
    assert_allclose(exp, obs, rtol=1e-12)

def test_int_loglog_only_interpolate_one_endpoint():
    # ln y = 2 ln x + 1
    # y = e * x ** 2
    # integral of y = e/3 * x**3
    exp_Eint = np.array([1, 3, 5, 7], dtype="float64")
    exp_xs = np.array([e, 9*e, 25*e, 49*e])
    obs = library.integrate_tab_range(5, exp_Eint, exp_xs, low = 2, high = 5)
    exp = e/3 * (5**3 - 2 **3) / (5-2)
    assert_allclose(exp, obs, rtol=1e-12)

def test_discretize():
    from os.path import isfile
    try:
        import urllib.request as urllib
    except ImportError:
        import urllib
    
    if not isfile("Ni59.txt"):
        urllib.urlretrieve("http://t2.lanl.gov/nis/data/data/ENDFB-VII.1-neutron/Ni/59",
                    "Ni59.txt")

    endfds = ENDFDataSource("Ni59.txt")
    nonelastic_rx = endfds.reaction("Ni59", "nonelastic")
    nonelastic_rx['dst_group_struct'] = np.logspace(7, -5, 33)
    nonelastic_c = endfds.discretize("Ni59", "nonelastic")
    exp = [0.54334609294912528, 0.21206255570566626,
           0.079089998725708668, 0.039061531003500925, 0.056193960028285306,
           0.062581135526972767, 0.086088778452663009, 0.1519375415918513,
           0.015156525895127398, 0.18503957567677801, 0.0039443417078627837,
           0.082573739674287688, 17.523219940338304, 0.97176481236488554,
           0.60307330340022303, 0.71684581122716162, 0.99386518962022252,
           1.4726882603418707, 2.2391970686479672, 3.405589441800994,
           5.2453926977834398, 8.0731410528834182, 12.384026334168054,
           19.175694435799141, 29.334824378652982, 45.254982026071197,
           74.217617672501689, 162.26091389706099, 218.90153743636509,
           312.62178192130619, 590.40136068709603, 724.64216445611373]
    assert_array_almost_equal(nonelastic_c, exp)

if __name__ == "__main__":
    nose.runmodule()
