#include <Python.h>
#include <time.h> /* For localtime,mktime,gmtime and struct tm */
#include <sys/types.h> /* For time_t */
#include <stdio.h> /* For snprintf */

static PyObject* _PosixTimeZone_tzset(PyObject* self,PyObject* args) {
    if (!PyArg_ParseTuple(args,""))
        return NULL;
    Py_INCREF(Py_None);
    return Py_None;
    }

struct tm tm; /* Shared since mktime etc. are not thread safe anyway */
time_t ticks;
time_t ticks2;

void _setException(void) {
    /* This should never happen, unless mktime(3) is busted or
        we manage to pass dud values to these C routines 
    */
    char msg[2048];
    
    snprintf(msg,2047,
        "mktime cannot handle %04d/%02d/%02d %02d:%02d:%02d %s",
        tm.tm_year + 1900,tm.tm_mon,tm.tm_mday,
        tm.tm_hour,tm.tm_min,tm.tm_sec,
	getenv("TZ")
        );
    PyErr_SetString(PyExc_ValueError,msg);
}

/* In this macro, we add an hour if mktime failed - some implementation
    cannot represent the hour after a DST transition (eg. Mac OS X 10.2).
    However, the Python datetime library can so we just have to ensure
    that out tzinfo implementation returns the correct utcoffset, tzname
    and dst which will still be correct the next hour.
*/
#define MAKETICKS(tm,ticks) \
    tm->tm_isdst = -1; \
    if ((ticks = mktime(tm)) == -1) { \
	tm->tm_hour++; \
	if ((ticks = mktime(tm)) == -1) { \
	    tm->tm_hour--; \
	    _setException(); \
	    return NULL; \
	    } \
        } 

static PyObject* _PosixTimeZone_tzname(PyObject* self,PyObject* args) {
    int is_dst;
    if (!PyArg_ParseTuple(args,"iiiiii",
        &tm.tm_sec, &tm.tm_min, &tm.tm_hour,
        &tm.tm_mday, &tm.tm_mon, &tm.tm_year
        ))
        return NULL;
    MAKETICKS((&tm),ticks);
    is_dst = (tm.tm_isdst == 1);
    return Py_BuildValue("s",tzname[is_dst]);
}

static PyObject* _PosixTimeZone_utcoffset(PyObject* self,PyObject* args) {
    struct tm* gtm;
    if (!PyArg_ParseTuple(args,"iiiiii",
        &tm.tm_sec, &tm.tm_min, &tm.tm_hour,
        &tm.tm_mday, &tm.tm_mon, &tm.tm_year
        ))
        return NULL;

    MAKETICKS((&tm),ticks); /* ticks now contains utc epoch */
    
    /* Abuse mktime to determine our UTC offset, since we don't have a 
       timegm(3) function on all systems */
    gtm = gmtime(&ticks);

    /* Propogate is_dst or mktime will be 1hour out in DST time */
    gtm->tm_isdst = tm.tm_isdst;

    ticks2 = mktime(gtm);
    if (ticks2 == -1) {
	/* If we fail to get a result, try again tomorrow */
	tm.tm_mday++;
	MAKETICKS((&tm),ticks);
	gtm = gmtime(&ticks);
	gtm->tm_isdst = tm.tm_isdst;
	ticks2 = mktime(gtm);
	if (ticks2 == -1) {
	    tm.tm_mday--;
	    _setException(); 
	    return NULL;
	}
    }
    
    return Py_BuildValue("i",(int) ticks - ticks2);
}

static PyObject* _PosixTimeZone_dst(PyObject* self,PyObject* args) {
    if (!PyArg_ParseTuple(args,"iiiiii",
        &tm.tm_sec, &tm.tm_min, &tm.tm_hour,
        &tm.tm_mday, &tm.tm_mon, &tm.tm_year
        ))
        return NULL;
    MAKETICKS((&tm),ticks);
    return Py_BuildValue("i",localtime(&ticks)->tm_isdst);
}

static PyMethodDef _PosixTimeZoneMethods[] = {
    {"tzset",_PosixTimeZone_tzset,METH_VARARGS, 
        "Calls the tzset(3) kernel call to change the local timezone" },

    {"tzname",_PosixTimeZone_tzname,METH_VARARGS, 
        "Returns the local timezone name" },

    {"utcoffset",_PosixTimeZone_utcoffset,METH_VARARGS, 
        "Returns the local UTC offset" },

    {"dst",_PosixTimeZone_dst,METH_VARARGS, 
        "Returns true if given moment is in daylight savings time" },

    {NULL,NULL,0,NULL}
    };

void init_PosixTimeZone(void) {
    (void) Py_InitModule("_PosixTimeZone",_PosixTimeZoneMethods);
    }

