"""
This code wraps the vendored appdirs module to so the return values are
compatible for the current pip code base.

The intention is to rewrite current usages gradually, keeping the tests pass,
and eventually drop this after all usages are changed.
"""

import os
import sys
from typing import List

from pip._vendor import platformdirs as _appdirs


def user_cache_dir(appname: str) -> str:
    return _appdirs.user_cache_dir(appname, appauthor=False)


def user_config_dir(appname: str, roaming: bool = True) -> str:
    path = _appdirs.user_config_dir(appname, appauthor=False, roaming=roaming)
    if sys.platform == "darwin" and not os.path.isdir(path):
        path = os.path.expanduser("~/.config/")
        if appname:
            path = os.path.join(path, appname)
    return path


# for the discussion regarding site_config_dir locations
# see <https://github.com/pypa/pip/issues/1733>
def site_config_dirs(appname: str) -> List[str]:
    dirval = _appdirs.site_config_dir(appname, appauthor=False, multipath=True)
    if sys.platform == "darwin":
        # always look in /Library/Application Support/pip as well
        return dirval.split(os.pathsep) + ["/Library/Application Support/pip"]
    elif sys.platform == "win32":
        return [dirval]
    else:
        # always look in /etc directly as well
        return dirval.split(os.pathsep) + ["/etc"]
