"""Persistent objects

This module defines:

pobject -- the base class for all persistent classes
ptype   -- the type of all persistent classes (i.e.: the metaclass)
"""
import pprop

class ptype(type):
    """The metclass for persistent objects

    The purpose of ptype is to register newly defined classes with the
    database and to delegate object instantiation to the database
    """
    def __init__(cls, name, bases, attributes):
        """Initialize a new persistent class

        Arguments
           name       -- the name of the new class (a str)
           bases      -- the baseclasses for the new class (a tuple)
           attributes -- the attributes of the new class (a dict)

        Class initialization will perform the following operations
        - Add the attribute name of the persistent properties to the
          property objects
        - Create lists of persistent attributes, stored in
            cls.persistent_values (for pval objects)
            cls.persistent_refs   (for pref objects)
        - Register the class with the database, using
            cls.database.register_class(cls)
        """
        # Do not initialize the base class 
        if name == 'pobject':
            return

        cls.persistent_values = bases[0].persistent_values[:]
        cls.persistent_refs = bases[0].persistent_refs[:]
            
        # Add the attribute names to the properties,
        # Add the key for instance.__dict__ to the properties
        # Add the properties to their respective property lists
        for attr_name, attr_val in attributes.items():
            if isinstance(attr_val, pprop.pprop):
                attr_val.name = attr_name
                attr_val.key  = cls.__name__+'__'+attr_name
                if isinstance(attr_val, pprop.pval):
                    cls.persistent_values.append(attr_val)
                if isinstance(attr_val, pprop.pref):
                    cls.persistent_refs.append(attr_val)

        cls.database.register_class(cls)
                
    def __call__(cls, *args, **kwargs):
        """Create an instance of a persistent class

        cls(*args, **kwargs) --> ptype.__call__(cls, *args, **kwargs)

        Instance creation is delegated to the database using
          cls.database.get_instance(cls, *args, **kwargs)
        """
        return cls.database.get_instance(cls, *args, **kwargs)

    def all_subclasses(cls):
        """Recursively generate all subclasses.

        Yields cls and then recursively generates classes by calling
        all_subclasses() on each of the subclasses of cls
        """
        yield cls
        for subcls in cls.__subclasses__():
            for i in subcls.all_subclasses():
                yield i

    def make_new_instance(cls, *args, **kwargs):
        """Create a new instance for an object that has not yet been stored
        in the database

        Mimics type.__call__, but calls instance._persistence_init() is called
        between cls.__new__(*args, **kwargs) and instance.__init__(*args,
        **kwargs), to initialize the support initialize support for persistent
        attributes
        """
        instance = cls.__new__(cls, *args, **kwargs)
        instance.oid = None
        instance._persistence_init()
        instance.__init__(*args, **kwargs)
        return instance
    
    def make_old_instance(cls, oid):
        """Create the bare instance for an object whose state will be
        retrieved from the database"""
        
        instance = object.__new__(cls)
        instance.oid = oid
        instance._persistence_init()
        return instance

class pobject(object):
    """The baseclass for all persistent objects

    A persistent object should:
    - be a subclass of pobject
    - define an attribute database, referring to a database interface (q.v.)
    - define one or more persistent attributes

    The purpose of pobject is to set the __metaclass__ of persistent objects
    to ptype, and to provide common methods to all persistent objects
    """
    __metaclass__ = ptype

    persistent_values = []
    persistent_refs = []

    def _persistence_init(self):
        """Initialize support for object persistence

        Currently only sets the dirty bit to False
        """
        self._dirty = False

    def commit(self):
        """Commit the persistent object to the database

        First, recursively, commit all objects to the database that this object
        refers to, then commit the object itself.
        """
        for prop in self.persistent_refs:
            val = prop.__get__(self)
            if val is not None:
                val.commit()
        self.database.commit(self)

